// communication.cpp

#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <config.h>
#include "global.h"
#include "communication.h"
#include "options.h"

// FIFOBuffer is used internally to buffer incoming and outgoing data

class FIFOBuffer {
public:
  FIFOBuffer (int maxsize = 4096);
  ~FIFOBuffer ();

  int len () {return bufferlength;}
  char *data () {return bufferdata;}

  bool append (const char *data, int size);
  bool remove (int size);

private:
  int bufferlength, maxlength;
  char *bufferdata;
};

FIFOBuffer::FIFOBuffer (int maxsize) {
  bufferlength = 0;
  bufferdata = new char [maxsize];
  maxlength = maxsize;
}

FIFOBuffer::~FIFOBuffer () {
  delete bufferdata;
}

bool FIFOBuffer::append (const char *data, int size) {
  if (size <= 0) return TRUE;
  bool correct = TRUE;
  if (size + bufferlength > maxlength) {
    correct = FALSE;
    size = maxlength - bufferlength;
  }
  qmemmove (bufferdata + bufferlength, data, size);
  bufferlength += size;
  return correct;
}

bool FIFOBuffer::remove (int size) {
  if (size <= 0) return TRUE;
  if (size > bufferlength) {
    bufferlength = 0;
    return FALSE;
  }
  qmemmove (bufferdata, bufferdata + size, bufferlength - size);
  bufferlength -= size;
  return TRUE;
}

Communication::Communication (QObject *parent, const char *name)
                    :QObject (parent, name) {
  communicationSocket = -1;
  readNotify = 0;
  writeNotify = 0;
  talkClient = client_unknown;
  version_main = version_sub = version_subsub = 0;
  status = st_editchars;
  incoming = new FIFOBuffer ();
  outgoing = new FIFOBuffer ();

  myEditChars [0] = '\b';     // Backspace
  myEditChars [1] = 21;       // CTRL-U
  myEditChars [2] = 23;       // CTRL-W

  connect (options, SIGNAL (fullNameChanged ()), 
                    SLOT (sendFullNameCommand ()));
  connect (this, SIGNAL (clientTypeChanged ()),
                 SLOT (sendFullNameCommand ()));
}

Communication::~Communication () {
  cleanup ();
}

void Communication::cleanup () {
  if (readNotify) delete readNotify;
  if (writeNotify) delete writeNotify;
  if (communicationSocket != -1) ::close (communicationSocket);
  status = st_disconnected;
  delete incoming;
  delete outgoing;
}

void Communication::startCommunication (int socket) {
  char welcomeMessage [] = "ktalk " VERSION ", supports file transfer\n";

  communicationSocket = socket;
  // ######### maybe make socket blocking and use QSocketNotifier to
  // see when writing is okay, and use buffered writing
  // fcntl (socket, F_SETFL, fcntl (socket, F_GETFL, 0) | O_NONBLOCK);

  readNotify = new QSocketNotifier (communicationSocket, 
                                    QSocketNotifier::Read, this);
  connect (readNotify, SIGNAL (activated (int)),
                       SLOT (incomingData (int)));
  writeNotify = new QSocketNotifier (communicationSocket, 
                                     QSocketNotifier::Write, this);
  connect (writeNotify, SIGNAL (activated (int)),
                        SLOT (readyToSendData (int)));
  writeNotify->setEnabled (FALSE);

  sendData (myEditChars, 3);
  sendData (welcomeMessage, strlen (welcomeMessage));
}

void Communication::sendData (QString data) {
  sendData (data.data (), data.length ());
}

void Communication::sendData (const char *data, int length) {
  if (!outgoing->append (data, length))
    KDEBUG (KDEBUG_WARN, 3900, "Buffer overflow on writing to socket");
  writeNotify->setEnabled (outgoing->len () > 0);
}

void Communication::sendCommand (QString command, QString parameters) {
  sendData ("\001" + command + ":" + parameters + '\002');
}

Communication::ClientType Communication::getTalkClient (int &main, int &sub, 
                                                        int &subsub) {
  if (talkClient == client_ktalk) {
    main = version_main;
    sub = version_sub;
    subsub = version_subsub;
  } else {
    main = sub = subsub = 0;
  }
  return talkClient;
}

int Communication::compareKTalkVersion (int main, int sub, int subsub) {
  if (talkClient != client_ktalk)
    return -1;       // any non-ktalk client is "older" (having less features)

  if (version_main   > main  ) return 1;
  if (version_main   < main  ) return -1;
  if (version_sub    > sub   ) return 1;
  if (version_sub    < sub   ) return -1;
  if (version_subsub > subsub) return 1;
  if (version_subsub < subsub) return -1;
  return 0;
}

void Communication::incomingData (int streamSocket) {
  char readBuffer [256];
  int readLength = read (streamSocket, readBuffer, 256);
  if (readLength <= 0) {
    emit connectionClosed (TRUE);
    return;
  }

  if (!incoming->append (readBuffer, readLength))
    KDEBUG (KDEBUG_WARN, 3900, "Buffer overflow on receiving data");
  while (incoming->len () > 0) {
    int welcomeLength = 0, result;
    int textLength = incoming->len ();
    switch (status) {
      case st_editchars:
        if (incoming->len () >= 3) {
          memcpy (hisEditChars, incoming->data (), 3);
          incoming->remove (3);
          if (hisEditChars [0] == 127)
            talkClient = client_talk;
          else if (((unsigned char) hisEditChars [0]) == 254)
            talkClient = client_ytalk;
          else if (hisEditChars [1] == 1 && hisEditChars [2] == 2)
            talkClient = client_chitchat;
          if (talkClient != client_unknown)
            emit clientTypeChanged ();
          status = st_welcome;
        }
        else return;   // not yet 3 edit characters read, wait for more
        break;
      case st_welcome:
        result = checkKTalkMessage (incoming->data (), incoming->len (),
                                    welcomeLength,
                                    version_main, version_sub, version_subsub);
        if (result == 1) return;  // uncomplete welcome message, wait for more
        if (result == 2) {
          talkClient = client_ktalk;
          emit clientTypeChanged ();
          incoming->remove (welcomeLength);
        }
        status = st_normal;
        break;
      case st_normal:
        // check if commands are understood
        if (compareKTalkVersion (0, 2, 7) >= 0) {
          if (incoming->data () [0] == '\001') {
            int colonPos = -1, j;
            QString command, params;
            for (j = 0; j < incoming->len () && 
                 incoming->data() [j] != '\002'; j++)
              if (incoming->data () [j] == ':')
                colonPos = j;
            if (incoming->data () [j] != '\002')
              return;    // command not complete yet
            if (colonPos == -1) {
              command = QString (incoming->data () + 1, j);
              params = QString ();
            } else {
              command = QString (incoming->data () + 1, colonPos);
              params = QString (incoming->data () + colonPos + 1, j - colonPos);
            }
            emit commandReceived (command, params);
            incoming->remove (j + 1);
            break;
          } else {
            int j;
            for (j = 0; j < incoming->len () &&
                        incoming->data () [j] != '\001'; j++);
            textLength = j;
          }
        }
        for (int i = 0; i < textLength; i++)
          for (int j = 0; j < 3; j++)
            if (incoming->data () [i] == hisEditChars [j]) {
              incoming->data () [i] = myEditChars [j];
              break;
            }
        emit dataReceived (QString (incoming->data (), textLength + 1));
        incoming->remove (textLength);
        break;
      case st_disconnected:
        incoming->remove (incoming->len ());
        break;
    }
  }
}

void Communication::readyToSendData (int streamSocket) {
  if (outgoing->len () > 0) {
    int result;
    result = write (streamSocket, outgoing->data (), outgoing->len ());
    if (result <= 0) {
      KDEBUG (KDEBUG_WARN, 3900, "Error writing into socket");
    } else
      outgoing->remove (result);
  }
  writeNotify->setEnabled (outgoing->len () > 0);
}

void Communication::sendFullNameCommand () {
  if (compareKTalkVersion (0, 2, 7) >= 0 && options->getSendFullName ())
    sendCommand ("FullName", options->fullName ());
}
