/*
   announcing.h

   Implements the handling of announcements from outside.
   The announcements are sent to ktalk via the unix domain socket named
   "/tmp/ktalk-username", where "username" is the name of the user who
   started ktalk.

   There are two different types of announcements:
   1. incoming talk requests, sent by the talk daemon to ktalk
   2. "start a talk connection to..." requests, sent to ktalk by another
      instance of ktalk.

   Furthermore there is a packet that queries the display name of the running
   ktalk client.

   For announcements of 1., a window pops up that tells the user he received
   a talk request, and provides two buttons to accept or reject.
   For announcements of 2., no window pops up, but the connection is tried to
   be started at once.

   To sent an announcement to ktalk, send a packet via a unix domain socket
   to the socket "/tmp/ktalk-username". You must "bind" the socket you use
   for sending the packet to a name (e.g. the result of tmpnam ()), so that
   ktalk can reply that he got the packet correctly.

   The sent packet must have the following form:
      struct {
        char announceType;     // value 1 or 2, for announcement type 1 or 2
                               // or 0 for looking up the display name
        char name_and_host [len];  // the username and host to talk to,
                                   // without '\0' at the end
                                   // is empty for packet type 0
      }

   If everything worked fine, ktalk sends a packet of 1 byte with contents 
   "42" back to the sender to show everythong worked fine for announcements
   of type 1 or 2, and returns the display name (without '\0' at the end)
   for packet of type 0.

*/

#ifndef _ANNOUNCING_H
#define _ANNOUNCING_H

#include <qdialog.h>
#include <qstring.h>
#include <qdict.h>
#include <qsocknot.h>

class AnnounceWidget: public QDialog {
Q_OBJECT

  public:
    AnnounceWidget (QString caller);
    ~AnnounceWidget ();

  signals:
    void decision (QString user, bool doConnect);

  protected:
    virtual void timerEvent (QTimerEvent *) 
       {emit decision (announceKey, FALSE);}

  private slots:
     void answer () {emit decision (announceKey, TRUE);}
     void ignore () {emit decision (announceKey, FALSE);}

  private:
    QString announceKey;
};

class AnnounceCenter: public QObject {
Q_OBJECT

public:
  AnnounceCenter (QObject *parent = 0, 
                  const char *name = 0);  
  ~AnnounceCenter ();
  bool startCenter (QStrList &parameters);

signals:
  void startTalk (const char *destination);

private slots:
  void announceReceived (int sock);
  void answerFromAnnounce (QString key, bool answer);

private:
  bool checkOtherClients (int sock, QString tempDir, QStrList &entryList, 
                          QStrList &parameters);
  bool transferList (QStrList &parameters, int sock,
                     struct sockaddr_un &destination);
  QString askClientForDisplay (int sock, struct sockaddr_un &destination);
  int chooseClient (const QStrList &displays);
  int sendToOtherClient (int sock, struct sockaddr_un &destination,
                         char *data, unsigned int dataLength,
                         char *answer, unsigned int maxAnswerLength);

  QDict <AnnounceWidget> announces;
  QString socketPath;
  int announceSocket;
  QSocketNotifier *notify;
  QString localDisplay;
};

#endif
