// announcing.cpp

#include <config.h>
#include "announcing.h"
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#include <qlabel.h>
#include <qlayout.h>
#include <qdir.h>
#include <qlistbox.h>
#include <kapp.h>
#include <kiconloader.h>
#include <kmsgbox.h>
#include "keasybutton.h"
#include "talkd.h"
#include "global.h"
#include "options.h"

#include <X11/Xlib.h>
#include <errno.h>

AnnounceWidget::AnnounceWidget (QString caller) : QDialog (0, 0, FALSE) {

  setCaption (i18n ("Talk Request"));
  announceKey = caller;

  QLabel *ktalkIcon = new QLabel (this);
  QLabel *requestText = new QLabel (this);
  KEasyButton *answerButton = new KEasyButton (i18n ("Answer"), this);
  KEasyButton *ignoreButton = new KEasyButton (i18n ("Ignore"), this);

  ktalkIcon->setPixmap (ICON ("ktalk.xpm"));
  ktalkIcon->setAlignment (AlignCenter);
  ktalkIcon->setMinimumSize (ktalkIcon->sizeHint ());
  requestText->setText (i18n ("Talk connection requested by\n") + caller);
  requestText->setAlignment (AlignCenter);
  requestText->setMinimumSize (requestText->sizeHint ());

  connect (answerButton, SIGNAL (clicked ()), SLOT (answer ()));
  connect (ignoreButton, SIGNAL (clicked ()), SLOT (ignore ()));

  QVBoxLayout *topLayout = new QVBoxLayout (this, 15);
  QHBoxLayout *upperLayout = new QHBoxLayout ();
  QHBoxLayout *lowerLayout = new QHBoxLayout ();

  topLayout->addLayout (upperLayout, 10);
  topLayout->addLayout (lowerLayout);

  upperLayout->addWidget (ktalkIcon);
  upperLayout->addWidget (requestText);

  lowerLayout->addStretch (10);
  lowerLayout->addWidget (answerButton);
  lowerLayout->addStretch (10);
  lowerLayout->addWidget (ignoreButton);
  lowerLayout->addStretch (10);

  topLayout->activate ();
  resize (minimumSize ());

  startTimer ((RING_WAIT + 10) * 1000);
}

AnnounceWidget::~AnnounceWidget () {
}

AnnounceCenter::AnnounceCenter (QObject *parent, const char *name) :
    QObject (parent, name), announces (17, FALSE, TRUE)
{
  announces.setAutoDelete (TRUE);
  announceSocket = -1;

  // Determine the local display name. An instance of QApplication must have
  // been created!
  localDisplay = DisplayString (kapp->getDisplay ());
}

AnnounceCenter::~AnnounceCenter () {
  if (announceSocket >= 0) {
    delete notify;
    ::close (announceSocket);
    announceSocket = -1;
    if (unlink (socketPath.data ()) == -1)
      KDEBUG (KDEBUG_WARN, 3900, "Unable to remove socket from file system");
  } 
}

bool AnnounceCenter::startCenter (QStrList &parameters) {

  QString tempDir = "/tmp";
  QString templ = "ktalk-" + localLoginName + "-";

  DIR *dir = opendir (tempDir);
  struct dirent *entry;
  QStrList dirList;
  while ((entry = readdir (dir)))
    if (strncmp (templ.data (), entry->d_name, templ.length ()) == 0)
      dirList.append (entry->d_name);

  if (dirList.count () > 0) {
    // create temporary socket
    int sendSocket = -1;
    struct sockaddr_un tempAddr;
    tempAddr.sun_family = AF_UNIX;
    if (tmpnam (tempAddr.sun_path) == 0 ||
        (sendSocket = socket (AF_UNIX, SOCK_DGRAM, 0)) < 0) {
      KDEBUG (KDEBUG_WARN, 3900, 
              "Unable to create a temporary unix domain socket!");
    } else if (bind (sendSocket, (struct sockaddr *) &tempAddr, 
                     sizeof (tempAddr)) == -1) {
      KDEBUG (KDEBUG_WARN, 3900, "Unable to bind to unix domain socket!");
      close (sendSocket);
    } else {
      bool result;
      result = checkOtherClients (sendSocket, tempDir, dirList, parameters);
      close (sendSocket);
      unlink (tempAddr.sun_path);
      if (result)
        return TRUE;
    }
  }

  // build up a new, unused unix domain socket
  int j = 0;
  QString fileName;
  do {
    j++;
    fileName.sprintf ("%s%d", templ.data (), j);
  } while (dirList.contains (fileName));
  socketPath = tempDir + "/" + fileName;
  announceSocket = socket (AF_UNIX, SOCK_DGRAM, 0);
  if (announceSocket == -1) {
    KDEBUG (KDEBUG_WARN, 3900, "Cannot open unix domain socket!");
    return FALSE;
  }
  struct sockaddr_un addr;
  addr.sun_family = AF_UNIX;
  strncpy (addr.sun_path, socketPath.data (), sizeof (addr.sun_path));
  if (bind (announceSocket, (struct sockaddr *) &addr, sizeof (addr)) == -1) {
    KDEBUG1 (KDEBUG_WARN, 3900, 
             "Can't bind to named unix domain socket '%s'!",
             socketPath.data ());
    close (announceSocket);
    announceSocket = -1;
    return FALSE;
  }

  notify = new QSocketNotifier (announceSocket, QSocketNotifier::Read, this);
  connect (notify, SIGNAL (activated (int)), SLOT (announceReceived (int)));    
  return FALSE;
}

bool AnnounceCenter::checkOtherClients (int sock, QString tempDir, 
           QStrList &entryList, QStrList &parameters)
{
  QStrList displayNames, socketNames;
  struct sockaddr_un destAddr;
  destAddr.sun_family = AF_UNIX;
  char *current;
  for (current = entryList.first (); current; current = entryList.next ()) {
    QString path = tempDir + "/" + current;
    strncpy (destAddr.sun_path, path.data (), sizeof (destAddr.sun_path));
    QString display = askClientForDisplay (sock, destAddr);
    if (display.isNull ()) {
      KDEBUG1 (KDEBUG_WARN, 3900, "unix domain socket %s doesn't answer, "
                                  "removing it!", destAddr.sun_path);
      if (unlink (destAddr.sun_path))
        KDEBUG (KDEBUG_WARN, 3900, "removing socket failed!");
    } else if (!display.isEmpty ()) {
      if (display == localDisplay)
        if (transferList (parameters, sock, destAddr))
          return TRUE;
      displayNames.append (i18n ("Client on display \"") + display + "\"");
      socketNames.append (destAddr.sun_path);
    }
  }
  if (displayNames.count () == 0)
    return FALSE;

  // let the user choose the client to use
  int selection = chooseClient (displayNames);
  if (selection == -1)
    return TRUE;            // quit the program
  if (selection == 0)
    return FALSE;           // start the new client

  // transfer the parameters to the chosen client
  strncpy (destAddr.sun_path, socketNames.at (selection - 1),
           sizeof (destAddr.sun_path));
  if (transferList (parameters, sock, destAddr))
    return TRUE;          // other client takes over, quit the program

  // Error occured transfering data
  return (KMsgBox::yesNo (0, i18n ("ktalk: Error"), 
                          i18n ("The client doesn't respond!\n"
                                "Do you want to start a new client?"),
                          KMsgBox::EXCLAMATION) != 1);
}

void AnnounceCenter::announceReceived (int sockt) {
  char buffer [256];
  struct sockaddr_un from;
  int len;
  ksize_t fromlen = sizeof (from);
  char answer = 42;
  if ((len = recvfrom (sockt, buffer, 256, 0, 
                      (struct sockaddr *) &from, &fromlen)) <= 0) {
    KDEBUG (KDEBUG_WARN, 3900, "Error receiving from announce socket!");
    return;
  }
  if (buffer [0] < 0 || buffer [0] > 2) {
    KDEBUG1 (KDEBUG_WARN, 3900, "Unknown announcement type %d!", buffer [0]);
    return;
  }

  // packet type 0: return display name
  if (buffer [0] == 0) {
    if (len != 1) {
      KDEBUG (KDEBUG_WARN, 3900, "Wrong length for announce packet of type 0!");
    } else {
      int length = localDisplay.length () + 1;
      if (sendto (sockt, localDisplay.data (), length, 0, 
                  (struct sockaddr *) &from, fromlen) != length)
        KDEBUG (KDEBUG_WARN, 3900, "Error sending to announce socket!");
    }
    return;
  }

  // packet type 1 or 2: reply to caller with acknowledge 42
  if (sendto (sockt, &answer, 1, 0, (struct sockaddr *) &from, fromlen) != 1)
  {
    KDEBUG (KDEBUG_WARN, 3900, "Error sending to announce socket!");
    return;
  }
  QString name (&(buffer [1]), len);

  // packet type 1: make announcement
  if (buffer [0] == 1) {
    options->playAnnounceSound ();
    if (announces [name.data ()]) {
      announces [name.data ()]->show ();
      announces [name.data ()]->raise ();
    } else {
      AnnounceWidget *newWidget = new AnnounceWidget (name);
      connect (newWidget, SIGNAL (decision (QString, bool)),
                          SLOT (answerFromAnnounce (QString, bool)));
      announces.insert (name.data (), newWidget);
      newWidget->show ();
    }
    return;
  }

  // packet type 2: start new talk connection
  emit startTalk (name.data ());
}

void AnnounceCenter::answerFromAnnounce (QString key, bool answer) {
  announces.remove (key.data ());
  if (answer)
    emit startTalk (key.data ());
}

bool AnnounceCenter::transferList (QStrList &parameters, int sock,
                                   struct sockaddr_un &destination) {
  char *current = parameters.first ();
  bool ready = FALSE;
  while (!ready) {
    if (!current) {
      ready = TRUE;
      current = "";  // send empty packet as signal for other client to raise
    }
    char buffer [256];
    buffer [0] = 2;
    strncpy (&(buffer [1]), current, 255);
    char answer;
    if (sendToOtherClient (sock, destination, buffer, strlen (current) + 1,
                           &answer, 1) != 1 ||
        answer != 42) {
      ready = FALSE;
      break;
    }
    if (!ready) current = parameters.next ();
  }
  return ready;
}

QString AnnounceCenter::askClientForDisplay (int sock, 
           struct sockaddr_un &destination)
{
  char query = 0;
  char answer [256];
  int len;
  len = sendToOtherClient (sock, destination, &query, 1, answer, 256);
  if (len < 0)
    return QString ((char *) 0);
  else
    return QString (answer, len);
}

int AnnounceCenter::chooseClient (const QStrList &displays)
{
  QDialog *dialog = new QDialog (0, 0, TRUE);
  QLabel *text = new QLabel (i18n ("There are already other ktalk clients "
      "running\non this machine for this user, "
      "but on other displays.\n"
      "Please choose the client to use!"), dialog);
  text->setMinimumSize (text->sizeHint ());
  QListBox *lb = new QListBox (dialog);
  lb->setMinimumSize (QSize (100, 5 * lb->fontMetrics ().height ()));
  KEasyButton *ok = new KEasyButton (i18n ("OK"), dialog, 0, -1,
                                     dialog, SLOT (accept ()));
  KEasyButton *can = new KEasyButton (i18n ("Cancel"), dialog, 0, -1,
                                      dialog, SLOT (reject ()));

  lb->insertItem ("Start a new Client");
  lb->insertStrList (&displays);
  lb->setCurrentItem (0);

  QVBoxLayout *topLayout = new QVBoxLayout (dialog, 10);
  QHBoxLayout *buttonLayout = new QHBoxLayout ();

  topLayout->addWidget (text);
  topLayout->addWidget (lb, 10);
  topLayout->addLayout (buttonLayout);

  buttonLayout->addStretch (10);
  buttonLayout->addWidget (ok);
  buttonLayout->addStretch (10);
  buttonLayout->addWidget (can);
  buttonLayout->addStretch (10);

  topLayout->activate ();
  dialog->resize (400, 230);
  int result = dialog->exec ();
  delete dialog;   
  if (result)
    return lb->currentItem ();
  else
    return -1;
}

int AnnounceCenter::sendToOtherClient (int sock, 
        struct sockaddr_un &destination,
        char *data, unsigned int dataLength,
        char *answer, unsigned int maxAnswerLength)
{
  unsigned int len;
  len = sendto (sock, data, dataLength, 0, 
                (struct sockaddr *) &destination, sizeof (destination));
  if (len != dataLength) {
    KDEBUG (KDEBUG_WARN, 3900, "Error sending via unix domain socket!");
    return -1;
  }

  fd_set readFDs;
  FD_ZERO (&readFDs);
  FD_SET (sock, &readFDs);

  struct timeval timeout;
  timeout.tv_sec = 0;
  timeout.tv_usec = 500000;  // Wait for answer at most 0.5 seconds

  if (!select (sock + 1, &readFDs, 0, 0, &timeout))
    return -1;
  return recv (sock, answer, maxAnswerLength, 0);
}
