#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <qwidget.h>
#include <qpushbt.h>
#include <qfiledlg.h> 
#include <qlabel.h> 
#include <qaccel.h> 
#include <qlist.h>
#include <qlayout.h>
#include <qgrpbox.h>
#include <qregexp.h>
#include <kmsgbox.h>
#include <kapp.h>
#include <kdebug.h>

#include "addrbook.h"

AddressBook::AddressBook ()
  : QDialog (0, 0, TRUE) {

  setCaption (i18n ("Addressbook"));  

  QLabel *label1 = new QLabel (i18n ("Entries :"), this);
  label1->setMinimumSize (label1->sizeHint ());

  lbEntries = new QListBox (this);
  lbEntries->setMinimumHeight (5 * lbEntries->itemHeight ());
  connect (lbEntries, SIGNAL (highlighted (int)), SLOT (selectEntry (int)));

  bAdd = new KEasyButton (i18n ("Add"), this, 0, -1, 
                          this, SLOT (addEntry ()),
                          i18n ("Adds a new entry to the addressbook"));
  bChange = new KEasyButton (i18n ("Change"), this, 0, -1,
                             this, SLOT (changeEntry ()),
                             i18n ("Changes the highlighted entry"));
  bChange->setEnabled (FALSE);
  bDelete = new KEasyButton (i18n ("Delete"), this, 0, -1,
                             this, SLOT (delEntry ()),
                             i18n ("Marks the highlighted entry for deletion"));
  bDelete->setEnabled (FALSE);
  bUndelete = new KEasyButton (i18n ("Undelete"), this, 0, -1,
                               this, SLOT (undelEntry ()),
                               i18n ("Removes the mark for deletion from"
                                     "the highlighted entry"));
  bUndelete->setEnabled (FALSE);

  QLabel *label2 = new QLabel (i18n ("Address(es) :"), this);
  label2->setMinimumSize (label2->sizeHint ());

  mleAddresses = new QMultiLineEdit (this);
  mleAddresses->setReadOnly (TRUE);
  mleAddresses->setMinimumHeight (4 * mleAddresses->fontMetrics().height ());

  bClose = new KEasyButton (i18n ("Close"), this, 0, -1,
                            this, SLOT (accept ()),
                            i18n ("Closes Addressbook and deletes "
                                  "all marked entries"));
  bReset = new KEasyButton (i18n ("Reset"), this, 0, -1,
                            this, SLOT (resetPressed ()),
                            i18n ("Restores the addressbook to the "
                                  "state it had before opening"));

  QBoxLayout *layout1 = new QVBoxLayout (this, 5);
  QBoxLayout *layout2 = new QHBoxLayout ();
  QBoxLayout *layout4 = new QVBoxLayout ();
  QBoxLayout *layout5 = new QHBoxLayout ();

  layout1->addWidget (label1);
  layout1->addLayout (layout2, 15);
  layout1->addSpacing (10);
  layout1->addWidget (label2);
  layout1->addWidget (mleAddresses, 10);
  layout1->addSpacing (10);
  layout1->addLayout (layout5);

  layout2->addWidget (lbEntries, 10);
  layout2->addLayout (layout4);

  layout4->addWidget (bAdd);
  layout4->addWidget (bChange);
  layout4->addWidget (bDelete);
  layout4->addWidget (bUndelete);
  layout4->addStretch (10);

  layout5->addStretch (10);
  layout5->addWidget (bClose);
  layout5->addStretch (10);
  layout5->addWidget (bReset);
  layout5->addStretch (10);

  layout1->activate ();

  addrList = new QDict <QStrList> (101);    // case sensitive and copy keys
  recentList = new QStrList;
  QString configFilename = KApplication::localconfigdir () + 
                           "/ktalk.addresses";
  abConfig = new KSimpleConfig (configFilename);
  readAddrConfig ();
  abConfig->setGroup ("RecentAddresses");
  abConfig->readListEntry ("Adresses", *recentList);
  lbEntries->setFocus ();
  resize (410, 400);
}

void AddressBook::insertRecentEntry (QString entry) {
  if (recentList->find (entry) != -1)
    return;
  if (recentList->count () > 5)
    recentList->removeLast ();
  recentList->insert (0, entry);
  abConfig->setGroup ("RecentAddresses");
  abConfig->writeEntry ("Adresses", *recentList);
}

void AddressBook::execAddressBook () {

  delEntries.clear ();
  markedEntry = -1;

  filloutDialog ();
  show ();
  for (char *str = delEntries.first (); str; str = delEntries.next ()) {
    if (!addrList->remove (str))
      KDEBUG1 (KDEBUG_ERROR, 3900, 
               "internal addressbook error (6) for entry %s", str);
    if (entries.find (str) != -1)
      entries.remove ();
    else
      KDEBUG1 (KDEBUG_ERROR, 3900, 
               "internal addressbook error (7) for entry %s", str);
  }
  writeAddrConfig ();
}

void AddressBook::addEntry () {

  AddChangeDialog dialog (&entries, recentList, this);
  if (dialog.exec () == Accepted) {
    QString newEntry = dialog.getEntry ();
    entries.inSort (newEntry);
    markedEntry = entries.find (newEntry);
    lbEntries->clearSelection ();
    lbEntries->insertItem (newEntry, markedEntry);
    buttonsForEntry ();
    addrList->insert (newEntry, dialog.getAddresses ());
    filloutDialog ();
  }
}

void AddressBook::changeEntry () {

  QString str = entries.at (markedEntry);
  if (str.isNull ()) return;
  AddChangeDialog dialog (str, getEntry (str),
                          &entries, recentList, this);
  if (dialog.exec () == Accepted) {
    if (!addrList->remove (str))
      KDEBUG1 (KDEBUG_ERROR, 3900, 
               "internal addressbook error (1) for entry %s", 
               (const char *) markedEntry);
    entries.remove (markedEntry);
    lbEntries->setUpdatesEnabled (FALSE);
    lbEntries->removeItem (markedEntry);
    QString newEntry = dialog.getEntry ();
    if (delEntries.find (newEntry) != -1)
      delEntries.remove ();
    entries.inSort (newEntry);
    markedEntry = entries.find (newEntry);
    lbEntries->insertItem (newEntry, markedEntry);
    lbEntries->setUpdatesEnabled (TRUE);
    lbEntries->repaint ();
    addrList->insert (newEntry, dialog.getAddresses ());
    filloutDialog ();
  }
}

void AddressBook::delEntry () {

  QString entry = entries.at (markedEntry);
  delEntries.append (entry);
  lbEntries->changeItem (i18n ("[delete] ") + entry, markedEntry);
  buttonsForEntry ();
}

void AddressBook::undelEntry () {
  QString entry = entries.at (markedEntry);
  if (delEntries.find (entry) != -1) {
    delEntries.remove ();
    lbEntries->changeItem (entry, markedEntry);
    buttonsForEntry ();
  } else
    KDEBUG1 (KDEBUG_ERROR, 3900, 
             "internal addressbook error (3) for entry %s",
             (const char *) entry);
}

void AddressBook::resetPressed () {
  readAddrConfig ();
  delEntries.clear ();
  markedEntry = -1;
  filloutDialog ();
}

void AddressBook::selectEntry (int i) {

  markedEntry = i;
  buttonsForEntry ();

  QStrList *adrs = getEntry (entries.at (markedEntry));
  mleAddresses->setUpdatesEnabled (FALSE);
  mleAddresses->clear ();
  for (char *line = adrs->first (); line; line = adrs->next ())
    mleAddresses->insertLine (line);
  mleAddresses->setUpdatesEnabled (TRUE);
  mleAddresses->repaint ();
}

void AddressBook::readAddrConfig () {

  entries.clear ();
  addrList->clear ();
  abConfig->setGroup ("Addressbook");
  KEntryIterator *iter = abConfig->entryIterator ("Addressbook");
  if (!iter) return;
  while (iter->current ()) {
     QString key = iter->currentKey ();
     if (key.left (4) == "Name") {
        key.replace (0, 4, "Addresses");
        if (!abConfig->hasKey (key)) {
          KDEBUG1 (KDEBUG_ERROR, 3900, 
                   "internal addressbook error (4) for entry %s",
                   (const char *) key);
        } else {
          QStrList *help = new QStrList ();
          abConfig->readListEntry (key, *help);
          addrList->replace (iter->current()->aValue, help);
          entries.inSort (iter->current()->aValue);
        }
     }
     ++(*iter);
  }
}

void AddressBook::writeAddrConfig () {

  abConfig->deleteGroup ("Addressbook");
  abConfig->setGroup ("Addressbook");
  QDictIterator <QStrList> iter (*addrList);
  int i = 1;
  while (iter.currentKey ()) {
     QString key;
     key.sprintf ("Name%d", i);
     abConfig->writeEntry (key, iter.currentKey ());
     key.sprintf ("Addresses%d", i);
     abConfig->writeEntry (key, *iter.current ());
     ++iter;
     i++;
  }
}

void AddressBook::filloutDialog () {
  int dummyMarked = markedEntry;
  lbEntries->setUpdatesEnabled (FALSE);
  lbEntries->clear ();
  for (char *str = entries.first (); str; str = entries.next ()) {
    if (delEntries.contains (str))
      lbEntries->insertItem (QString (i18n ("[delete] ")) + str);
    else
      lbEntries->insertItem (str);
  }
  if (dummyMarked < 0)
    mleAddresses->clear ();
  else {
    lbEntries->setCurrentItem (dummyMarked);
    lbEntries->centerCurrentItem ();
  }
  lbEntries->setUpdatesEnabled (TRUE);
  lbEntries->repaint ();
}

void AddressBook::buttonsForEntry () {
  if (markedEntry < 0) {
    bChange->setEnabled (FALSE);
    bDelete->setEnabled (FALSE);
    bUndelete->setEnabled (FALSE);
  } else {
    bChange->setEnabled (TRUE);
    QString entry = entries.at (markedEntry);
    int deleted = delEntries.contains (entry);
    bDelete->setEnabled (!deleted);
    bUndelete->setEnabled (deleted);
  }
}    

// -----------------------------------------------------------------

AddChangeDialog::AddChangeDialog (const QStrList *usedEntries, 
   const QStrList *recent, QWidget *parent, const char *name) :
   QDialog (parent, name, TRUE)
{
  fillDialog (FALSE);
  _recent->insertStrList (recent);
  _usedEntries = usedEntries;
}

AddChangeDialog::AddChangeDialog (const QString entry, 
   QStrList *addresses, const QStrList *usedEntries, 
   const QStrList *recent, QWidget *parent, const char *name) :
   QDialog (parent, name, TRUE)
{
  fillDialog (TRUE);
  _recent->insertStrList (recent);
  _entry->setText (entry);
  oldEntry = entry;
  char *dummy;
  for (dummy = addresses->first (); dummy; dummy = addresses->next ())
     _addresses->insertLine (dummy);
  _usedEntries = usedEntries;
}

void AddChangeDialog::acceptPressed ()
{
  QString entry = QString (_entry->text ()).stripWhiteSpace ();
  _entry->setText (entry);
  if (entry.isEmpty ())
    KMsgBox::message (0, i18n ("Error: no name"), 
                      i18n ("You have to specify a name\n"
                            "for the addressbook entry!"),
                      KMsgBox::EXCLAMATION);
  else if (entry != oldEntry && _usedEntries->contains (entry))
    KMsgBox::message (0, i18n ("Error: entry exists"),
                      i18n ("An addressbook entry with this "
                            "name already exists,\n"
                            "please use a different name!"),
                      KMsgBox::EXCLAMATION);
  else accept ();
}

void AddChangeDialog::recentSelected (const char *address) {
  _addresses->insertLine (address);
}

void AddChangeDialog::entryReturnPressed () {
  _addresses->setFocus ();
}

QStrList *AddChangeDialog::getAddresses () {
  QString help = _addresses->text();
  help.replace (QRegExp ("\n"), ",");
  QStrList *result = new QStrList (stripCommaList (help));
  return result;
}

void AddChangeDialog::fillDialog (bool change)
{
  if (change)
    setCaption (i18n ("Change existing addressbook entry"));
  else
    setCaption (i18n ("Add new entry to addressbook"));

  QLabel *label1 = new QLabel (i18n ("Entry name :"), this);
  label1->setMinimumSize (label1->sizeHint ());

  _entry = new QLineEdit (this);
  _entry->setMinimumHeight (_entry->sizeHint ().height ());
  connect (_entry, SIGNAL (returnPressed ()), SLOT (entryReturnPressed ()));

  QLabel *label2 = new QLabel (i18n ("Addresses :"), this);
  label2->setMinimumSize (label2->sizeHint ());

  _addresses = new QMultiLineEdit (this);
  _addresses->setMinimumHeight (4 * _addresses->fontMetrics().height ());

  QLabel *label3 = new QLabel (i18n ("recently used addresses :"), this);
  label3->setMinimumSize (label3->sizeHint ());

  _recent = new QListBox (this); 
  _recent->setMinimumHeight (3 * _recent->itemHeight ());
  connect (_recent, SIGNAL (selected (const char *)), 
                    SLOT (recentSelected (const char *))); 

  KEasyButton *ok = new KEasyButton 
     (change ? i18n ("Change") : i18n ("Add"), this, 0, -1, 
      this, SLOT (acceptPressed ()), 
      change ? i18n ("Changes the existing addressbook entry"):
               i18n ("Adds this entry to the addressbook"));

  KEasyButton *cancel = new KEasyButton 
     (i18n ("Cancel"), this, 0, -1, this, SLOT (reject ()),
      i18n ("Closes this dialog without changing addressbook"));

  QBoxLayout *layout1 = new QVBoxLayout (this, 5);
  QBoxLayout *layout2 = new QHBoxLayout ();
  layout1->addWidget (label1);
  layout1->addWidget (_entry);
  layout1->addWidget (label2);
  layout1->addWidget (_addresses, 20);
  layout1->addWidget (label3);
  layout1->addWidget (_recent, 10);
  layout1->addLayout (layout2);
  layout2->addStretch (10);
  layout2->addWidget (ok);
  layout2->addStretch (10);
  layout2->addWidget (cancel);
  layout2->addStretch (10);
  layout1->activate ();

  resize (430, 350);
}

QStrList stripCommaList (const QString &line) {
  QStrList result (TRUE);                 // deep copy
  result.setAutoDelete (TRUE);            // auto delete
  uint pos = 0;

  while (pos < line.length()) {
    int commapos = line.find (",", pos);
    if (commapos < 0) 
      commapos = line.length ();
    QString item = line.mid (pos, commapos - pos);
    item = item.simplifyWhiteSpace ();
    if (!item.isEmpty ())
      result.append (item);
    pos = commapos + 1;
  }
  return result;
}

QString buildCommaList (QStrList &list) {
  QString result;
  char *dummy = list.first ();
  while (dummy) {
    if (!result.isEmpty ())
      result.append (","); 
    result.append (dummy);
    dummy = list.next ();
  }
  return result;
}

