/*
** Copyright (C) 1999 Erik de Castro Lopo <erikd@zip.com.au>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/



#include	<stdio.h>
#include	<unistd.h>
#include	<sys/stat.h>
#include	<fcntl.h>

#include	<sndfile.h>

static	unsigned char  wav_ieee [] =
{	0x52, 0x49, 0x46, 0x46, 0x64, 0x00, 0x00, 0x00, 
	0x57, 0x41, 0x56, 0x45, 0x66, 0x6D, 0x74, 0x20,
	0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00, 
	0x34, 0x12, 0x00, 0x00, 0xD0, 0x48, 0x00, 0x00,
	0x04, 0x00, 0x20, 0x00, 0x64, 0x61, 0x74, 0x61, 
	0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x80, 0x3F, 0x00, 0x00, 0x00, 0x40, 
	0x00, 0x00, 0x40, 0x40, 0x00, 0x00, 0x80, 0x40,
	0x00, 0x00, 0xA0, 0x40, 0x00, 0x00, 0xC0, 0x40, 
	0x00, 0x00, 0xE0, 0x40, 0x00, 0x00, 0x00, 0x41,
	0x00, 0x00, 0x10, 0x41, 0x00, 0x00, 0x20, 0x41, 
	0x00, 0x00, 0x30, 0x41, 0x00, 0x00, 0x40, 0x41,
	0x00, 0x00, 0x50, 0x41, 0x00, 0x00, 0x60, 0x41, 
	0x00, 0x00, 0x70, 0x41, 0x49, 0x4E, 0x46, 0x4F,
	0x04, 0x00, 0x00, 0x00, 0xFE, 0xED, 0xC0, 0xDE
} ; /* wav_ieee */

static	unsigned char  wav_pcm32 [] =
{	0x52, 0x49, 0x46, 0x46, 0x64, 0x00, 0x00, 0x00, 
	0x57, 0x41, 0x56, 0x45, 0x66, 0x6D, 0x74, 0x20,
	0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 
	0x34, 0x12, 0x00, 0x00, 0xD0, 0x48, 0x00, 0x00,
	0x04, 0x00, 0x20, 0x00, 0x64, 0x61, 0x74, 0x61, 
	0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 
	0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 
	0x07, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
	0x09, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 
	0x0B, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00,
	0x0D, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 
	0x0F, 0x00, 0x00, 0x00, 0x49, 0x4E, 0x46, 0x4F,
	0x04, 0x00, 0x00, 0x00, 0xFE, 0xED, 0xC0, 0xDE
} ; /* wav_pcm32 */

static	unsigned char  wav_pcm24 [] =
{	0x52, 0x49, 0x46, 0x46, 0x54, 0x00, 0x00, 0x00, 
	0x57, 0x41, 0x56, 0x45, 0x66, 0x6D, 0x74, 0x20,
	0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 
	0x34, 0x12, 0x00, 0x00, 0x9C, 0x36, 0x00, 0x00, 
	0x03, 0x00, 0x18, 0x00, 0x64, 0x61, 0x74, 0x61, 
	0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x03, 0x00, 0x00, 
	0x04, 0x00, 0x00, 0x05, 0x00, 0x00, 0x06, 0x00, 
	0x00, 0x07, 0x00, 0x00, 0x08, 0x00, 0x00, 0x09, 
	0x00, 0x00, 0x0A, 0x00, 0x00, 0x0B, 0x00, 0x00, 
	0x0C, 0x00, 0x00, 0x0D, 0x00, 0x00, 0x0E, 0x00, 
	0x00, 0x0F, 0x00, 0x00, 0x49, 0x4E, 0x46, 0x4F,
	0x04, 0x00, 0x00, 0x00, 0xFE, 0xED, 0xC0, 0xDE
} ; /* wav_pcm24 */

static	unsigned char  wav_pcm16 [] =
{	0x52, 0x49, 0x46, 0x46, 0x44, 0x00, 0x00, 0x00, 
	0x57, 0x41, 0x56, 0x45, 0x66, 0x6D, 0x74, 0x20, 
	0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 
	0x34, 0x12, 0x00, 0x00, 0x24, 0x68, 0x00, 0x00, 
	0x02, 0x00, 0x10, 0x00, 0x64, 0x61, 0x74, 0x61, 
	0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 
	0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 
	0x06, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 
	0x0A, 0x00, 0x0B, 0x00, 0x0C, 0x00, 0x0D, 0x00, 
	0x0E, 0x00, 0x0F, 0x00, 0x49, 0x4E, 0x46, 0x4F,
	0x04, 0x00, 0x00, 0x00, 0xFE, 0xED, 0xC0, 0xDE
} ; /* wav_pcm16 */

static	unsigned char  wav_pcm8 [] =
{	0x52, 0x49, 0x46, 0x46, 0x34, 0x00, 0x00, 0x00, 
	0x57, 0x41, 0x56, 0x45, 0x66, 0x6D, 0x74, 0x20, 
	0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 
	0x34, 0x12, 0x00, 0x00, 0x34, 0x12, 0x00, 0x00, 
	0x01, 0x00, 0x08, 0x00, 0x64, 0x61, 0x74, 0x61, 
	0x10, 0x00, 0x00, 0x00, 0x80, 0x81, 0x82, 0x83, 
	0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 
	0x8C, 0x8D, 0x8E, 0x8F, 0x49, 0x4E, 0x46, 0x4F,
	0x04, 0x00, 0x00, 0x00, 0xFE, 0xED, 0xC0, 0xDE
} ; /* wav_pcm8 */

static	unsigned char  aiff_pcm32 [] =
{	0x46, 0x4F, 0x52, 0x4D, 0x00, 0x00, 0x00, 0x6E, 
	0x41, 0x49, 0x46, 0x46, 0x43, 0x4F, 0x4D, 0x4D,
	0x00, 0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0x00, 
	0x00, 0x10, 0x00, 0x20, 0x40, 0x0B, 0x91, 0xA0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 
	0x4E, 0x44, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
	0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
	0x00, 0x06, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 
	0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00,
	0x00, 0x0A, 0x00, 0x00, 0x00, 0x0B, 0x00, 0x00, 
	0x00, 0x0C, 0x00, 0x00, 0x00, 0x0D, 0x00, 0x00,
	0x00, 0x0E, 0x00, 0x00, 0x00, 0x0F
} ; /* aiff_pcm32 */

static	unsigned char  aiff_pcm24 [] =
{	0x46, 0x4F, 0x52, 0x4D, 0x00, 0x00, 0x00, 0x5E, 
	0x41, 0x49, 0x46, 0x46, 0x43, 0x4F, 0x4D, 0x4D, 
	0x00, 0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0x00, 
	0x00, 0x10, 0x00, 0x18, 0x40, 0x0B, 0x91, 0xA0, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 
	0x4E, 0x44, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x03, 0x00, 0x00, 0x04, 0x00, 0x00, 0x05, 
	0x00, 0x00, 0x06, 0x00, 0x00, 0x07, 0x00, 0x00, 
	0x08, 0x00, 0x00, 0x09, 0x00, 0x00, 0x0A, 0x00, 
	0x00, 0x0B, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x0D, 
	0x00, 0x00, 0x0E, 0x00, 0x00, 0x0F
} ; /* aiff_pcm24 */

static	unsigned char  aiff_pcm16 [] =
{	0x46, 0x4F, 0x52, 0x4D, 0x00, 0x00, 0x00, 0x4E, 
	0x41, 0x49, 0x46, 0x46, 0x43, 0x4F, 0x4D, 0x4D, 
	0x00, 0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0x00, 
	0x00, 0x10, 0x00, 0x10, 0x40, 0x0B, 0x91, 0xA0, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x53, 
	0x4E, 0x44, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 
	0x00, 0x05, 0x00, 0x06, 0x00, 0x07, 0x00, 0x08, 
	0x00, 0x09, 0x00, 0x0A, 0x00, 0x0B, 0x00, 0x0C, 
	0x00, 0x0D, 0x00, 0x0E, 0x00, 0x0F
} ; /* aiff_pcm16 */


static	unsigned char  au_pcm32 [] =
{	0x2E, 0x73, 0x6E, 0x64, 0x00, 0x00, 0x00, 0x18, 
	0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x05,
	0x00, 0x00, 0x12, 0x34, 0x00, 0x00, 0x00, 0x01, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05,
	0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x07,
	0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09,
	0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x0B,
	0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x0D,
	0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x0F
} ; /* au_pcm32 */

static	unsigned char  au_pcm24 [] =
{	0x2E, 0x73, 0x6E, 0x64, 0x00, 0x00, 0x00, 0x18, 
	0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x04,
	0x00, 0x00, 0x12, 0x34, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x03, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x05, 0x00, 0x00, 0x06, 0x00, 0x00, 0x07,
	0x00, 0x00, 0x08, 0x00, 0x00, 0x09, 0x00, 0x00,
	0x0A, 0x00, 0x00, 0x0B, 0x00, 0x00, 0x0C, 0x00,
	0x00, 0x0D, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x0F
} ; /* au_pcm24 */

static	unsigned char  au_pcm16 [] =
{	0x2E, 0x73, 0x6E, 0x64, 0x00, 0x00, 0x00, 0x18, 
	0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x03,
	0x00, 0x00, 0x12, 0x34, 0x00, 0x00, 0x00, 0x01, 
	0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03,
	0x00, 0x04, 0x00, 0x05, 0x00, 0x06, 0x00, 0x07, 
	0x00, 0x08, 0x00, 0x09, 0x00, 0x0A, 0x00, 0x0B,
	0x00, 0x0C, 0x00, 0x0D, 0x00, 0x0E, 0x00, 0x0F
} ; /* au_pcm16 */

static
void	write_file (char *filename, unsigned char *data, size_t len)
{	FILE* fp = NULL ;

	fp = fopen (filename,"wb") ;
	
	if (fp == NULL)
	{	printf ("Could not open file '%s' for writing.\n", filename) ;
		exit (1) ;
		} ;
		
	if (fwrite (data, 1, len, fp) != len)
	{	printf ("Write to file '%s' failed.\n", filename) ;
		exit (1) ;
		} ;
		
	fclose (fp) ;
} /* write_file */

static
void	double_test (char *filename)
{	int		k, count ;
	SNDFILE	*file ;
	SF_INFO	sfinfo ;
	double	*readdata ;

	printf ("    double_test (%s) ... ", filename) ;
	
	if (! (file = sf_open_read (filename, &sfinfo)))
	{	printf ("Could not open file '%s' for read.\n", filename) ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
		
	if (sfinfo.samples != 16)
	{	printf ("Expecting 16 samples in file, found %d.\n", sfinfo.samples) ;
		exit (1) ;
		} ;
		
	readdata = (double*) malloc (sfinfo.samples * sizeof (double)) ;

	if ((count = sf_read_double (file, readdata, sfinfo.samples, 0)) != sfinfo.samples)
	{	printf ("sf_read_double failed (%d items read).\n", count) ;
		exit (1) ;
		} ;
	
	for (k = 0 ; k < sfinfo.samples ; k++)
		if (readdata [k] != (double) (k))
		{	printf ("Incorrect sample (#%d => %g).\n", k, readdata [k]) ;
			exit (1) ;
			} ;

	/* Check seek from start of file. */
	k = sf_seek (file, 1, SEEK_SET) ;
	sf_read_double (file, readdata, 1, 0) ;
	if (readdata [0] != 1 || k != 1)
	{	printf ("sf_seek (SEEK_SET) followed by sf_read_XXXX failed (%d, %d).\n", ((int) readdata [0]), k) ;
		exit (1) ;
		} ;
	
	/* Check seek forward from current position. */
	k = sf_seek (file, 2, SEEK_CUR) ;
	sf_read_double (file, readdata, 1, 0) ;
	if (readdata [0] != 4 || k != 4)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_XXXX failed (%d, %d).\n", ((int) readdata [0]), k) ;
		exit (1) ;
		} ;
	
	/* Check seek backward from current position. */
	k = sf_seek (file, -2, SEEK_CUR) ;
	sf_read_double (file, readdata, 1, 0) ;
	if (readdata [0] != 3 || k != 3)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_XXXX failed (%d, %d).\n", ((int) readdata [0]), k) ;
		exit (1) ;
		} ;
	
	/* Check that read past end of file returns number of items. */
	if ((k = sf_read_double (file, readdata, sfinfo.samples, 0)) != 12)
	{	printf ("Return value from short sf_read_XXXX incorrect (%d).\n", k) ;
		exit (1) ;
		} ;
	
	/* Check seek backward from end. */
	k = sf_seek (file, -14, SEEK_END) ;
	sf_read_double (file, readdata, 1, 0) ;
	if (readdata [0] != 2 || k != 2)
	{	printf ("sf_seek (SEEK_END) followed by sf_read_XXXX failed (%d, %d).\n", ((int) readdata [0]), k) ;
		exit (1) ;
		} ;
	
	/* Check multiple reads. */
	k = sf_seek (file, 0, SEEK_SET) ;
	count = 0 ;
	while ((k = sf_read_double (file, readdata, 3, 0)))
		count += k ;
	if (count != sfinfo.samples)
	{	printf ("failed multiple read test (%d, %d).\n", count, sfinfo.samples) ;
		exit (1) ;
		} ;
	
	free (readdata) ;
	sf_close (file) ;
	
	printf ("ok\n") ;

} /* double_test */


static
void	int_test (char *filename)
{	int		count, k ;
	SNDFILE	*file ;
	SF_INFO	sfinfo ;
	int		*readdata ;

	printf ("    int_test    (%s) ... ", filename) ;
	
	if (! (file = sf_open_read (filename, &sfinfo)))
	{	printf ("Could not open file '%s' for read.\n", filename) ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
		
	if (sfinfo.samples != 16)
	{	printf ("Expecting 16 samples in file, found %d.\n", sfinfo.samples) ;
		exit (1) ;
		} ;
		
	readdata = (int*) malloc (sfinfo.samples * sizeof (int)) ;

	if ((count = sf_read_int (file, readdata, sfinfo.samples)) != sfinfo.samples)
	{	printf ("sf_read_int failed (%d items read).\n", count) ;
		exit (1) ;
		} ;
	
	for (k = 0 ; k < sfinfo.samples ; k++)
		if (readdata [k] != (int) (k))
		{	printf ("Incorrect sample (#%d => %d).\n", k, readdata [k]) ;
			exit (1) ;
			} ;
	
	/* Check seek from start of file. */
	k = sf_seek (file, 1, SEEK_SET) ;
	sf_read_int (file, readdata, 1) ;
	if (readdata [0] != 1 || k != 1)
	{	printf ("sf_seek (SEEK_SET) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check seek forward from current position. */
	k = sf_seek (file, 2, SEEK_CUR) ;
	sf_read_int (file, readdata, 1) ;
	if (readdata [0] != 4 || k != 4)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check seek backward from current position. */
	k = sf_seek (file, -2, SEEK_CUR) ;
	sf_read_int (file, readdata, 1) ;
	if (readdata [0] != 3 || k != 3)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check that read past end of file returns number of items. */
	if ((k = sf_read_int (file, readdata, sfinfo.samples)) != 12)
	{	printf ("Return value from short sf_read_XXXX incorrect (%d).\n", k) ;
		exit (1) ;
		} ;
	
	/* Check seek backward from end. */
	k = sf_seek (file, -14, SEEK_END) ;
	sf_read_int (file, readdata, 1) ;
	if (readdata [0] != 2 || k != 2)
	{	printf ("sf_seek (SEEK_END) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check multiple reads. */
	k = sf_seek (file, 0, SEEK_SET) ;
	count = 0 ;
	while ((k = sf_read_int (file, readdata, 3)))
		count += k ;
	if (count != sfinfo.samples)
	{	printf ("failed multiple read test (%d, %d).\n", count, sfinfo.samples) ;
		exit (1) ;
		} ;
	
	free (readdata) ;
	sf_close (file) ;
	
	printf ("ok\n") ;

} /* int_test */

static
void	short_test (char *filename)
{	int		count, k ;
	SNDFILE	*file ;
	SF_INFO	sfinfo ;
	short	*readdata ;

	printf ("    short_test  (%s) ... ", filename) ;
	
	if (! (file = sf_open_read (filename, &sfinfo)))
	{	printf ("Could not open file '%s' for read.\n", filename) ;
		exit (1) ;
		} ;
		
	if (sfinfo.samples != 16)
	{	printf ("Expecting 16 samples in file, found %d.\n", sfinfo.samples) ;
		exit (1) ;
		} ;
		
	readdata = (short*) malloc (sfinfo.samples * sizeof (short)) ;

	if ((count = sf_read_short (file, readdata, sfinfo.samples)) != sfinfo.samples)
	{	printf ("sf_read_short failed (%d items read).\n", count) ;
		exit (1) ;
		} ;
	
	for (k = 0 ; k < sfinfo.samples ; k++)
		if (readdata [k] != (short) (k))
		{	printf ("Incorrect sample (#%d => %d).\n", k, readdata [k]) ;
			exit (1) ;
			} ;
			
	/* Check seek from start of file. */
	k = sf_seek (file, 1, SEEK_SET) ;
	sf_read_short (file, readdata, 1) ;
	if (readdata [0] != 1 || k != 1)
	{	printf ("sf_seek (SEEK_SET) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check seek forward from current position. */
	k = sf_seek (file, 2, SEEK_CUR) ;
	sf_read_short (file, readdata, 1) ;
	if (readdata [0] != 4 || k != 4)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check seek backward from current position. */
	k = sf_seek (file, -2, SEEK_CUR) ;
	sf_read_short (file, readdata, 1) ;
	if (readdata [0] != 3 || k != 3)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check that read past end of file returns number of items. */
	if ((k = sf_read_short (file, readdata, sfinfo.samples)) != 12)
	{	printf ("Return value from short sf_read_XXXX incorrect (%d).\n", k) ;
		exit (1) ;
		} ;
	
	/* Check seek backward from end. */
	k = sf_seek (file, -14, SEEK_END) ;
	sf_read_short (file, readdata, 1) ;
	if (readdata [0] != 2 || k != 2)
	{	printf ("sf_seek (SEEK_END) followed by sf_read_XXXX failed (%d, %d).\n", readdata [0], k) ;
		exit (1) ;
		} ;
	
	/* Check multiple reads. */
	k = sf_seek (file, 0, SEEK_SET) ;
	count = 0 ;
	while ((k = sf_read_short (file, readdata, 3)))
		count += k ;
	if (count != sfinfo.samples)
	{	printf ("failed multiple read test (%d, %d).\n", count, sfinfo.samples) ;
		exit (1) ;
		} ;
	
	free (readdata) ;
	sf_close (file) ;
	
	printf ("ok\n") ;

} /* short_test */

int		main (int argc, char *argv [])
{	char	*filename ;
	int		bDoAll = 0 ;
	int		nTests = 0 ;

	if (argc != 2)
	{	printf ("Usage : %s <test>\n", argv [0]) ;
		printf ("    Where <test> is one of the following:\n") ;
		printf ("           wav      - test WAV file functions (little endian)\n") ;
		printf ("           aiff     - test AIFF file functions (big endian)\n") ;
		printf ("           au       - test AU file functions (big endian)\n") ;
		printf ("           all      - perform all tests\n") ;
		exit (1) ;
		} ;
		
	bDoAll = !strcmp (argv [1], "all") ;

	if (bDoAll || ! strcmp (argv [1], "wav"))
	{	filename = "wav_pcm8.wav" ;
		write_file (filename, wav_pcm8, sizeof (wav_pcm8)) ;
		short_test	(filename) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;

		filename = "wav_pcm16.wav" ;
		write_file (filename, wav_pcm16, sizeof (wav_pcm16)) ;
		short_test	(filename) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;

		filename = "wav_pcm24.wav" ;
		write_file (filename, wav_pcm24, sizeof (wav_pcm24)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;

		filename = "wav_pcm32.wav" ;
		write_file	(filename, wav_pcm32, sizeof (wav_pcm32)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;

		filename = "wav_ieee.wav" ;
		write_file	(filename, wav_ieee, sizeof (wav_ieee)) ;
		short_test	(filename) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
		nTests++ ;
		} ;
	
	if (bDoAll || ! strcmp (argv [1], "aiff"))
	{	filename = "aiff_pcm16.aiff" ;
		write_file (filename, aiff_pcm16, sizeof (aiff_pcm16)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
		
		filename = "aiff_pcm24.aiff" ;
		write_file (filename, aiff_pcm24, sizeof (aiff_pcm24)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
	
		filename = "aiff_pcm32.aiff" ;
		write_file (filename, aiff_pcm32, sizeof (aiff_pcm32)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
		nTests++ ;
		} ;
	
	if (bDoAll || ! strcmp (argv [1], "au"))
	{	filename = "au_pcm16.au" ;
		write_file (filename, au_pcm16, sizeof (au_pcm16)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
		
		filename = "au_pcm24.au" ;
		write_file (filename, au_pcm24, sizeof (au_pcm24)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
	
		filename = "au_pcm32.au" ;
		write_file (filename, au_pcm32, sizeof (au_pcm32)) ;
		int_test	(filename) ;
		double_test (filename) ;
		unlink (filename) ;
		nTests++ ;
		} ;
	
	if (nTests==0)
	{	printf ("************************************\n") ;
		printf ("*  No '%s' test defined.\n", argv [1]) ;
		printf ("************************************\n") ;
		return 1 ;
		}

	return 0;
} /* main */

/*============================================================================================
 *	Here are the test functions.
 */ 

