/*
** Copyright (C) 1999 Erik de Castro Lopo <erikd@zip.com.au>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include	<stdio.h>
#include	<string.h>
#include	<unistd.h>
#include	<math.h>

#include	<sndfile.h>

#ifndef M_PI
        #define M_PI 3.14159
#endif

#define	BUFFER_SIZE		(1<<14)

#define		ADPCM_MAP(x) 	((int) (32000 * sin (2.0 * M_PI * ((double) x) / ((double) BUFFER_SIZE))))

static	void	adpcm_test_short	(char *str, char *filename, int typemajor, int typeminor) ;
static	void	adpcm_test_int		(char *str, char *filename, int typemajor, int typeminor) ;
static	void	adpcm_test_double	(char *str, char *filename, int typemajor, int typeminor) ;

/* Force the start of this buffer to be double aligned. Sparc-solaris will
** choke if its not.
*/
static	double	test_buffer [BUFFER_SIZE + 1] ;

int		main (int argc, char *argv[])
{	char	*filename ;
	int		bDoAll = 0 ;
	int		nTests = 0 ;

	if (argc != 2)
	{	printf ("Usage : %s <test>\n", argv [0]) ;
		printf ("    Where <test> is one of the following:\n") ;
		printf ("           wav_ima  - test WAV IAM ADPCM file functions\n") ;
		printf ("           all      - perform all tests\n") ;
		exit (1) ;
		} ;
		
	bDoAll = !strcmp (argv [1], "all") ;

	if (bDoAll || ! strcmp (argv [1], "wav_ima"))
	{	filename = "test.wav" ;
		adpcm_test_short	("wav_ima", filename, SF_FORMAT_WAV, SF_FORMAT_IMA_ADPCM) ;
		adpcm_test_int		("wav_ima", filename, SF_FORMAT_WAV, SF_FORMAT_IMA_ADPCM) ;
		adpcm_test_double	("wav_ima", filename, SF_FORMAT_WAV, SF_FORMAT_IMA_ADPCM) ;
		unlink (filename) ;
		nTests++;
		} ;

	if (nTests == 0)
	{	printf ("************************************\n") ;
		printf ("*  No '%s' test defined.\n", argv [1]) ;
		printf ("************************************\n") ;
		return 1 ;
		} ;

	return 0;
} /* main */

/*============================================================================================
 *	Here are the test functions.
 */ 
 
static	
void	adpcm_test_short (char *str, char *filename, int typemajor, int typeminor)
{	SNDFILE			*file ;
	SF_INFO			sfinfo ;
	int				k, m, seekpos ;
	unsigned int	datalen ;
	short			*data ;

	printf ("    adpcm_test_short  : %s ... ", str) ;
	
	datalen = BUFFER_SIZE ;

	data = (short*) test_buffer ;
	for (k = 0 ; k < datalen ; k++)
		data [k] = ADPCM_MAP (k) ;
		
	sfinfo.samplerate  = 44100 ;
	sfinfo.samples     = 123456789 ;	/* Ridiculous value. */
	sfinfo.channels    = 1 ;
	sfinfo.pcmbitwidth = 16 ;
	sfinfo.format 	   = (typemajor | typeminor) ;

	if (! (file = sf_open_write (filename, &sfinfo)))
	{	printf ("sf_open_write failed with error : ") ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
	
	if ((k = sf_write_short (file, data, datalen)) != datalen)
	{	printf ("sf_write_short failed with short write (%d => %d).\n", datalen, k) ;
		exit (1) ;
		} ;
	sf_close (file) ;
	
	memset (data, 0, datalen * sizeof (short)) ;
	memset (&sfinfo, 0, sizeof (sfinfo)) ;
	
	if (! (file = sf_open_read (filename, &sfinfo)))
	{	printf ("sf_open_read failed with error : ") ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
	
	if (sfinfo.format != (typemajor | typeminor))
	{	printf ("Returned format incorrect (0x%08X => 0x%08X).\n", (typemajor | typeminor), sfinfo.format) ;
		exit (1) ;
		} ;
	
	if (sfinfo.samples < datalen)
	{	printf ("Too few samples in file. (%d => %d)\n", datalen, sfinfo.samples) ;
		exit (1) ;
		} ;
	
	if (sfinfo.channels != 1)
	{	printf ("Incorrect number of channels in file.\n") ;
		exit (1) ;
		} ;

	if (sfinfo.pcmbitwidth != 16)
	{	printf ("Incorrect bit width (%d).\n", sfinfo.pcmbitwidth) ;
		exit (1) ;
		} ;

	if ((k = sf_read_short (file, data, datalen)) != datalen)
	{	printf ("short read (%d).\n", k) ;
		exit (1) ;
		} ;

	for (k = 0 ; k < datalen ; k++)
		if (abs (data [k] - ADPCM_MAP (k)) > 3)
		{	printf ("Incorrect sample (A #%d : %d should be %d).\n", k, data [k], ADPCM_MAP (k)) ;
			exit (1) ;
			} ;

	if ((k = sf_read_short (file, data, datalen)) != sfinfo.samples - datalen)
	{	printf ("Incorrect read length (%d should be %d).\n", sfinfo.samples - datalen, k) ;
		exit (1) ;
		} ;
		
	for (k = 0 ; k < sfinfo.samples - datalen ; k++)
		if (abs (data [k]) > 3)
		{	printf ("Incorrect sample (B #%d : %d != 0).\n", datalen + k, data [k]) ;
			exit (1) ;
			} ;

	/* Now test sf_seek function. */
	
	if ((k = sf_seek (file, 0, SEEK_SET)) != 0)
	{	printf ("Seek to start of file failed (%d).\n", k) ;
		exit (1) ;
		} ;

	for (m = 0 ; m < 3 ; m++)
	{	if ((k = sf_read_short (file, data, datalen/7)) != datalen / 7)
		{	printf ("Incorrect read length (%d => %d).\n", datalen / 7, k) ;
			exit (1) ;
			} ;

		for (k = 0 ; k < datalen/7 ; k++)
			if (abs (data [k] - ADPCM_MAP (k + m * (datalen / 7))) > 3)
			{	printf ("Incorrect sample (C #%d : %d => %d).\n", k + m * (datalen / 7), ADPCM_MAP (k + m * (datalen / 7)), data [k]) ;
				for (m = 0 ; m < 10 ; m++)
					printf ("%d ", data [k]) ;
				printf ("\n") ;
				exit (1) ;
				} ;
		} ;

	/* Now test sf_seek function. */
	
	seekpos = BUFFER_SIZE / 10 ;
	
	/* Check seek from start of file. */
	if ((k = sf_seek (file, seekpos, SEEK_SET)) != seekpos)
	{	printf ("Seek to start of file + %d failed (%d).\n", seekpos, k) ;
		exit (1) ;
		} ;
	if ((k = sf_read_short (file, data, 1)) != 1)
	{	printf ("sf_read_short (file, data, 1) returned %d.\n", k) ;
		exit (1) ;
		} ;
	
	if (abs (data [0] != ADPCM_MAP (seekpos)) > 3)
	{	printf ("sf_seek (SEEK_SET) followed by sf_read_short failed (%d, %d).\n", ADPCM_MAP (1), data [0]) ;
		exit (1) ;
		} ;
	
	seekpos += BUFFER_SIZE / 5 ;
	k = sf_seek (file, BUFFER_SIZE / 5, SEEK_CUR) ;
	sf_read_short (file, data, 1) ;
	if (abs (data [0] - ADPCM_MAP (seekpos)) > 4 || k != seekpos + 1)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_short failed (%d, %d) (%d, %d).\n", data [0], ADPCM_MAP (seekpos), k, seekpos) ;
		exit (1) ;
		} ;
	
	seekpos -= 20 ;
	/* Check seek backward from current position. */
	k = sf_seek (file, -20, SEEK_CUR) ;
	sf_read_short (file, data, 1) ;
	if (abs (data [0] - ADPCM_MAP (seekpos)) > 8 || k != seekpos + 2)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_short failed (%d, %d) (%d, %d).\n", data [0], ADPCM_MAP (seekpos), k, seekpos) ;
		exit (1) ;
		} ;
	
	/* Check that read past end of file returns number of items. */
	sf_seek (file, (int) datalen, SEEK_SET) ;

 	if ((k = sf_read_short (file, data, datalen)) != sfinfo.samples - datalen)
 	{	printf ("Return value from sf_read_short past end of file incorrect (%d).\n", k) ;
 		exit (1) ;
 		} ;
	
	/* Check seek backward from end. */
	k = sf_seek (file, 5 - (int) sfinfo.samples, SEEK_END) ;
	sf_read_short (file, data, 1) ;
	if (abs (data [0] != ADPCM_MAP (5)) > 3 || k != 5)
	{	printf ("sf_seek (SEEK_END) followed by sf_read_short failed (%d, %d).\n", data [0], k) ;
		exit (1) ;
		} ;

	sf_close (file) ;

	printf ("ok\n") ;
} /* adpcm_test_short */


 
static	
void	adpcm_test_int (char *str, char *filename, int typemajor, int typeminor)
{	SNDFILE			*file ;
	SF_INFO			sfinfo ;
	int				k, m, seekpos ;
	unsigned int	datalen ;
	int				*data ;

	printf ("    adpcm_test_int    : %s ... ", str) ;
	
	datalen = BUFFER_SIZE ;

	data = (int*) test_buffer ;
	for (k = 0 ; k < datalen ; k++)
		data [k] = ADPCM_MAP (k) ;
		
	sfinfo.samplerate  = 44100 ;
	sfinfo.samples     = 123456789 ;	/* Ridiculous value. */
	sfinfo.channels    = 1 ;
	sfinfo.pcmbitwidth = 16 ;
	sfinfo.format 	   = (typemajor | typeminor) ;

	if (! (file = sf_open_write (filename, &sfinfo)))
	{	printf ("sf_open_write failed with error : ") ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
	
	if ((k = sf_write_int (file, data, datalen)) != datalen)
	{	printf ("sf_write_int failed with int write (%d => %d).\n", datalen, k) ;
		exit (1) ;
		} ;
	sf_close (file) ;
	
	memset (data, 0, datalen * sizeof (int)) ;
	memset (&sfinfo, 0, sizeof (sfinfo)) ;
	
	if (! (file = sf_open_read (filename, &sfinfo)))
	{	printf ("sf_open_read failed with error : ") ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
	
	if (sfinfo.format != (typemajor | typeminor))
	{	printf ("Returned format incorrect (0x%08X => 0x%08X).\n", (typemajor | typeminor), sfinfo.format) ;
		exit (1) ;
		} ;
	
	if (sfinfo.samples < datalen)
	{	printf ("Too few samples in file. (%d => %d)\n", datalen, sfinfo.samples) ;
		exit (1) ;
		} ;
	
	if (sfinfo.channels != 1)
	{	printf ("Incorrect number of channels in file.\n") ;
		exit (1) ;
		} ;

	if (sfinfo.pcmbitwidth != 16)
	{	printf ("Incorrect bit width (%d).\n", sfinfo.pcmbitwidth) ;
		exit (1) ;
		} ;

	if ((k = sf_read_int (file, data, datalen)) != datalen)
	{	printf ("int read (%d).\n", k) ;
		exit (1) ;
		} ;

	for (k = 0 ; k < datalen ; k++)
		if (abs (data [k] - ADPCM_MAP (k)) > 3)
		{	printf ("Incorrect sample (A #%d : %d should be %d).\n", k, data [k], ADPCM_MAP (k)) ;
			exit (1) ;
			} ;

	if ((k = sf_read_int (file, data, datalen)) != sfinfo.samples - datalen)
	{	printf ("Incorrect read length (%d should be %d).\n", sfinfo.samples - datalen, k) ;
		exit (1) ;
		} ;
		
	for (k = 0 ; k < sfinfo.samples - datalen ; k++)
		if (abs (data [k]) > 3)
		{	printf ("Incorrect sample (B #%d : %d != 0).\n", datalen + k, data [k]) ;
			exit (1) ;
			} ;

	/* Now test sf_seek function. */
	
	if ((k = sf_seek (file, 0, SEEK_SET)) != 0)
	{	printf ("Seek to start of file failed (%d).\n", k) ;
		exit (1) ;
		} ;

	for (m = 0 ; m < 3 ; m++)
	{	if ((k = sf_read_int (file, data, datalen/7)) != datalen / 7)
		{	printf ("Incorrect read length (%d => %d).\n", datalen / 7, k) ;
			exit (1) ;
			} ;

		for (k = 0 ; k < datalen/7 ; k++)
			if (abs (data [k] - ADPCM_MAP (k + m * (datalen / 7))) > 3)
			{	printf ("Incorrect sample (C #%d : %d => %d).\n", k + m * (datalen / 7), ADPCM_MAP (k + m * (datalen / 7)), data [k]) ;
				for (m = 0 ; m < 10 ; m++)
					printf ("%d ", data [k]) ;
				printf ("\n") ;
				exit (1) ;
				} ;
		} ;

	/* Now test sf_seek function. */
	
	seekpos = BUFFER_SIZE / 10 ;
	
	/* Check seek from start of file. */
	if ((k = sf_seek (file, seekpos, SEEK_SET)) != seekpos)
	{	printf ("Seek to start of file + %d failed (%d).\n", seekpos, k) ;
		exit (1) ;
		} ;
	if ((k = sf_read_int (file, data, 1)) != 1)
	{	printf ("sf_read_int (file, data, 1) returned %d.\n", k) ;
		exit (1) ;
		} ;
	
	if (abs (data [0] != ADPCM_MAP (seekpos)) > 3)
	{	printf ("sf_seek (SEEK_SET) followed by sf_read_int failed (%d, %d).\n", ADPCM_MAP (1), data [0]) ;
		exit (1) ;
		} ;
	
	seekpos += BUFFER_SIZE / 5 ;
	k = sf_seek (file, BUFFER_SIZE / 5, SEEK_CUR) ;
	sf_read_int (file, data, 1) ;
	if (abs (data [0] - ADPCM_MAP (seekpos)) > 4 || k != seekpos + 1)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_int failed (%d, %d) (%d, %d).\n", data [0], ADPCM_MAP (seekpos), k, seekpos) ;
		exit (1) ;
		} ;
	
	seekpos -= 20 ;
	/* Check seek backward from current position. */
	k = sf_seek (file, -20, SEEK_CUR) ;
	sf_read_int (file, data, 1) ;
	if (abs (data [0] - ADPCM_MAP (seekpos)) > 8 || k != seekpos + 2)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_int failed (%d, %d) (%d, %d).\n", data [0], ADPCM_MAP (seekpos), k, seekpos) ;
		exit (1) ;
		} ;
	
	/* Check that read past end of file returns number of items. */
	sf_seek (file, (int) datalen, SEEK_SET) ;

 	if ((k = sf_read_int (file, data, datalen)) != sfinfo.samples - datalen)
 	{	printf ("Return value from sf_read_int past end of file incorrect (%d).\n", k) ;
 		exit (1) ;
 		} ;
	
	/* Check seek backward from end. */
	k = sf_seek (file, 5 - (int) sfinfo.samples, SEEK_END) ;
	sf_read_int (file, data, 1) ;
	if (abs (data [0] != ADPCM_MAP (5)) > 3 || k != 5)
	{	printf ("sf_seek (SEEK_END) followed by sf_read_int failed (%d, %d).\n", data [0], k) ;
		exit (1) ;
		} ;

	sf_close (file) ;

	printf ("ok\n") ;
} /* adpcm_test_int */


static	
void	adpcm_test_double (char *str, char *filename, int typemajor, int typeminor)
{	SNDFILE			*file ;
	SF_INFO			sfinfo ;
	int				k, m, seekpos ;
	unsigned int	datalen ;
	double			*data ;

	printf ("    adpcm_test_double : %s ... ", str) ;
	
	datalen = BUFFER_SIZE ;

	data = (double*) test_buffer ;
	for (k = 0 ; k < datalen ; k++)
		data [k] = ADPCM_MAP (k) ;
		
	sfinfo.samplerate  = 44100 ;
	sfinfo.samples     = 123456789 ;	/* Ridiculous value. */
	sfinfo.channels    = 1 ;
	sfinfo.pcmbitwidth = 16 ;
	sfinfo.format 	   = (typemajor | typeminor) ;

	if (! (file = sf_open_write (filename, &sfinfo)))
	{	printf ("sf_open_write failed with error : ") ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
	
	if ((k = sf_write_double (file, data, datalen, 0)) != datalen)
	{	printf ("sf_write_double failed with double write (%d => %d).\n", datalen, k) ;
		exit (1) ;
		} ;
	sf_close (file) ;
	
	memset (data, 0, datalen * sizeof (double)) ;
	memset (&sfinfo, 0, sizeof (sfinfo)) ;
	
	if (! (file = sf_open_read (filename, &sfinfo)))
	{	printf ("sf_open_read failed with error : ") ;
		sf_perror (NULL) ;
		exit (1) ;
		} ;
	
	if (sfinfo.format != (typemajor | typeminor))
	{	printf ("Returned format incorrect (0x%08X => 0x%08X).\n", (typemajor | typeminor), sfinfo.format) ;
		exit (1) ;
		} ;
	
	if (sfinfo.samples < datalen)
	{	printf ("Too few samples in file. (%d => %d)\n", datalen, sfinfo.samples) ;
		exit (1) ;
		} ;
	
	if (sfinfo.channels != 1)
	{	printf ("Incorrect number of channels in file.\n") ;
		exit (1) ;
		} ;

	if (sfinfo.pcmbitwidth != 16)
	{	printf ("Incorrect bit width (%d).\n", sfinfo.pcmbitwidth) ;
		exit (1) ;
		} ;

	if ((k = sf_read_double (file, data, datalen, 0)) != datalen)
	{	printf ("double read (%d).\n", k) ;
		exit (1) ;
		} ;

	for (k = 0 ; k < datalen ; k++)
		if (abs ((int)(data [k] - ADPCM_MAP (k))) > 3)
		{	printf ("Incorrect sample (A #%d : %d should be %d).\n", k, (int) data [k], ADPCM_MAP (k)) ;
			exit (1) ;
			} ;

	if ((k = sf_read_double (file, data, datalen, 0)) != sfinfo.samples - datalen)
	{	printf ("Incorrect read length (%d should be %d).\n", sfinfo.samples - datalen, k) ;
		exit (1) ;
		} ;
		
	for (k = 0 ; k < sfinfo.samples - datalen ; k++)
		if (abs ((int) data [k]) > 3)
		{	printf ("Incorrect sample (B #%d : %d != 0).\n", datalen + k, (int) data [k]) ;
			exit (1) ;
			} ;

	/* Now test sf_seek function. */
	
	if ((k = sf_seek (file, 0, SEEK_SET)) != 0)
	{	printf ("Seek to start of file failed (%d).\n", k) ;
		exit (1) ;
		} ;

	for (m = 0 ; m < 3 ; m++)
	{	if ((k = sf_read_double (file, data, datalen/7, 0)) != datalen / 7)
		{	printf ("Incorrect read length (%d => %d).\n", datalen / 7, k) ;
			exit (1) ;
			} ;

		for (k = 0 ; k < datalen/7 ; k++)
			if (abs ((int)(data [k] - ADPCM_MAP (k + m * (datalen / 7)))) > 3)
			{	printf ("Incorrect sample (C #%d : %d => %d).\n", k + m * (datalen / 7), ADPCM_MAP (k + m * (datalen / 7)), (int) data [k]) ;
				for (m = 0 ; m < 10 ; m++)
					printf ("%d ", (int) data [k]) ;
				printf ("\n") ;
				exit (1) ;
				} ;
		} ;

	/* Now test sf_seek function. */
	
	seekpos = BUFFER_SIZE / 10 ;
	
	/* Check seek from start of file. */
	if ((k = sf_seek (file, seekpos, SEEK_SET)) != seekpos)
	{	printf ("Seek to start of file + %d failed (%d).\n", seekpos, k) ;
		exit (1) ;
		} ;
	if ((k = sf_read_double (file, data, 1, 0)) != 1)
	{	printf ("sf_read_double (file, data, 1) returned %d.\n", k) ;
		exit (1) ;
		} ;
	
	if (abs (data [0] != ADPCM_MAP (seekpos)) > 3)
	{	printf ("sf_seek (SEEK_SET) followed by sf_read_double failed (%d, %d).\n", ADPCM_MAP (1), (int) data [0]) ;
		exit (1) ;
		} ;
	
	seekpos += BUFFER_SIZE / 5 ;
	k = sf_seek (file, BUFFER_SIZE / 5, SEEK_CUR) ;
	sf_read_double (file, data, 1, 0) ;
	if (abs ((int)(data [0] - ADPCM_MAP (seekpos))) > 4 || k != seekpos + 1)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_double failed (%d, %d) (%d, %d).\n", (int) data [0], ADPCM_MAP (seekpos), k, seekpos) ;
		exit (1) ;
		} ;
	
	seekpos -= 20 ;
	/* Check seek backward from current position. */
	k = sf_seek (file, -20, SEEK_CUR) ;
	sf_read_double (file, data, 1, 0) ;
	if (abs ((int)(data [0] - ADPCM_MAP (seekpos))) > 8 || k != seekpos + 2)
	{	printf ("sf_seek (SEEK_CUR) followed by sf_read_double failed (%d, %d) (%d, %d).\n", (int) data [0], ADPCM_MAP (seekpos), k, seekpos) ;
		exit (1) ;
		} ;
	
	/* Check that read past end of file returns number of items. */
	sf_seek (file, (int) datalen, SEEK_SET) ;

 	if ((k = sf_read_double (file, data, datalen, 0)) != sfinfo.samples - datalen)
 	{	printf ("Return value from sf_read_double past end of file incorrect (%d).\n", k) ;
 		exit (1) ;
 		} ;
	
	/* Check seek backward from end. */
	k = sf_seek (file, 5 - (int) sfinfo.samples, SEEK_END) ;
	sf_read_double (file, data, 1, 0) ;
	if (abs (data [0] != ADPCM_MAP (5)) > 3 || k != 5)
	{	printf ("sf_seek (SEEK_END) followed by sf_read_double failed (%d, %d).\n", (int) data [0], k) ;
		exit (1) ;
		} ;

	sf_close (file) ;

	printf ("ok\n") ;
} /* adpcm_test_double */



