// DSTART 
// SmIRC - an X11R6/Motif 2.0 IRC client for Linux 
//  
// Current version is 0.70 
//  
// Copyright 1997-1999, Double Precision, Inc. 
//  
// This program is distributed under the terms of the GNU General Public 
// License. See COPYING for additional information. 
//  
// DEND 
#ifndef	afxtempl_h
#define	afxtempl_h

static const char afxtempl_h_rcsid[]="$Id: afxtempl.h,v 1.20 1999/05/07 02:56:12 mrsam Exp $";

#include	"afx.h"

#define		AFXPAGESIZE	1024

#define	AFX_USE_PRIVATE_ALLOC	1

#if	AFX_USE_PRIVATE_ALLOC

////////////////////////////////////////////////////////////////////////
//
// Private allocation for internal structures is done in blocks to reduce
// memory fragmentation

#define	AFXOVERHEAD	64

#define	AFXBLOCKSIZE(T)	\
	( AFXPAGESIZE - AFXOVERHEAD ) / sizeof(T) > 3 ?			\
	( AFXPAGESIZE - AFXOVERHEAD ) / sizeof(T) :			\
		(((sizeof(T)*4 + AFXOVERHEAD ) / AFXPAGESIZE + 1)	\
			 * AFXPAGESIZE - AFXOVERHEAD) / sizeof(T)

template<class T> class CAfxAllocPool;
template<class T> class CAfxAllocUnit;
template<class T> class CAfxAllocBlock;

// When using internal allocation each unit is prefixed either by:
//   A) pointer to its block when it is allocated
//   B) pointer to the next unit in the block's freelist when it is not
//      allocated

template<class T> class CAfxAllocUnit {
public:
	union {
		CAfxAllocBlock<T>	*block;
		CAfxAllocUnit<T>	*freelist;
		} u;
	char	instance[sizeof(T)];
} ;

template<class T> class CAfxAllocBlock {
public:
	CAfxAllocBlock<T>	*prev, *next;
	CAfxAllocUnit<T>	*freelist;
	size_t			cnt;	// # of free blocks here

	CAfxAllocUnit<T>	blocks[ AFXBLOCKSIZE(CAfxAllocUnit<T>) ];
} ;

// All blocks are kept in this pool:

template<class T> class CAfxAllocPool {
public:
	CAfxAllocBlock<T> *first, *last;
	CAfxAllocPool() : first(0), last(0)	{}
	~CAfxAllocPool() { while (first)
			{
			CAfxAllocBlock<T> *p=first;

				Unlink(p);
				delete p;
			}
		}
	char *Alloc();
	void Free(const char *);
private:

	// Unlink block from pool

	void Unlink(CAfxAllocBlock<T> *p)
		{
			if (p->prev)	p->prev->next=p->next;
			else first=p->next;
			if (p->next)	p->next->prev=p->prev;
			else last=p->prev;
		}

	// All blocks with at least one free unit are kept at the beginning
	// of the list

	void LinkFirst(CAfxAllocBlock<T> *p)
		{
			p->next=first;
			p->prev=0;
			first=p;
			if (p->next)
				p->next->prev=p;
			else
				last=p;
		}

	// All blocks which are completely full are kept at the end of the list

	void LinkLast(CAfxAllocBlock<T> *p)
		{
			p->next=0;
			p->prev=last;
			last=p;
			if (p->prev)
				p->prev->next=p;
			else
				first=p;
		}

} ;

template<class T> char *CAfxAllocPool<T>::Alloc()
{
	// Since all blocks with at least one free unit are kept at the
	// beginning of the list, if the first block has nothing full,
	// we need to allocate another block

	if (!first || !first->freelist)
	{
	CAfxAllocBlock<T>	*p=new CAfxAllocBlock<T>;

		if (!p)		AfxThrowMemoryException();

		LinkFirst(p);
		p->freelist=0;
		for (p->cnt=0; p->cnt < sizeof(p->blocks)/sizeof(p->blocks[0]);
			p->cnt++)
		{
			p->blocks[p->cnt].u.freelist=p->freelist;
			p->freelist=&p->blocks[p->cnt];
		}
	}

CAfxAllocBlock<T>	*p=first;
CAfxAllocUnit<T>	*u=p->freelist;

	p->freelist=u->u.freelist;
	if (--p->cnt == 0)
	{
		Unlink(p);
		LinkLast(p);
	}
	u->u.block=p;
	return (u->instance);
}

template<class T> void CAfxAllocPool<T>::Free(const char *p)
{
	if (!p)	AfxThrowInternalException();

	// Calculate my unit pointer

CAfxAllocUnit<T>	*u= (CAfxAllocUnit<T> *) ( p -
	(int) ((CAfxAllocUnit<T> *)0)->instance);

CAfxAllocBlock<T>	*b= u->u.block;

	if (!b)	AfxThrowInternalException();	// Sanity check

	if (u < &b->blocks[0] || u >= &b->blocks[
			sizeof(b->blocks)/sizeof(b->blocks[0])] ||
		((char *)u - (char *)&b->blocks[0]) % sizeof(b->blocks[0]))
		AfxThrowInternalException();	// Sanity check

	// If this is the first unit that became free in this block, move
	// this block to the beginning of the pool list.

	if (b->cnt == 0)
	{
		Unlink(b);
		LinkFirst(b);
	}
	u->u.freelist=b->freelist;
	b->freelist=u;

	// If this block is completely free, delete it.

	if (++b->cnt == sizeof(b->blocks)/sizeof(b->blocks[0]))
	{
		Unlink(b);
		delete b;
	}
} ;

#endif

/////////////////////////////////////////////////////////////////////
//
//  Here's my Array template

template<class T, class ArgT> class CArray {

	T *aryp;
	int size;		// # of elements in the array
	int alloc;		// How much we allocated (could be >size)
	int growby;
public:
	CArray() : aryp((T *)NULL), size(0), alloc(0), growby(-1) {}
				// When growby < 0, we calculate optimum size
	virtual ~CArray() { RemoveAll(); if (aryp) delete[] ( (char *)aryp); }
	int GetSize() const { return (size); }
	int GetUpperBound() const { return (size-1); }
	void RemoveAll()
		{
			SetSize(0);
		}

	void SetSize(int newSize, int growBy= -1);
private:
	void realloc(int newAlloc);	// Resize array
public:
        T GetAt(int i) const
		{
			if (i < 0 || i >= size)	AfxThrowInternalException();
			return (aryp[i]);
		}

        void SetAt(int i, ArgT val)
		{
			if (i < 0 || i >= size)	AfxThrowInternalException();
			aryp[i]=val;
		}
        T &ElementAt(int i)
		{
			if (i < 0 || i >= size)	AfxThrowInternalException();
			return (aryp[i]);
		}

        const T *GetData() const { return (aryp); }
        T * GetData() { return (aryp); }

        void SetAtGrow(int i, ArgT val)
		{
			if (i >= size)
				SetSize(i+1, growby);
			aryp[i]=val;
		}
        void Add(ArgT val) { SetSize(size+1, growby); aryp[size-1]=val; }

        int Append(const CArray &src)
		{
		int s=size;

			SetSize(s+src.size, growby);

		int i;

			for (i=0; i<src.size; i++)
				aryp[s+i]=src.aryp[i];

			return (size);
		}

	void Copy(const CArray &src)
		{
			SetSize(src.size, growby);

		int i;

			for (i=0; i<src.size; i++)
				aryp[i]=src.aryp[i];
		}
        T operator[](int i) const { return (GetAt(i)); }
        T &operator[](int i)
		{
			if (i < 0 || i >= size)	AfxThrowInternalException();
			return (aryp[i]);
		}

        void InsertAt(int i, ArgT newVal, int n = 1)
		{
			if (n < 0 || i < 0 || i > size)	AfxThrowInternalException();
			if (n == 0)	return;
			SetSize(size+n, growby);

		int	j;

			for (j=size-n; j>i; )
			{
				--j;
				aryp[j+n]=aryp[j];
			}
			for (j=0; j<n; j++)
				aryp[j+i]=newVal;
		}

        void RemoveAt(int i, int n = 1)
		{
			if (n < 0 || i < 0 || i >= size || size-i < n)
				AfxThrowInternalException();
			if (n == 0)	return;
		int j;
			for (j=i+n; j<size; j++)
				aryp[j-n]=aryp[j];
			SetSize(size-n);
		}


        void InsertAt(int i, const CArray<T, ArgT> *src)
		{
			if (i < 0 || i > size) AfxThrowInternalException();
			SetSize(size+src->size, growby);

		int j;
			for (j=size-src->size; j > i; )
			{
				--j;
				aryp[j+src->size]=aryp[j];
			}

			for (j=0; j<src->size; j++)
				aryp[i+j]=src->aryp[j];
		}

	CArray(const CArray<T, ArgT> &src)
		: aryp((T *)NULL), size(0), alloc(0), growby(-1)
	{
		Copy(src);
	}

	CArray<T, ArgT> &operator=(const CArray<T, ArgT> &src)
	{
		Copy(src);
		return (*this);
	}
} ;

template<class T, class ArgT> inline void CArray<T, ArgT>::
	realloc(int newAlloc)	// Resize array
{
	if (growby > 0)
	{
		newAlloc += growby-1;
		newAlloc -= newAlloc % growby;
	}
	if (growby < 0)
	{
	int s=newAlloc*sizeof(T) + 32 + AFXPAGESIZE-1;

		s -= s % AFXPAGESIZE;
		s -= 32;

		newAlloc = s / sizeof(T);
	}

T *p= (T *) ::new char[newAlloc * sizeof(T)];

	if (!p)	AfxThrowMemoryException();

int i=0;

	try
	{
		while (i < size)
		{
			AFXCONSTRUCT(p+i, T(aryp[i]));
			++i;
		}
	}
	catch (...)
	{
		while (i)
		{
			--i;
			(p+i)-> ~T();
		}
		delete (char *)p;
		throw;
	}

T *oldp=aryp;

	aryp=p;

	for (i=0; i<size; i++)
		(oldp+i)->~T();

	if (oldp)
		delete (char *)oldp;
	alloc=newAlloc;
}

template<class T, class ArgT> inline void CArray<T, ArgT>::
	SetSize(int newSize, int growBy)
{
	growby=growBy;

	while (size > newSize)
		(aryp + --size) -> ~T();

	if (newSize > alloc)
	{
		realloc(newSize);
	}

	while (size < newSize)
	{
		AFXCONSTRUCT(aryp+size, T);
		++size;
	}
}

//////////////////////////////////////////////////////////////////////
//
//  Here's my List class

template<class T, class ArgT> class CListItem {
public:
	CListItem *m_next, *m_prev;
	T m_item;
	CListItem(ArgT t);
	~CListItem() {}

#if	AFX_USE_PRIVATE_ALLOC

static void destroy( CListItem<T,ArgT> *p )
	{
		p->~CListItem();
	}
#endif

} ;

template<class T, class ArgT> CListItem<T, ArgT>::
	CListItem(ArgT t) : m_item(t)
{
}

#ifndef	AFXPOSITION_DEFINED
#define	AFXPOSITION_DEFINED

typedef void *POSITION;

#endif

template<class T, class ArgT> class CList {

	CListItem<T,ArgT> *m_head, *m_tail;
	size_t m_cnt;

#if	AFX_USE_PRIVATE_ALLOC
	CAfxAllocPool<CListItem<T, ArgT> > m_alloc_pool;
#endif

public:
	CList() : m_head((CListItem<T,ArgT> *)NULL),
		m_tail((CListItem<T,ArgT> *)NULL), m_cnt(0)	{}
	~CList() { RemoveAll(); }
	CList(CList<T, ArgT> &src);		// NOT DEFINED
	CList<T, ArgT> &operator=(const CList<T, ArgT> &src)
		{
			RemoveAll();
			AddTail(&src);
			return (*this);
		}
	AFXBOOL IsEmpty() const { return (m_head == NULL); }
	size_t GetCount() const { return (m_cnt); }

	// Gimme the head or the tail

	T &GetHead() { if (!m_head) AfxThrowInternalException();
				return (m_head->m_item); }

	T GetHead() const { if (!m_head) AfxThrowInternalException();
				return (m_head->m_item); }

	T &GetTail() { if (!m_tail) AfxThrowInternalException();
				return (m_tail->m_item); }

	T GetTail() const { if (!m_tail) AfxThrowInternalException();
				return (m_tail->m_item); }

	T RemoveHead() { T save_item=GetHead(); RemoveItem(m_head);
				return (save_item); }
	T RemoveTail() { T save_item=GetTail(); RemoveItem(m_tail);
				return (save_item); }

	void RemoveAll() { while (!IsEmpty())
				RemoveItem(m_head);
			}
private:
	void RemoveItem(CListItem<T, ArgT> *p);
public:
	void     AddHead(const CList<T, ArgT> *b)
	{
	POSITION	p;

		for (p=b->GetTailPosition(); p; )
		{
		T	t(b->GetPrev(p));

			AddHead( t );
		}
	}
		
	void     AddTail(const CList<T, ArgT> *b)
	{
	POSITION	p;

		for (p=b->GetHeadPosition(); p; )
		{
		T	t(b->GetNext(p) );

			AddTail( t );
		}
	}

	POSITION AddHead(ArgT p) { return (InsertBefore(NULL, p)); }
	POSITION AddTail(ArgT p) { return (InsertAfter(NULL, p)); }

private:
	CListItem<T, ArgT> *Alloc( ArgT elem);
public:
	POSITION InsertBefore(POSITION pos, ArgT elem)
	{
	CListItem<T, ArgT> *newp=Alloc(elem);
	CListItem<T, ArgT> *nodep;

		if (!pos || !(nodep=(CListItem<T, ArgT> *)pos)->m_prev)
		{
			newp->m_prev=NULL;
			newp->m_next=m_head;
			if (m_head)	m_head->m_prev=newp;
			else		m_tail=newp;
			m_head=newp;
		}
		else
		{
			newp->m_prev=nodep->m_prev;
			newp->m_next=nodep;
			newp->m_prev->m_next=newp;
			nodep->m_prev=newp;
		}
		++m_cnt;
		return ( (POSITION)newp );
	}

	POSITION InsertAfter(POSITION pos, ArgT elem)
	{
	CListItem<T, ArgT> *newp=Alloc(elem);
	CListItem<T, ArgT> *nodep;

		if (!pos || !(nodep=(CListItem<T, ArgT> *)pos)->m_next)
		{
			newp->m_next=(CListItem<T,ArgT> *)NULL;
			newp->m_prev=m_tail;
			if (m_tail)	m_tail->m_next=newp;
			else		m_head=newp;
			m_tail=newp;
		}
		else
		{
			newp->m_next=nodep->m_next;
			newp->m_prev=nodep;
			newp->m_next->m_prev=newp;
			nodep->m_next=newp;
		}
		++m_cnt;
		return ( (POSITION)newp );
	}

	POSITION GetHeadPosition() const { return (POSITION)m_head; }
	POSITION GetTailPosition() const { return (POSITION)m_tail; }

	T &GetNext(POSITION &p) { if (!p) AfxThrowInternalException();
			CListItem<T, ArgT> *nodep=(CListItem<T, ArgT> *)p;

				p=(POSITION)nodep->m_next;
				return (nodep->m_item);
			}

	T GetNext(POSITION &p) const { if (!p) AfxThrowInternalException();
			CListItem<T, ArgT> *nodep=(CListItem<T, ArgT> *)p;

				p=(POSITION)nodep->m_next;
				return (nodep->m_item);
			}

	T &GetPrev(POSITION &p) { if (!p) AfxThrowInternalException();
			CListItem<T, ArgT> *nodep=(CListItem<T, ArgT> *)p;

				p=(POSITION)nodep->m_prev;
				return (nodep->m_item);
			}

	T GetPrev(POSITION &p) const { if (!p) AfxThrowInternalException();
			CListItem<T, ArgT> *nodep=(CListItem<T, ArgT> *)p;

				p=(POSITION)nodep->m_prev;
				return (nodep->m_item);
			}
	POSITION FindIndex(int n) const {
		POSITION p=GetHeadPosition();

			for (;;)
			{
				if (!p)	AfxThrowInternalException();
				if (!n)	break;
				p=(POSITION)
					( ((CListItem<T, ArgT> *)p)->m_next );
				--n;
			}
			return (p);
		}

	T &GetAt(POSITION p)
		{
			if (!p)	AfxThrowInternalException();
			return (((CListItem<T, ArgT> *)p)->m_item);
		}

	T GetAt(POSITION p) const
		{
			if (!p)	AfxThrowInternalException();
			return (((CListItem<T, ArgT> *)p)->m_item);
		}

	T &SetAt(POSITION p, ArgT t)
		{
			if (!p)	AfxThrowInternalException();
			((CListItem<T, ArgT> *)p)->m_item=t;
		}
	void RemoveAt(POSITION p)
		{
			if (!p)	AfxThrowInternalException();
			RemoveItem((CListItem<T, ArgT> *)p);
		}
} ;

template<class T, class ArgT> void CList<T, ArgT>::
	RemoveItem(CListItem<T, ArgT> *p)
{
	if (p->m_next)	p->m_next->m_prev=p->m_prev;
	else	m_tail=p->m_prev;

	if (p->m_prev)	p->m_prev->m_next=p->m_next;
	else	m_head=p->m_next;
	--m_cnt;

#if	AFX_USE_PRIVATE_ALLOC
	try
	{
		p->destroy(p);
	}
	catch (...)
	{
		m_alloc_pool.Free((char *)p);
		throw;
	}
	m_alloc_pool.Free((char *)p);
#else
	delete p;
#endif
}

template<class T, class ArgT> CListItem<T, ArgT> *
	CList<T, ArgT>::Alloc( ArgT elem)
{
CListItem<T, ArgT> *newp;

#if	AFX_USE_PRIVATE_ALLOC

char *allocp= m_alloc_pool.Alloc();

	if (!allocp)	AfxThrowMemoryException();

	try
	{
		newp=AFXCONSTRUCT(allocp, CListItem<T AFXCOMMA ArgT>(elem) );
	}
	catch(...)
	{
		m_alloc_pool.Free(allocp);
		throw;
	}
#else
	newp=new CListItem<T, ArgT>(elem);
	if (!newp)	AfxThrowInternalException();
#endif
	return (newp);
}

///////////////////////////////////////////////////////////////////////

typedef CArray<CString, const CString> CStringArray;
typedef CArray<unsigned int, unsigned int> CUIntArray;

typedef CList<CString, const CString> CStringList;
typedef CList<unsigned int, unsigned int> CUIntList;

// EXTRA!!!

CStringArray CStringTok(CString, CString, AFXBOOL smart=FALSE);
		// strtok(), if smart, understands quotes
CStringArray CStringSplit(CString, char fld);
CString CStringPaste(CStringArray a, char fld);

/////////////////////////////////////////////////////////////////////////
//
//  Here's my CMap class.
//
//  My CMap uses a balanced binary tree, instead of a hash table, by
//  you-know-who.  It is pretty clear that my approach does much, much,
//  better on large maps, since it will take O(log n), EVEN with consecutive
//  keys.  It may seem that I have a large overhead, and I will not be
//  efficient with small maps.  However, take notice that I DO NOT USE
//  a hash function, which saves a lot of time.
//
//  My only requirements are that the key and the value classes have
//  a copy constructor, and have operator< operator> and operator== defined.
//  None of these methods are allowed to throw an exception, otherwise I will
//  end up with a corrupted map.  I don't _think_ that I need an operator=,
//  give it a try.
//
//  This is basically a straight lift from the algorythm described by
//  Adelson-Velsky and Landis in "Doklad Akademiy Nauk SSSR 146", (USSR, 1962).
//  Noteworthy to my implementation is that I eliminated tail recursion
//  on the Insert function, by using a parent pointer in each node.
//  The addition of a parent pointer to the algorythm complicates things a
//  bit, especially for the Delete operation, when I need to swap a leaf with
//  a non-leaf successor.  I do this with two dummy nodes.  The two dummy nodes
//  are substituted for the original nodes, then the original nodes are
//  substituted back into each other's position.  That's the only way to
//  maintain the linkage correctly.
//
//  The algorythm is too arcane to be commented in-code, you are directed
//  to the original source for information.

template<class Key, class ArgKey, class Val, class ArgVal>
	class CMapNode {
public:
	CMapNode<Key, ArgKey, Val, ArgVal> *m_parent, *m_left, *m_right;

	POSITION m_pos;		// In CMap::m_nodes;

	Key	m_key;
	Val	m_val;
	short	m_balance;	// I use -1, 0, 1 to indicate left-tall, even
				// and right-tall subtrees.

	CMapNode()
		: m_parent(NULL), m_left(NULL), m_right(NULL), m_balance(0)
		{
		}

	CMapNode(ArgKey key, ArgVal val)
		: m_parent(NULL), m_left(NULL), m_right(NULL), m_key(key), m_val(val), m_balance(0)
		{
		}
	CMapNode(ArgKey key) : m_parent(NULL), m_left(NULL), m_right(NULL), m_key(key), m_balance(0)
		{
		}
	~CMapNode()
		{
		}
	CMapNode(const CMapNode<Key, ArgKey, Val, ArgVal> &p);
	void operator=(const CMapNode<Key, ArgKey, Val, ArgVal> &p);


#if	AFX_USE_PRIVATE_ALLOC

static void destroy( CMapNode<Key, ArgKey, Val, ArgVal> *p)
	{
		p->~CMapNode();
	}
#endif

} ;

template<class Key, class ArgKey, class Val, class ArgVal> class CMap {

	CMapNode<Key, ArgKey, Val, ArgVal> *m_root;

	// Locate the pointer that points to this node.

	CMapNode<Key, ArgKey, Val, ArgVal> **ParentPtr(
			CMapNode<Key, ArgKey, Val, ArgVal> *p)
	{
		return ( !p->m_parent ? &m_root:
			p->m_parent->m_right &&
				p->m_parent->m_right->m_key == p->m_key
			? &p->m_parent->m_right:&p->m_parent->m_left);
	}

	// In order to be able to quickly delete a large hash table, I keep
	// a list of all allocated nodes in here.

	CList<
		CMapNode<Key, ArgKey, Val, ArgVal> *,
		CMapNode<Key, ArgKey, Val, ArgVal> *> m_nodes;

#if	AFX_USE_PRIVATE_ALLOC

	CAfxAllocPool<CMapNode<Key, ArgKey, Val, ArgVal> > m_alloc_pool;

#endif

public:
	CMap() : m_root(NULL)	{}
	~CMap()				{ RemoveAll(); }

	size_t	GetCount() const { return (m_nodes.GetCount()); }

	AFXBOOL	IsEmpty() const
	{
		return (GetCount() == 0);
	}

public:
	AFXBOOL	Lookup(ArgKey key, Val &value) const
		{
		CMapNode<Key, ArgKey, Val, ArgVal> *root;

			for (root= m_root; root; )
			{
				if ( root->m_key > key)
					root=root->m_left;
				else if ( root->m_key < key)
					root=root->m_right;
				else
					break;
			}

			if (!root)
				return (FALSE);
			value= root->m_val;
			return (TRUE);
		}

	POSITION GetStartPosition() const;
	void GetNextAssoc(POSITION &p, Key &key, Val &value) const;
	Val &operator[](ArgKey key);
	void SetAt(ArgKey key, ArgVal val);
	AFXBOOL RemoveKey(ArgKey key);
	void RemoveAll();

	// EXTRA!!!
	POSITION GetStartPositionSorted() const;
	void GetNextAssocSorted(POSITION &p, Key &key, Val &value) const;

//
//  Validate is a sanity check.  This should be considered a debugging function,
//  and not used in production code, since it uses recursion.
//
	void	Validate() { Validate(m_root); }

	size_t	Validate(CMapNode<Key, ArgKey, Val, ArgVal> *p)
	{
		if (!p)	return (0);

		if (p->m_left)
			if (p->m_left->m_key >= p->m_key ||
				!p->m_left->m_parent ||
				p->m_left->m_parent->m_key != p->m_key)
			{
				AfxThrowInternalException();
			}

		if (p->m_right)
			if (p->m_right->m_key <= p->m_key ||
				!p->m_right->m_parent ||
				p->m_right->m_parent->m_key != p->m_key)
			{
				AfxThrowInternalException();
			}

	size_t	l=Validate(p->m_left), r=Validate(p->m_right);

		if (p->m_balance < -1 || p->m_balance > 1 ||
			l + p->m_balance != r)
		{
			AfxThrowInternalException();
		}

		return 1+(l > r ? l:r);
	}
private:
	void	DeleteNode( CMapNode<Key, ArgKey, Val, ArgVal> *p)
	{
#if	AFX_USE_PRIVATE_ALLOC

		try
		{
			p->destroy(p);
		}
		catch (...)
		{
			m_alloc_pool.Free( (const char *)p);
			throw;
		}
		m_alloc_pool.Free( (char *)p);
#else
		delete p;
#endif
	}

	// Here's the beef:

	CMapNode<Key, ArgKey, Val, ArgVal> **Find(ArgKey key,
		CMapNode<Key, ArgKey, Val, ArgVal> *&parent);
	void	Insert(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	LeftBalance(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	RightBalance(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	LeftBalanceDeleted(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	RightBalanceDeleted(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	RotateLeft(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	RotateRight(CMapNode<Key, ArgKey, Val, ArgVal> **);
	void	RemoveSwap(
			CMapNode<Key, ArgKey, Val, ArgVal> *,
			CMapNode<Key, ArgKey, Val, ArgVal> *);
	void	Substitute(
			CMapNode<Key, ArgKey, Val, ArgVal> *,
			CMapNode<Key, ArgKey, Val, ArgVal> *);
	CMap(const CMap<Key, ArgKey, Val, ArgVal> &);	// Undefined
	void operator=(const CMap<Key, ArgKey, Val, ArgVal> &);
							// Undefined
} ;

// Common function to search the binary tree.

template<class Key, class ArgKey, class Val, class ArgVal>
	CMapNode<Key, ArgKey, Val, ArgVal>
		** CMap<Key, ArgKey, Val, ArgVal>::Find(ArgKey key,
			CMapNode<Key, ArgKey, Val, ArgVal> *&parent)
{
CMapNode<Key, ArgKey, Val, ArgVal> **root;

	parent=NULL;
	for (root= &m_root; *root; )
	{
		if ( (*root)->m_key > key)
		{
			parent= *root;
			root= &parent->m_left;
		}
		else if ( (*root)->m_key < key)
		{
			parent= *root;
			root= &parent->m_right;
		}
		else
			break;
	}
	return (root);
}

// Two functions to either add or update the map

template<class Key, class ArgKey, class Val, class ArgVal>
	Val &CMap<Key, ArgKey, Val, ArgVal>::operator[](ArgKey key)
{
#if	AFX_USE_PRIVATE_ALLOC
char	*p=0;
#endif
CMapNode<Key, ArgKey, Val, ArgVal> *parent;
CMapNode<Key, ArgKey, Val, ArgVal> **root=Find(key, parent);
CMapNode<Key, ArgKey, Val, ArgVal> *pp;

	if (!*root)
	{
#if	AFX_USE_PRIVATE_ALLOC
		p=m_alloc_pool.Alloc();
		if (!p)	AfxThrowMemoryException();
		try
		{
			(*root)=AFXCONSTRUCT(p,
				CMapNode<Key AFXCOMMA ArgKey AFXCOMMA
					Val AFXCOMMA ArgVal>(key) );
		}
		catch (...)
		{
			m_alloc_pool.Free(p);
			throw;
		}
#else

		(*root)=new CMapNode<Key, ArgKey, Val, ArgVal>(key);
		if (!*root)	AfxThrowMemoryException();
#endif
		try
		{
			(*root)->m_pos=m_nodes.AddTail( *root );
		}
		catch (...)	// Can only be a memory exception
		{
		CMapNode<Key, ArgKey, Val, ArgVal> *ppp= *root;

			*root=NULL;
			DeleteNode (ppp);
			throw;
		}

		(*root)->m_parent = parent;
		pp= *root;
		Insert(root);
	}
	else
	{
		pp= *root;
	}

	return (pp->m_val);
}

template<class Key, class ArgKey, class Val, class ArgVal>
void CMap<Key, ArgKey, Val, ArgVal>::SetAt(ArgKey key, ArgVal val)
{
CMapNode<Key, ArgKey, Val, ArgVal> *parent;
CMapNode<Key, ArgKey, Val, ArgVal> **root=Find(key, parent);

	if (*root)
	{
		(*root)->m_val=val;
		return;
	}

#if	AFX_USE_PRIVATE_ALLOC

	{
	char *p=m_alloc_pool.Alloc();

		if (!p)	AfxThrowMemoryException();
		try
		{
			(*root)=AFXCONSTRUCT(p,
				CMapNode<Key AFXCOMMA ArgKey AFXCOMMA
					Val AFXCOMMA ArgVal>(key, val) );
		}
		catch (...)
		{
			m_alloc_pool.Free(p);
			throw;
		}
	}
#else

	(*root)=new CMapNode<Key, ArgKey, Val, ArgVal>(key, val);
	if (!*root)	AfxThrowMemoryException();
#endif

	try
	{
		(*root)->m_pos=m_nodes.AddTail( *root );
	}
	catch (...)	// Can only be a memory exception
	{
	CMapNode<Key, ArgKey, Val, ArgVal> *p= *root;

		*root=NULL;
		DeleteNode (p);
		throw;
	}

	(*root)->m_parent = parent;

	Insert(root);
}

// When we get to Insert(), the new leaf node has already been
// added to the tree.  Update and rebalance the binary tree.

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		Insert(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
	while ( (*root)->m_parent)
	{
		if ( (*root)->m_parent->m_key < (*root)->m_key )
			++(*root)->m_parent->m_balance;
		else
			--(*root)->m_parent->m_balance;

	CMapNode<Key, ArgKey, Val, ArgVal> *p= (*root)->m_parent;

		root= ParentPtr(p);

		if (!*root || (*root)->m_key != p->m_key)
			AfxThrowInternalException();

		if (p->m_balance < -1)
		{
			LeftBalance(root);
			return;
		}

		if (p->m_balance > 1)
		{
			RightBalance(root);
			return;
		}

		if (p->m_balance == 0)	break;
	}
}

// Left/Right rotation

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		RotateLeft(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
CMapNode<Key, ArgKey, Val, ArgVal> *temp, *rootparent= (*root)->m_parent;

	temp= (*root)->m_right;
	(*root)->m_right=temp->m_left;
	temp->m_left= *root;
	*root=temp;

	(*root)->m_parent=rootparent;
	temp= (*root)->m_left;
	temp->m_parent= *root;
	if (temp->m_right)	temp->m_right->m_parent=temp;
}

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		RotateRight(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
CMapNode<Key, ArgKey, Val, ArgVal> *temp, *rootparent= (*root)->m_parent;

	temp= (*root)->m_left;
	(*root)->m_left=temp->m_right;
	temp->m_right= *root;
	*root=temp;

	(*root)->m_parent=rootparent;
	temp= (*root)->m_right;
	temp->m_parent= *root;
	if (temp->m_left)	temp->m_left->m_parent=temp;
}

// Rebalance long right/left subtree after addition

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		RightBalance(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
CMapNode<Key, ArgKey, Val, ArgVal> *x, *w;

	x= (*root)->m_right;
	if (x->m_balance > 0)
	{
		(*root)->m_balance=0;
		x->m_balance=0;
		RotateLeft(root);
		return;
	}
	if (x->m_balance < 0)
	{
		w=x->m_left;
		switch (w->m_balance)	{
		case 0:
			(*root)->m_balance=0;
			x->m_balance=0;
			break;
		case -1:
			(*root)->m_balance=0;
			x->m_balance=1;
			break;
		case 1:
			(*root)->m_balance= -1;
			x->m_balance=0;
			break;
		}
		w->m_balance=0;
		RotateRight(&x);
		(*root)->m_right=x;
		RotateLeft(root);
		return;
	}
	AfxThrowInternalException();
}

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		LeftBalance(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
CMapNode<Key, ArgKey, Val, ArgVal> *x, *w;

	x= (*root)->m_left;
	if (x->m_balance < 0)
	{
		(*root)->m_balance=0;
		x->m_balance=0;
		RotateRight(root);
		return;
	}
	if (x->m_balance > 0)
	{
		w=x->m_right;
		switch (w->m_balance)	{
		case 0:
			(*root)->m_balance=0;
			x->m_balance=0;
			break;
		case 1:
			(*root)->m_balance=0;
			x->m_balance= -1;
			break;
		case -1:
			(*root)->m_balance= 1;
			x->m_balance=0;
			break;
		}
		w->m_balance=0;
		RotateLeft(&x);
		(*root)->m_left=x;
		RotateRight(root);
		return;
	}
	AfxThrowInternalException();
}

template<class Key, class ArgKey, class Val, class ArgVal>
	AFXBOOL CMap<Key, ArgKey, Val, ArgVal>::RemoveKey(ArgKey key)
{
CMapNode<Key, ArgKey, Val, ArgVal> *parent;
CMapNode<Key, ArgKey, Val, ArgVal> **root=Find(key, parent);

	if (!*root)	return (FALSE);

	if ( (*root)->m_left && (*root)->m_right )
	{
		// Swap the node for its immediate successor, then delete
		// the successor instead.

	CMapNode<Key, ArgKey, Val, ArgVal> *oldparent= *root;

		parent= *root;
		root= &parent->m_left;
		while ( (*root)->m_right)
		{
			parent= *root;
			root= &parent->m_right;
		}

	CMapNode<Key, ArgKey, Val, ArgVal> *newparent= *root;
		RemoveSwap(oldparent, newparent);

		parent=oldparent->m_parent;
		root=ParentPtr(oldparent);

		if (!*root || (*root)->m_key != oldparent->m_key)
			AfxThrowInternalException();
	}

CMapNode<Key, ArgKey, Val, ArgVal> *delnode= *root;
short	balchange;

	if (parent && parent->m_right
		&& parent->m_right->m_key == delnode->m_key)
		balchange=-1;
	else
		balchange=1;

	*root= delnode->m_left ? delnode->m_left:delnode->m_right;
	if (*root)
		(*root)->m_parent=parent;

	m_nodes.RemoveAt(delnode->m_pos);

	DeleteNode(delnode);

	while (parent)
	{
		parent->m_balance += balchange;

		if (parent->m_parent == NULL)
		{
			parent=NULL;
			root= &m_root;
			balchange=0;
		}
		else if (parent->m_parent->m_key < parent->m_key)
		{
			parent=parent->m_parent;
			root= &parent->m_right;
			balchange= -1;
		}
		else
		{
			parent=parent->m_parent;
			root= &parent->m_left;
			balchange= 1;
		}

		if ((*root)->m_balance < -1)
		{
		short	doContinue= (*root)->m_left->m_balance;

			LeftBalanceDeleted(root);
			if (doContinue)
				continue;
			break;
		}

		if ((*root)->m_balance > 1)
		{
		short	doContinue= (*root)->m_right->m_balance;

			RightBalanceDeleted(root);
			if (doContinue)
				continue;
			break;
		}

		if ((*root)->m_balance)	break;
	}
}

// Remove all nodes.  Don't individually delete each node one by node with
// RemoveKey, since the rebalancing will take too long.
// Instead, just delete nodes from their link list.

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::RemoveAll()
{
POSITION p;

	for (p=m_nodes.GetHeadPosition(); p; )
	{
	POSITION save_pos=p;

		DeleteNode (m_nodes.GetNext(p));
		m_nodes.RemoveAt(save_pos);
	}
	m_root=NULL;
}

// Rebalance long right/left subtree after deletion

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		RightBalanceDeleted(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
CMapNode<Key, ArgKey, Val, ArgVal> *x, *w;

	x= (*root)->m_right;
	if ( x->m_balance > 0)
	{
		(*root)->m_balance=0;
		x->m_balance=0;
		RotateLeft(root);
		return;
	}

	if ( x->m_balance < 0)
	{
		w=x->m_left;
		switch (w->m_balance)	{
		case 0:
			(*root)->m_balance=0;
			x->m_balance=0;
			break;
		case -1:
			(*root)->m_balance=0;
			x->m_balance=1;
			break;
		case 1:
			(*root)->m_balance= -1;
			x->m_balance=0;
			break;
		}
		w->m_balance=0;
		RotateRight(&x);
		(*root)->m_right=x;
		RotateLeft(root);
		return;
	}

	(*root)->m_balance= 1;
	x->m_balance= -1;
	RotateLeft(root);
}

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::
		LeftBalanceDeleted(CMapNode<Key, ArgKey, Val, ArgVal> **root)
{
CMapNode<Key, ArgKey, Val, ArgVal> *x, *w;

	x= (*root)->m_left;
	if ( x->m_balance < 0)
	{
		(*root)->m_balance=0;
		x->m_balance=0;
		RotateRight(root);
		return;
	}

	if ( x->m_balance > 0)
	{
		w=x->m_right;
		switch (w->m_balance)	{
		case 0:
			(*root)->m_balance=0;
			x->m_balance=0;
			break;
		case 1:
			(*root)->m_balance=0;
			x->m_balance=-1;
			break;
		case -1:
			(*root)->m_balance= 1;
			x->m_balance=0;
			break;
		}
		w->m_balance=0;
		RotateLeft(&x);
		(*root)->m_left=x;
		RotateRight(root);
		return;
	}

	(*root)->m_balance= -1;
	x->m_balance=1;
	RotateRight(root);
}

// RemoveSwap is called when a non-leaf node is to be deleted.  This node,
// and its immediate successor, which is always a leaf node, are exchanged
// places, so we'll always end up with a leaf node to be deleted.  For a
// short time the tree is invalid, but that is quickly remedied.  Here,
// we just want to swap position.  There's some tricky linkage here, and
// the only way to make this kosher is to first substitute two dummy nodes
// for the two nodes we are swapping, then substitute the two nodes back into
// the tree, for the two dummies, in the swapped positions.

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>:: RemoveSwap(
			CMapNode<Key, ArgKey, Val, ArgVal> *a,
			CMapNode<Key, ArgKey, Val, ArgVal> *b)
{
CMapNode<Key, ArgKey, Val, ArgVal> asub(a->m_key, a->m_val),
	bsub(b->m_key, b->m_val);

	Substitute(a, &asub);
	Substitute(b, &bsub);
	Substitute(&bsub, a);
	Substitute(&asub, b);

short	bal=a->m_balance;

	a->m_balance=b->m_balance;
	b->m_balance=bal;
}

template<class Key, class ArgKey, class Val, class ArgVal>
	void CMap<Key, ArgKey, Val, ArgVal>::Substitute(
			CMapNode<Key, ArgKey, Val, ArgVal> *node,
			CMapNode<Key, ArgKey, Val, ArgVal> *sub)
{
	*ParentPtr(node)=sub;
	sub->m_parent=node->m_parent;
	if ((sub->m_left=node->m_left) != NULL)
		sub->m_left->m_parent=sub;

	if ((sub->m_right=node->m_right) != NULL)
		sub->m_right->m_parent=sub;
}

template<class Key, class ArgKey, class Val, class ArgVal>
POSITION CMap<Key, ArgKey, Val, ArgVal>::GetStartPosition() const
{
	return (m_nodes.GetHeadPosition());
}

template<class Key, class ArgKey, class Val, class ArgVal>
POSITION CMap<Key, ArgKey, Val, ArgVal>::GetStartPositionSorted() const
{
CMapNode<Key, ArgKey, Val, ArgVal> *p=m_root;

	while (p && p->m_left)	p=p->m_left;
	return (p ? (POSITION)p:(POSITION)(NULL) );
}

template<class Key, class ArgKey, class Val, class ArgVal>
void CMap<Key, ArgKey, Val, ArgVal>::
	GetNextAssoc(POSITION &pos, Key &key, Val &value) const
{
	if (!pos)	AfxThrowInternalException();

CMapNode<Key, ArgKey, Val, ArgVal> *p=m_nodes.GetNext(pos);

	key=p->m_key;
	value=p->m_val;
}

template<class Key, class ArgKey, class Val, class ArgVal>
void CMap<Key, ArgKey, Val, ArgVal>::
	GetNextAssocSorted(POSITION &pos, Key &key, Val &value) const
{
	if (!pos)	AfxThrowInternalException();

CMapNode<Key, ArgKey, Val, ArgVal> *p=(CMapNode<Key, ArgKey, Val, ArgVal> *)pos;

	key=p->m_key;
	value=p->m_val;

	if (p->m_right)
	{
		p=p->m_right;
		while (p->m_left)	p=p->m_left;
		pos=(POSITION)p;
		return;
	}

CMapNode<Key, ArgKey, Val, ArgVal> *parent=p;
	do
	{
		p=parent;
		parent=p->m_parent;

		if (!parent)	break;

	} while (parent->m_key < p->m_key);

	pos=(POSITION)parent;
}

typedef CMap<CString, CString, CString, CString> CMapStringToString;

/////////////////////////////////////////////////////////////////
//
// Template used to have member objects of a class that are dynamically
// allocated, and automatically deleted when the instance of an object
// is destroyed.

template <class T> class CAfxAutoObj {

	T	*m_p;
public:
	CAfxAutoObj() : m_p(0)	{}
	~CAfxAutoObj();
	T *Init(T *p) { return (m_p=p); }
	T *Ptr() { return (m_p); }
	T *operator ->() {
#ifdef	DEBUG
			if (!m_p)	AfxThrowInternalException();
#endif
			return (Ptr());
		}
	operator T *() { return (Ptr()); }
			// So we can use CAfxAutoObj-> member notation
} ;

template <class T> CAfxAutoObj<T>::~CAfxAutoObj()
{
	if (m_p)	delete m_p;
}

#endif
