#!/usr/bin/guile -s
scirc, an IRC Client in Guile Scheme
Copyright (C) 1998 forcer <forcer@mindless.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
!#

(define SCIRC-VERSION "0.40")
(define CHANNEL-ID (list #\# #\& #\+ #\!))

(display (string-append
"scirc v"  SCIRC-VERSION " by forcer <forcer@mindless.com>
scirc comes with ABSOLUTELY NO WARRANTY; for details see file COPYING
This is free software, and you are welcome to redistribute it
under certain conditions; see the file COPYING for details"))
(newline) (newline)

;;; The basic features
(define (channel? c)
	(cond
		((= (string-length c) 0)
			#f)
		((memv (string-ref c 0) CHANNEL-ID)
			#t)
		(#t
			#f)))

(define CURRENT-CHANNEL-VAR "")
(define (set-cur-chan! newchan)
	(cond
		((string? newchan)
			(set! CURRENT-CHANNEL-VAR newchan)
			(set-input-prompt (current-target) "> ")
			(do-status))))
(define (current-channel)
	CURRENT-CHANNEL-VAR)

(define CURRENT-QUERY-VAR "")
(define (set-cur-query! newquery)
	(cond
		((string? newquery)
			(set! CURRENT-QUERY-VAR newquery)
			(set-input-prompt (current-target) "> ")
			(do-status))))
(define (current-query)
	CURRENT-QUERY-VAR)

(define (current-target)
	(let ((query (current-query)))
		(if (= (string-length query) 0)
			(current-channel)
			query)))

(define (set-cur-nick! newnick)
	(if (string? newnick)
		(set-var! 'NICK newnick)))
(define (current-nick)
	(get-var 'NICK))

;;; setvar - SETable variables
;; (create-var name value)	=> ( name . value )
;; (rm-var     name)       	=> [ old-value | #f ]
;; (set-var    name value)	=> [ old-value | #f ]
;; (get-var    name)		=> [ value | #f ]

(define SET-LIST '())

(define (create-var name value)
	(let creat-loop ((lis SET-LIST))
		(cond
			((null? lis)
				(set! SET-LIST (cons (cons name value) SET-LIST))
				value)
			((eqv? name (caar lis))
				#f)
			(#t
				(creat-loop (cdr lis))))))

(define (rm-var name)
	(let rm-loop ((prev '())
				  (cur  SET-LIST))
		(cond
			((null? cur)
				#f)
			((eqv? name (caar cur))
				(set! SET-LIST (append prev (cdr cur)))
				(cdar cur))
			(#t
				(rm-loop (cons (car cur) prev) (cdr cur))))))

(define (set-var! name value)
	(let set-loop ((lis SET-LIST))
		(cond
			((null? lis)
				#f)
			((eqv? name (caar lis))
				(let ((old-value (cdar lis)))
				(set-cdr! (car lis) value)
				old-value))
			(#t
				(set-loop (cdr lis))))))

(define (get-var name)
	(let get-loop ((lis SET-LIST))
		(cond
			((null? lis)
				#f)
			((eqv? name (caar lis))
				(cdar lis))
			(#t
				(get-loop (cdr lis))))))

(define (disp-var)
	(let disp-var-loop ((lis SET-LIST))
		(cond
			((not (null? lis))
				(write-to #f "*** Current value of " (caar lis)
					" is " (cdar lis))
				(disp-var-loop (cdr lis)))
			(#t
				(write-to #f "*** * End of /SET list")))))

;; Text marking:
(define (mark-it mark-char stuff)
	(let ((mark (make-string 1 mark-char)))
		(if (null? stuff)
			mark
			(string-append mark (apply string-append stuff) mark))))
(define (underline . stuff)
	(mark-it #\037 stuff))
(define (bold . stuff)
	(mark-it #\002 stuff))
(define (inverse . stuff)
	(mark-it #\026 stuff))
(define (plain . stuff)
	(mark-it #\017 stuff))
(define (as-ctcp . stuff)
	(mark-it #\001 stuff))

;;; String handling
(define (skip-space string)
	(cond
		((= (string-length string) 0)
			"")
		((not (eqv? #\space (string-ref string 0)))
			string)
		(#t
			(skip-space (substring string 1)))))

(define (strip-string string char)
	(let strip-string-loop ((done "")
							(todo string))
		(cond
			((= (string-length todo) 0)
				done)
			((eqv? (string-ref todo 0) char)
				(strip-string-loop done (substring todo 1)))
			(#t
				(strip-string-loop
					(string-append done (substring todo 0 1))
					(substring todo 1))))))

(define (strip-colon string)
	(if (and (> (string-length string) 0) (eqv? (string-ref string 0) #\:))
		(substring string 1)
		string))

(define (nextword string . divider)
	(let nextword-loop ((sofar "")
						(str   (skip-space string))
						(divi  (if (null? divider) #\space (car divider))))
		(if (= (string-length str) 0)
			(cons sofar "")
			(let ((s (substring str 0 1)))
				(if (eqv? (string-ref s 0) divi)
					(cons sofar (substring str 1))
					(nextword-loop (string-append sofar s)
						(substring str 1)
						divi))))))

(define (string-upcase string)
	(let string-upcase-loop ((todo string)
							 (done ""))
		(if (= 0 (string-length todo))
			done
			(string-upcase-loop (substring todo 1)
				(string-append done (make-string 1 (char-upcase
					(string-ref todo 0))))))))

(define (string-upcase-one-word string)
	(let ((c (nextword string)))
		(string-append (string-upcase (car c)) " " (cdr c))))

(define (split-serverline line)
	(let split-serv-loop ((str  line) (from '()) (cmd  '())
						  (to   '())  (args '()))
		(let ((w (nextword str)))
			(cond
				((or (= (string-length (car w)) 0) (not (null? args)))
					(list
						(if (null? from) "" from) (if (null? cmd)  "" cmd)
						(if (null? to)   "" to)   (if (null? args) "" args)))
				((null? from)
					(if (eqv? #\: (string-ref (car w) 0))
						(split-serv-loop (cdr w)
							(substring (car w) 1) '() '() '())
						(split-serv-loop (cdr w) "" (car w) '() '())))
				((null? cmd)
					(split-serv-loop (cdr w) from (car w) '() '()))
				((null? to)
					(if (eqv? #\: (string-ref (car w) 0))
						(list from cmd "" (substring str 1))
						(split-serv-loop (cdr w) from cmd (car w) '())))
				(#t
					(if (eqv? #\: (string-ref str 0))
						(list from cmd to (substring str 1))
						(list from cmd to str)))))))

(define (split-nuh nuh)
	(nextword nuh #\!))
(define (nuh->n nuh)
	(car (split-nuh nuh)))
(define (nuh->u@h nuh)
	(cdr (split-nuh nuh)))
(define (nice-from nuh)
	(let ((nuhcons (split-nuh nuh)))
		(string-append (car nuhcons)
			(if (= (string-length (cdr nuhcons)) 0)
				""
				(string-append " (" (cdr nuhcons) ")")))))

(define (xsplat string)
	(let ((s (nextword string)))
		(if (string-ci=? (car s) "*")
			(string-append (current-channel) " " (cdr s))
			string)))

(define (addcolon string position)
	(let addcolon-loop ((todo	string)
						(before "")
						(pos	position))
		(cond
			((= pos 0)
				(string-append before ":" todo))
			(#t
				(let ((s (nextword todo)))
					(addcolon-loop (cdr s) (string-append before (car s) " ")
						(- pos 1)))))))

;;; Path handling
;; expands ~ in paths
(define (xpath path)
	(define (getuserhome nameoruid)
		(array-ref
			(if (number? nameoruid)
				(getpwuid nameoruid)
				(getpwnam nameoruid))
			5))
	(if (and (> (string-length path) 0)
			 (char=? #\~ (string-ref path 0)))
		(let xpath-loop ((rest	(substring path 1))
						 (user	""))
		     (cond
			 	((or (= 0 (string-length rest))
					 (char=? (string-ref rest 0) #\/))
					(string-append
						(if (= 0 (string-length user)) ;; only ~
							(or (getenv "HOME")
								(getuserhome (or (getenv "LOGNAME")
												 (getuid))))
							(getuserhome user))
						rest))
				(#t
					(xpath-loop
						(substring rest 1)
						(string-append user (substring rest 0 1))))))
		path))

;; scirc-load uses xpath
(define (scirc-load path)
	(let ((file (xpath path)))
		(cond
			((access? file R_OK)
				(write-to #f "*** Loading " file)
				(load file))
			(#t
				(write-to #f "*** Can't read " file)))))

;;; Ignore handling
;(add-ignore pattern)
;(rm-ignore pattern)		; pattern can also be a reference count
;(ignore? string)
;(disp-ignore)

(define IGNORE-LIST '())

(define (add-ignore pattern)
	(set! IGNORE-LIST
		(append IGNORE-LIST
			(list (list (ignore:make-matcher pattern) pattern)))))
(define (rm-ignore pattern)
	(define (ignore-equal? pat ent i)
		(if (number? pat)
			(= pat i)
			(string-ci=? pat ent)))
	(let rm-ignore-loop ((lis  IGNORE-LIST)
						 (done '())
						 (i    0))
		(cond
			((null? lis)
				(set! IGNORE-LIST done))
			((ignore-equal? pattern (cadar lis) i)
				(rm-ignore-loop (cdr lis) done (+ 1 i)))
			(#t
				(rm-ignore-loop (cdr lis) (append done
					(list (car lis))) (+ 1 i))))))

(define (ignore? string)
	(let ignore?-loop ((lis	IGNORE-LIST))
		(cond
			((null? lis)
				#f)
			(((caar lis) string)
					#t)
			(#t
				(ignore?-loop (cdr lis))))))

(define (disp-ignore)
	(let disp-ignore-loop ((lis IGNORE-LIST)
						   (i   0))
		(cond
			((null? lis)
				(write-to #f "*** * End of /IGNORE list"))
			(#t
				(write-to #f "*** Ignore #" (number->string i) ": "
					(cadar lis))
				(disp-ignore-loop (cdr lis) (+ 1 i))))))

; This is the function from slib.
; i changed this one to reflect my needs (read, I cut out everything i didn't
; need)
;;; glob.scm: String matching for filenames (a la BASH).
;;; Copyright (C) 1998 Radey Shouman.
;
;Permission to copy this software, to redistribute it, and to use it
;for any purpose is granted, subject to the following restrictions and
;understandings.
;
;1.  Any copy made of this software must include this copyright notice
;in full.
;
;2.  I have made no warrantee or representation that the operation of
;this software will be error-free, and I am under no obligation to
;provide any services, by way of maintenance, update, or otherwise.
;
;3.  In conjunction with products arising from the use of this
;material, there shall be no use of my name in any advertising,
;promotional, or sales literature without prior written consent in
;each case.

(define (ignore:make-matcher pat)
  (define (match-end str k)
    (= k (string-length str)))
  (define (match-char ch nxt)
    (lambda (str k)
      (and (< k (string-length str))
	   (char-ci=? ch (string-ref str k))
	   (nxt str (+ k 1)))))
  (define (match-? nxt)
    (lambda (str k)
      (and (< k (string-length str))
	   (nxt str (+ k 1)))))
  (define (match-* nxt)
    (lambda (str k)
      (let loop ((kk (string-length str)))
	(and (>= kk k)
	     (or (nxt str kk)
		 (loop (- kk 1)))))))

  (let ((matcher
	 (let recur ((i 0))
	   (if (= i (string-length pat))
	       match-end
	       (let ((pch (string-ref pat i)))
		 (case pch
		   ((#\?)
		    (let ((nxt (recur (+ i 1))))
		      (match-? nxt)))
		   ((#\*)
		    (let ((nxt (recur (+ i 1))))
		      (match-* nxt)))
		   (else (let ((nxt (recur (+ i 1))))
			   (match-char pch nxt)))))))))
    (lambda (str) (matcher str 0))))

;;; Time
(define (ctime ut)
	(strftime "%a %b %d %H:%M:%S %Y" (localtime ut)))
(define (tdiff secs)
	(let tdiff-loop ((n		secs)
					 (days	#f)
					 (hrs	#f)
					 (mins	#f))
		(cond
			((eqv? days #f)
				(tdiff-loop (modulo n (* 60 60 24))
					(inexact->exact (truncate (/ n (* 60 60 24))))
					#f #f))
			((eqv? hrs #f)
				(tdiff-loop (modulo n (* 60 60))
					days (inexact->exact (truncate (/ n (* 60 60))))
					#f))
			((eqv? mins #f)
				(tdiff-loop (modulo n 60)
					days hrs (inexact->exact (truncate (/ n 60)))))
			(#t
				(string-append
					(if (not (= 0 days))
						(string-append (number->string days) " day"
							(if (not (= 1 days)) "s" "")
							(if (not (= 0 hrs mins n)) " " ""))
						"")
					(if (not (= 0 hrs))
						(string-append (number->string hrs) " hour"
							(if (not (= 1 hrs)) "s" "")
							(if (not (= 0 mins n)) " " ""))
						"")
					(if (not (= 0 mins))
						(string-append (number->string mins) " minute"
							(if (not (= 1 mins)) "s" "")
							(if (not (= 0 n)) " " ""))
						"")
					(if (or (not (= 0 n)) (= 0 days hrs mins))
						(string-append (number->string n) " second"
							(if (not (= 1 n)) "s" ""))
						""))))))

;;; Interface type: 'dumb 'ssfe 'tkirc
(define INTERFACE-TYPE-VAR 'dumb)
(define (set-interface-type! type)
	(if (memv type '(dump ssfe))
		(set! INTERFACE-TYPE-VAR type)))
(define (interface-type)
	INTERFACE-TYPE-VAR)

;; stuff that depends on the interface: user-I/O
(define (write-to dst . text)
	(case (interface-type)
		((dumb ssfe)
			(if (or (eqv? dst #f)
					(not (channel? dst))
					(string-ci=? dst (current-channel)))
				(write-line (apply string-append text))
				(write-line (string-append "[" dst "]"
					(apply string-append text)))))))
(define (ssfe-do cmd . string)
	(write-line
		(string-append "`#ssfe#" cmd
			(apply string-append string))))
(define (set-status . string)
	(case (interface-type)
		((ssfe)
			(ssfe-do "s" (apply string-append string)))))
(define (add-tab-nick nick)
	(case (interface-type)
		((ssfe)
			(ssfe-do "t" (string-append "/msg " nick " ")))))
(define (set-input-prompt . prompt)
	(case (interface-type)
		((ssfe)
			(ssfe-do "I" (apply string-append prompt)))))
(define (do-status)
	(case (interface-type)
		((ssfe)
			(let ((chan (current-channel))
				  (quer (current-query)))
				(ssfe-do "s"
					"[" (strftime "%H:%M" (localtime (current-time))) "] "
					(current-nick) " "
					(if (> (string-length quer) 0)
						(string-append "[Query: " quer "] ")
						"")
					(if (> (string-length chan) 0)
						(string-append "on " chan " ")
						"")
					(get-var 'STATUS-USER))))))

;;; Network I/O
(define (net-connect server port)
	(let ((sock (socket AF_INET SOCK_STREAM 0)))
		(connect sock AF_INET (car (array-ref (gethostbyname server) 4))
			port)
		sock))

;;; Server I/O
(define CURRENT-SERVER-VAR 0)
(define (set-current-server! port)
	(if (input-port? port)
		(set! CURRENT-SERVER-VAR port)))
(define (current-server)
	CURRENT-SERVER-VAR)

(define (send-server . stuff)
	(let ((serv (current-server)))
		(if (output-port? serv)
			(write-line (apply string-append stuff) serv)
			(write-to #f "*** Not connected to a server"))))
(define (msg dst . message)
	(if (channel? dst)
		(write-to dst "> " (apply string-append message))
		(write-to dst "-> *" (bold dst) "* " (apply string-append message)))
	(send-server "PRIVMSG " dst " :" (apply string-append message)))
(define (notice dst . message)
	(if (channel? dst)
		(write-to dst "- " (apply string-append message))
		(write-to dst "-> -" (bold dst) "- " (apply string-append message)))
	(send-server "NOTICE " dst " :" (apply string-append message)))
(define (say . message)
	(msg (current-target) (apply string-append message)))
(define (ctcp nick . command)
	(send-server "PRIVMSG " nick " :"
		(as-ctcp (string-upcase-one-word (apply string-append command)))))
(define CTCPR-LAST 0)
(define CTCPR-IGNO #f)
(define (ctcpr nick . message)
	(let ((time (current-time))
		  (last CTCPR-LAST))
		(set! CTCPR-LAST time)
		(cond
			((> (- time last) 5)
				(cond ((eqv? CTCPR-IGNO #t)
					(write-to #f "*** CTCP flood protection: resuming replies")
					(set! CTCPR-IGNO #f)))
				(send-server "NOTICE " nick " :"
					(as-ctcp (string-upcase-one-word
						(apply string-append message)))))
			(#t
				(cond ((eqv? CTCPR-IGNO #f)
					(write-to #f
						"*** CTCP flood protection: suspending replies")
					(set! CTCPR-IGNO #t)))))))
(define (describe dst . description)
	(if (channel? dst)
		(write-to dst "* " (current-nick) " "
			(apply string-append description))
		(write-to dst "* -> " (bold dst) " "
			(apply string-append description)))
	(ctcp dst "ACTION " (apply string-append description)))
(define (me . action)
	(describe (current-target) (apply string-append action)))
(define (new-server host port pass nick)
	(write-to #f "*** Connecting to " host ":" (number->string port))
	(let ((sock (net-connect host port)))
 		(if (> (string-length pass) 0)
			(write-line (string-append "PASS " pass) sock))
  		(write-line (string-append "NICK " nick) sock)
  		(write-line (string-append "USER"
				" " (getenv "LOGNAME")
				" " (inet-ntoa (array-ref (getsockname sock) 1))
				" " host
				" :" (get-var 'IRCNAME)) sock)
		(add-port sock host server-command)))

;;; Descriptor handling
(define PORT-LIST '())
(define (add-port port name proc)
	(let add-port-loop ((lis PORT-LIST))
		(cond
			((null? lis)
				(set! PORT-LIST (cons (list proc name port) PORT-LIST)))
			((eqv? (cdar lis) port)
				#f)
			(#t
				(add-port-loop (cdr lis))))))
(define (get-fd-list)
	(let get-fd-list-loop ((todo PORT-LIST)
						   (done '()))
		(if (null? todo)
			done
			(get-fd-list-loop (cdr todo) (cons (fileno (caddar todo)) done)))))
(define (call-port-func port line)
	(let call-port-func-loop ((lis PORT-LIST)
							  (done '()))
		(if (not (null? lis))
			(cond
				((eqv? (caddar lis) port)
					((caar lis) (cadar lis) port line)
					(if (eof-object? line)
						(set! PORT-LIST (append done (cdr lis)))))
				(#t
					(call-port-func-loop (cdr lis)
					(append done (list (car lis)))))))))

;;; Now the heart of the client

(define (main-loop)
	(let main-loop-loop ((lis (car (select (get-fd-list) '() '() 45))))
		(do-status)
		(cond
			((not (null? lis))
				(let ((ports (fdes->ports (car lis))))
					(if (not (null? ports))
						(call-port-func (car ports) (read-line (car ports))))
					(main-loop-loop (cdr lis))))))
	 (main-loop))

;;; server commands
(define (srv-ping from cmd to args)
	(send-server "PONG :" args))
(define (srv-privmsg from cmd to args)
	(if (and (< 0 (string-length args)) (eqv? #\001 (string-ref args 0)))
		(srv-ctcp from cmd to (strip-string args #\001))
		(cond
			((channel? to)
				(write-to to "<" (nuh->n from) "> " args))
			(#t
				(add-tab-nick (nuh->n from))
				(write-to (nuh->n from) "*" (bold (nuh->n from)) "* " args)))))
(define (srv-ctcp from cmd to args)
	(let* ((ctcp-cmd (nextword args))
		   (show-ctcp (lambda ()
					(write-to (if (channel? to) to (nuh->n from))
						"*** CTCP " (car ctcp-cmd) " from " (nice-from from)
						" to " to ": " (cdr ctcp-cmd)))))
		(case (string->symbol (car ctcp-cmd))
			((ACTION)
				(write-to (if (channel? to) to (nuh->n from))
				(if (channel? to)
					(string-append "* " (nuh->n from) " " (cdr ctcp-cmd))
					(string-append "*> " (bold (nuh->n from))
					" " (cdr ctcp-cmd)))))
			((VERSION)
				(show-ctcp)
				(let ((my-uname (uname)))
					(ctcpr (nuh->n from) "VERSION Scirc : " SCIRC-VERSION " : "
						(utsname:sysname my-uname) " "
						(utsname:release my-uname)
						" : " (get-var 'CLIENT-INFO))))
			((PING)
				(ctcpr (nuh->n from) "PING " (cdr ctcp-cmd))
				(write-to (if (channel? to) to (nuh->n from))
					"*** CTCP PING from " (nice-from from) " to " to
					": "
					(cond
						((string->number (cdr ctcp-cmd)) =>
							(lambda (n) (string-append
								(tdiff (- (current-time) n)) " ago")))
						(#t
							(cdr ctcp-cmd)))))
			((CLIENTINFO)
				(show-ctcp)
				(ctcpr (nuh->n from) "CLIENTINFO "
					"ACTION CLIENTINFO FINGER PING TIME USERINFO VERSION"))
			((USERINFO)
				(show-ctcp)
				(ctcpr (nuh->n from) "USERINFO " (get-var 'USER-INFO)))
			((ERRMSG)
				(show-ctcp))
			((FINGER)
				(show-ctcp)
				(ctcpr (nuh->n from) "FINGER " 
					(car (nextword (passwd:gecos
						(getpwnam (getenv "LOGNAME"))) #\,))
					" (" (getenv "LOGNAME") "@" (or (getenv "HOSTNAME") "host")
					")"))
			((TIME)
				(show-ctcp)
				(ctcpr (nuh->n from) "TIME " (ctime (current-time))))
			((DCC)
				(srv-dcc from "DCC" to (cdr ctcp-cmd)))
			(else
				(write-to (if (channel? to) to (nuh->n from))
					"*** Unknown CTCP " (car ctcp-cmd) " from "
					(nice-from from) " to " to ": " (cdr ctcp-cmd))))))
(define (srv-dcc from cmd to args)
	(write-to (nuh->n from) "*** DCC not yet implemented! (" (nuh->n from)
		" sent " args))
(define (srv-mode from cmd to args)
	(write-to to "*** Mode change \"" args "\" on " to " by "
		(nice-from from)))
(define (srv-join from cmd to args)
	(let ((channel+o (nextword args #\007)))
		(if (string-ci=? (nuh->n from) (current-nick))
			(set-cur-chan! (car channel+o)))
		(write-to (car channel+o)
			"*** " (nice-from from) " has joined " (car channel+o))
		(if (> (string-length (cdr channel+o)) 0)
			(callhooks from "MODE" (car channel+o)
				(string-append "+" (substring (cdr channel+o) 0 1) " "
					(nuh->n from))))))
(define (srv-part from cmd to args)
	(write-to to "*** " (nice-from from) " has left " to
		" (" (strip-colon args) ")")
	(if (and (string-ci=? (nuh->n from) (current-nick))
			 (string-ci=? to (current-channel)))
		(set-cur-chan! "")))
(define (srv-kick from cmd to args)
	(let ((kicked+reason (nextword args)))
		(write-to to "*** "
			(if (string-ci=? (current-nick) (car kicked+reason))
				"You have"
				(string-append (car kicked+reason) " has"))
			" been kicked of channel "
			to " by " (nice-from from) " (" (strip-colon (cdr kicked+reason))
				")")
		(if (and (string-ci=? (car kicked+reason) (current-nick))
				 (string-ci=? to (current-channel)))
			(set-cur-chan! ""))))
(define (srv-nick from cmd to args)
	(write-to (nuh->n from) "*** " (nice-from from) " is now known as " args))
(define (srv-invite from cmd to args)
	(write-to (nuh->n from) "*** " (nice-from from)
		" invites " to " to " args))
(define (srv-notice from cmd to args)
	(if (and (< 0 (string-length args)) (eqv? #\001 (string-ref args 0)))
			 (srv-ctcpr from cmd to (strip-string args #\001))
		(if (channel? to)
			(write-to to "+" (nuh->n from) "+ " args)
			(write-to (nuh->n from) "-" (bold (nuh->n from)) "- " args))))
(define (srv-ctcpr from cmd to args)
	(let* ((ctcpr-cmd (nextword args))
		   (show-ctcpr (lambda () (write-to (if (channel? to) to (nuh->n from))
					"*** CTCP " (car ctcpr-cmd) " reply from " (nice-from from)
					" to "
					to ": " (cdr ctcpr-cmd)))))
		(case (string->symbol (car ctcpr-cmd))
			((PING)
				(let ((num (string->number (cdr ctcpr-cmd))))
					(cond
						(num
							(write-to (if (channel? to) to (nuh->n from))
								"*** CTCP PING reply from " (nice-from from)
								": " (tdiff (- (current-time) num))))
						(#t
							(show-ctcpr)))))
			(else
				(show-ctcpr)))))
(define (srv-quit from cmd to args)
	(write-to (nuh->n from)
		"*** " (nice-from from) " has left IRC (" args ")"))
(define (srv-topic from cmd to args)
	(write-to to "*** " (nice-from from) " has changed the topic on channel "
		to " to " (strip-colon args)))
(define (srv-error from cmd to args)
	(write-to #f "*** " args))
(define (srv-001 from cmd to args)
	(write-to #f "*** " args))
(define (srv-221 from cmd to args)
	(write-to #f "*** Your user mode is \"" args "\""))
(define (srv-311 from cmd to args)
	(let* ((nick (nextword args))
		   (user (nextword (cdr nick)))
		   (host (nextword (cdr user)))
		   (real (cdr (nextword (cdr host)))))
		(write-to #f "*** " (car nick) " is " (car user) "@" (car host)
			" (" (strip-colon real) ")")))
(define (srv-312 from cmd to args)
	(let* ((nick   (nextword args))
		   (server (nextword (cdr nick)))
		   (info   (cdr server)))
		(write-to #f "*** " (car nick) " using " (car server)
			" (" (strip-colon info) ")")))
(define (srv-314 from cmd to args)
	(let* ((nick (nextword args))
		   (user (nextword (cdr nick)))
		   (host (nextword (cdr user)))
		   (real (cdr (nextword (cdr host)))))
		(write-to #f "*** " (car nick) " was " (car user) "@" (car host)
			" (" real ")")))
(define (srv-317 from cmd to args)
	(let* ((nick (nextword args))
		   (idle (car (nextword (cdr nick)))))
		(write-to #f "*** " (car nick) " has been idle for "
			(tdiff (string->number idle)))))
(define (srv-319 from cmd to args)
	(let ((nick+chans (nextword args)))
		(write-to #f "*** " (car nick+chans) " on channels: "
			(strip-colon (cdr nick+chans)))))
(define (srv-324 from cmd to args)
	(let ((chan+mode (nextword args)))
		(write-to (car chan+mode) "*** Mode for " (car chan+mode) " is \""
			(cdr chan+mode) "\"")))
(define (srv-329 from cmd to args)
	(let ((chan+time (nextword args)))
		(write-to (car chan+time) "*** Channel " (car chan+time) " created "
			(ctime (string->number (cdr chan+time))))))
(define (srv-332 from cmd to args)
	(let ((chan+topic (nextword args)))
		(write-to (car chan+topic) "*** Topic for " (car chan+topic) ": "
			(strip-colon (cdr chan+topic)))))
(define (srv-333 from cmd to args)
	(let* ((chan (nextword args))
		   (setter+time (nextword (cdr chan))))
		(write-to (car chan) "*** Topic for " (car chan) " set by "
			(car setter+time) " on "
			(ctime (string->number (cdr setter+time))))))
(define (srv-341 from cmd to args)
	(let ((chan+inick (nextword args)))
		(write-to (car chan+inick) "*** Invited " (cdr chan+inick) " to "
			(car chan+inick))))
(define (srv-367 from cmd to args)
	(let ((chan+banid (nextword args)))
		(write-to (car chan+banid) "*** Ban on " (car chan+banid) ": "
			(cdr char+banid))))

(define SERVER-COMMANDS
		'((PING		. srv-ping)
		  (PRIVMSG	. srv-privmsg)
		  (MODE		. srv-mode)
		  (JOIN		. srv-join)
		  (PART		. srv-part)
		  (KICK		. srv-kick)
		  (NICK		. srv-nick)
		  (INVITE	. srv-invite)
		  (NOTICE	. srv-notice)
		  (QUIT		. srv-quit)
		  (TOPIC	. srv-topic)
		  (ERROR	. srv-error)
		  (001		. srv-001)
		  (221		. srv-221)
		  (311		. srv-311)
		  (312		. srv-312)
		  (314		. srv-314)
		  (317		. srv-317)
		  (319		. srv-319)
		  (324		. srv-324)
		  (329		. srv-329)
		  (332		. srv-332)
		  (333		. srv-333)
		  (341		. srv-341)
		  (367		. srv-367)))

(define (server-command name port line)
	(if (eof-object? line)
		(write-to #f "*** EOF from server " name)
		(let ((fcta (split-serverline (strip-string line #\cr))))
			(cond
				((not (ignore? (car fcta)))
					(set-current-server! port)
					(set-var! 'SERVER name)
					(apply callhooks fcta))))))
(define (addhook event func)
	(set! SERVER-COMMANDS (cons (cons event func) SERVER-COMMANDS)))
(define (rmhook event func)
	(let rmhook-loop ((lis	SERVER-COMMANDS)
					  (done '()))
		(cond
			((null? lis)
				(set! SERVER-COMMANDS done))
			((and (eqv? event (caar lis)) (eqv? func (cdar lis)))
				(rmhook-loop (cdr lis) done))
			(#t
				(rmhook-loop (cdr lis) (append done (list (car lis))))))))

(define (callhooks from cmd to args)
	(let* ((num (string->number cmd))
		   (event (if num num (string->symbol (string-upcase cmd)))))
		(let callhooks-loop ((lis  SERVER-COMMANDS)
							 (done 0))
			(if (not (null? lis))
				(cond
					((eqv? event (caar lis))
						((eval (cdar lis)) from cmd to args)
						(callhooks-loop (cdr lis) 1))
					(#t
						(callhooks-loop (cdr lis) done)))
				(cond ((= done 0)
					(write-to #f "*** " args)))))))

;;; Client commands
(define (cmd-ctcp cmd args)
	(let ((arg (nextword args)))
		(ctcp (car arg) (cdr arg))))
(define (cmd-describe cmd args)
	(let ((arg (nextword args)))
		(describe (car arg) (cdr arg))))
(define (cmd-exit cmd args)
	(quit))
(define (cmd-help cmd args)
	(write-to #f "*** Hey! It says DON'T type /HELP!"))
(define (cmd-ignore cmd args)
	(let ((pat (car (nextword args))))
		(add-ignore pat)
		(write-to #f "*** Added " pat " to the ignore list")))
(define (cmd-join cmd args)
	(send-server "JOIN " args))
(define (cmd-kick cmd args)
	(send-server "KICK " (addcolon (xsplat args) 2)))
(define (cmd-kill cmd args)
	(send-server "KILL " (addcolon args 1)))
(define (cmd-load cmd args)
	(scirc-load args))
(define (cmd-me cmd args)
	(me args))
(define (cmd-mode cmd args)
	(send-server "MODE " (xsplat args)))
(define (cmd-msg cmd args)
	(let ((arg (nextword args)))
		(add-tab-nick (car arg))
		(msg (car arg) (cdr arg))))
(define (cmd-names cmd args)
	(send-server "NAMES " (xsplat args)))
(define (cmd-notice cmd args)
	(let ((arg (nextword args)))
		(notice (car arg) (cdr arg))))
(define (cmd-part cmd args)
	(send-server "PART " (xsplat args)))
(define (cmd-ping cmd args)
	(let ((arg (nextword args)))
		(ctcp (car arg) "PING " (number->string (current-time)))))
(define (cmd-query cmd args)
	(set-cur-query! args))
(define (cmd-quit cmd args)
	(send-server "QUIT :" args))
(define (cmd-quote cmd args)
	(send-server args))
(define (cmd-say cmd args)
	(msg (current-target) args))

;;; The cmd-scheme ... it's a bit messy ;)
(define SCHEME-CMD-CONTINUATION #f)
(define SCHEME-CMD-STRING       "")
(define SCHEME-CMD-SEMAPHORE	#f)
(define (scheme-cmd-exception key . rest)
	(display "*** An exception occured while evaluating a /SCHEME command:")
	(newline)
	(display "*** Key  : ") (write key) (newline)
	(display "*** Other: ") (write rest) (newline)
	(if SCHEME-CMD-CONTINUATION
		(SCHEME-CMD-CONTINUATION key))
	(display "*** No continuation found. HELP!"))
(define (scheme-cmd-do-eval)
	(let ((result (eval (call-with-input-string SCHEME-CMD-STRING read))))
		(display "*** => ")
		(write result)
		(newline)))
(define (cmd-scheme cmd args)
	(cond
		(SCHEME-CMD-SEMAPHORE
			(write-to #f "*** Sorry, only one /SCHEME command at a time"))
		(#t
			(set! SCHEME-CMD-SEMAPHORE #t)
			(call-with-current-continuation (lambda (return)
				(set! SCHEME-CMD-CONTINUATION return)
				(set! SCHEME-CMD-STRING args)
				(catch #t scheme-cmd-do-eval scheme-cmd-exception)))
			(set! SCHEME-CMD-SEMAPHORE #f))))
(define (cmd-server cmd args)
	(let* ((srvc	(nextword args #\:))
		   (portc	(nextword (cdr srvc) #\:))
		   (passc	(nextword (cdr portc) #\:))
		   (nickc	(nextword (cdr passc) #\:)))
		(new-server
			(car srvc)
			(if (= 0 (string-length (car portc)))
				6667
				(string->number (car portc)))
			(if (= 0 (string-length (car passc)))
				""
				(car passc))
			(if (= 0 (string-length (car nickc)))
				(get-var 'NICK)
				(car nickc)))))
(define (cmd-set cmd args)
	(let ((arg (nextword args)))
		(cond
			((= 0 (string-length (car arg)))
				(disp-var))
			((= 0 (string-length (cdr arg)))
				(write-to #f "*** Current value of " (car arg) " is "
					(get-var (string->symbol (car arg)))))
			(#t
				(if (set-var! (string->symbol (car arg)) (cdr arg))
					(write-to #f "*** Value of " (string->symbol (car arg))
						" set to " (cdr arg))
					(write-to #f "*** No such variable \""
						(string->symbol (car arg)) "\""))))))
(define (cmd-squit cmd args)
	(send-server "SQUIT " (addcolon args 1)))
(define (cmd-topic cmd args)
	(send-server "TOPIC " (addcolon (xsplat args) 1)))
(define (cmd-unignore cmd args)
	(let ((arg (car (nextword args))))
		(cond
			((string->number arg) => (lambda (n)
				(write-to #f "*** Removed ignore #" (number->string n))
				(rm-ignore n)))
			(#t
				(write-to #f "*** Removed ignore " arg)
				(rm-ignore arg)))))
(define (cmd-version cmd args)
	(send-server "VERSION" args)
	(write-to #f "*** This is Scirc v" SCIRC-VERSION))
(define (cmd-whois cmd args)
	(send-server "WHOIS " args))
(define (cmd-whowas cmd args)
	(send-server "WHOWAS " args))
(define (cmd-xquote cmd args)
	(send-server cmd " " args))
(define (cmd-xyzzy cmd args)
	(write-to #f "*** Nothing happens"))

(define CLIENT-COMMAND
	'((SAY		. cmd-say)
	  (JOIN		. cmd-join)
	  (VERSION	. cmd-version)
	  (CTCP		. cmd-ctcp)
	  (DESCRIBE	. cmd-describe)
	  (EXIT		. cmd-exit)
	  (HELP		. cmd-help)
	  (IGNORE	. cmd-ignore)
	  (KICK		. cmd-kick)
	  (KILL		. cmd-kill)
	  (LOAD		. cmd-load)
	  (PART		. cmd-part)
	  (ME		. cmd-me)
	  (MSG		. cmd-msg)
	  (MODE		. cmd-mode)
	  (NAMES	. cmd-names)
	  (NOTICE	. cmd-notice)
	  (PING		. cmd-ping)
	  (QUERY	. cmd-query)
	  (QUIT		. cmd-quit)
	  (QUOTE	. cmd-quote)
	  (SCHEME	. cmd-scheme)
	  (SERVER	. cmd-server)
	  (SET		. cmd-set)
	  (SQUIT	. cmd-squit)
	  (TOPIC	. cmd-topic)
	  (UNIGNORE	. cmd-unignore)
	  (WHOIS	. cmd-whois)
	  (WHOWAS	. cmd-whowas)
	  (XYZZY	. cmd-xyzzy)
	  (ADMIN	. cmd-xquote)
	  (AWAY		. cmd-xquote)
	  (CONNECT	. cmd-xquote)
	  (DIE		. cmd-xquote)
	  (INFO		. cmd-xquote)
	  (INVITE	. cmd-xquote)
	  (LINKS	. cmd-xquote)
	  (NICK		. cmd-xquote)
	  (MOTD		. cmd-xquote)
	  (OPER		. cmd-xquote)
	  (REHASH	. cmd-xquote)
	  (RESTART	. cmd-xquote)
	  (STATS	. cmd-xquote)
	  (SUMMON	. cmd-xquote)
	  (TIME		. cmd-xquote)
	  (TRACE	. cmd-xquote)
	  (USERS	. cmd-xquote)))

(define (user-command name port line)
	(cond
		((eof-object? line)
			(write-to #f "*** EOF from " name "! HELP!"))
		((= 0 (string-length line))
			#f)
		(#t
			(let* ((cmd   (nextword line))
		    	   (chars (get-var 'CMDCHARS))
				   (len   (string-length chars)))
				(if (string=? line "@ssfe@i")
					(set-interface-type! 'ssfe)
					(if (and (> (string-length (car cmd)) len)
						     (string-ci=? chars (substring (car cmd) 0 len)))
						(apply callcommands (list (substring (car cmd) len)
							(cdr cmd)))
						(apply callcommands (list "SAY" line))))))))

(define (addcommand cmd func)
	(let addcommand-loop ((lis CLIENT-COMMAND))
		(cond
			((null? lis)
				(set! CLIENT-COMMAND (cons (cons cmd func) CLIENT-COMMAND)))
			((eqv? cmd (caar lis))
				(set-cdr! (car lis) func))
			(#t
				(addcommand-loop (cdr lis))))))
(define (rmcommand cmd func)
	(let rmcommand-loop ((lis	CLIENT-COMMAND)
						 (done  '()))
		(cond
			((null? lis)
				(set! CLIENT-COMMAND done))
			((and (eqv? cmd (caar lis)) (eqv? func (cdar lis)))
				(rmcommand-loop (cdr lis) done))
			(#t
				(rmcommand-loop (cdr lis) (append done (list (car lis))))))))

(define (callcommands cmd args)
	(let ((event (string->symbol (string-upcase cmd))))
		(let callcommands-loop ((lis  CLIENT-COMMAND)
								(done 0))
			(cond
				((null? lis)
					(if (= done 0)
						(write-to #f "*** Unknown command: " cmd)))
				(#t
					(cond
						((eqv? event (caar lis))
							(apply (eval (cdar lis)) (list cmd args))
							(callcommands-loop (cdr lis) 1))
						(#t
							(callcommands-loop (cdr lis) done))))))))

;;; Now initialize the stuff

(create-var 'NICK			(getenv "LOGNAME"))
(create-var 'IRCNAME		(or (getenv "IRCNAME")
	(car (nextword (passwd:gecos (getpwnam (getenv "LOGNAME"))) #\,))))
(create-var 'USER-INFO		"none set")
(create-var 'CLIENT-INFO
	"A Guile Scheme Client. To iterate is human, to recurse, divine.")
(create-var 'ADD-ONS		"")
(create-var 'STATUS-USER	"* don't type /help for help")
(create-var 'SERVER			"")
(create-var 'CMDCHARS		"/")

(add-port (fdopen 0 "r") "user" user-command)

(if (not (isatty? (current-output-port)))
	(write-line "`#ssfe#i"))
(setvbuf (current-output-port) _IOLBF)
(sigaction SIGPIPE SIG_IGN)
(sigaction SIGQUIT SIG_IGN)
(do-status)


(define (catch-main-loop)
	(catch #t main-loop exception-handler))

(define (exception-handler key . rest)
	(if (eqv? key 'quit)
		(quit))
	(display "*** EXCEPTION CAUGHT")
	(newline)
	(display "*** This means that you probably found a bug in scirc.")
	(newline)
	(display
		"*** Please send the following information together with a")
	(newline)
	(display
		"*** description of what you just did to forcer@mindless.com:")
	(newline)
	(display "*** Key : ") (write key)  (newline)
	(display "*** Args: ") (write rest) (newline)
	(display "*** Though for now i'll try to continue :)")
	(newline)
	(catch-main-loop))

(let ((file (xpath "~/.scircrc")))
	(if (access? file R_OK)
		(load file)))
(cond ((getenv "IRCSERVER") => (lambda (n)
	(cmd-server "/SERVER" n))))

(catch-main-loop)


