//   $Id: kvi_voice.cpp,v 1.4 1998/10/23 22:12:22 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this library; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CLASS_NAME_ "KviVoiceWnd"

#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_voice.h"
#include "kvi_view.h"
#include "kvi_frame.h"
#include "kvi_uparser.h"
#include "kvi_debug.h"
#include "kvi_socket.h"
#include "kvi_opt.h"
#include "kvi_ident.h"
#include "kvi_event.h"
#include "kvi_mdi.h"
#include "kvi_app.h"
#include "kvi_taskbar.h"
#include "kvi_int.h"
#include "kvi_support.h"

#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>             

#include <signal.h>

#ifndef SIGUSR1
#define SIGUSR1 10
#endif

#ifndef SIGUSR2
#define SIGUSR2 12
#endif

KviVoiceWnd::KviVoiceWnd(KviMdiManager *parent,KviFrame *frame,const char *aname,int aid)
:KviMdiChild(parent,frame,aname,aid,KVI_WND_TYPE_VOICE)
{
	m_lpOutput=new KviView(this,frame,"KviViewClass");

	m_lpOpt=m_lpFrm->m_lpOpt;
	m_lpKviPhone=new KProcess();
	QObject::connect(m_lpKviPhone,SIGNAL(processExited(KProcess *)),this,SLOT(childExited(KProcess *)));
	QObject::connect(m_lpKviPhone,SIGNAL(receivedStdout(KProcess *,char *,int)),this,SLOT(gotStdout(KProcess *,char *,int)));
	QObject::connect(m_lpKviPhone,SIGNAL(receivedStderr(KProcess *,char *,int)),this,SLOT(gotStdout(KProcess *,char *,int)));
	m_lpTalk=new QPushButton(i18n("Talk"),this);
	m_lpTalk->setEnabled(false);
	connect(m_lpTalk,SIGNAL(pressed()),this,SLOT(talk()));
	connect(m_lpTalk,SIGNAL(released()),this,SLOT(endTalk()));
	m_bCanTalk=false;
	m_lpProgress=new KProgress(0,300,0,KProgress::Horizontal,this);
	m_lpProgress->setBarStyle(KProgress::Solid);
	m_lpProgress->setTextEnabled(false);
	QColor my_green(20,120,0);
	m_lpProgress->setBarColor(my_green);
	m_lpSendProgress=new KProgress(0,300,0,KProgress::Horizontal,this);
	m_lpSendProgress->setBarStyle(KProgress::Solid);
	m_lpSendProgress->setTextEnabled(false);
	QColor my_red(150,20,0);
	m_lpSendProgress->setBarColor(my_red);
	setFocusOwner(m_lpOutput);
	connect(this,SIGNAL(closeButtonPressed()),this,SLOT(closeSlot()));
	m_lpTalkTimer=new QTimer();
	connect(m_lpTalkTimer,SIGNAL(timeout()),this,SLOT(talkinPlusPlus()));
	applyOptions();
	m_szLastIncompleteOutput="";
}
KviVoiceWnd::~KviVoiceWnd()
{
	if(m_lpKviPhone->isRunning())m_lpKviPhone->kill();
	delete m_lpOutput;
	delete m_lpKviPhone;
	delete m_lpTalk;
	delete m_lpProgress;
	delete m_lpSendProgress;
}

//============ talkinPlusPlus ============//

void KviVoiceWnd::talkinPlusPlus()
{
	_debug_entertrace("talkinPlusPlus");
	m_lpSendProgress->setValue(m_lpSendProgress->value()+16); //16 packets each time
	_debug_leavetrace("talkinPlusPlus");
}

//============ talk ============//

void KviVoiceWnd::talk()
{
	_debug_entertrace("talk");
	if(m_bCanTalk){
		if(!m_lpKviPhone->kill(SIGUSR1))doOutput(KVI_OUT_ERROR,i18n("WARNING : Could not deliver child signal."));
		else {
			m_lpTalkTimer->start(500); //250 msec = 16*512 bytes
		}
	}
	_debug_leavetrace("talk");
}

//============ endTalk ============//

void KviVoiceWnd::endTalk()
{
	_debug_entertrace("endTalk");
	if(!m_lpKviPhone->kill(SIGUSR2))doOutput(KVI_OUT_ERROR,i18n("WARNING : Could not deliver child signal."));
	else {
		m_lpTalkTimer->stop();
		m_lpSendProgress->setValue(0);
	}
	_debug_leavetrace("endTalk");
}

//============ childExited ============//

void KviVoiceWnd::childExited(KProcess *)
{
	_debug_entertrace("childExited");
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Conversation terminated : child process exited."));
	m_bCanTalk=false;
	m_lpTalk->setEnabled(false);
	_debug_leavetrace("childExited");
}

//============ gotStdout ============//

void KviVoiceWnd::gotStdout(KProcess *,char *buffer,int buflen)
{
	_debug_entertrace("gotStdout");
	if(buflen<1)return;
	QString szData(buffer,buflen+1);
	szData.prepend(m_szLastIncompleteOutput.data());
	m_szLastIncompleteOutput="";
	int idx=szData.find("\n");
	while(idx != -1){
		QString szOut=szData.left(idx);
		szData.remove(0,idx+1);
		processOutput(szOut);
		idx=szData.find("\n");
	}
	if(!szData.isEmpty())m_szLastIncompleteOutput=szData.copy();
	_debug_leavetrace("gotStdout");
}

//============ processOutput ============//

void KviVoiceWnd::processOutput(QString &szOut)
{
	_debug_entertrace("processOutput");
	char first=szOut[0];
	szOut.remove(0,1);
	int iVal=0;
	bool bOk=false;
	switch(first){
	case 'I':
		//LOCK
		m_bCanTalk=false;
		m_lpTalk->setEnabled(false);
		break;
	case 'U':
		//UNLOCK
		m_lpTalk->setEnabled(true);
		m_bCanTalk=true;
		break;
	case 'V':
		//received data
		bOk=false;
		iVal=szOut.toInt(&bOk);
		if(iVal>300)iVal=300;
		if(!bOk)iVal=0;
		m_lpProgress->setValue(iVal);
		break;
//	case 'B': eliminated
//		//received data
//		bOk=false;
//		iVal=szOut.toInt(&bOk);
//		if(iVal>300)iVal=300;
//		if(!bOk)iVal=0;
//		m_lpSendProgress->setValue(iVal);
//		break;
	case 'E':
		//fatal error
		doOutput(KVI_OUT_ERROR,szOut.data());
		m_bCanTalk=false;
		m_lpTalk->setEnabled(false);
		break;
	case 'M':
		//normal message
		szOut.remove(0,4);
		doOutput(KVI_OUT_INTERNAL,szOut.data());
		break;
	case 'C':
		//connected
		doOutput(KVI_OUT_CONNECT,szOut.data());
		m_bCanTalk=true;
		m_lpTalk->setEnabled(true);
		break;
	case 'L':
		//is listening
		QString szAddr;
		szAddr.setNum(m_myAddress);
		m_lpFrm->m_lpSock->sendFmtData("PRIVMSG %s :%cDCC VOICE adpcm %s %s%c",m_szRemoteNick.data(),0x01,
					szAddr.data(),szOut.data(),0x01);
		doFmtOutput(KVI_OUT_INTERNAL,i18n("Sent DCC VOICE request to %s [%s:%s]...waiting for reply"),
					m_szRemoteNick.data(),szAddr.data(),szOut.data());
		break;
	}
	_debug_leavetrace("processOutput");
}

//============ startListeningKviphone ============//

bool KviVoiceWnd::startListeningKviphone()
{
	_debug_entertrace("startListeningKviphone");
	(*m_lpKviPhone) << "kviphone" << "listen";
	return m_lpKviPhone->start(KProcess::NotifyOnExit,KProcess::AllOutput);
	_debug_leavetrace("startListeningKviphone");
}

//============ startConnectingKviphone ============//

bool KviVoiceWnd::startConnectingKviphone(QString &szIp,QString &szPort)
{
	_debug_entertrace("startConnectingKviphone");
	(*m_lpKviPhone) << "kviphone";
	(*m_lpKviPhone) << szIp.data();
	(*m_lpKviPhone) << szPort.data();
	return m_lpKviPhone->start(KProcess::NotifyOnExit,KProcess::AllOutput);
	_debug_leavetrace("startConnectingKviphone");
}

//============ requestConnection ============//

bool KviVoiceWnd::requestConnection(const char *szNick)
{
	_debug_entertrace("requestConnection");
	QString szStat;
	szStat.sprintf(i18n("Requesting DCC VOICE (ADPCM) connection to %s"),szNick);
	m_lpFrm->setStatusText(szStat.data());
	m_szRemoteNick=QString(szNick);
//#if 0
	if(!m_lpFrm->m_lpSock->m_bConnected){
		doOutput(KVI_OUT_ERROR,i18n("DCC VOICE Requires an active IRC connection"));
		m_lpFrm->setStatusText(i18n("DCC VOICE Request failed"));
		return false;
	}
	m_myAddress=m_lpFrm->m_lpSock->getSockAddress();
	if(!m_myAddress){
		doOutput(KVI_OUT_ERROR,i18n("Unable to resolve localhost, can't request DCC VOICE"));
		m_lpFrm->setStatusText(i18n("DCC VOICE Request failed"));
		return false;
	}
//#endif
	if(!startListeningKviphone()){
		doOutput(KVI_OUT_ERROR,i18n("Unable to start KVIPhone program : can't request DCC VOICE"));
		m_lpFrm->setStatusText(i18n("DCC VOICE Request failed"));
		return false;
	} else doOutput(KVI_OUT_INTERNAL,i18n("Child process started."));

	_debug_leavetrace("requestConnection");
	return true;
}

//============ acceptConnection ============//

void KviVoiceWnd::acceptConnection(QString &szPort,QString &szIp,const char *szNick)
{
	_debug_entertrace("acceptConnection");
	QString szStat;
	szStat.sprintf(i18n("Accepting DCC VOICE connection from %s"),szNick);
	m_lpFrm->setStatusText(szStat.data());
	m_szRemoteNick=QString(szNick);
	bool bOK=false;
	szPort.toUShort(&bOK);
	if(!bOK){
		doOutput(KVI_OUT_ERROR,i18n("Can't retrieve the port to connect. DCC VOICE failed."));
		m_lpFrm->setStatusText(i18n("DCC VOICE Accept failed"));
		return;	
	}
	szIp=szIp.stripWhiteSpace();
	unsigned long iAddr=szIp.toULong(&bOK);
	if(!bOK){
		doFmtOutput(KVI_OUT_ERROR,i18n("Can't retrieve the address to connect. DCC VOICE failed (%s)."),szIp.data());
		m_lpFrm->setStatusText(i18n("DCC VOICE Accept failed"));
		return;	
	}
	struct in_addr inAddress;
	inAddress.s_addr=ntohl(iAddr);
	QString szRealIp=QString(inet_ntoa(inAddress));
	if(!startConnectingKviphone(szRealIp,szPort)){
		doOutput(KVI_OUT_ERROR,i18n("Unable to start KVIPhone program. DCC VOICE failed."));
		m_lpFrm->setStatusText(i18n("DCC VOICE Accept failed"));
		return;	
	} else doOutput(KVI_OUT_INTERNAL,i18n("Child process started."));
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connecting to %s on port %s"),szRealIp.data(),szPort.data());
	_debug_leavetrace("acceptConnection");
}

// ------------------------------->
//                STANDARD SECTION
// ------------------------------->
void KviVoiceWnd::closeSlot()
{
	_debug_entertrace("closeSlot");
	m_lpTalk->setEnabled(false);
	m_bCanTalk=false;
	hide();
	_macro_kviApplication->processEvents();
	blockSignals(true);
	m_lpMdi->m_lpTaskBar->removeButton(id());
	m_lpMdi->m_lpChildList->setAutoDelete(false);
	m_lpMdi->m_lpChildList->removeRef(this);
	m_lpMdi->focusTopChild();
	m_lpMdi->fillWinListPopup();
	m_lpFrm->queryExternalDestroy(this);
	_debug_leavetrace("closeSlot");
}
void KviVoiceWnd::applyOptions()
{
	_debug_entertrace("applyOptions");
	m_lpOutput->m_bTimestamp=m_lpFrm->m_lpOpt->bTimestamp;
	m_lpOutput->m_bShowPixmaps=m_lpFrm->m_lpOpt->bShowPixmaps;
	m_lpOutput->setFont(m_lpInt->fnt_output);
	const QSize sz=size();
	QResizeEvent e(sz,sz);
	resizeEvent(&e);
	_debug_leavetrace("applyOptions");
}
void KviVoiceWnd::resizeEvent(QResizeEvent *)
{
	_debug_entertrace("resizeEvent");
	updateRects();
	QRect rct=viewSizeHint();
	QFontMetrics fnt=m_lpTalk->fontMetrics();
	int input_hght=fnt.lineSpacing()+8;
	m_lpOutput->setGeometry(rct.left(),rct.top()+((KVI_MDI_VIEW_SEP+input_hght)*3),
			rct.width(),rct.height()-((KVI_MDI_VIEW_SEP+input_hght)*3));
	m_lpProgress->setGeometry(rct.left(),rct.top(),rct.width(),input_hght);
	m_lpSendProgress->setGeometry(rct.left(),rct.top()+(KVI_MDI_VIEW_SEP+input_hght),rct.width(),input_hght);
	m_lpTalk->setGeometry(rct.left(),rct.top()+((KVI_MDI_VIEW_SEP+input_hght)*2),
			rct.width(),input_hght);
	_debug_leavetrace("resizeEvent");
}
void KviVoiceWnd::doFmtOutput(int nType,const char *szFmt,...)
{
	_debug_entertrace("doFmtOutput");
	char szText[600]; //It should be big enough... I hope...
	va_list list;
	va_start(list,szFmt);
	if(vsnprintf(szText,600,szFmt,list)==-1)debug("WARNING : Output string truncated"); // Fritz: vsnprintf + realloc
	va_end(list);
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,(const char *) &szText);
	_debug_leavetrace("doFmtOutput");
}
void KviVoiceWnd::doOutput(int nType,const char *szText)
{
	_debug_entertrace("doOutput");
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,szText);
	_debug_leavetrace("doOutput");
}

#include "m_kvi_voice.moc"
