//   $Id: kvi_srv.cpp,v 1.5 1998/09/25 15:58:36 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviServerManager"
#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_macros.h"
#include "kvi_srv.h"
#include "kvi_app.h"
#include "kvi_debug.h"
#include "kvi_defs.h"

#include <kmsgbox.h>



//============ KviServerManager ============//

KviServerManager::KviServerManager()
{
	_debug_entertrace("KviServerManager");
	m_lpNetworkList=new QList<KviIrcNetwork>;
	m_lpNetworkList->setAutoDelete(true);
	m_lpProxyList=new QList<KviProxyEntry>;
	m_lpProxyList->setAutoDelete(true);
	iCurNetwork=-1;
	iTimeout=60;
	cacheLookup=true;
	reconnectOnDisconnect=true;
	reconnectOnConnectFailed=true;
	bUseProxy=false;
	bUseSocksV5=true;
	readSettings();
	_debug_leavetrace("KviServerManager");
}

//============ ~KviServerManager ============//

KviServerManager::~KviServerManager()
{
	_debug_entertrace("~KviServerManager");
	saveSettings();
	clearNetworkList();
	clearProxyList();
	delete m_lpNetworkList;
	delete m_lpProxyList;
	_debug_leavetrace("~KviServerManager");
}

//============ clearProxyList ============//

void KviServerManager::clearProxyList()
{
	_debug_entertrace("clearProxyList");
	while(!m_lpProxyList->isEmpty())m_lpProxyList->removeFirst();
	_debug_leavetrace("clearProxyList");
}

//============ clearNetworkList ============//

void KviServerManager::clearNetworkList()
{
	_debug_entertrace("clearNetworkList");
	while(!m_lpNetworkList->isEmpty()){
		KviIrcNetwork *lpN=m_lpNetworkList->last();
		_range_valid(lpN->serverList);
		lpN->serverList->setAutoDelete(true);
		while(!lpN->serverList->isEmpty())lpN->serverList->removeLast();
		delete lpN->serverList;
		m_lpNetworkList->removeLast();
	}
	_debug_leavetrace("clearNetworkList");
}

//============ readSettings ============//

void KviServerManager::readSettings()
{
	_debug_entertrace("readSettings");
	QString szFileName(_macro_getKVircHomeDirectory("Config/kvi.server.conf"));

	QFile config(szFileName.data());
	if(!config.exists()){
		config.open(IO_WriteOnly);
		config.close();
		return;
	}

	if(!config.open(IO_ReadOnly)){ cantReadError(); return; }
	uint cnt;
	char dummyBuf[100];
	if(config.readBlock(dummyBuf,9)<0)return;
	dummyBuf[9]=0;
	QString szSign(dummyBuf,10);
	if(szSign != "KVIRCSRV\n"){ //Not our file
		cantReadError(); return;
	}
	if(config.readBlock(dummyBuf,46)<0)return;
	if(config.readBlock((char *)&cnt,sizeof(cnt))<0){ cantReadError(); return;}

	for(uint i=0;i<cnt;i++){
		KviIrcNetwork *lpN=new KviIrcNetwork;
		if(!readNetworkEntry(lpN,&config)){ delete lpN; cantReadError(); return; }
		else m_lpNetworkList->append(lpN);
	}

	if(config.readBlock((char *)&cnt,sizeof(cnt))<0){ cantReadError(); return;};

	for(uint i=0;i<cnt;i++){
		KviProxyEntry *lpE=new KviProxyEntry;
		if(!readProxyEntry(lpE,&config)){ delete lpE; cantReadError(); return; }
		else m_lpProxyList->append(lpE);
	}

	//save the bool variables
	if(config.readBlock((char *)&iTimeout,sizeof(iTimeout))<0){ cantReadError(); return; };
	if(config.readBlock((char *)&cacheLookup,sizeof(cacheLookup))<0){ cantReadError(); return; };
	if(config.readBlock((char *)&reconnectOnDisconnect,sizeof(reconnectOnDisconnect))<0){ cantReadError(); return; };
	if(config.readBlock((char *)&iCurNetwork,sizeof(iCurNetwork))<0){ cantReadError(); return; };
	if(config.readBlock((char *)&reconnectOnConnectFailed,sizeof(reconnectOnConnectFailed))<0){ cantReadError(); return; };
	if(config.readBlock((char *)&bUseProxy,sizeof(bUseProxy))<0){ cantReadError(); return; };
	if(config.readBlock((char *)&bUseSocksV5,sizeof(bUseSocksV5))<0){ cantReadError(); return; };

	config.close();
	_debug_leavetrace("readSettings");
}

//============ cantWriteError ============//
void KviServerManager::cantWriteError()
{
	_debug_entertrace("cantWriteError");
	KMsgBox::message(0,i18n("WARNING File I/O Error"),
			 i18n("Can't write to the kvi.server.conf file.\nThe current server list will be lost."));
	_debug_leavetrace("cantWriteError");
}

//============ cantReadError ============//
void KviServerManager::cantReadError()
{
	_debug_entertrace("cantReadError");
	KMsgBox::message(0,i18n("WARNING File I/O Error"),i18n("Can't read the kvi.server.conf file.\nThe current server list will be lost."));
	_debug_leavetrace("cantReadError");
}

//============ saveSettings ============//

void KviServerManager::saveSettings()
{
	_debug_entertrace("saveSettings");
	QString szFileName(_macro_getKVircHomeDirectory("Config/kvi.server.conf"));

	QFile config(szFileName.data());
	if(!config.open(IO_WriteOnly|IO_Truncate)){ cantWriteError(); return; }
	if(config.writeBlock("KVIRCSRV\n",9)<0){ cantWriteError(); return; };
	if(config.writeBlock("KVIrc CONFIGURATION FILE - PLEASE DO NOT EDIT\n",46)<0){ cantWriteError(); return; };

	uint cnt=m_lpNetworkList->count();
	if(config.writeBlock((char *)&cnt,sizeof(cnt))<0){ cantWriteError(); return; };
	KviIrcNetwork *lpN;
	for(lpN=m_lpNetworkList->first();lpN;lpN=m_lpNetworkList->next()){
		if(!writeNetworkEntry(lpN,&config)){ cantWriteError(); return; };
	}
	cnt=m_lpProxyList->count();
	if(config.writeBlock((char *)&cnt,sizeof(cnt))<0){ cantWriteError(); return; };
	KviProxyEntry *lpE;
	for(lpE=m_lpProxyList->first();lpE;lpE=m_lpProxyList->next()){
		if(!writeProxyEntry(lpE,&config)){ cantWriteError(); return; };
	}
	//save the bool variables
	if(config.writeBlock((char *)&iTimeout,sizeof(iTimeout))<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&cacheLookup,sizeof(cacheLookup))<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&reconnectOnDisconnect,sizeof(reconnectOnDisconnect))<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&iCurNetwork,sizeof(iCurNetwork))<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&reconnectOnConnectFailed,sizeof(reconnectOnConnectFailed))<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&bUseProxy,sizeof(bUseProxy))<0){ cantWriteError(); return; };
	if(config.writeBlock((char *)&bUseSocksV5,sizeof(bUseSocksV5))<0){ cantWriteError(); return; };
	if(m_lpNetworkList->isEmpty())iCurNetwork=-1;
	else {
		if((iCurNetwork<0)||(iCurNetwork>=((int)m_lpNetworkList->count())))iCurNetwork=0;
	}
	config.close();
	_debug_leavetrace("saveSettings");
}

//============ writeNetworkEntry ============//

bool KviServerManager::writeNetworkEntry(KviIrcNetwork *lpN,QFile *f)
{
	_debug_entertrace("writeNetworkEntry");
	_range_valid(f);
	_range_valid(lpN);
	if(!writeString(lpN->szName,f))return false;
	uint cnt=lpN->serverList->count();
	if(f->writeBlock((char *)&cnt,sizeof(cnt))<0)return false;
	KviServerEntry *lpE;
	for(lpE=lpN->serverList->first();lpE;lpE=lpN->serverList->next()){
		if(!writeServerEntry(lpE,f))return false;
	}
	_debug_leavetrace("writeNetworkEntry");
	return true;
}


//============ readNetworkEntry ============//

bool KviServerManager::readNetworkEntry(KviIrcNetwork *lpN,QFile *f)
{
	_debug_entertrace("readNetworkEntry");
	_range_valid(f);
	_range_valid(lpN);
	if(!readString(lpN->szName,f))return false;
	uint cnt;
	if(f->readBlock((char *)&cnt,sizeof(cnt))<0)return false;
	lpN->serverList=new QList<KviServerEntry>;
	lpN->serverList->setAutoDelete(true);
	for(uint i=0;i<cnt;i++){
		KviServerEntry *lpE=new KviServerEntry;
		if(!readServerEntry(lpE,f)){
			while(!lpN->serverList->isEmpty())lpN->serverList->removeFirst();
			delete lpN->serverList;
			delete lpE;
			return false;
		} else lpN->serverList->append(lpE);
	}
	_debug_leavetrace("readNetworkEntry");
	return true;
}


//============ readString ============//

bool KviServerManager::readString(QString &szString,QFile *f)
{
	_debug_entertrace("readString");
	_range_valid(f);
	uint len;
	if(f->readBlock((char *)&len,sizeof(len))<0)return false;
	char buf[4096];
	if(f->readBlock((char *)&buf,len)<0)return false;
	buf[len]=0;
	szString=QString(buf,len+1);

	_debug_leavetrace("readString");
	return true;
}

//============ writeString ============//

bool KviServerManager::writeString(QString &szString,QFile *f)
{
	_debug_entertrace("writeString");
	_range_valid(f);
	uint len=szString.length();
	if(f->writeBlock((char *)&len,sizeof(len))<0)return false;
	if(f->writeBlock((char *)szString.data(),len)<0)return false;
	_debug_leavetrace("writeString");
	return true;
}

//============ writeServerEntry ============//

bool KviServerManager::writeServerEntry(KviServerEntry *lpE,QFile *f)
{
	_debug_entertrace("writeServerEntry");
	_range_valid(f);
	_range_valid(lpE);
	if(!writeString(lpE->szAddress,f))return false;
	if(!writeString(lpE->szIp,f))return false;
	if(!writeString(lpE->szPassword,f))return false;
	if(!writeString(lpE->szDescription,f))return false;
	if(f->writeBlock((char *)&(lpE->iPort),sizeof(lpE->iPort))<0)return false;
	_debug_leavetrace("writeServerEntry");
	return true;
}

//============ readServerEntry ============//

bool KviServerManager::readServerEntry(KviServerEntry *lpE,QFile *f)
{
	_debug_entertrace("readServerEntry");
	_range_valid(f);
	_range_valid(lpE);
	if(!readString(lpE->szAddress,f))return false;
	if(!readString(lpE->szIp,f))return false;
	if(!readString(lpE->szPassword,f))return false;
	if(!readString(lpE->szDescription,f))return false;
	if(f->readBlock((char *)&(lpE->iPort),sizeof(lpE->iPort))<0)return false;
	_debug_leavetrace("readServerEntry");
	return true;
}

//============ copyProxyList ============//

void KviServerManager::copyProxyList(QList<KviProxyEntry> *dst,QList<KviProxyEntry> *src)
{
	_debug_entertrace("copyProxyList");
	_range_valid(src && dst);
	KviProxyEntry *lpE;
	for(lpE=src->first();lpE;lpE=src->next()){
		KviProxyEntry *lpN=new KviProxyEntry;
		copyProxyEntry(lpN,lpE);
		dst->append(lpN);
	}
	_debug_leavetrace("copyProxyList");
}

//============ copyNetworkList ============//

void KviServerManager::copyNetworkList(QList<KviIrcNetwork> *dst,QList<KviIrcNetwork> *src)
{
	_debug_entertrace("copyNetworkList");
	_range_valid(src && dst);
	KviIrcNetwork *lpE;
	for(lpE=src->first();lpE;lpE=src->next()){
		KviIrcNetwork *lpN=new KviIrcNetwork;
		copyNetworkEntry(lpN,lpE);
		dst->append(lpN);
	}
	_debug_leavetrace("copyNetworkList");
}

//============ copyNetworkEntry ============//

void KviServerManager::copyNetworkEntry(KviIrcNetwork *dst,KviIrcNetwork *src)
{
	_debug_entertrace("copyNetworkEntry");
	_range_valid(src && dst);
	dst->szName = src->szName.data();
	dst->serverList=new QList<KviServerEntry>;
	dst->serverList->setAutoDelete(true);
	KviServerEntry *lpE;
	for(lpE=src->serverList->first();lpE;lpE=src->serverList->next()){
		KviServerEntry *lpN=new KviServerEntry;
		copyServerEntry(lpN,lpE);
		dst->serverList->append(lpN);
	}
	_debug_leavetrace("copyNetworkEntry");
}

//============ copyServerEntry ============//

void KviServerManager::copyServerEntry(KviServerEntry *dst,KviServerEntry *src)
{
	_debug_entertrace("copyServerEntry");
	_range_valid(src && dst);
	dst->szAddress     = src->szAddress.data();
	dst->szIp          = src->szIp.data();
	dst->szPassword    = src->szPassword.data();
	dst->szDescription = src->szDescription.data();
	dst->iPort         = src->iPort;
	_debug_leavetrace("copyServerEntry");
}

//============ copyProxyEntry ============//

void KviServerManager::copyProxyEntry(KviProxyEntry *dst,KviProxyEntry *src)
{
	_debug_entertrace("copyProxyEntry");
	_range_valid(src && dst);
	dst->szAddress  = src->szAddress.data();
	dst->szIp       = src->szIp.data();
	dst->szPassword = src->szPassword.data();
	dst->szUsername = src->szUsername.data();
	dst->iPort      = src->iPort;
	_debug_leavetrace("copyProxyEntry");
}

//============ writeProxyEntry ============//

bool KviServerManager::writeProxyEntry(KviProxyEntry *lpE,QFile *f)
{
	_debug_entertrace("writeProxyEntry");
	_range_valid(f);
	_range_valid(lpE);
	if(!writeString(lpE->szAddress,f))return false;
	if(!writeString(lpE->szIp,f))return false;
	if(!writeString(lpE->szPassword,f))return false;
	if(!writeString(lpE->szUsername,f))return false;
	if(f->writeBlock((char *)&lpE->iPort,sizeof(lpE->iPort))<0)return false;
	_debug_leavetrace("writeProxyEntry");
	return true;
}

//============ readProxyEntry ============//

bool KviServerManager::readProxyEntry(KviProxyEntry *lpE,QFile *f)
{
	_debug_entertrace("readProxyEntry");
	_range_valid(f);
	_range_valid(lpE);
	if(!readString(lpE->szAddress,f))return false;
	if(!readString(lpE->szIp,f))return false;
	if(!readString(lpE->szPassword,f))return false;
	if(!readString(lpE->szUsername,f))return false;
	if(f->readBlock((char *)&(lpE->iPort),sizeof(lpE->iPort))<0)return false;
	_debug_leavetrace("readProxyEntry");
	return true;
}

//============ getServer ============//

KviServerEntry * KviServerManager::getServer(uint index)
{
	_debug_entertrace("getServer");
	if(m_lpNetworkList->isEmpty())return 0;
	if((iCurNetwork<0) || (iCurNetwork>=((int)m_lpNetworkList->count())))iCurNetwork=0;
	KviIrcNetwork *lpN=m_lpNetworkList->at(iCurNetwork);
	if(!lpN)return 0;
	if(lpN->serverList->isEmpty())return 0;
	if(index>=lpN->serverList->count())return 0;
	_debug_leavetrace("getServer");
	return lpN->serverList->at(index);
}

//============ setCurrentServer ============//

void KviServerManager::setCurrentServer(QString &szServer,unsigned short int iPort )
{
	_debug_entertrace("setCurrentServer");
	KviIrcNetwork *lpI;
	iCurNetwork=0;
	if((iPort>60000)||(iPort<1))iPort=6667;
	for(lpI=m_lpNetworkList->first();lpI;lpI=m_lpNetworkList->next()){
		KviServerEntry *lpE;
		for(lpE=lpI->serverList->first();lpE;lpE=lpI->serverList->next()){
			if(!strcasecmp(lpE->szAddress.data(),szServer.data())){
				//got it
				lpE->iPort=iPort;
				lpI->serverList->setAutoDelete(false);
				lpI->serverList->removeRef(lpE);
				lpI->serverList->insert(0,lpE);
				lpI->serverList->setAutoDelete(true);
				return;
			}
		}
		iCurNetwork++;
	}
	//Not found till now
	//check if we have a __UNKNOWNNET entry
	iCurNetwork=0;
	KviIrcNetwork *lpT=0;
	for(lpI=m_lpNetworkList->first();lpI && (!lpT);lpI=m_lpNetworkList->next()){
		if(lpI->szName=="__UNKNOWNNET")lpT=lpI;
		else iCurNetwork++;
	}
	if(!lpT){ //new network : __UNKNOWNNET <---first place in alphabetic order
		lpT=new KviIrcNetwork;
		lpT->szName="__UNKNOWNNET";
		lpT->serverList=new QList<KviServerEntry>;
		lpT->serverList->setAutoDelete(true);
		iCurNetwork=0;
	}
	KviServerEntry *lpEntry=new KviServerEntry;
	lpEntry->szAddress=szServer.copy();
	lpEntry->iPort=iPort;
	lpEntry->szIp=KVI_EMPTY_IP;
	lpEntry->szPassword="";
	lpEntry->szDescription="Unknown location : set by /server command";
	lpT->serverList->insert(0,lpEntry);
	m_lpNetworkList->insert(0,lpT);
	_debug_leavetrace("setCurrentServer");
}


//============ getCurNetworkName ============//

void KviServerManager::getCurNetworkName(QString &szNet)
{
	_debug_entertrace("getCurNetworkName");
	szNet="No network";
	if(m_lpNetworkList->isEmpty())return;
	if((iCurNetwork<0) || (iCurNetwork>=((int)m_lpNetworkList->count())))iCurNetwork=0;
	KviIrcNetwork *lpN=m_lpNetworkList->at(iCurNetwork);
	szNet=lpN->szName.copy();
	_debug_leavetrace("getCurNetworkName");
}


//
// $Log: kvi_srv.cpp,v $
// Revision 1.5  1998/09/25 15:58:36  pragma
// Moving ti use the KviProxy class.
//
// Revision 1.4  1998/09/20 20:23:53  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.3  1998/09/16 17:17:00  fritz
// Starting i18n.
//
// Revision 1.2  1998/09/16 16:13:24  pragma
// Moving to use a dynamic kvirc home directory.
// Big commit :)
//
//
