//   $Id: kvi_socket.cpp,v 1.6 1998/09/25 15:58:32 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this library; see the file COPYING.LIB.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviIrcSocket"

#include "kvi_support.h"
#include "kvi_socket.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_app.h"
#include "kvi_status.h"
#include "kvi_stbar.h"
#include "kvi_sparser.h"
#include "kvi_proxy.h"

#include <qtimer.h>

#include <qstring.h> //includes <string.h> too

#include <stdio.h>
#include <stdarg.h> //for va_list

#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdarg.h>
#include <arpa/inet.h>
#include <netdb.h>

// False i18n for xgettext
#define __i18n__(___text) ___text

const char * KviIrcSocket::errTable[KVI_SOCKERR_ERR_NUM]={
	__i18n__("Success"),
	__i18n__("You're already connected to a server"),
	__i18n__("Another connection attempt in progress"),
	__i18n__("System panic : Unable to create a inet/stream socket"),
	__i18n__("System panic : Unable to set the socket as non blocking (fcntl failed)"),
	__i18n__("System panic : Bad file descriptor"),
	__i18n__("System panic : The address is out of our address space (Unable to resolve hosts that aren't on the earth)"),
	__i18n__("System panic : The socket is not a socket (kernel is drunk?)"),
	__i18n__("Connection refused by remote host"),
	__i18n__("Operation timed out at socket level"),
	__i18n__("Network is unreachable"),
	__i18n__("Address already in use..."),
	__i18n__("Another socket operation in progress (try rebooting your system)"),
	__i18n__("Non blocking operation in progress"),
	__i18n__("Panic : a horrible UNKNOWN error occured"),
	__i18n__("Can't complete operation : no active connection"),
	__i18n__("Can't write to socket ,remote end has reading disabled"),
	__i18n__("Operation interrupted by a signal"),
	__i18n__("Low level I/O error"),
	__i18n__("Connection attempt timed out"),
	__i18n__("Connection closed by remote host"),
	__i18n__("Connection attempt aborted"),
	__i18n__("Unable to resolve localhost"),
	__i18n__("Host not found"),
	__i18n__("The host have no DNS entry"),
	__i18n__("DNS Fatal error : nameserver crashed?"),
	__i18n__("DNS Temporaneous fault (nameserver sleeping , or your modem is off?)"),
	__i18n__("Can't resolve hostname"),
	__i18n__("No route to host"),
	__i18n__("System call interrupted by a signal"),
	__i18n__("Permission denied"),
	__i18n__("Machine is not on the network (Drinkin' a beer in the pub on the other side of the road...)"),
	__i18n__("Protocol error"),
	__i18n__("Communication error on send"),
	__i18n__("Bad protocol"),
	__i18n__("Unsupported protocol"),
	__i18n__("Unsupported socket type"),
	__i18n__("Operation not supported on transport endpoint"),
	__i18n__("Unsupported address family"),
	__i18n__("Network is down"),
	__i18n__("Network dropped connection because of reset"),
	__i18n__("Software caused connection abort"),
	__i18n__("No buffer space available"),
	__i18n__("Remote host is down"),
	__i18n__("Remote I/O error (server crashed?)"),
	__i18n__("Error 0 - operation failed")
};

const twoErr KviIrcSocket::trnTable[KVI_TRANSLATE_TAB_NUM]={
	{ EBADF           , KVI_SOCKERR_BADDESCRIPTOR     },
	{ EFAULT          , KVI_SOCKERR_OUTOFADDRESSSPACE },
	{ ENOTSOCK        , KVI_SOCKERR_NOTASOCKET        },
	{ EISCONN         , KVI_SOCKERR_ALREADY           },
	{ ECONNREFUSED    , KVI_SOCKERR_CONNREFUSED       },
	{ ETIMEDOUT       , KVI_SOCKERR_SOCKTIMEOUT       },
	{ ENETUNREACH     , KVI_SOCKERR_NETUNREACH        },
	{ EADDRINUSE      , KVI_SOCKERR_ADDRINUSE         },
	{ EALREADY        , KVI_SOCKERR_PANICINPROGRESS   },
	{ EINPROGRESS     , KVI_SOCKERR_NONBLOCK          },
	{ EINVAL          , KVI_SOCKERR_CANTWRITE         },
	{ EAGAIN          , KVI_SOCKERR_NONBLOCK          },
	{ EINTR           , KVI_SOCKERR_INTERRUPT         },
	{ EIO             , KVI_SOCKERR_IOERROR           },
	{ EHOSTUNREACH    , KVI_SOCKERR_HOSTUNREACH       },
	{ EINTR           , KVI_SOCKERR_SIGINTERRUPT      },
	{ EACCES          , KVI_SOCKERR_NOPERMISSION      },
	{ ENONET          , KVI_SOCKERR_NOTONNET          },
	{ EPROTO          , KVI_SOCKERR_PROTOERR          },
	{ ECOMM           , KVI_SOCKERR_COMMSEND          },
	{ EPROTOTYPE      , KVI_SOCKERR_BADPROTOCOL       },
	{ ENOPROTOOPT     , KVI_SOCKERR_UNSUPPPROTO       },
	{ EPROTONOSUPPORT , KVI_SOCKERR_UNSUPPPROTO       },
	{ ESOCKTNOSUPPORT , KVI_SOCKERR_UNSUPPSOCKET      },
	{ EOPNOTSUPP      , KVI_SOCKERR_OPNOTSUPP         },
	{ EPFNOSUPPORT    , KVI_SOCKERR_UNSUPPPROTO       },
	{ EAFNOSUPPORT    , KVI_SOCKERR_UNSUPADDRFAM      },
	{ ENETDOWN        , KVI_SOCKERR_NETDOWN           },
	{ ENETRESET       , KVI_SOCKERR_NETRESET          },
	{ ECONNABORTED    , KVI_SOCKERR_SOFTWCONNABORT    },
	{ ENOBUFS         , KVI_SOCKERR_NOBUFFERSPACE     },
	{ ENOTCONN        , KVI_SOCKERR_NOTCONNECTED      },
	{ EHOSTDOWN       , KVI_SOCKERR_HOSTDOWN          },
	{ EREMOTEIO       , KVI_SOCKERR_REMOTEIOERROR     },
	{ 0               , KVI_SOCKERR_ERROR0            }
};

//============ KviIrcSocket ============//

KviIrcSocket::KviIrcSocket(KviFrame *lpFrame)
{
	//setup
	_debug_entertrace("KviIrcSocket");
	m_lpFrm                  =lpFrame;
	m_bConnected             =false;
	m_bInProgress            =false;
	m_iTimeout               =30;
	m_lpTimer                =new QTimer(this);
	m_lpWriteNotifier        =0;
	m_lpReadNotifier         =0;
	m_szLastIncompleteMessage="";
	m_lpStat                 =m_lpFrm->m_lpStatusBar;
	m_lpServerParser         =m_lpFrm->m_lpServerParser;
	connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(timeoutSlot()));
	_debug_leavetrace("KviIrcSocket");
}

//============ ~KviIrcSocket ============//

KviIrcSocket::~KviIrcSocket()
{
	//destructor
	_debug_entertrace("~KviIrcSocket");
	if(m_lpTimer->isActive())m_lpTimer->stop();
	delete m_lpTimer;
	killNotifiers();
	if(m_bConnected||m_bInProgress) ::close(m_Sock);
	_debug_leavetrace("~KviIrcSocket");
}

//============ disableNotifiers ============//

void KviIrcSocket::disableNotifiers()
{
	_debug_entertrace("disableNotifiers");
	if(m_lpReadNotifier){
		if(m_lpReadNotifier->isEnabled())m_lpReadNotifier->setEnabled(false);
	}
	if(m_lpWriteNotifier){
		if(m_lpWriteNotifier->isEnabled())m_lpWriteNotifier->setEnabled(false);
	}
	_debug_leavetrace("disableNotifiers");
}

//============ killNotifiers ============//

void KviIrcSocket::killNotifiers()
{
	_debug_entertrace("killNotifiers");
	if(m_lpReadNotifier){
		if(m_lpReadNotifier->isEnabled())m_lpReadNotifier->setEnabled(false);
		delete m_lpReadNotifier;
		m_lpReadNotifier=0;
	}
	if(m_lpWriteNotifier){
		if(m_lpWriteNotifier->isEnabled())m_lpWriteNotifier->setEnabled(false);
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	_debug_leavetrace("killNotifiers");
}

//============ resetSocket ============//

void KviIrcSocket::resetSocket()
{
	_debug_entertrace("resetSocket");
	if(m_bConnected||m_bInProgress) ::close(m_Sock);
	m_bConnected=false;
	m_bInProgress=false;
	if(m_lpTimer->isActive())m_lpTimer->stop();
	killNotifiers();
	m_szLastIncompleteMessage="";
	_debug_leavetrace("resetSocket");
}

//============ connectTo ============//

int KviIrcSocket::connectTo(struct in_addr servAddr,unsigned short int iPort)
{
	_debug_entertrace("connectTo");
	if(m_bConnected)return KVI_SOCKERR_ALREADY;
	if(m_bInProgress)return KVI_SOCKERR_INPROGRESS;
	//Create the socket
	m_Sock = ::socket(PF_INET,SOCK_STREAM,0);
	if (m_Sock < 0)return KVI_SOCKERR_NOSOCKET;
	m_bInProgress=true;
	//Make the socket non blocking
	if(fcntl(m_Sock, F_SETFL, O_NONBLOCK) < 0){
		resetSocket();
		return KVI_SOCKERR_NONBLOCKFAIL;
	}; 
	fillSockAddr(&m_sAddr,servAddr,iPort);
	if(m_iTimeout<3)m_iTimeout=3; //At least 3 seconds...
	m_lpTimer->start(m_iTimeout * 1000);
	//Let's go...
	m_uiSentBytes=0;
	m_uiSentPackets=0;
	m_uiReceivedBytes=0;
	m_uiReceivedPackets=0;
	//	debug("connecting to %u on port %d",servAddr.s_addr,iPort);
	if(::connect( m_Sock, (struct sockaddr*)(&m_sAddr), sizeof( m_sAddr) )<0){
		if(errno != EINPROGRESS){
			int nSockErr=errno;
			ksize_t iSize=sizeof(int);
			if(errno==0){
				if(getsockopt(m_Sock,SOL_SOCKET,SO_ERROR,(void *)&nSockErr,&iSize)==-1)nSockErr=0;
			}
			resetSocket();
			return translateFromSystemError(nSockErr);
		}
	}
	//	debug("connect exited");
	//Create the notifiers...
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read);
	m_lpWriteNotifier= new QSocketNotifier(m_Sock, QSocketNotifier::Write);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(socketReadData()));
	QObject::connect(m_lpWriteNotifier, SIGNAL(activated(int)), this, SLOT(socketWriteData()));
	m_lpReadNotifier->setEnabled(true);
	m_lpWriteNotifier->setEnabled(true);
	_debug_leavetrace("connectTo");
	return KVI_SOCKERR_SUCCESS;
}

//============ translateFromSystemError ============//

int KviIrcSocket::translateFromSystemError(int err)
{
	_debug_entertrace("translateFromSystemError");
	for(int i=0;i<KVI_TRANSLATE_TAB_NUM;i++){
		if(err==trnTable[i].sysError)return trnTable[i].kviError;
	}
	//	debug("\nGOT ERROR = %d\n",err);
	return KVI_SOCKERR_UNKNOWN;
	_debug_leavetrace("translateFromSystemError");
}

//============ timeoutSlot ============//

void KviIrcSocket::timeoutSlot()
{
	_debug_entertrace("timeoutSlot");
	m_lpTimer->stop();
	if((!m_bConnected)&&(m_bInProgress)){
		m_lpFrm->handleSocketError(KVI_SOCKERR_TIMEDOUT);
		m_lpFrm->handleConnectFailed();
		resetSocket();
	}
	_debug_leavetrace("timeoutSlot");
}

//============= fillSockAddr =============//

void KviIrcSocket::fillSockAddr(struct sockaddr_in *lpSA,struct in_addr in,unsigned short int port)
{
	_debug_entertrace("fillSockAddr");
	lpSA->sin_family	=	AF_INET;
	lpSA->sin_port		=	htons(port);
	lpSA->sin_addr		=	in;
	_debug_leavetrace("fillSockAddr");
}

//============ writeRawData ============//

void KviIrcSocket::writeRawData(const char *buffer,int len)
{
	_debug_entertrace("writeRawData");
	int bySnt=write(m_Sock,buffer,len);
	if(bySnt<len)debug("WARNING : Unable to send raw data to socket (proxy connection?)");
	_debug_leavetrace("writeRawData");
}

//============ send ============//

int KviIrcSocket::sendData(const char *buffer)
{
	_debug_entertrace("send");
	if(!m_bConnected)return KVI_SOCKERR_NOTCONNECTED;
	int bytSnt;
	QString szBuf(buffer);
	szBuf+="\r\n";
	bytSnt=write(m_Sock,szBuf.data(),strlen(buffer)+2);
	if(bytSnt<0){
		disableNotifiers();
		m_lpFrm->handleSocketError(translateFromSystemError(errno));
		m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_ERROR,
		i18n("You have lost your connection , probably due to a transmission hardware error,please restart KVIrc."));
		QTimer::singleShot(300,this,SLOT(connectionLost()));
		return translateFromSystemError(errno);
	} else {
		m_uiSentBytes+=bytSnt;
		m_uiSentPackets++;
		m_lpStat->increaseLedState();
		return KVI_SOCKERR_SUCCESS;
	}
	_debug_leavetrace("send");
}

//============ connectionLost ============//

void KviIrcSocket::connectionLost()
{
	_debug_entertrace("connectionLost");
	m_lpFrm->handleSocketDisconnected();
	resetSocket();
	_debug_leavetrace("connectionLost");
}

int kvi_irc_vsnprintf(char *buffer,const char *fmt,va_list list,bool *bTruncated)
{
//	_range_valid(fmt);
//	_range_valid(buffer);
	register char *p = buffer;
	char *argString;
	long argValue;
	unsigned long argUValue;
	char numberBuffer[64]; //enough ? 10 is enough for 32bit unsigned int...
	char *pNumBuf;
	unsigned int tmp;
	*bTruncated = false;
	int len = 512;

	for (p=buffer ; *fmt ; ++fmt) {
		if(len < 3)goto truncate;
		//copy up to a '%'
		if (*fmt != '%') {
			*p++ = *fmt;
			--len;
			continue;
		}
		++fmt; //skip this '%'
		switch(*fmt){
			case 's': //string
				argString = va_arg(list,char *);
				if(!argString)argString = "[!NULL!]";
				//check for space...
				while(*argString){
					*p++ = *argString++;
					if(--len < 3)goto truncate;
				}
				continue;
			case 'd': //signed integer
				argValue = va_arg(list,int);
				if(argValue < 0){ //negative integer
					*p++ = '-';
					if(--len < 3)goto truncate; //place just for CRLF
					argValue = -argValue; //need to have it positive
				}
				//write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argValue / 10;
					*pNumBuf++ = argValue - (tmp * 10) + '0';
				} while((argValue = tmp));
				//copy now....
				do {
					*p++ = *--pNumBuf;
					if(--len < 3)goto truncate;
				} while(pNumBuf != numberBuffer);
				continue;
			case 'u': //unsigned integer
				argUValue = va_arg(list,unsigned int); //many implementations place int here
				//write the number in a temporary buffer
				pNumBuf = numberBuffer;
				do {
					tmp = argUValue / 10;
					*pNumBuf++ = argUValue - (tmp * 10) + '0';
				} while((argUValue = tmp));
				//copy now....
				if(--len < 3)goto truncate; //no place for digits
				do {
					*p++ = *--pNumBuf;
					if(--len < 3)goto truncate;
				} while(pNumBuf != numberBuffer);
				continue;
			case 'c': //char
				//
				// I'm not sure about this...
				// In the linux kernel source the
				// unsigned char is extracted from an integer type.
				// We assume that gcc stacks a char argument
				// as sizeof(int) bytes value.
				// Is this always true ?
				//
				*p++ = (char)va_arg(list,int);
				--len;
				continue;
			default: //a normal percent
				*p++ = '%';  //write it
				if(--len < 3)goto truncate; //not enough space for next char
				if(*fmt){        //this if is just in case that we have a % at the end of the string.
					*p++ = *fmt; //and write this char
					--len;
				}
				continue;
		}
	}
	//succesfull finish
//	__range_valid(len >= 2);
	*p++ = '\r';
	*p   = '\n';
	return ((p-buffer)+1);
truncate:
//	__range_valid(len >= 2);
	*bTruncated = true;
	*p++ = '\r';
	*p   = '\n';
	return ((p-buffer)+1);
}

//============ sendFmtData ============//

int KviIrcSocket::sendFmtData(const char *fmtBuf,...)
{
	_debug_entertrace("sendFmtData");
	if(!m_bConnected)return KVI_SOCKERR_NOTCONNECTED;
	char szText[524];
	va_list list;
	va_start(list,fmtBuf);
	bool bTruncated;
	int len = kvi_irc_vsnprintf(szText,fmtBuf,list,&bTruncated);
	if(bTruncated)debug("WARNING: Data truncated to first 512 bytes");
	va_end(list);
	int bytSnt=write(m_Sock,szText,len);
	if(bytSnt<0){
		disableNotifiers();
		m_lpFrm->handleSocketError(translateFromSystemError(errno));
		m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_ERROR,
		i18n("You have lost your connection , probably due to a transmission hardware error,please restart KVIrc."));
		QTimer::singleShot(300,this,SLOT(connectionLost()));
		return translateFromSystemError(errno);
	}
	m_uiSentBytes+=bytSnt;
	m_uiSentPackets++;
	m_lpStat->increaseLedState();
	return KVI_SOCKERR_SUCCESS;
	_debug_leavetrace("sendFmtData");
}

//============ getErr ============//

void KviIrcSocket::getErr(int nErr,QString &szErr)
{
	_debug_entertrace("getErr");
	if(nErr<KVI_SOCKERR_ERR_NUM)szErr=klocale->translate(errTable[nErr]);
	else szErr=i18n("Unknown error");
	_debug_leavetrace("getErr");
}

//============ disconnect ============//

void KviIrcSocket::disconnect()
{
	_debug_entertrace("disconnect");
	if(m_bInProgress){
		m_lpFrm->handleSocketError(KVI_SOCKERR_ABORTED);
		m_lpFrm->handleConnectFailed();
		resetSocket();
		return;
	}
	if(!m_bConnected){
		_debug_fatal("KviSocket::Disconnect() : Socket not connected!!!!!");
	} else {
		m_lpFrm->handleSocketDisconnected();
		resetSocket();
	}
	_debug_leavetrace("disconnect");
}

unsigned long KviIrcSocket::getSockAddress(){
	if((!m_bConnected)&&(!m_bInProgress))return 0;
	//Don't ask me why...but the ksize_t don't want
	//to work on my RedHat 5... it defined as
	//int * , and my getsockname wants uint *
	struct sockaddr_in name; 
	ksize_t len = sizeof(name);
	getsockname(m_Sock, (struct sockaddr *) &name, &len);
	return htonl(name.sin_addr.s_addr);
}

int KviIrcSocket::getLocalHostIp(QString &szName){
	if((!m_bConnected)&&(!m_bInProgress)){
		return KVI_SOCKERR_NOSOCKNAME;
	}
	if(m_Sock<0)return KVI_SOCKERR_NOSOCKET;
	struct sockaddr_in name; ksize_t len = sizeof(name);
	if(getsockname(m_Sock, (struct sockaddr *) &name, &len)<0){
		return KVI_SOCKERR_NOSOCKNAME;
	}
	szName=QString(inet_ntoa(name.sin_addr));
	return KVI_SOCKERR_SUCCESS;
}

int KviIrcSocket::getHostByName(const char *szName,KviHostDescription *lpkHD){

	struct hostent *lpHostInfo;
	lpHostInfo = gethostbyname( szName );
	if ( !lpHostInfo ){
		switch(h_errno){
			case HOST_NOT_FOUND:
				return KVI_SOCKERR_HOSTNOTFOUND;
				break;
			case NO_ADDRESS:
				return KVI_SOCKERR_NOIPADDRESS;
				break;
			case NO_RECOVERY:
				return KVI_SOCKERR_DNSFATAL;
				break;
			case TRY_AGAIN:
				return KVI_SOCKERR_DNSTRYAGAIN;
				break;
			default:
				return KVI_SOCKERR_UNABLETORESOLVE;
				break;
		}
	}
	lpkHD->szName		= QString(lpHostInfo->h_name);
	if(lpHostInfo->h_aliases[0]){
		lpkHD->szAlias1	= QString(lpHostInfo->h_aliases[0]);
		if(lpHostInfo->h_aliases[1]){
			lpkHD->szAlias2	= QString(lpHostInfo->h_aliases[1]);
		} else lpkHD->szAlias2="";
	} else lpkHD->szAlias1="";
	lpkHD->szIp	= QString( inet_ntoa( *(struct in_addr*) lpHostInfo->h_addr ) );
	lpkHD->iAddress= *(struct in_addr*) lpHostInfo->h_addr;
	//	lpkHD->iPort=0;
	return KVI_SOCKERR_SUCCESS;
}

void KviIrcSocket::socketWriteData(){
	if(m_bInProgress){
		int nSockErr;
		ksize_t iSize=sizeof(int);
		if(getsockopt(m_Sock,SOL_SOCKET,SO_ERROR,(void *)&nSockErr,&iSize)==-1)nSockErr=EBADF;
		if(nSockErr!=0){
			if(m_lpFrm->m_lpProxy->isActive())
					m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_INTERNAL,
					i18n("[ SOCKS4 ] : Warning : connection failed with error %d"),nSockErr);
			disableNotifiers();
			m_lpFrm->handleSocketError(translateFromSystemError(nSockErr));
			m_lpFrm->handleConnectFailed();
			QTimer::singleShot(300,this,SLOT(resetSocket()));
		} else { //Succesfully connected...
			m_bInProgress=false;
			m_bConnected=true;
			m_lpWriteNotifier->setEnabled(false);
			delete m_lpWriteNotifier;
			m_lpWriteNotifier=0;
			m_lpFrm->handleSocketConnected();
		}
	} else debug("Socket write notifier should be disabled!!"); 
}

//============ dealWithProxy ============//

void KviIrcSocket::dealWithProxy(unsigned char reply,int len)
{
	_debug_entertrace("dealWithProxy");
	//ok..so check what we need now...
	if(len<2){  //too small packet...it should never happen that we read 1 byte...
				//but if it happens just fail...
		m_lpFrm->m_lpConsole->doOutput(KVI_OUT_INTERNAL,
			i18n("[ SOCKS ] : Warning , Received broken reply packet"));
		disableNotifiers();
		m_lpFrm->handleConnectFailed();
		m_lpFrm->m_lpProxy->clear();
		QTimer::singleShot(300,this,SLOT(resetSocket()));
		return;
	}
	if(m_lpFrm->m_lpProxy->state() == KVI_PROXY_SELECT){
		//need a selection packet : 2 bytes , second is method selected or FF if something is wrong
		//however for us 0xff is a secure failure , but anything else that is not equal to 0 or 2
		//indicates a wrong proxy version (tried to contact a V4 server with V5 proto ?)
		if(reply<0xff){
			m_lpFrm->m_lpProxy->methodSelected(reply);
			return;
		} else m_lpFrm->m_lpConsole->doOutput(KVI_OUT_INTERNAL,
			i18n("[ SOCKS ] : Warning , received reply : 0xff - No acceptable methods"));
	} else if(m_lpFrm->m_lpProxy->state()==KVI_PROXY_AUTH){
		//need a auth ack packet : 2 bytes , second is 0 on success
		if(reply==0){
			m_lpFrm->m_lpProxy->authComplete();
			return;
		} else m_lpFrm->m_lpConsole->doOutput(KVI_OUT_INTERNAL,i18n("[ SOCKS ] : Authorisation failed"));
	} else { //waiting for final response
		m_lpFrm->handleProxyReply(reply);
		//assure that the proxy is inactive now
		m_lpFrm->m_lpProxy->clear();
		return;	
	}
	//we failed
	disableNotifiers();
	m_lpFrm->handleConnectFailed();
	m_lpFrm->m_lpProxy->clear();
	QTimer::singleShot(300,this,SLOT(resetSocket()));
	_debug_leavetrace("dealWithProxy");
}


void KviIrcSocket::socketReadData(){
	if(m_bInProgress){
		socketWriteData();
		return;
	}
	//fprintf(stdout,"\nSOCKET READ DATA\n##################\n");
	char szIncomingData[1025];
   	bzero(szIncomingData, 1025); //just a space for the terminator
	int readLength = read(m_Sock,szIncomingData, 1024);
	if (readLength<= 0) {
		if(m_lpReadNotifier->isEnabled()){ //Do it ony once
			disableNotifiers();
			m_lpFrm->handleSocketError(KVI_SOCKERR_REMOTECLOSE);
			m_lpFrm->handleSocketDisconnected();
			QTimer::singleShot(300,this,SLOT(resetSocket()));
		} else _debug_fatal("OOpps....received read signal from an inexisting notifier....");
  	} else {
		if(m_lpFrm->m_lpProxy->isActive()){
			if(readLength>=2)dealWithProxy(szIncomingData[1],readLength);
			else dealWithProxy(0xFF,0);
			return;
		}
		m_uiReceivedBytes+=readLength;
		m_uiReceivedPackets++;
		QString szIncomingString(szIncomingData,readLength+1);
		//fprintf(stdout,"READ    :\n%s\n",szIncomingString.data());
		szIncomingString.prepend(m_szLastIncompleteMessage);
		//fprintf(stdout,"PREPEND :\n%s\n",szIncomingString.data());
		m_szLastIncompleteMessage = "";
		int lenToCopy;
		char *lpStrData;
		const char *d;
		int idx;
		//Split lines and remove the CRLF
		//all this gnary hack just to save a couple 
		//(but maybe 50...) of ms each time :)
		int strSize=0;
		while (!szIncomingString.isEmpty()){
			//find '\n' in the string
			//subst: int idx= szIncomingString.find('\n');
			lpStrData=szIncomingString.data();
			d = strchr(lpStrData,'\n' );
			idx = ( d ? (int)(d-lpStrData) : -1 );	
			if (idx > -1){
				if(idx > 1){
					//Subst: szNewDataLine=szIncomingString.left(idx);
					//cut also the CR so (idx-1)=CR index
					//
					// FUNNY NOTE:
					//   rfc1459 says that an IRC MESSAGE MUST END WITH A <CRLF>
					//   so If the server sends more messages in one buffer
					//   should always separate it by <CRLF>'s.
					//   well...try irc.nijenrode.nl :)
					//
					//lets say that idx=10=(LF index) so CR index=9 and is the 10th char..
					strSize=idx;
					if(szIncomingString[idx-1]!='\r')strSize++; //Add the 9th char too if there is no CR
					QString szNewDataLine(strSize); //up to CR (excl.) and space for \0 so here 10 (9+1) chars
					strncpy(szNewDataLine.data(),lpStrData,strSize-1 ); //copy 9 chars
					*(szNewDataLine.data()+(strSize-1)) = '\0'; //add the 10th
					//Disable the notifier while processing the data...
					//Just to have no overlapping calls in dialogs etc...
					m_lpReadNotifier->setEnabled(false);
					m_lpServerParser->parseMessage(szNewDataLine);
					m_lpReadNotifier->setEnabled(true);
					//Subst:szincomingString.remove(0,idx+2);
					lenToCopy=szIncomingString.length()-idx; //lets say the string is long 100 so lenToCopy=90
					if(lenToCopy)memmove(lpStrData,lpStrData+(idx+1),lenToCopy ); //start from the char after the LF and copy 90 (89+1)
					else szIncomingString="";
					((QByteArray) (szIncomingString)).resize(lenToCopy); //and resize to 90 (89+ '\0')
				} else {
					//fprintf(stdout,"\nWe found just a CRLF!\n");
					//I noticed that we newer run here...but it is possible...
					szIncomingString.remove(0,idx+1); //Just a (CR)LF...
				}
			} else {
				//fprintf(stdout,"NEWLINE NOT FOUND : %s\n",szIncomingString.data());
				m_szLastIncompleteMessage = szIncomingString.copy();
				szIncomingString = "";
			}
		}

	}
}
#include "m_kvi_socket.moc"
