//
//   File : kvi_mdi.cpp (/usr/cvs/kvirc/kvirc/kvi_mdi.cpp)
//   Last modified : Sun Nov 22 1998 16:31:49 by root@localhost.localdomain
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998-1999 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation ; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING. If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//


#define _KVI_DEBUG_CLASS_NAME_ "KviMdiManager"
#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_macros.h"
#include "kvi_mdi.h"
#include "kvi_frame.h"
#include "kvi_debug.h"
#include "kvi_taskbar.h"
#include "kvi_int.h"
#include "kvi_app.h"

#include <math.h>

#include <X11/Xlib.h>

#include <kwm.h>
#include <kiconloader.h>
#include <kmsgbox.h>

//============ KviMdiManager ============//
KviMdiManager::KviMdiManager(KviFrame *parent) : QFrame(parent,"KviMdiManagerClass")
{
	_debug_entertrace("KviMdiManager");
	m_lpFrm=parent;
	m_bMinimizingAll=false;
	m_lpInt	=	parent->m_lpInt;
	setFrameStyle(WinPanel | Sunken);
	m_lpChildList = new QList<KviMdiChild>; //In fact a stack...
	m_lpChildList->setAutoDelete(false);
	m_lpTaskBar = new KviNewTaskBar(parent,this);
	m_lpFrm->addToolBar(m_lpTaskBar);
//	m_lpTaskBar->setBarPos(KToolBar::Bottom);
	linkChildren(m_lpTaskBar,this);
	m_lpWinListPopup = new QPopupMenu();
	setFocusPolicy(QWidget::StrongFocus);
	setFocusProxy(0);
	connect(m_lpTaskBar,SIGNAL(itemActivated(int,bool)),this,SLOT(taskbarButtonClicked(int,bool)));
//	connect(m_lpTaskBar,SIGNAL(itemDeactivated(int)),this,SLOT(taskbarButtonDeactivated(int)));
	connect(m_lpTaskBar,SIGNAL(taskBarPopupRequest(QPoint,int)),this,SLOT(taskbarPopupRequest(QPoint,int)));
	connect(m_lpWinListPopup,SIGNAL(activated(int)),this,SLOT(menuButtonClicked(int)));
	_debug_leavetrace("KviMdiManager");
}
//================= linkChildren ==============//
void KviMdiManager::linkChildren(QWidget *widget,QWidget *focusedOne)
{
	_debug_entertrace_huge("linkChildren");
	QList<QObject> *list = (QList<QObject> *)(widget->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QWidget *w = (QWidget *)list->at(i);
			linkChildren(w,focusedOne);
		}
	}
	widget->installEventFilter(this);
	if(widget != focusedOne){
		widget->setFocusPolicy(NoFocus);
		widget->setFocusProxy(focusedOne);
	}
	_debug_leavetrace_huge("linkChildren");
}
//============ ~KviMdiManager ============//
KviMdiManager::~KviMdiManager()
{
	_debug_entertrace("~KviMdiManager");
	while(!m_lpChildList->isEmpty()){
		KviMdiChild *lpC=m_lpChildList->last();
		m_lpChildList->removeLast();
		lpC->killWindow();
	}
	delete m_lpChildList;
	delete m_lpTaskBar;
	delete m_lpWinListPopup;
	_debug_leavetrace("~KviMdiManager");
}
//============ applyOptions ============//
void KviMdiManager::applyOptions()
{
	_debug_entertrace("applyOptions");
	if(!m_lpInt->pix_bk_mdi.isNull())setBackgroundPixmap(m_lpInt->pix_bk_mdi);
	else setBackgroundColor(m_lpInt->clr_bk_mdi);
	KviMdiChild *lpC=0;
	KviMdiChild *lastCh=0;
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		lpC->applyOptionsInternal();
		lpC->applyOptions();
		lastCh=lpC;
	}
	if(lastCh)lastCh->setFocus();
	_debug_leavetrace("applyOptions");
}
//============ focusTopChild ============//
void KviMdiManager::focusTopChild()
{
	_debug_entertrace("focusTopChild");
	//work around a qt(?) bug...right clicking
	//in the window when the application is not active
	//the main widget gets focused, but for some strange
	//reason it is not raised to the top of the windows stack...
	//Window wnd=(Window)topLevelWidget()->winId();
	Window wnd=(Window)m_lpFrm->winId();
	if(KWM::desktop(wnd)==KWM::currentDesktop()){
		if(!KWM::isActive(wnd))KWM::raise(wnd);
		KviMdiChild *lpC=topChildInZOrder();
		if(lpC){
//			if(lpC->m_bMinimized)lpC->minimizeWindow(); //force to restore
			lpC->raise();
			lpC->setFocus();
			if(!lpC->m_bLabelRaised)lpC->raiseLabel(true);
		}
	}
	_debug_leavetrace("focusTopChild");
}
//============ focusInEvent ============//
void KviMdiManager::focusInEvent(QFocusEvent *)
{
	_debug_entertrace("focusInEvent");
	focusTopChild();
	_debug_leavetrace("focusInEvent");
}
//============ eventFiter ============//
bool KviMdiManager::eventFilter(QObject *,QEvent *e)
{
	_debug_entertrace_huge("eventFiter");
	if(e->type()==Event_FocusIn){
//		debug("Event filter : object=%s [%s], focus in",ob->name(),ob->className());
		focusTopChild();
		return true;
	}
	if(e->type()==Event_FocusOut){
		return true;
	}
	_debug_leavetrace_huge("eventFiter");
	return false;
}
//============ mousePressEvent ============//
void KviMdiManager::mousePressEvent(QMouseEvent *e)
{
	_debug_entertrace("mousePressEvent");
	if(e->button() & RightButton)m_lpWinListPopup->popup(mapToGlobal(e->pos()));
	_debug_leavetrace("mousePressEvent");
}
//============ addChild ============//
void KviMdiManager::addChild(KviMdiChild *lpC,bool setGeom)
{
	_debug_entertrace("addChild");
	KviMdiChild *lpChild=0;
	m_lpChildList->append(lpC);
	m_lpTaskBar->addButton(lpC->id(),lpC->name(),m_lpFrm->m_lpIcon[lpC->type()],(lpC->type()==KVI_WND_TYPE_SEND));
	_macro_kviApplication->processEvents();
	//position the new child...(like cascade...)
	int chx=0;
	int chy=0;
	if(setGeom){
			for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
				if(lpChild==lpC && (lpChild->id()!=0))
					lpChild->setGeometry(chx,chy,KVI_MDI_CHILD_CASCADE_WIDTH,KVI_MDI_CHILD_CASCADE_HEIGHT);
				chx+=KVI_MDI_CHILD_CASCADE_STEP; 			chy+=KVI_MDI_CHILD_CASCADE_STEP;
				if(chy>(height()-(KVI_MDI_CHILD_CASCADE_STEP*2))){
					chy=0;
					chx+=KVI_MDI_CHILD_CASCADE_JUMPX;
				}
			}
	}
	lpC->show();
	lpC->raise();
	lpC->setFocus();
	lpC->raiseLabel(true);
	connect(lpC,SIGNAL(windowMaximized(KviMdiChild *)),this,SLOT(childMaximized(KviMdiChild *)));
	connect(lpC,SIGNAL(windowMinimized(KviMdiChild *)),this,SLOT(childMinimized(KviMdiChild *)));
	fillWinListPopup();
	_debug_leavetrace("addChild");
}
//============ addChildMinimized ============//
void KviMdiManager::addChildMinimized(KviMdiChild *lpC,bool setGeom)
{
	_debug_entertrace("addChildMinimized");
	m_lpChildList->insert(0,lpC);
	m_lpTaskBar->addButton(lpC->id(),lpC->name(),m_lpFrm->m_lpIcon[lpC->type()],(lpC->type()==KVI_WND_TYPE_SEND));
	_macro_kviApplication->processEvents();
	//position the new child...(like cascade...)
	int chx=0;
	int chy=0;
	KviMdiChild *lpChild=0;
	if(setGeom){
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			if(lpChild==lpC && (lpChild->id()!=0))lpChild->setGeometry(chx,chy,KVI_MDI_CHILD_CASCADE_WIDTH,KVI_MDI_CHILD_CASCADE_HEIGHT);
			chx+=KVI_MDI_CHILD_CASCADE_STEP;
			chy+=KVI_MDI_CHILD_CASCADE_STEP;
			if(chy>(height()-(KVI_MDI_CHILD_CASCADE_STEP*2))){
				chy=0;
				chx+=KVI_MDI_CHILD_CASCADE_JUMPX;
			}
		}
	}
	lpC->m_bMinimized=true;
	lpC->lower();
	lpC->hide();
	connect(lpC,SIGNAL(windowMaximized(KviMdiChild *)),this,SLOT(childMaximized(KviMdiChild *)));
	connect(lpC,SIGNAL(windowMinimized(KviMdiChild *)),this,SLOT(childMinimized(KviMdiChild *)));
	fillWinListPopup();
	focusTopChild();
	_debug_leavetrace("addChildMinimized");
}
//============ topChildInZOrder ============//
KviMdiChild * KviMdiManager::topChildInZOrder()
{
	_debug_entertrace("topChildInZOrder");
	return (m_lpChildList->isEmpty() ? 0 : m_lpChildList->last());
	_debug_leavetrace("topChildInZOrder");
}
//============ childFocusIn ============//
void KviMdiManager::childFocusIn(KviMdiChild *lpC)
{
	_debug_entertrace("childFocusIn");
//	debug("Gains focus = %s",lpC->name());
	KviMdiChild *lpFoc=0;
	for(lpFoc=m_lpChildList->first();lpFoc;lpFoc=m_lpChildList->next()){
		if(lpFoc != lpC){
			if(lpFoc->hasFocus())lpFoc->clearFocus(); //this may be senseless
			if(lpFoc->m_bLabelRaised)lpFoc->raiseLabel(false);
		}
	}
	if(m_lpChildList->removeRef(lpC)){
		m_lpChildList->append(lpC);
		m_lpTaskBar->setActiveButton(lpC->id());
	}
	lpC->raise();
//	debug("Focus gain done for %s",lpC->name());
	_debug_leavetrace("childFocusIn");
}
//============ childFocusOut ============//
void KviMdiManager::childFocusOut(KviMdiChild *)
{
	_debug_entertrace("childFocusOut");
	//do Nothing 'til now...
	_debug_leavetrace("childFocusOut");
}
KviMdiChild * KviMdiManager::findWindow(const char *szName){
	_debug_entertrace("findWindow");
	if(m_lpChildList->isEmpty() || (!szName))return 0;
	KviMdiChild *lpC=0;
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		if(!strcasecmp(lpC->name(),szName)){
			_debug_leavetrace("findWindow...found.");
			return lpC;
		}
	}
	debug("Unable to find child window %s",szName);
	return 0;
}
KviMdiChild * KviMdiManager::findWindow(const QString &szName){
	_debug_entertrace("findWindow");
	if(m_lpChildList->isEmpty() || szName.isEmpty())return 0;
	KviMdiChild *lpC=0;
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		QString szC(lpC->name());
		if(szName.lower()==szC.lower()){
			_debug_leavetrace("findWindow...found.");
			return lpC;
		}
	}
	debug("Unable to find child window %s",szName.data());
	return 0;
}
KviMdiChild * KviMdiManager::findWindow(int winId){
	_debug_entertrace("findWindow");
	if(m_lpChildList->isEmpty())return 0;
	KviMdiChild *lpC=0;
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		if(winId==lpC->id()){
			_debug_leavetrace("findWindow....found.");
			return lpC;
		}
	}
	debug("Unable to find child window %d",winId);
	return 0;
}
//============ removeWindow ============//
void KviMdiManager::removeWindow(int winId)
{
	_debug_entertrace("removeWindow");
	KviMdiChild *lpC=findWindow(winId);
	if(lpC)removeWindow(lpC);
	_debug_leavetrace("removeWindow");
}
//============ removeWindow ============//
void KviMdiManager::removeWindow(const char *szName)
{
	_debug_entertrace("removeWindow");
	KviMdiChild *lpC=findWindow(szName);
	if(lpC)removeWindow(lpC);
	_debug_leavetrace("removeWindow");
}
//============ removeWindow ============//
void KviMdiManager::removeWindow(KviMdiChild *lpC)
{
	_debug_entertrace("removeWindow");
	_range_valid(lpC);
	m_lpTaskBar->removeButton(lpC->id());
	lpC->hide();
	m_lpChildList->setAutoDelete(false);
	m_lpChildList->removeRef(lpC);
	lpC->killWindow();
	focusTopChild();
	fillWinListPopup();
	_debug_leavetrace("removeWindow");
}
void KviMdiManager::taskbarButtonClicked(int nId,bool bIsOn)
{
	//A taskbar button was clicked
	//If it bIsOn , focus the selected window (restoring if necessary)
	//else lower it (if possible, else minimize it) and focus a new one.
	_debug_entertrace("taskbarButtonClicked");
	KviMdiChild *lpC=findWindow(nId);
	if(lpC){
		if(bIsOn){
			if((!lpC->hasFocus())||(lpC->m_bMinimized)){
				if(lpC->m_bMinimized)lpC->minimizeWindow(); 
				lpC->setFocus();
			}
		} else {
			if(getVisibleChildCount() >= 2){
				//we are sure to have at least TWO non minimized windows...
				//this one and another one somewhere lower in the Z order
				//move this one on the bottom of the stack
				if(m_lpChildList->removeRef(lpC)){
					m_lpChildList->insert(0,lpC);
				}
				//and find the next non minimized
				KviMdiChild *lpX=m_lpChildList->last();
				while(lpX->m_bMinimized){
					m_lpChildList->removeRef(lpX);
					m_lpChildList->insert(0,lpX);
					lpX=m_lpChildList->last();
					//just in case....(but should not happen , I presume)
					if(lpX==lpC)break;
				}
				//k we have a good window on the top
				focusTopChild();
			} else if(!lpC->m_bMinimized)lpC->minimizeWindow();
		}
	}
	_debug_leavetrace("taskbarButtonClicked");
}
void KviMdiManager::menuButtonClicked(int nId){
	_debug_entertrace("menuButtonClicked");
	if(nId>=0){
		KviMdiChild *lpC=findWindow(nId);
		if(lpC){
			if(lpC->m_bMinimized)lpC->minimizeWindow(); //restore :)
			lpC->setFocus();
		}
	}
	_debug_leavetrace("menuButtonClicked");
}
void KviMdiManager::taskbarPopupRequest(QPoint pnt,int nId){
	_debug_entertrace("taskbarPopupRequest");
	KviMdiChild *lpC=findWindow(nId);
	if(lpC){
//		if(topChildInZOrder() != lpC)taskbarButtonClicked(nId);
		lpC->showSystemMenu(pnt);
	}
	_debug_leavetrace("taskbarPopupRequest");
}
void KviMdiManager::childMaximized(KviMdiChild *){
	_debug_entertrace("childMaximized");
	_debug_leavetrace("childMaximized");
}
void KviMdiManager::childMinimized(KviMdiChild *){
	_debug_entertrace("childMinimized");
	m_lpTaskBar->unpressAllButtons();
	if(!m_bMinimizingAll)focusNextChild();
	_debug_leavetrace("childMinimized");
}
void KviMdiManager::focusNextChild(){
	_debug_entertrace("focusNextChild");
	KviMdiChild *lpC;
	lpC = m_lpChildList->last(); //top of the stack...
	if(!lpC)return; //can't focus nothing....
	while(lpC->m_bMinimized||(!lpC->isVisible())){
		lpC=m_lpChildList->prev();
		if(!lpC){
			_debug_warning("NULL : There should be no windows visible...");
			return;
		}
	}
	lpC->raise();
	lpC->setFocus();
	_debug_leavetrace("focusNextChild");
}
void KviMdiManager::resizeEvent(QResizeEvent *)
{
	_debug_entertrace("resizeEvent");
	KviMdiChild *lpC=0;
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		if(lpC->m_bMaximized)lpC->setGeometry(rect());
	}
	_debug_leavetrace("resizeEvent");
}

void KviMdiManager::cascadeWindows()
{
	_debug_entertrace("cascadeWindows");
	if(!m_lpChildList->isEmpty()){
		KviMdiChild *lpChild=0;
		int chx=0;
		int chy=0;
		KviMdiChild *prevChild=0;
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			if(lpChild->m_bMaximized)lpChild->maximizeWindow(); //restore windows if any
			if(!lpChild->m_bMinimized){ //skip the minimized windows
				lpChild->setGeometry(chx,chy,KVI_MDI_CHILD_CASCADE_WIDTH,KVI_MDI_CHILD_CASCADE_HEIGHT);
				chx+=KVI_MDI_CHILD_CASCADE_STEP;
				chy+=KVI_MDI_CHILD_CASCADE_STEP;
				if(chy>(height()-(KVI_MDI_CHILD_CASCADE_STEP*2))){
					chy=0;
					chx+=KVI_MDI_CHILD_CASCADE_JUMPX;
				}
				if(prevChild)prevChild->clearFocus();
				prevChild=lpChild;
			}
		}
		if(prevChild){
			prevChild->raise();
			prevChild->setFocus();
		}
	}
	_debug_leavetrace("cascadeWindows");
}
void KviMdiManager::restoreAllWindows()
{
	_debug_entertrace("restoreAllWindows");
	if(!m_lpChildList->isEmpty()){
		KviMdiChild *lpChild=0;
		KviMdiChild *lastChild=0;
		QList<KviMdiChild> list(*m_lpChildList);
		for(lpChild=list.first();lpChild;lpChild=list.next()){
			if(lpChild->m_bMaximized)lpChild->maximizeWindow(); //restore windows if any
			if(lpChild->m_bMinimized)lpChild->minimizeWindow();
			if(lastChild)lastChild->clearFocus();
			if(lpChild)lastChild=lpChild;
		}
		if(lastChild){
			lastChild->setFocus();
		}
	}
	_debug_leavetrace("restoreAllWindows");
}
void KviMdiManager::minimizeAllWindows()
{
	_debug_entertrace("minimizeAllWindows");
	if(!m_lpChildList->isEmpty()){
		KviMdiChild *lpChild;
		m_bMinimizingAll=true;
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			if(!lpChild->m_bMinimized){
				lpChild->minimizeWindow();
			}
		}
		m_bMinimizingAll=false;
		m_lpTaskBar->unpressAllButtons();
	}
	_debug_leavetrace("minimizeAllWindows");
}
void KviMdiManager::highlightWindow(int nID)
{
	_debug_entertrace("highlightWindow");
	m_lpTaskBar->highlightButton(nID);
	_debug_leavetrace("highlightWindow");
}
void KviMdiManager::fillWinListPopup(){
	_debug_entertrace("fillWinListPopup");
	KviMdiChild *lpChild;
	KIconLoader *lpIL=_macro_kviIconLoader;
	m_lpWinListPopup->clear();
	m_lpWinListPopup->insertItem(i18n("&Toolbars"),m_lpFrm->m_lpToolbarsPopup);
	m_lpWinListPopup->insertSeparator();
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_casc.gif"),i18n("&Cascade"),this,SLOT(cascadeWindows()));
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_rstr.gif"),i18n("&Restore all"),this,SLOT(restoreAllWindows()));
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_mnmz.gif"),i18n("&Minimize all"),this,SLOT(minimizeAllWindows()));
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_horz.gif"),i18n("Tile &horizontally"),this,SLOT(tileHorz()));
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_vert.gif"),i18n("Tile &vertically"),this,SLOT(tileVert()));
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_tall.gif"),i18n("&Anodine's full tile"),this,SLOT(tileAnodine()));
	m_lpWinListPopup->insertItem(lpIL->loadIcon("kvii_tall.gif"),i18n("&Pragma's overlapped tile"),this,SLOT(tilePragma()));
	m_lpWinListPopup->insertSeparator();
	if(!m_lpChildList->isEmpty()){
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			m_lpWinListPopup->insertItem((*lpChild->getIcon()),lpChild->name(),lpChild->id());
		}
	}
	_debug_leavetrace("fillWinListPopup");
}
void KviMdiManager::tileVert(){
	_debug_entertrace("tileVert");
	int cnt=getVisibleChildCount();
	KviMdiChild *lpChild;
	if(cnt){
		int wdt=(width()/cnt);
		int ax=0;
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			if(!lpChild->m_bMinimized){
				if(lpChild->m_bMaximized)lpChild->maximizeWindow(); //restore;
				lpChild->move(ax,0);
				lpChild->resize(wdt,height());
				ax+=wdt;
			}
		}
	}
	_debug_leavetrace("tileVert");
}
void KviMdiManager::tileHorz(){
	_debug_entertrace("tileHorz");
	int cnt=getVisibleChildCount();;
	KviMdiChild *lpChild;
	if(cnt){
		int hgh=(height()/cnt)-1;
		int ay=0;
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			if(!lpChild->m_bMinimized){
				if(lpChild->m_bMaximized)lpChild->maximizeWindow(); //restore;
				lpChild->move(0,ay);
				lpChild->resize(width(),hgh);
				ay+=hgh;
			}
		}
	}
	_debug_leavetrace("tileVert");
}
//============ getVisibleChildCount ============//
int KviMdiManager::getVisibleChildCount()
{
	_debug_entertrace("getVisibleChildCount");
	int cnt=0;
	KviMdiChild *lpChild;
	if(!m_lpChildList->isEmpty()){
		for(lpChild=m_lpChildList->first();lpChild;lpChild=m_lpChildList->next()){
			if(!lpChild->m_bMinimized)cnt++;
		}
	}
	_debug_leavetrace("getVisibleChildCount");
	return cnt;
}
//============ tilePragma ============//
void KviMdiManager::tilePragma()
{
	_debug_entertrace("tilePragma");
	tileAllInternal(9);
	_debug_leavetrace("tilePragma");
}
//============ tileAllInternal ============//
void KviMdiManager::tileAllInternal(int maxWnds)
{
	_debug_entertrace("tileAllInternal");
	//NUM WINDOWS =           1,2,3,4,5,6,7,8,9
	static int colstable[9]={ 1,1,1,2,2,2,3,3,3 }; //num columns
	static int rowstable[9]={ 1,2,3,2,3,3,3,3,3 }; //num rows
	static int lastwindw[9]={ 1,1,1,1,2,1,3,2,1 }; //last window multiplier
	static int colrecall[9]={ 0,0,0,3,3,3,6,6,6 }; //adjust self
	static int rowrecall[9]={ 0,0,0,0,4,4,4,4,4 }; //adjust self

	KviMdiChild *lpTop=topChildInZOrder();
	int numVisible=getVisibleChildCount();
	if(numVisible<1)return;
	int numToHandle=((numVisible > maxWnds) ? maxWnds : numVisible);
	int xQuantum=width()/colstable[numToHandle-1];
	if(xQuantum < KVI_MDI_CHILD_MIN_WIDTH){
		if(colrecall[numToHandle-1]==0)KMsgBox::message(this,i18n("Tile"),i18n("Not enough space."));
		else tileAllInternal(colrecall[numToHandle-1]);
		return;
	}
	int yQuantum=height()/rowstable[numToHandle-1];
	if(yQuantum < KVI_MDI_CHILD_MIN_HEIGHT){
		if(rowrecall[numToHandle-1]==0)KMsgBox::message(this,i18n("Tile"),i18n("Not enough space."));
		else tileAllInternal(rowrecall[numToHandle-1]);
		return;
	}
	int curX=0;
	int curY=0;
	KviMdiChild  *lpC=0;
	int curRow=1;
	int curCol=1;
	int curWin=1;
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		if(!lpC->m_bMinimized){
			//restore the window
			if(lpC->m_bMaximized)lpC->maximizeWindow();
			if((curWin%numToHandle)==0)lpC->setGeometry(curX,curY,xQuantum * lastwindw[numToHandle-1],yQuantum);
			else lpC->setGeometry(curX,curY,xQuantum,yQuantum);
			//example : 12 windows : 3 cols 3 rows
			if(curCol<colstable[numToHandle-1]){ //curCol<3
				curX+=xQuantum; //add a column in the same row
				curCol++;       //increase current column
			} else {
				curX=0;         //new row
				curCol=1;       //column 1
				if(curRow<rowstable[numToHandle-1]){ //curRow<3
					curY+=yQuantum; //add a row
					curRow++;       //
				} else {
					curY=0;         //restart from beginning
					curRow=1;       //
				}
			}
			curWin++;
		}
	}
	if(lpTop)lpTop->setFocus();
	_debug_leavetrace("tileAllInternal");
}
//============ tileAnodine ============//
void KviMdiManager::tileAnodine()
{
	_debug_entertrace("tileAnodine");
	KviMdiChild *lpTop=topChildInZOrder();
	int numVisible=getVisibleChildCount(); // count visible windows
	if(numVisible<1)return;
	int numCols=int(sqrt(numVisible)); // set columns to square root of visible count
	// create an array to form grid layout
	int *numRows=new int[numCols]; // Pragma :porting prob :moving runtime array sizing to new memory allocation
	int numCurCol=0;
	while(numCurCol<numCols){
		numRows[numCurCol]=numCols; // create primary grid values
		numCurCol++;
	}
	int numDiff=numVisible-(numCols*numCols); // count extra rows
	int numCurDiffCol=numCols; // set column limiting for grid updates
	while(numDiff>0){
		numCurDiffCol--;
		numRows[numCurDiffCol]++; // add extra rows to column grid
		if(numCurDiffCol<1)numCurDiffCol=numCols; // rotate through the grid
		numDiff--;
	}
	numCurCol=0;
	int numCurRow=0;
	int curX=0;
	int curY=0;
	KviMdiChild  *lpC=0;
	// the following code will size everything based on my grid above
	// there is no limit to the number of windows it will handle
	// it's great when a kick-ass theory works!!!                      // Pragma :)
	int xQuantum=width()/numCols;
	int yQuantum=height()/numRows[numCurCol];
	for(lpC=m_lpChildList->first();lpC;lpC=m_lpChildList->next()){
		if(!lpC->m_bMinimized){
			if(lpC->m_bMaximized)lpC->maximizeWindow();
			lpC->setGeometry(curX,curY,xQuantum,yQuantum);
			numCurRow++;
			curY+=yQuantum;
			if(numCurRow==numRows[numCurCol]){
				numCurRow=0;
				numCurCol++;
				curY=0;
				curX+=xQuantum;
				if(numCurCol!=numCols)yQuantum=height()/numRows[numCurCol];
			}
		}
	}
	delete[] numRows; // Pragma : runtime array sizing
	if(lpTop)lpTop->setFocus();
	_debug_leavetrace("tileAnodine");
}

void KviMdiManager::drawDragRectangle( QRect &rectToDraw,int style)
{
	if (!rectToDraw.isNull()){
		XGCValues gcvals;
		gcvals.foreground=black.pixel();
		gcvals.subwindow_mode=IncludeInferiors;
		gcvals.function=GXinvert;
        
		GC gc=XCreateGC(x11Display(),handle(),GCForeground|GCSubwindowMode|GCFunction,&gcvals);
		XDrawRectangle(x11Display(),handle(),gc,rectToDraw.x(),rectToDraw.y(),rectToDraw.width(),rectToDraw.height());
		if(style > 0){
			XDrawRectangle(x11Display(),handle(),gc,rectToDraw.x()+2,rectToDraw.y()+2,rectToDraw.width()-4,rectToDraw.height()-4);
		}
		XDrawRectangle(x11Display(),handle(),gc,rectToDraw.x()+1,rectToDraw.y()+1,rectToDraw.width()-2,rectToDraw.height()-2);
		XFreeGC(x11Display(),gc);
	}
}

#include "m_kvi_mdi.moc"

//
// $Log: kvi_mdi.cpp,v $
// Revision 1.6  1998/10/24 13:21:12  pragma
// Added Anodine's tile code.
//
// Revision 1.5  1998/09/20 20:23:07  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.4  1998/09/18 03:57:08  pragma
// Not so much changes...only a couple of bug fixes.
//
// Revision 1.3  1998/09/16 17:16:36  fritz
// Starting i18n.
//
// Revision 1.2  1998/09/15 13:19:37  pragma
// Taskbar buttons progress view fix.
//
//
