//   $Id: kvi_listbox.cpp,v 1.5 1998/10/06 14:42:31 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviListBox"
#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_listbox.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_int.h"
#include "kvi_mdi.h"
#include "kvi_global.h"
#include "kvi_uglobal.h"
#include "kvi_uparser.h"
#include "kvi_event.h"
#include "kvi_status.h"

#include <qpainter.h>

//============ KviListBox ============//

KviListBox::KviListBox(QWidget *parent,KviMdiChild *mdiParent,KviFrame *frame,const char *name)
:QFrame(parent,name)
{
	_debug_entertrace("KviListBox");
	m_lpMdiParent   = mdiParent;
	m_lpFrm         = frame;
	m_lpInt         = frame->m_lpInt;
	m_lpGUList      = frame->m_lpGlobalUserList;
	m_lpNickList    = new QList<KviListBoxEntry>;
	m_lpMemBuffer   = new QPixmap(width(),height());
	m_lpPainter     = new QPainter();
	setFrameStyle(WinPanel|Sunken);
	setBackgroundMode(NoBackground);
	m_lpNickList->setAutoDelete(true);
	m_iOp=0;
	m_iVoice=0;
	m_iTotal=0;
	m_bAutoUpdate=true;
	m_lpScrollBar=new QScrollBar(this);
	m_lpScrollBar->setOrientation(QScrollBar::Vertical);
	m_lpScrollBar->setTracking(true);
	m_lpScrollBar->setRange(0,0);
	m_lpScrollBar->hide();
	connect(m_lpScrollBar,SIGNAL(valueChanged(int)),this,SLOT(scrollBarMoved(int)));
	m_lpOpPixmap=frame->m_lpOpPixmap;
	m_lpVoicePixmap=frame->m_lpVoicePixmap;
	m_iOffset=0;
	m_iLastUnderCursor=-1;
	updateCellSize();
	_debug_leavetrace("KviListBox");
}

//============ ~KviListBox ============//

KviListBox::~KviListBox()
{
	_debug_entertrace("~KviListBox");
	clearList();
	delete m_lpNickList;
	delete m_lpMemBuffer;
	delete m_lpPainter;
	delete m_lpScrollBar;
	_debug_leavetrace("~KviListBox");
}
//============ clearList ============//
void KviListBox::clearList()
{
	_debug_entertrace("clearList");
	while(!m_lpNickList->isEmpty()){
		KviListBoxEntry *lpE=m_lpNickList->first();
		m_lpGUList->removeRef(lpE->lpNick);
		m_lpNickList->removeFirst();
	}
	m_iOp=0;
	m_iVoice=0;
	m_iTotal=0;
	_debug_leavetrace("clearList");
}
//============ removeAll ============//
//ALIAS FOR CLEARLIST
void KviListBox::removeAll()
{
	_debug_entertrace("removeAll");
	clearList();
	_debug_leavetrace("removeAll");
}
//============ findEntry ============//
KviListBoxEntry * KviListBox::findEntry(const char *szNick)
{
	_debug_entertrace("findEntry");
	_range_valid(szNick);
	KviListBoxEntry *lpE;
	for(lpE=m_lpNickList->first();lpE;lpE=m_lpNickList->next()){
		_range_valid(lpE->lpNick);
		if(!strcasecmp(lpE->lpNick->szNick.data(),szNick))return lpE;
	}
	_debug_leavetrace("findEntry");
	return 0;
}
//============ nickJoin ============//
void KviListBox::nickJoin(KviNewNick &kn,bool bIsOp,bool bIsVoice)
{
	_debug_entertrace("nickJoin");
	_range_invalid(findEntry(kn.szNick.data())); //Must not be already here!!!
	_range_invalid(bIsOp && bIsVoice); //can not be voice and op
	KviListBoxEntry *lpE=new KviListBoxEntry;
	lpE->lpNick=m_lpGUList->addNick(kn);
	_range_valid(lpE->lpNick);
	lpE->bOp=bIsOp;
	lpE->bVoice=bIsVoice;
	lpE->bSelected=false;
	insertEntry(lpE);
	m_iTotal++;
	if(bIsOp)m_iOp++;
	else if(bIsVoice)m_iVoice++;
	if(m_bAutoUpdate){
		repaint();
	}
	_debug_leavetrace("nickJoin");
}
//============ insertEntry ============//
void KviListBox::insertEntry(KviListBoxEntry *lpE)
{
	_debug_entertrace("insertEntry");
	_range_valid(lpE);
	if(m_lpNickList->isEmpty())m_lpNickList->append(lpE);
	else {
		KviListBoxEntry *lpO;
		int i=0;
		for(lpO=m_lpNickList->first();lpO;lpO=m_lpNickList->next()){
			if(lpE->bOp){ //the new one is an op
				if(lpO->bOp){ //compare and insert
					if(strcasecmp(lpO->lpNick->szNick.data(),lpE->lpNick->szNick.data())>=0){ //old one is greater
						m_lpNickList->insert(i,lpE);
						return;
					}		
				} else { //the first one is not op , insert before
					m_lpNickList->insert(i,lpE);
					return;
				}
			} else if(lpE->bVoice){ //the new one is a voiced user ,skip ops
				if(lpO->bVoice){ //compare and insert
					if(strcasecmp(lpO->lpNick->szNick.data(),lpE->lpNick->szNick.data())>=0){ //old one is greater
						m_lpNickList->insert(i,lpE);
						return;
					}							
				} else {
					if(!lpO->bOp){ //the first one is not op or voice
						m_lpNickList->insert(i,lpE);
						return;
					} //else is op , so skip
				}
			} else {
				if(!(lpO->bOp||lpO->bVoice)){ //the first one is not voice nor op
					if(strcasecmp(lpO->lpNick->szNick.data(),lpE->lpNick->szNick.data())>=0){ //old one is greater
						m_lpNickList->insert(i,lpE);
						return;
					}												
				} //else skip
			}
			i++;
		}
		m_lpNickList->append(lpE);
	}
	_debug_leavetrace("insertEntry");
}
//============ nickPart ============//
void KviListBox::nickPart(const char *szNick)
{

	_debug_entertrace("nickPart");
	_range_valid(szNick);
	KviListBoxEntry *lpE=findEntry(szNick);
	_range_valid(lpE);
	if(lpE){
		m_iTotal--;
		if(lpE->bOp)m_iOp--;
		else if(lpE->bVoice)m_iVoice--;
		m_lpGUList->removeRef(lpE->lpNick);
		m_lpNickList->removeRef(lpE);
	}
	repaint();
	_debug_leavetrace("nickPart");
}
//============ opNick ============//
void KviListBox::opNick(const char *szNick,bool bOp)
{
	_debug_entertrace("opNick");
	_range_valid(szNick);
	KviListBoxEntry *lpE=findEntry(szNick);
	_range_valid(lpE);
	if(lpE){
		if(lpE->bOp != bOp){
			if(bOp){
				m_iOp++;
				if(lpE->bVoice)m_iVoice--;
			} else m_iOp--;
			m_lpNickList->setAutoDelete(false);
			m_lpNickList->removeRef(lpE);
			m_lpNickList->setAutoDelete(true);
			lpE->bOp=bOp;
			lpE->bVoice=false;
			insertEntry(lpE);
			repaint();
		}
	}
	_debug_leavetrace("opNick");
}
//============ getEntryByPointer ============//
KviListBoxEntry * KviListBox::getEntryByPointer(KviNewNick *lpGlobalNickPointer)
{
	// returns our local entry that has that nick pointer
	// to the global users list inside
	_debug_entertrace("getEntryByPointer");
	_range_valid(lpGlobalNickPointer);
	KviListBoxEntry *lpE;
	for(lpE=m_lpNickList->first();lpE;lpE=m_lpNickList->next()){
		if(lpE->lpNick==lpGlobalNickPointer)return lpE;
	}
	_debug_leavetrace("getEntryByPointer");
	return 0;
}
//============ resortNick ============//
bool KviListBox::resortNick(KviNewNick *lpGlobalNickPointer)
{
	// removes and reinserts an user that changed nickname
	// it is called from KviServerParser::handleNick
	// and the nickname is updated once globally
	// we only remove and reinsert to get new alphabetic/op/voice order
	_debug_entertrace("resortNick");
	_range_valid(lpGlobalNickPointer);
	KviListBoxEntry *lpE=getEntryByPointer(lpGlobalNickPointer);
	if(!lpE)return false;
	m_lpNickList->setAutoDelete(false);
	m_lpNickList->removeRef(lpE);
	m_lpNickList->setAutoDelete(true);
	insertEntry(lpE);
	repaint();
	_debug_leavetrace("resortNick");
	return true;
}
//============ voiceNick ============//
void KviListBox::voiceNick(const char *szNick,bool bVoice)
{
	_debug_entertrace("voiceNick");
	_range_valid(szNick);
	KviListBoxEntry *lpE=findEntry(szNick);
	_range_valid(lpE);
	if(lpE){
		if(lpE->bOp)return;
		if(lpE->bVoice != bVoice){
			if(bVoice)m_iVoice++;
			else m_iVoice--;
			m_lpNickList->setAutoDelete(false);
			m_lpNickList->removeRef(lpE);
			m_lpNickList->setAutoDelete(true);
			lpE->bVoice=bVoice;
			insertEntry(lpE);
			repaint();
		}
	}
	_debug_leavetrace("voiceNick");
}
//============ isOp ============//
bool KviListBox::isOp(const char *szNick)
{
	_debug_entertrace("isOp");
	_range_valid(szNick);
	KviListBoxEntry *lpE=findEntry(szNick);
	_range_valid(lpE);
	if(lpE)return lpE->bOp;
	else return false;
	_debug_leavetrace("isOp");
}
//============ isVoice ============//
bool KviListBox::isVoice(const char *szNick)
{
	_debug_entertrace("isVoice");
	_range_valid(szNick);
	KviListBoxEntry *lpE=findEntry(szNick);
	_range_valid(lpE);
	if(lpE)return lpE->bVoice;
	else return false;
	_debug_leavetrace("isVoice");
}
//============ resizeEvent ============//
void KviListBox::resizeEvent(QResizeEvent *)
{
	_debug_entertrace("resizeEvent");
	m_lpMemBuffer->resize(width(),height());
	QRect the_Rect=rect();
	m_lpScrollBar->setGeometry(the_Rect.right()-17,the_Rect.top()+1,16,the_Rect.height()-2);
	updateScrollBar();
	_debug_leavetrace("resizeEvent");
}
//============ scrollBarMoved ============//
void KviListBox::scrollBarMoved(int value)
{
	_debug_entertrace("scrollBarMoved");
	m_iOffset=value;
	//skip repaint if we're already painting
	//drawContents calls updateScrollBar
	//if draw contents was called by nickJoin or nickPart
	//updateScrollBar sets new range and a signal connected to this slot is emitted
	//if the signal arrives immediately , we have the painting active
	//from the previous drawContents call ---> crash :)
	if(this->paintingActive())return;
	if(isVisibleToTLW()){
		QPaintEvent *e=new QPaintEvent(rect());
		paintEvent(e);
		delete e;
	}
	_debug_leavetrace("scrollBarMoved");
}
//============ updateScrollBar ============//
void KviListBox::updateScrollBar()
{
	_debug_entertrace("updateScrollBar");
	uint maxVisible=contentsRect().height() / m_iCellHeight;
	if(maxVisible < m_iTotal){
		m_lpScrollBar->show();
		m_lpScrollBar->setRange(0,m_iTotal-maxVisible);
	} else {
		m_lpScrollBar->hide();
		m_iOffset=0;
	}
	_debug_leavetrace("updateScrollBar");
}
//============ drawContents ============//
void KviListBox::drawContents(QPainter *p)
{
	_debug_entertrace("drawContents");
	updateScrollBar();
	if(!m_iTotal){
		QRect internalR=contentsRect();
		QBrush bkBrush(m_lpInt->clr_bk_listbox); //back
		if((!(m_lpInt->pix_bk_listbox.isNull())))bkBrush.setPixmap(m_lpInt->pix_bk_listbox);
		p->fillRect(internalR,bkBrush);
		return;
	}
	if(m_lpMemBuffer->paintingActive())return;
	m_lpPainter->begin(m_lpMemBuffer);
	m_lpPainter->setFont(font());

	//draw the backgrounrd
	QRect internalR=contentsRect();
	if(m_lpScrollBar->isVisible())internalR.setWidth(internalR.width()-16);
	QBrush bkBrush(m_lpInt->clr_bk_listbox); //back
	if((!(m_lpInt->pix_bk_listbox.isNull())))bkBrush.setPixmap(m_lpInt->pix_bk_listbox);
	m_lpPainter->fillRect(internalR,bkBrush);

	QFontMetrics fm=m_lpPainter->fontMetrics();
	int yLineOffset=fm.ascent()+(fm.leading()/2);
	int maxY = internalR.bottom()+yLineOffset+1;
  	int yStart=3+yLineOffset;
	int xStart=20;
	uint curLine=m_iOffset;
	int cellWdth=width()-20;
	if(m_lpScrollBar->isVisible())cellWdth-=16;
	KviListBoxEntry *lpE;
    while ((yStart < maxY)&&(curLine < m_iTotal)) {
		//PAINT ITEM
		lpE=m_lpNickList->at(curLine);
		_range_valid(lpE);
		if(lpE->bSelected){
			m_lpPainter->fillRect(xStart-19,yStart-(yLineOffset+1),cellWdth+16,m_iCellHeight,m_lpInt->clr_bk_sel_listbox); //select back
			m_lpPainter->setPen(m_lpInt->clr_fr_sel_listbox); //select fore
	   	} else {
			if(lpE->bOp)m_lpPainter->setPen(m_lpInt->clr_fr_op_listbox); //op fore
			else if(lpE->bVoice)m_lpPainter->setPen(m_lpInt->clr_fr_voice_listbox); //voice fore
			else m_lpPainter->setPen(m_lpInt->clr_fr_normal_listbox); //fore
		}
		if(lpE->bOp)m_lpPainter->drawPixmap(3,yStart-yLineOffset,*m_lpOpPixmap);
		else if(lpE->bVoice)m_lpPainter->drawPixmap(3,yStart-yLineOffset,*m_lpVoicePixmap);
		m_lpPainter->drawText(xStart,yStart,lpE->lpNick->szNick.data());
		//PAINT ITEM
		curLine++;
		yStart+=m_iCellHeight;
    }
	m_lpPainter->end();
	bitBlt(this,internalR.top(),internalR.left(),
		m_lpMemBuffer,internalR.top(),internalR.left(),internalR.width(),internalR.height(),CopyROP,false);
	_debug_leavetrace("drawContents");
}

//============ updateCellSize ============//

void KviListBox::updateCellSize()
{
	_debug_entertrace("updateCellSize");
	QFontMetrics fm(font());
	m_iCellHeight=fm.lineSpacing();
	if(m_iCellHeight<14)m_iCellHeight=14;
	_debug_leavetrace("updateCellSize");
}

//============ setFont ============//

void KviListBox::setFont(const QFont & font)
{
	_debug_entertrace("setFont");
	QFrame::setFont(font);
	updateCellSize();
	repaint();
	_debug_leavetrace("setFont");
}

//============ getEntryAt ============//

KviListBoxEntry * KviListBox::getEntryAt(uint index)
{
	_debug_entertrace("getEntryAt");
	return m_lpNickList->at(index);
	_debug_leavetrace("getEntryAt");
}

//============ clearSelection ============//

void KviListBox::clearSelection()
{
	_debug_entertrace("clearSelection");
	KviListBoxEntry *lpE;
	for(lpE=m_lpNickList->first();lpE;lpE=m_lpNickList->next())lpE->bSelected=false;
	_debug_leavetrace("clearSelection");
}

//============ mousePressEvent ============//

void KviListBox::mousePressEvent(QMouseEvent *e)
{
	_debug_entertrace("mousePressEvent");
	//calculate the rowe->pos().y();
	int nActLine=((e->pos().y()-4)/m_iCellHeight)+m_iOffset;
	if((nActLine<((int)m_iTotal))&&(nActLine>=0)){
		m_iLastUnderCursor=nActLine;
		KviListBoxEntry *lpE=getEntryAt(nActLine);
		if(lpE){
			if(e->button() & LeftButton){
				if((e->state() & ShiftButton)||(e->state() & ControlButton))lpE->bSelected=!lpE->bSelected;
				else {
					clearSelection();
					lpE->bSelected=true;
				}
				repaint();
				infoToStatus(lpE->lpNick);
			} else if(e->button() & RightButton){
				if(!lpE->bSelected){
					clearSelection();
					lpE->bSelected=true;
					repaint();
				}
				if(m_lpMdiParent->type()==KVI_WND_TYPE_CHAN){
					m_lpFrm->m_lpGlb->lpCurrentChan=(KviChanWnd *)m_lpMdiParent;
					m_lpFrm->m_lpGlb->szCurrentSelection="";
					for(lpE=m_lpNickList->first();lpE;lpE=m_lpNickList->next()){
						if(lpE->bSelected){
							if(!m_lpFrm->m_lpGlb->szCurrentSelection.isEmpty())m_lpFrm->m_lpGlb->szCurrentSelection+=",";
							_range_valid(lpE->lpNick);
							m_lpFrm->m_lpGlb->szCurrentSelection+=lpE->lpNick->szNick.data();
						}
					}
					if(!m_lpFrm->m_lpGlb->szCurrentSelection.isEmpty())m_lpFrm->requestNickListPopup(mapToGlobal(e->pos()));
				} else if(m_lpMdiParent->type()==KVI_WND_TYPE_STATUS){
					m_lpFrm->m_lpGlb->szCurrentNotifySelection="";
					for(lpE=m_lpNickList->first();lpE;lpE=m_lpNickList->next()){
						if(lpE->bSelected){
							if(!m_lpFrm->m_lpGlb->szCurrentNotifySelection.isEmpty())m_lpFrm->m_lpGlb->szCurrentNotifySelection+=",";
							_range_valid(lpE->lpNick);
							m_lpFrm->m_lpGlb->szCurrentNotifySelection+=lpE->lpNick->szNick.data();
						}
					}
					if(!m_lpFrm->m_lpGlb->szCurrentNotifySelection.isEmpty())m_lpFrm->requestNotifyListPopup(mapToGlobal(e->pos()));
				}
			}
		}
	}
	_debug_leavetrace("mousePressEvent");
}

//============ mouseDoubleClickEvent ============//

void KviListBox::mouseDoubleClickEvent(QMouseEvent *e)
{
	_debug_entertrace("mouseDoubleClickEvent");
	if(e->button() & LeftButton){
		int nActLine=((e->pos().y()-4)/m_iCellHeight)+m_iOffset;
		if((nActLine<((int)m_iTotal))&&(nActLine>=0)){
			KviListBoxEntry *lpE=getEntryAt(nActLine);
			if(lpE){
//				debug("azz %s",lpE->lpNick->szNick.data());
				clearSelection();
				lpE->bSelected=true;
				m_iLastUnderCursor=nActLine;
				repaint();
				QString szCmd="/QUERY ";
				_range_valid(lpE->lpNick);
				szCmd+=lpE->lpNick->szNick.data();
				m_lpFrm->m_lpUserParser->parseUserInput(m_lpMdiParent,m_lpMdiParent->type(),m_lpMdiParent->name(),szCmd);
			}
		}
	} else if(e->button() & MidButton){
		int nActLine=((e->pos().y()-4)/m_iCellHeight)+m_iOffset;
		if((nActLine<((int)m_iTotal))&&(nActLine>=0)){
			KviListBoxEntry *lpE=getEntryAt(nActLine);
			if(lpE){
				clearSelection();
				lpE->bSelected=true;
				m_iLastUnderCursor=nActLine;
				repaint();
				if(m_lpMdiParent->type()==KVI_WND_TYPE_STATUS){
					//copy to status selection
					m_lpFrm->m_lpGlb->szCurrentNotifySelection=lpE->lpNick->szNick.data();
					QString szCmd="/WHOIS -i ";
					_range_valid(lpE->lpNick);
					szCmd+=lpE->lpNick->szNick.data();
					m_lpFrm->m_lpUserParser->parseUserInput(m_lpMdiParent,m_lpMdiParent->type(),m_lpMdiParent->name(),szCmd);
				} else if(m_lpMdiParent->type()==KVI_WND_TYPE_CHAN){
					m_lpFrm->m_lpGlb->szCurrentSelection=lpE->lpNick->szNick.data();
					if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnMiddleDoubleClick]->bEnabled){
						QString szPara=m_lpMdiParent->name();
						szPara+=" ";
						_range_valid(lpE->lpNick);
						szPara+=lpE->lpNick->szNick.data();
						m_lpFrm->m_lpUserParser->executeEvent(m_lpFrm->m_lpConsole,
							m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnMiddleDoubleClick],szPara);
					}		
				}
			}
		}
	}
	_debug_leavetrace("mouseDoubleClickEvent");
}

//============ infoToStatus ============//

void KviListBox::infoToStatus(KviNewNick *lpN)
{
	_debug_entertrace("infoToStatus");
	_range_valid(lpN);
	_range_invalid(lpN->szNick.isEmpty());
	_range_invalid(lpN->szAddress.isEmpty());
	_range_invalid(lpN->szUserName.isEmpty());
	QString szIdent;
	szIdent=lpN->szNick.copy();
	szIdent+=" is ";
	szIdent+=lpN->szNick;
	szIdent+='!';
	int nlen=strlen(KVI_STR_NULL);
	if(strncmp(lpN->szUserName.data(),KVI_STR_NULL,nlen))szIdent+=lpN->szUserName;
	else szIdent+='*';
	szIdent+='@';
	if(strncmp(lpN->szAddress.data(),KVI_STR_NULL,nlen))szIdent+=lpN->szAddress;
	else szIdent+='*';
	m_lpFrm->setStatusText(szIdent.data());
	_debug_leavetrace("infoToStatus");
}

//============ mouseMoveEvent ============//

void KviListBox::mouseMoveEvent(QMouseEvent *e)
{
	_debug_entertrace("mouseMoveEvent");
	int nActLine=((e->pos().y()-4)/m_iCellHeight)+m_iOffset;
	if((nActLine<((int)m_iTotal))&&(nActLine>=0)){
		if(nActLine != m_iLastUnderCursor){
			m_iLastUnderCursor=nActLine;
			if(e->state() & LeftButton){
				KviListBoxEntry *lpE=m_lpNickList->at(nActLine);
				if(lpE){
					if((e->state() & ShiftButton)||(e->state() & ControlButton))lpE->bSelected=!lpE->bSelected;
					else {
						clearSelection();
						lpE->bSelected=true;
					}
					repaint();
					infoToStatus(lpE->lpNick);
				}
			}
		}
	} else m_iLastUnderCursor=-1;
	_debug_leavetrace("mouseMoveEvent");
}

#include "m_kvi_listbox.moc"

//
// $Log: kvi_listbox.cpp,v $
// Revision 1.5  1998/10/06 14:42:31  pragma
// Tons of changes
//
// Revision 1.4  1998/09/25 15:58:31  pragma
// Moving ti use the KviProxy class.
//
// Revision 1.3  1998/09/23 23:15:08  pragma
// Ignore list. Still needs testing.
// Added some ASSERT-Like macros to kvi_chan.cpp and kvi_listbox.cpp
// to discover a strange bug :
// Sometimes the a single nick on a random channel gets wrong username...
// Just like a hollow pointer...but still can't guess how to reproduce it.
// It happened only a couple of times...
//
//
