//   $Id: kvi_input.cpp,v 1.6 1998/09/30 03:11:33 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this library; see the file COPYING.LIB.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
//#define _KVI_DEBUG_CLASS_NAME_ "KviInput"

#include "kvi_input.h"
#include "kvi_child.h"
#include "kvi_translate.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_app.h"
#include "kvi_clrbox.h"
#include "kvi_opt.h"
#include "kvi_uparser.h"
#include "kvi_nick.h"
#include "kvi_chan.h"
#include "kvi_mdi.h"
#include "kvi_listbox.h"
#include "kvi_view.h"

#include <qkeycode.h>
#include <qclipboard.h>
#include <qpainter.h>
#include <qdrawutil.h>

#include <ctype.h>

KviInput::KviInput(KviMdiChild *parent,int parentID,int parentType,
					const char *szParentName,KviFrame *lpFrame,const char *name): QWidget(parent,name)
{
	_debug_entertrace("KviInput");
	//set up internal state
	m_ParentType=parentType;
	m_lpParent=parent;
	m_lpFrame=lpFrame;
	m_ParentID=parentID;
	m_szParentName=QString(szParentName);
	m_lpUserParser=lpFrame->m_lpUserParser;
	//Well...the QLineEdit does it...why?
	initMetaObject();
	//Context popup menu
	m_lpContextPopup = new QPopupMenu();
	m_lpContextPopup->insertItem(i18n("Copy"),this,SLOT(copySlot()));
	m_lpContextPopup->setId(0,1);
	m_lpContextPopup->insertItem(i18n("Paste"),this,SLOT(pasteSlot()));
	m_lpContextPopup->setId(1,2);
	m_lpContextPopup->setFocusPolicy(NoFocus);
	m_lpContextPopup->setFocusProxy(this);
	//more internal state
	m_lpMemBuffer      = new QPixmap(width(),height());
	m_cursorPos        = 0;
	m_iLastTabCursorPos= 0;
	m_offset           = 0;
	m_bCursorActive    = true;
	m_markAnchor       = 0;
	m_markDrag         = 0;
	m_bDragScrolling   = false;
	m_bScrollLeft      = false;
	m_bColorBoxVisible =false;
	m_bNickCompleted=false;
	m_bColorBoxEnabled =true;
	m_textBuffer	   = "";
	//set the default colors....will be overriden later by the parent
	m_backgroundColor  =QColor(0,0,0);
	m_foregroundColor  =QColor(40,160,30);
	m_markBackColor    =QColor(0,100,190);
	m_markForeColor    =QColor(0,0,0);
	m_cursorColor      =QColor(255,0,0);
	//Yep
	setFocusPolicy(StrongFocus);
	setCursor(ibeamCursor);
	setBackgroundMode( PaletteBase );
	//set up the command list
	m_lpCommandList=new QList<QString>;
	m_lpCommandList->setAutoDelete(TRUE);
	m_lpCommandList->append(new QString(""));
	m_CommandListIndex=0;
	//Nick completion support
	m_iLastTabIndex=-1;
	m_szLastNickCompletionBuffer="";
	//And the colorbox
	m_lpClrBox=new KviClrBox(parent,m_lpFrame);
	_debug_leavetrace("KviInput");
}
void KviInput::setDefaultBackgroundColor(const QColor &acolor)
{
	m_backgroundColor=acolor;
	setBackgroundColor(acolor);
}
void KviInput::setDefaultForegroundColor(const QColor &acolor)
{
	m_foregroundColor=acolor;
}
void KviInput::setMarkBackgroundColor(const QColor &acolor)
{
	m_markBackColor=acolor;
}
void KviInput::setMarkForegroundColor(const QColor &acolor)
{
	m_markForeColor=acolor;
}
void KviInput::setCursorColor(const QColor &acolor)
{
	m_cursorColor=acolor;
}
void KviInput::enableColorBox(bool bEnable)
{
	m_bColorBoxEnabled=bEnable;
}
void KviInput::returnPressed(){
	_debug_entertrace("returnPressed");
	if((!m_textBuffer.isNull())&&(m_textBuffer != "")){
		QString *lpStr=m_lpCommandList->last();
		bool bAppend=true;
		if(lpStr){ //check if the last is a repetition of the new command
			if(!strcmp(lpStr->data(),m_textBuffer.data()))bAppend=false;
		}
		if(bAppend){
			m_lpCommandList->append(new QString(m_textBuffer));
			if(m_lpCommandList->count()>KVI_INPUT_MAX_COMMANDS)m_lpCommandList->remove(1);
		}
		m_CommandListIndex=0;
		QString szCommand=m_textBuffer.copy();
		if(m_lpFrame->m_lpOpt->bUseTranslation){
			if(m_ParentType==KVI_WND_TYPE_CHAT){
				if(!m_lpFrame->m_lpOpt->bNotTranslateDCC)m_lpFrame->m_lpTranslator->translateToServer(szCommand);
			} else m_lpFrame->m_lpTranslator->translateToServer(szCommand);
		}
		m_lpUserParser->parseUserInput(m_lpParent,m_ParentType,m_szParentName.data(),szCommand);
	}
	setText("");
	_debug_leavetrace("returnPressed");
}

//================ tabPressed() ===============//
//
//  Tab nick completion support
//  patch by Andras Szigethy
//
//  12 Aug 98 : Added multi Tab press support...now it can find also the 'next' matching nick
//  FIXME : This will not work well if for example the nick just before the last
//          one completed parts...we may skip one.
//          Hard to fix because the nicks are not in strict alphabetic order.(@ is before +)
void KviInput::tabPressed()
{
    if( m_ParentType != KVI_WND_TYPE_CHAN )return;
	if(m_iLastTabIndex < 0){
		//new nick completion session
		//copy the buffer and the cursor position
		m_szLastNickCompletionBuffer=m_textBuffer.copy();
		m_iLastTabCursorPos=m_cursorPos;
	} // else old session , searching for the next nick

	if(m_szLastNickCompletionBuffer.isEmpty())return;

    m_markAnchor = m_cursorPos;
    m_markDrag   = m_markAnchor;
	//test the last nick completion buffer
    QString test( m_szLastNickCompletionBuffer.copy());
    uint cp = m_iLastTabCursorPos;
	if( cp == 0 ){ //check that position is not 0 (no nick before the cursor)
		m_iLastTabIndex=-1;
		return;
	}
    if( test.data()[cp-1] == ' ' ){ //cp-1 can NOT be -1 now...
									//
									//Andras ,my doubt : 
									// (cp==0||test.data()[cp-1])
									// What happens if cp==0 ?
									// test.data()[-1] is evaluated or not? :)
									// I think that this may be compiler dependant...
									// so I separated it...but I may be wrong...
									// Szymon
		//tab completion cannot be made
		//reset to begin (useless?)
		m_iLastTabIndex=-1;
		return;
	}
    uint wordStart = test.findRev( ' ', cp - 1 ) + 1;
    int tmp = test.find( ' ', cp );
    uint wordEnd = (tmp != -1) ? tmp : test.length();
    uint len = wordEnd - wordStart;
    QList<KviListBoxEntry> *nl = ((KviChanWnd *)m_lpParent)->m_lpListBox->m_lpNickList;
    QString word( test.mid( wordStart, len ) );
    KviNewNick *n=0;
	KviListBoxEntry *lpE;
	int idx=0;
    for( lpE = nl->first(); lpE != 0; lpE = nl->next() ) {
		n=lpE->lpNick;
		if(idx>m_iLastTabIndex){ //find the 'next' one
	        if( strnicmp( word.data(), n->szNick.data(), len ) == 0 ) {
				//found the next nick to complete
				m_iLastTabIndex=idx;
				test.remove( wordStart, len );
				test.insert( wordStart, n->szNick );
				cp = wordStart + n->szNick.length();
				if( wordStart == 0 ){
					test.insert( cp, ": " );
					cp += 2;
					m_bNickCompleted=true;
				}
				break;
			}
		} //else skip it
		idx++;
    }
	if(!n){
		//not found , the next tab will start from beginning
		//this time we will reset to the LAST WORD TO COMPLETE
		//eg m_szLastNickCompletionBuffer
		m_iLastTabIndex=-1;
		m_bNickCompleted=false;
	}
    if ( test != m_textBuffer && test.length() < KVI_INPUT_MAX_LENGTH ) {
		m_cursorPos = cp;
		m_textBuffer = test;
		kviInputRepaint;
    }
}
//================ autoCompleteNick was NickComp() ===============//
// by RADKade1
// A quick hack of the above tab completion
// to make it automatic.
// (Anyone wanna bother cleaning this up? ;)
void KviInput::autoCompleteNick()
{
	_debug_entertrace("autoCompleteNick");
	if( m_ParentType != KVI_WND_TYPE_CHAN )return;
	QString test( m_textBuffer.copy());
	uint cp = m_iLastTabCursorPos;
	int len = test.find( ':', cp );
	if(len != -1 && len<9 && len>0) {
		QList<KviListBoxEntry> *nl = ((KviChanWnd *)m_lpParent)->m_lpListBox->m_lpNickList;
		nl->setAutoDelete(false);
		QString word( test.mid( 0, len ) );
		KviNewNick *n=0;
		KviListBoxEntry *lpE;
		for( lpE = nl->first(); lpE != 0; lpE = nl->next() ) {
			n=lpE->lpNick;
			if(!strnicmp( word.data(), n->szNick.data(), len )) {
				test.remove( 0, len );
				test.insert( 0, n->szNick );
				cp = n->szNick.length();
				test.insert( cp, "" );
				cp += 2;
				break;
			}
		}
		if ((test != m_textBuffer) && (test.length() < KVI_INPUT_MAX_LENGTH)) {
			m_cursorPos = cp;
			m_textBuffer = test;
		}
	}
	_debug_leavetrace("autoCompleteNick");
}
static int xPosToCursorPos( char *s, const QFontMetrics &fm,int xPos, int width )
{
	char *tmp;
	int dist;
	if ( xPos > width )xPos = width;
 	if ( xPos <= 0 )return 0;
	dist = xPos;
	tmp = s;
	while ((*tmp)&&(dist > 0))dist -= fm.width( tmp++, 1 );
	if ( dist < 0 && ( xPos - dist > width || fm.width( tmp - 1, 1)/2 < -dist))tmp--;
	return tmp - s;
}

static int showLastPartOffset( char *s, const QFontMetrics &fm, int width )
{
	if( !s || s[0] == '\0' )return 0;
	char *tmp = &s[strlen( s ) - 1];
	do {
		width -= fm.width( tmp--, 1 );
	} while ( tmp >=s && width >=0 );
	return width < 0 ? tmp - s + 2 : 0;
}

KviInput::~KviInput()
{
	_debug_entertrace("~KviInput");
	while(!(m_lpCommandList->isEmpty()))m_lpCommandList->removeLast();
	delete m_lpCommandList;
	delete m_lpMemBuffer;
	delete m_lpClrBox;
	delete m_lpContextPopup;
	_debug_leavetrace("~KviInput");
}

void KviInput::setText( const char *text )
{
	QString oldText(m_textBuffer);
	oldText.detach();
	m_textBuffer = text ? text : "";
	if ( (int)m_textBuffer.length() > KVI_INPUT_MAX_LENGTH ) {
		m_textBuffer.resize( KVI_INPUT_MAX_LENGTH+1 );
		m_textBuffer[KVI_INPUT_MAX_LENGTH] = '\0';
	}
	m_offset    = 0;
	m_cursorPos = 0;
	m_markAnchor = 0;
	m_markDrag = 0;
	end(false);
	kviInputRepaint;
}

const char *KviInput::text() const
{
	return m_textBuffer;
}

void KviInput::keyPressEvent( QKeyEvent *e )
{
	_debug_entertrace("keyPressEvent");
	//Hide the color box
	if(m_bColorBoxVisible){
		m_lpClrBox->hide();
		m_bColorBoxVisible=false;
	}
	if(e->key()==Key_Tab){
		tabPressed();
		e->ignore();
		return;
	} else m_iLastTabIndex=(-1); //Next time we will look for a different nick

	if ((e->key()==Key_Enter)||(e->key()==Key_Return)) {
		//emit returnPressed();
		if((m_lpFrame->m_lpOpt->bAutoNickCompletion)&&(!m_bNickCompleted))autoCompleteNick();
		m_bNickCompleted=false;
		returnPressed();
		e->ignore();
		return;
	}
	if(e->key()==Key_Escape){
		KviMdiChild *lpC=m_lpFrame->m_lpMdi->topChildInZOrder();
		if(lpC)lpC->minimizeWindow();
		m_lpFrame->m_lpMdi->focusTopChild();
		e->ignore();
		return;
	}
 	if( e->key() == Key_Up ){
		if(!m_lpCommandList->isEmpty()){
			if(m_CommandListIndex)m_CommandListIndex--;
			else m_CommandListIndex=m_lpCommandList->count()-1;
			setText( (m_lpCommandList->at(m_CommandListIndex) )->data() );
		}
		return;
	}
	if( e->key() == Key_Down ){
		if(!m_lpCommandList->isEmpty()){
			if(m_CommandListIndex < m_lpCommandList->count()-1)m_CommandListIndex++;
			else m_CommandListIndex=0;
			setText( (m_lpCommandList->at(m_CommandListIndex) )->data() );
		}
		return;
	}

	// Scroll view pages.
	// Patch by Pedro Martinez <pedro@stil.acad.bg>
	if( e->key() == Key_PageUp && e->state() == 0 ){
		m_lpParent->m_lpOutput->m_lpScrollBar->subtractPage();
		return;
	}
	if( e->key() == Key_PageDown && e->state() == 0 ){
		m_lpParent->m_lpOutput->m_lpScrollBar->addPage();
		return;
	}
	//

	int unknown = 0;
	if( e->state() & ControlButton ) {
		switch ( e->key() ){
			case Key_R: //Reverse text
				m_textBuffer.insert(m_cursorPos,KVI_TEXT_REVERSE);
				moveCursorRight(false);
				break;
			case Key_O: //reset attributes
				m_textBuffer.insert(m_cursorPos,KVI_TEXT_RESET);
				moveCursorRight(false);
				break;
			case Key_B: //Bold
				m_textBuffer.insert(m_cursorPos,KVI_TEXT_BOLD);
				moveCursorRight(false);
				break;
			case Key_Right:
				end( e->state() & ShiftButton );
				break;
			case Key_K: //Kolors
				if((!m_bColorBoxVisible)&&(m_bColorBoxEnabled)){
					m_lpClrBox->move(x(),y()-20);
					m_lpClrBox->show();
					m_lpClrBox->raise();
					m_bColorBoxVisible=TRUE;
				}
				m_textBuffer.insert(m_cursorPos,KVI_TEXT_COLOR);
				moveCursorRight(false);
				break;
			case Key_U: //Underlined
				m_textBuffer.insert(m_cursorPos,KVI_TEXT_UNDERLINE);
				moveCursorRight(false);
				break;
			case Key_C:	{//Copy
				QString t="";
				if(kviInputHasSelection)t=m_textBuffer.mid(minMark(),maxMark()-minMark());
   				 if ( !t.isEmpty() )QApplication::clipboard()->setText( t );
				}
				break;
			case Key_V:
			{			// paste
				QString txt = QApplication::clipboard()->text();
				if ( !txt.isEmpty() ) {
					txt.detach();
					bool bMultiline=false;
					while(!txt.isEmpty()){
						int idx = txt.find( '\n' );
						if(idx != -1)bMultiline=true;
						QString t=((idx>=0) ? txt.left(idx) : txt);
						if(idx>=0)txt.remove(0,idx+1);
						else txt="";
						int tlen = t.length();
						int blen;
						int cp = m_cursorPos;

						QString test( m_textBuffer.copy() );
						if ( kviInputHasSelection ) {
							test.remove( minMark(), maxMark() - minMark() );
		    					if ( cp > maxMark() )cp -= (maxMark() - minMark());
		    					else if ( cp > minMark() )cp = minMark();
						}
						blen = test.length();
						if ( tlen+blen >= KVI_INPUT_MAX_LENGTH ) {
		    					if ( blen >= KVI_INPUT_MAX_LENGTH )break;
		    					if ( tlen > KVI_INPUT_MAX_LENGTH )tlen = KVI_INPUT_MAX_LENGTH;
		    					t.truncate( KVI_INPUT_MAX_LENGTH-blen+1 );
		    					t[KVI_INPUT_MAX_LENGTH-blen] = '\0';
						}
						test.insert( cp, t );
						cp += t.length();
	
						m_textBuffer = test;
						m_cursorPos = cp;
						repaint(false);
						if(bMultiline)returnPressed();
					}
				}
			}
				break;
			case Key_0:
				pasteUserBuffer(0);
				break;
			case Key_1:
				pasteUserBuffer(1);
				break;
			case Key_2:
				pasteUserBuffer(2);
				break;
			case Key_3:
				pasteUserBuffer(3);
				break;
			case Key_4:
				pasteUserBuffer(4);
				break;
			case Key_5:
				pasteUserBuffer(5);
				break;
			case Key_6:
				pasteUserBuffer(6);
				break;
			case Key_7:
				pasteUserBuffer(7);
				break;
			case Key_8:
				pasteUserBuffer(8);
				break;
			case Key_9:
				pasteUserBuffer(9);
				break;
			default:
				unknown++;
		}
	} else {
		bool shiftPressed=(e->state() & ShiftButton);
		switch ( e->key() ) {
		case Key_Left:		moveCursorLeft(shiftPressed);	break;
		case Key_Right:		moveCursorRight(shiftPressed);	break;
		case Key_Backspace:	backspace();				break;
		case Key_Home:		home(shiftPressed);			break;
		case Key_End:		end(shiftPressed);			break;
		case Key_Delete:	del();						break;
		default:
		    unknown++;
		}
	}
	if (unknown && (e->ascii())) {
		QString test( m_textBuffer.copy() );
		int cp = m_cursorPos;
		if (kviInputHasSelection){
			test.remove( minMark(), maxMark() - minMark() );
			cp = minMark();
		}
		if ( (int)test.length() < KVI_INPUT_MAX_LENGTH )test.insert( cp, e->ascii() );
		if ( test != m_textBuffer ) {
			m_cursorPos = cp;
			m_textBuffer = test;
			moveCursorRight( FALSE );
		}
		return;
	}
	_debug_leavetrace("keyPressEvent");
}


//============ pasteUserBuffer ============//

void KviInput::pasteUserBuffer(int nBuf)
{
	_debug_entertrace("pasteUserBuffer");
	QString txt = m_lpFrame->m_lpOpt->szTextUserBuffer[nBuf].data();
	if(!txt.isEmpty()){
		int len=m_textBuffer.length()+txt.length();
		if(len < KVI_INPUT_MAX_LENGTH){
			m_textBuffer.insert(m_cursorPos,txt);
			m_cursorPos += txt.length();
			repaint(false);
		}
	}
	_debug_leavetrace("pasteUserBuffer");
}

void KviInput::focusInEvent( QFocusEvent * )
{
	_debug_entertrace("focusInEvent");
    if ( style() == WindowsStyle && m_markAnchor == m_markDrag ){
			m_markAnchor = 0;
			m_markDrag   = 0;
	}
    killTimers();
    startTimer( KVI_INPUT_BLINK_TIME );
    m_bCursorActive = true;
    kviInputRepaint;
	_debug_leavetrace("focusInEvent");
}

void KviInput::focusOutEvent( QFocusEvent * )
{
	_debug_entertrace("focusOutEvent");
    if ((focusWidget()!=this)||(qApp->focusWidget()==0)
			||(qApp->focusWidget()->topLevelWidget()!=topLevelWidget()))
	{
 			m_markAnchor = m_cursorPos;
			m_markDrag   = m_cursorPos;
			kviInputRepaint;
			//deselect();
	}
    killTimers();
    m_bCursorActive	  = false;
    m_bDragScrolling = false;
    kviInputRepaint;
	_debug_leavetrace("focusOutEvent");
}

void KviInput::paintEvent( QPaintEvent *event )
{
	if (hasFocus())pixmapPaint(event->rect());
    else {
		QPainter thePainter;
		thePainter.begin( this );
		thePainter.setClipRect(event->rect());
		paintText( &thePainter, size());
		thePainter.end();
	}
}

void KviInput::timerEvent( QTimerEvent * )
{
//	_debug_entertrace_huge("timerEvent");
    if ( hasFocus() ) {
		if ( m_bDragScrolling ) {
		    if ( m_bScrollLeft )moveCursorLeft(true);	// mark left
		    else moveCursorRight(true);	// mark right
		} else {
		    m_bCursorActive = !m_bCursorActive;
		    kviInputRepaint;
		}
    }
//	_debug_leavetrace_huge("timerEvent");
}

void KviInput::resizeEvent( QResizeEvent *e )
{
    m_lpMemBuffer->resize( e->size() );
    int max = lastCharVisible();
    if ( m_cursorPos > max ) {
		QFontMetrics fm = fontMetrics();
		int w = width() - 8;
		int i = m_cursorPos;
		while ( w > 0 && i > 0 ) {
		    i--;
		    w -= fm.width( m_textBuffer.at(i) );
		}
		if ( w < 0 && i != m_cursorPos )i++;
		m_offset = i;
    } else if ( m_offset ) {
		int i = showLastPartOffset( m_textBuffer.data(), fontMetrics(),width() - 8);
		if ( i < m_offset )m_offset = i;
    }
    kviInputRepaint;
}
void KviInput::copySlot(){
	QKeyEvent k( Event_KeyPress,Key_C,0,ControlButton);
	keyPressEvent(&k);
}
void KviInput::pasteSlot(){
	QKeyEvent k( Event_KeyPress,Key_V,0,ControlButton);
	keyPressEvent(&k);
}
void KviInput::mousePressEvent( QMouseEvent *e )
{
	_debug_entertrace("mousePressEvent");

    killTimers();
	if ( e->button() & RightButton ) {		//popup the menu
		if(!m_lpContextPopup->isVisible()){
			m_lpContextPopup->setItemEnabled(1,kviInputHasSelection);
			m_lpContextPopup->setItemEnabled(2,QApplication::clipboard()->text());
			m_lpContextPopup->popup(mapToGlobal(e->pos()));
		}
		return;
	}

    int margin = 4;
    m_cursorPos=m_offset+xPosToCursorPos(&m_textBuffer[(int)m_offset],fontMetrics(),e->pos().x()-margin,width()-2*margin);
    if ( e->button() & MidButton ) {		// paste
		pasteSlot();
		return;
    }
    m_markAnchor = m_cursorPos;
    newMark( m_markAnchor);
    m_bCursorActive	  = TRUE;
    m_bDragScrolling = FALSE;
    startTimer( KVI_INPUT_BLINK_TIME );
	repaint(false);
	_debug_leavetrace("mousePressEvent");
}

void KviInput::mouseMoveEvent( QMouseEvent *e )
{
	_debug_entertrace("mouseMoveEvent");
    int margin = 4;

    if ( e->pos().x() < margin || e->pos().x() > width() - margin) {
	m_bScrollLeft =	 ( e->pos().x() < margin );
	if ( !m_bDragScrolling ) {
	    if ( m_bScrollLeft )
		newMark(m_offset);
	    else
		newMark(lastCharVisible());
	    killTimers();
	    m_bDragScrolling = TRUE;
	    m_bCursorActive	  = TRUE;
	    startTimer( KVI_INPUT_SCROLL_TIME );
	    kviInputRepaint;
	} else {
	    if ( m_bScrollLeft ) {
		int steps = -(e->pos().x() + margin) / 15 + 2;
		moveCursorLeft( TRUE, steps );
	    } else {
		int steps = (e->pos().x() - width() +  margin) / 15 + 2;
		moveCursorRight( TRUE, steps );
	    }
	}
    } else {
	m_bDragScrolling = FALSE;
	int mousePos  = m_offset +
	    xPosToCursorPos( &m_textBuffer[(int)m_offset], fontMetrics(),e->pos().x()- margin,width()-margin-margin);
	newMark( mousePos);
	m_bCursorActive  = TRUE;
	killTimers();
	startTimer( KVI_INPUT_BLINK_TIME );
	kviInputRepaint;
    }
	_debug_leavetrace("mouseMoveEvent");
}

void KviInput::mouseReleaseEvent( QMouseEvent * )
{
	_debug_entertrace("mouseReleaseEvent");
    if ( m_bDragScrolling ) {
		m_bDragScrolling = FALSE;
		killTimers();
		startTimer( KVI_INPUT_BLINK_TIME );
    }
	_debug_leavetrace("mouseReleaseEvent");
}

void KviInput::mouseDoubleClickEvent( QMouseEvent * )
{
	_debug_entertrace("mouseDoubleClickEvent");
    if ( m_bDragScrolling ) {
		m_bDragScrolling = false;
		killTimers();
		startTimer( KVI_INPUT_BLINK_TIME );
    }
    markWord( m_cursorPos );
    kviInputRepaint;
	_debug_leavetrace("mouseDoubleClickEvent");
}

void KviInput::pixmapPaint( const QRect& clip )
{
    QPainter painter( m_lpMemBuffer );
    painter.setClipRect( clip );
    painter.setFont( font() );
    paintText( &painter, m_lpMemBuffer->size());
    painter.end();
    bitBlt( this, 0, 0, m_lpMemBuffer, 0, 0, width(), height() );
}

void KviInput::setAttributesFromString(QString &szData){
	m_bBold			=false;
	m_bUnderline	=false;
	m_curForeColor	=120; //default
	m_curBackColor	=100;

	if(szData.isEmpty())return;
	register char *pC=szData.data();
	while(*pC)
	{	
		char appColor;
		switch((*pC)){
		case KVI_TEXT_BOLD:
			m_bBold			=(!m_bBold);
			break;
		case KVI_TEXT_UNDERLINE:
			m_bUnderline	=(!m_bUnderline);
			break;
		case KVI_TEXT_RESET:
			m_bUnderline	=false;
			m_bBold			=false;
			m_curForeColor	=120;
			m_curBackColor	=100;
			break;
		case KVI_TEXT_REVERSE:
			appColor		=m_curForeColor;
			m_curForeColor	=m_curBackColor;
			m_curBackColor	=appColor;
			break;
		case KVI_TEXT_COLOR: //COLORS...
#define _DO_SET_1_(_var1,_var2) { m_curBackColor=_var1;m_curForeColor=_var2; }
#define _DO_SET_2_(_var) { m_curBackColor=_var; }
			{ /////////////////////////////////////////////COLORS BLOCK
				if(!(*(++pC))) _DO_SET_1_(100,120)
				if(((*pC)>='0')&&((*pC)<='9')){
					QString szINKColor(2); szINKColor[0]=(*pC);
					if((*pC)=='1'){
						if(!(*(++pC)))_DO_SET_1_(100,((char)szINKColor.toLong()))
						if(((*pC)>='0')&&((*pC)<'6')){
							szINKColor.resize(3); szINKColor[1]=(*pC);
							if(!(*(++pC)))_DO_SET_1_(100,((char)szINKColor.toLong()))
						}
					} else {
						if(!(*(++pC)))_DO_SET_1_(100,((char)szINKColor.toLong()))
					}
					m_curForeColor=((char)szINKColor.toLong());
					if((*pC)==','){
						if(!(*(++pC)))_DO_SET_2_(100)
						if(((*pC)>='0')&&((*pC)<='9')){
							QString szPAPERColor(2);  szPAPERColor[0]=(*pC);
							if((*pC)=='1'){
								if(!(*(++pC)))_DO_SET_2_(((char)szPAPERColor.toInt()))
								if(((*pC)>='0')&&((*pC)<'6')){
									szPAPERColor.resize(3); szPAPERColor[1]=(*pC);
								} else --pC;
								m_curBackColor=(char)szPAPERColor.toLong();
							} else m_curBackColor=(char)szPAPERColor.toLong();
						} else pC-=2;
					} else --pC;
				} else {
					m_curBackColor=100;
					m_curForeColor=120;
					--pC;
				}
			} ///////////////////////////////////////////////////////////////// Colors Block
			break;
		}
		++pC;
	}
}
bool KviInput::getTextPieceAndSetAttributes(QString &szData,QString &szText)
{
	if(szData.isEmpty())return false;

	register const char *pC=szData.data(); //gcc,please PUT IN IN A REGISTEEEEER!!!! :)

	////////////////////////////////////////////// GET ATTRIBUTES INFO
	while(	((*pC)==KVI_TEXT_BOLD)||((*pC)==KVI_TEXT_UNDERLINE)||((*pC)==KVI_TEXT_COLOR)||
			((*pC)==KVI_TEXT_REVERSE)||((*pC)==KVI_TEXT_RESET))
	{	
		char appColor;
		switch((*pC)){
		case KVI_TEXT_BOLD:
			m_bBold			=(!m_bBold);
			break;
		case KVI_TEXT_UNDERLINE:
			m_bUnderline	=(!m_bUnderline);
			break;
		case KVI_TEXT_RESET:
			m_bUnderline	=false;
			m_bBold			=false;
			m_curForeColor	=120;
			m_curBackColor	=100;
			break;
		case KVI_TEXT_REVERSE:
			appColor		=m_curForeColor;
			m_curForeColor	=m_curBackColor;
			m_curBackColor	=appColor;
			break;
		default: //COLORS...
#define _DO_RETURN_1_(_var1,_var2) { szText=szData.left(szData.length()-strlen(pC));szData=pC;m_curBackColor=_var1;m_curForeColor=_var2;return true; }
#define _DO_RETURN_2_(_var) { szText=szData.left(szData.length()-strlen(pC));szData=pC;m_curBackColor=_var;return true;}

			{ /////////////////////////////////////////////COLORS BLOCK
				if(!(*(++pC))) _DO_RETURN_1_(100,120)
				if(((*pC)>='0')&&((*pC)<='9')){
					QString szINKColor(2); szINKColor[0]=(*pC);
					if((*pC)=='1'){
						if(!(*(++pC)))_DO_RETURN_1_(100,((char)szINKColor.toLong()))
						if(((*pC)>='0')&&((*pC)<'6')){
							szINKColor.resize(3); szINKColor[1]=(*pC);
							if(!(*(++pC)))_DO_RETURN_1_(100,((char)szINKColor.toLong()))
						}
					} else {
						if(!(*(++pC)))_DO_RETURN_1_(100,((char)szINKColor.toLong()))
					}
					m_curForeColor=((char)szINKColor.toLong());
					if((*pC)==','){
						if(!(*(++pC)))_DO_RETURN_2_(100)
						if(((*pC)>='0')&&((*pC)<='9')){
							QString szPAPERColor(2);  szPAPERColor[0]=(*pC);
							if((*pC)=='1'){
								if(!(*(++pC)))_DO_RETURN_2_(((char)szPAPERColor.toInt()))
								if(((*pC)>='0')&&((*pC)<'6')){
									szPAPERColor.resize(3); szPAPERColor[1]=(*pC);
								} else --pC;
								m_curBackColor=(char)szPAPERColor.toLong();
							} else m_curBackColor=(char)szPAPERColor.toLong();
						} else pC-=2;
					} else --pC;
				} else {
					m_curBackColor=100;
					m_curForeColor=120;
					--pC;
				}
			} ///////////////////////////////////////////////////////////////// Colors Block

			break;	
		}
		
		if(!(*(++pC))){
			szText=szData.left(szData.length()-strlen(pC));szData=pC; return true;
	 	}		//Only attribute changes....no text...
	}
	if(pC != szData.data()){
		szText=szData.left(szData.length()-strlen(pC));szData=pC; return true;	
	}
#undef _DO_RETURN_1_
#undef _DO_RETURN_2_

#define _isControlChar(__pchar)	(	((__pchar)==KVI_TEXT_BOLD)  || ((__pchar)==KVI_TEXT_UNDERLINE)||  \
									((__pchar)==KVI_TEXT_COLOR) || ((__pchar)==KVI_TEXT_RESET) ||    \
									((__pchar)==KVI_TEXT_REVERSE) )
	while(! (_isControlChar(*pC))){
		if(!(*(++pC))){ szText=szData;szData="";return true; }
	}
	QString szApp(pC);
	szText=szData.left(szData.length()-szApp.length());
	szData=szApp;
	return true; //OUHHHHHHHHHHH :))))))))
}


void KviInput::paintText( QPainter *p, const QSize &s)
{
	bool hasPixmap;
	QColorGroup g = colorGroup();
	if(backgroundPixmap()){
		if(backgroundPixmap()->isNull())hasPixmap=FALSE;
		else hasPixmap=TRUE;
     } else hasPixmap=FALSE;
    
	QFontMetrics fm(font());
    int markBegin 		= minMark();
    int markEnd			= maxMark();
    int margin			= 4;
    QBrush fill(m_backgroundColor);
    
	if(hasPixmap)fill.setPixmap(*(backgroundPixmap()));	
	
	qDrawWinPanel( p, 0, 0, s.width(), s.height(), g,true, &fill );

	//Ok let's draw the text......
	QString displayText;
	displayText = m_textBuffer.mid( m_offset, m_textBuffer.length() );
	QString firstPart=m_textBuffer.left(m_offset);
	int idxK=firstPart.findRev(KVI_TEXT_COLOR);
	if(idxK != -1){
		uint idx=0;
		while((firstPart.length()-(idxK+1))<6){
			if(displayText.length()>idx){
				char ch=displayText[idx];
				if(!((ch == KVI_TEXT_BOLD)||(ch == KVI_TEXT_REVERSE)||(ch == KVI_TEXT_UNDERLINE)||
					(ch == KVI_TEXT_RESET)||(ch == KVI_TEXT_COLOR))){
					firstPart+=ch;
					idx++;
				} else idxK-=6;
			} else idxK-=6;
		}
	}
	setAttributesFromString(firstPart);
   

    int ypos = s.height()-margin- fm.descent() - 1 -(s.height() - 2*margin - fm.height())/2;

    if ( !displayText.isEmpty() ) {
		p->setClipRect( margin, margin,s.width()- 2*margin,s.height() - 2*margin );	

		int charsVisible = lastCharVisible() - m_offset;
		if ( displayText[ charsVisible ] != '\0' )charsVisible++;

		int invertStart,invertEnd; // start and end of inverted text, 0 = leftmost char

		if ( markBegin > m_offset ) {
		    if ( markBegin <  m_offset + charsVisible )invertStart = markBegin - m_offset;
		    else invertStart = charsVisible;
		} else invertStart = 0;

		if ( markEnd > m_offset ) {
		    if ( markEnd <	m_offset + charsVisible )invertEnd = markEnd - m_offset;
		    else invertEnd = charsVisible;
		} else {
		    invertEnd = 0;
		}
		QString marked;
		int xpos1=0;
		int xpos2=0;
		if ( invertEnd != invertStart ){
			marked=displayText.mid( invertStart, (invertEnd - invertStart ) );
			xpos1 =  margin + fm.width( displayText, invertStart );
		    xpos2 =  xpos1 + fm.width( marked ) - 1;
		} else marked="";

		QString szPiece;
		int xLeft=margin;
		int xRight=0;
		int xWidth=0;
		QString szDisplay=displayText.copy();
		while(!szDisplay.isEmpty()){
			if(getTextPieceAndSetAttributes(szDisplay,szPiece)){
				//debug("%s:fore:%d:back:%d",szPiece.data(),m_curForeColor,m_curBackColor);
				if(m_curForeColor>15){
					if(m_curForeColor==100)p->setPen(m_backgroundColor);
					else p->setPen(m_foregroundColor);
				}
				else p->setPen(*(m_lpFrame->m_lpColor[m_curForeColor]));
				QFont fnt(font());
				//fnt.setBold(m_bBold);
				fnt.setUnderline(m_bUnderline);
				p->setFont(fnt);
				QFontMetrics theFm(fnt);
				xWidth=theFm.width(szPiece.data());
				xRight=xLeft+xWidth;
				if(m_curBackColor<16){
					p->fillRect(xLeft,ypos-fm.ascent(),xRight-xLeft,fm.height(),(*(m_lpFrame->m_lpColor[m_curBackColor])));
				} else if(m_curBackColor==120){
					p->fillRect(xLeft,ypos-fm.ascent(),xRight-xLeft,fm.height(),m_foregroundColor);	
				}
				p->drawText(xLeft,ypos,szPiece);
				if(m_bBold)p->drawText(xLeft+1,ypos,szPiece);
				if(_isControlChar(szPiece[0])){
					p->setPen(m_cursorColor);
					p->drawRect(xLeft,ypos-fm.ascent(),xRight-xLeft,fm.height());
				}
				xLeft=xRight;
			} else szDisplay="";
		}
		if (!marked.isEmpty()) {
		    p->fillRect( xpos1, ypos - fm.ascent(),xpos2 - xpos1, fm.height(),m_markBackColor);
			p->setPen(m_markForeColor);
		    p->drawText( xpos1, ypos, marked );
		}
    }

	p->setPen(m_cursorColor);
    p->setClipping(false);

    if (m_bCursorActive) {
		int curXPos = margin;
		curXPos += fm.width( displayText, m_cursorPos - m_offset ) - 1;
		//debug("%s ; %d ;",displayText.data(),curXPos);
		int curYPos   = ypos - fm.ascent() -1;
		if ( hasFocus() ) {
			int aHeight=fm.height();
			int curYEnd=curYPos+aHeight;
		    p->drawLine(curXPos, curYPos, curXPos, curYEnd);
			p->drawLine(curXPos - 2, curYPos,curXPos + 2, curYPos );
			p->drawLine(curXPos - 2,curYEnd,curXPos + 2,curYEnd);
			if(m_bColorBoxVisible){
				p->drawLine(curXPos,curYEnd-(aHeight/2),curXPos+7,curYPos);
				p->drawLine(curXPos,curYEnd-(aHeight/2),curXPos+7,curYEnd);
				p->drawLine(curXPos +5, curYPos,curXPos + 9, curYPos );
				p->drawLine(curXPos +5,curYEnd,curXPos + 9,curYEnd);
			}
		}
    }
}

#undef _isControlChar

void KviInput::moveCursorLeft( bool mark, int steps )
{
    if ( steps < 0 ) {
	moveCursorRight( mark, -steps );
	return;
    }
    if ( m_cursorPos != 0 || !mark && kviInputHasSelection ) {
		killTimers();
		m_bCursorActive=true;
		m_cursorPos -= steps;
		if ( m_cursorPos < 0 )m_cursorPos = 0;
		if ( mark ) {
		    newMark( m_cursorPos );
		} else {
		    m_markAnchor = m_cursorPos;
		    m_markDrag   = m_markAnchor;
		}
		if ( m_cursorPos < m_offset )m_offset = m_cursorPos;
		startTimer( m_bDragScrolling ? KVI_INPUT_SCROLL_TIME : KVI_INPUT_BLINK_TIME );
		kviInputRepaint;
    }
}

void KviInput::moveCursorRight( bool mark, int steps )
{
    int margin = 4;
    if ( steps < 0 ) {
		moveCursorLeft( mark, -steps );
		return;
    }
    int len = (int)strlen( m_textBuffer );
    if ( len > m_cursorPos || !mark && kviInputHasSelection ) {
		QFontMetrics fm = fontMetrics();
		killTimers();
		m_bCursorActive   = TRUE;
		m_cursorPos += steps;
		if ( m_cursorPos > len )m_cursorPos = len;
		if ( mark ) {
		    newMark( m_cursorPos );
		} else {
		    m_markAnchor = m_cursorPos;
		    m_markDrag   = m_markAnchor;
		}
		int surplusWidth = width() - 2*margin- fm.width( &m_textBuffer[ m_offset ], m_cursorPos - m_offset);
		if ( surplusWidth < 0 ) {
		    while ( surplusWidth < 0 && m_offset < m_cursorPos - 1 ) {
				surplusWidth += fm.width( &m_textBuffer[ m_offset ], 1 );
				m_offset++;
		    }
		}
	startTimer( m_bDragScrolling ? KVI_INPUT_SCROLL_TIME : KVI_INPUT_BLINK_TIME );
	kviInputRepaint;
    }
}

void KviInput::backspace()
{
    if ( kviInputHasSelection ) {
		del();
    } else {
		if ( m_cursorPos > 0 ) {
		    moveCursorLeft( FALSE );
		    del();
		}
    }
}

void KviInput::del()
{
    QString test( m_textBuffer.copy() );

    if ( kviInputHasSelection ) {
	test.remove( minMark(), maxMark() - minMark() );
	int cp = minMark();
	m_textBuffer = test;
	m_cursorPos = cp;
	m_markAnchor = m_cursorPos;
	m_markDrag   = m_cursorPos;
	if ( m_cursorPos < m_offset )
	    m_offset = m_cursorPos;
	repaint( !hasFocus() );
    } else if ( m_cursorPos != (int)strlen(m_textBuffer) ) {
		test.remove( m_cursorPos, 1 );
		int curPos = m_cursorPos;
		m_textBuffer = test;
		m_cursorPos = curPos;
		kviInputRepaint;
    }
}

void KviInput::home( bool mark )
{
    if ( m_cursorPos != 0 || !mark && kviInputHasSelection ) {
	killTimers();
	m_cursorPos = 0;
	if ( mark ) {
	    newMark( m_cursorPos );
	} else {
	    m_markAnchor = 0;
	    m_markDrag   = m_markAnchor;
	}
	m_offset	 = 0;
	m_bCursorActive = TRUE;
	startTimer( m_bDragScrolling ? KVI_INPUT_SCROLL_TIME : KVI_INPUT_BLINK_TIME );
	kviInputRepaint;
    }
}

void KviInput::end( bool mark )
{
	int tlen = strlen( m_textBuffer );
	if ( m_cursorPos != tlen || !mark && kviInputHasSelection ) {
		killTimers();
		m_offset += showLastPartOffset( &m_textBuffer[m_offset], fontMetrics(),width() - 8 );
		m_cursorPos = tlen;
		if ( mark ) {
			newMark( m_cursorPos );
		} else {
			m_markAnchor = m_cursorPos;
			m_markDrag   = m_markAnchor;
		}
		m_bCursorActive  = TRUE;
		startTimer( m_bDragScrolling ? KVI_INPUT_SCROLL_TIME : KVI_INPUT_BLINK_TIME );
		kviInputRepaint;
	}
}

void KviInput::newMark( int pos)
{
    m_markDrag  = pos;
    m_cursorPos = pos;
}


void KviInput::markWord( int pos )
{
    int i = pos - 1;
    while ( i >= 0 && isprint(m_textBuffer.at(i)) && !isspace(m_textBuffer.at(i)) )i--;
    i++;
    m_markAnchor = i;

    int lim = m_textBuffer.length();
    i = pos;
    while ( i < lim && isprint(m_textBuffer.at(i)) && !isspace(m_textBuffer.at(i)) )i++;
    m_markDrag = i;

    int maxVis	  = lastCharVisible();
    int markBegin = minMark();
    int markEnd	  = maxMark();
    if ( markBegin < m_offset || markBegin > maxVis ) {
		if ( markEnd >= m_offset && markEnd <= maxVis ) {
		    m_cursorPos = markEnd;
		} else {
		    m_offset    = markBegin;
		    m_cursorPos = markBegin;
		}
    } else {
	m_cursorPos = markBegin;
    }
}

int KviInput::lastCharVisible() const
{
	int tDispWidth = width() - 8;
	return m_offset + xPosToCursorPos( &m_textBuffer[(int)m_offset], fontMetrics(),tDispWidth, tDispWidth );
}

int KviInput::minMark() const
{
    return m_markAnchor < m_markDrag ? m_markAnchor : m_markDrag;
}

int KviInput::maxMark() const
{
    return m_markAnchor > m_markDrag ? m_markAnchor : m_markDrag;
}

#include "m_kvi_input.moc"


//
// $Log: kvi_input.cpp,v $
// Revision 1.6  1998/09/30 03:11:33  pragma
// Bugfixes
//
// Revision 1.5  1998/09/29 14:03:21  pragma
// Minor hacks all around
//
//
