//
//   File : kvi_chat.cpp (/usr/cvs/kvirc/kvirc/kvi_chat.cpp)
//   Last modified : Fri Dec 4 1998 02:15:44 by root@localhost.localdomain
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998-1999 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation ; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING. If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviChatWnd"

#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_chat.h"
#include "kvi_translate.h"
#include "kvi_input.h"
#include "kvi_view.h"
#include "kvi_frame.h"
#include "kvi_uparser.h"
#include "kvi_debug.h"
#include "kvi_listen.h"
#include "kvi_socket.h"
#include "kvi_opt.h"
#include "kvi_ident.h"
#include "kvi_event.h"
#include "kvi_mdi.h"
#include "kvi_app.h"
#include "kvi_taskbar.h"
#include "kvi_int.h"
#include "kvi_support.h"
#include "kvi_global.h"
#include "kvi_usr.h"
#include "kvi_socket.h"
//
//#ifdef __BSD #include <sys/socket.h> #include <sys/types.h>
//
#include <sys/types.h>
#include <sys/socket.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <errno.h>


KviChatWnd::KviChatWnd(KviMdiManager *parent,KviFrame *frame,const char *aname,int aid)
:KviMdiChild(parent,frame,aname,aid,KVI_WND_TYPE_CHAT)
{
	_debug_entertrace("KviChatWnd");
	m_lpInput   =new KviInput(this,aid,KVI_WND_TYPE_CHAT,aname,frame,"KviInputClass");
	m_lpOutput  =new KviView(this,frame,"KviViewClass");
	m_lpOutput->m_lpChatParent =this;

	setFocusOwner(m_lpInput);
	connect(this,SIGNAL(closeButtonPressed()),this,SLOT(closeSlot()));
	applyOptions();

	m_Sock              = -1;
	m_lpListeningSocket = 0;
	m_lpReadNotifier    = 0;
	m_lpWriteNotifier   = 0;
	m_bConnected        = false;
	m_lpOpt             = m_lpFrm->m_lpOpt;
	m_lpTimer           = 0;
	m_bWaitForAck       = false;
	_debug_leavetrace("KviChatWnd");
}

KviChatWnd::~KviChatWnd()
{
	_debug_entertrace("~KviChatWnd");
	m_lpFrm->m_lpUserParser->m_szLastDCCTarget ="";
	killSock();
	delete m_lpInput;
	delete m_lpOutput;
	_debug_leavetrace("~KviChatWnd");
}


//============ killNotifiers ============//

void KviChatWnd::killNotifiers()
{
	_debug_entertrace("killNotifiers");
	if(m_lpReadNotifier){
		delete m_lpReadNotifier;
		m_lpReadNotifier=0;
	}
	if(m_lpWriteNotifier){
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	_debug_leavetrace("killNotifiers");
}

//============ killSock ============//

void KviChatWnd::killSock()
{
	_debug_entertrace("killSock");
	if(m_lpListeningSocket){
		m_lpListeningSocket->stopListening();
		delete m_lpListeningSocket;
		m_lpListeningSocket=0;
	}
	killNotifiers();
	if(m_lpTimer){
		if(m_lpTimer->isActive())m_lpTimer->stop();
		delete m_lpTimer;
		m_lpTimer=0;
	}
	if(m_Sock != -1)::close(m_Sock);
	m_bConnected=false;
	_debug_leavetrace("killSock");
}

//============ requestChat =============//

bool KviChatWnd::requestChat(const char *szNick)
{
	//
	// Sends a DCC Chat request to a specified nick on IRC
	//
	_debug_entertrace("requestChat");
	m_szRemoteNick=szNick;
	// Status text
	QString szStat;
	szStat.sprintf(i18n("Requesting DCC CHAT connection to %s"),szNick);
	m_lpFrm->setStatusText(szStat.data());
	// Fail if we have no IRC connection
	if(!m_lpFrm->m_lpSock->m_bConnected){
		requestFailed(i18n("DCC CHAT requires an active IRC connection"));
		return false;
	}
	// Find our address
	unsigned long myAddr=m_lpFrm->m_lpSock->getSockAddress();
	if(!myAddr){
		// Can not find it by using a system call
		// try to use the user supplied address
		if(!m_lpOpt->szLocalIp.isEmpty()){
			struct in_addr inAddress;
			if(inet_aton(m_lpOpt->szLocalIp.data(),&inAddress)){
				myAddr=inAddress.s_addr;
				doFmtOutput(KVI_OUT_INTERNAL,i18n("Warning : unable to resolve localhost, using user supplied IP address : %s"),m_lpOpt->szLocalIp.data());
			}
		}
		if(!myAddr){
			// Nothing....just fail
			requestFailed(i18n("Unable to resolve localhost. (You can try by setting your IP address in the options dialog.)"));
			return false;
		}
	}
	// Setup the listening socket
	m_lpListeningSocket=new KviListeningSocket();
	if(!m_lpListeningSocket->listenForConnection(0,&m_Sock,m_lpFrm)){
		requestFailed(i18n("Unable to setup a listening socket, can't request DCC CHAT"));
		killSock();
		return false;
	}

	QObject::connect(m_lpListeningSocket,SIGNAL(gotConnection()),this,SLOT(connectedSlot()));
	QObject::connect(m_lpListeningSocket,SIGNAL(connectTimedOut()),this,SLOT(connectTimedOut()));
	// Yes a bit senseless the string here...
	QString szAddr;
	QString szPort;
	szAddr.setNum(myAddr);
	szPort.setNum(m_lpListeningSocket->m_iPort);
	// OK , send the request via IRC
	m_lpFrm->m_lpSock->sendFmtData("PRIVMSG %s :%cDCC CHAT chat %s %s%c",szNick,0x01,szAddr.data(),szPort.data(),0x01);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Sent DCC CHAT request to %s...waiting for reply"),szNick);
	_debug_leavetrace("requestChat");
	return true;
}


//============ acceptFailed ============//

void KviChatWnd::acceptFailed(const char *text)
{
	_debug_entertrace("acceptFailed");
	doOutput(KVI_OUT_ERROR,text);
	doOutput(KVI_OUT_ERROR,i18n("DCC CHAT Accept failed"));
	m_lpFrm->setStatusText(i18n("DCC CHAT Accept failed"));
	_debug_leavetrace("acceptFailed");
}

//============ requestFailed ============//

void KviChatWnd::requestFailed(const char *text)
{
	_debug_entertrace("requestFailed");
	doOutput(KVI_OUT_ERROR,text);
	doOutput(KVI_OUT_ERROR,i18n("DCC CHAT Request failed"));
	m_lpFrm->setStatusText(i18n("DCC CHAT Request failed"));
	_debug_leavetrace("requestFailed");
}

//============ connectToDaemon ============//

void KviChatWnd::connectToDaemon(QString &szPort,QString &szIp)
{
	//
	// Connects to a specified host and port (DCC without IRC)
	//
	_debug_entertrace("connectToDaemon");
	QString szStat;
	m_lpFrm->setStatusText((szStat.sprintf(i18n("Connecting to %s"),szIp.data())).data());
	m_szRemoteNick=i18n("Unknown");
	bool bOK=false;
	unsigned short int iPort=(unsigned short int)szPort.toUInt(&bOK);
	if(!bOK){
		requestFailed(i18n("Can't retrieve the port to connect. DCC CHAT failed."));
		return;	
	}
	szIp=szIp.stripWhiteSpace();
	struct in_addr inAddress;
	if(!inet_aton(szIp.data(),&inAddress)){
		requestFailed(i18n("Can't retrieve the address to connect. DCC CHAT failed."));
		return;	
	}
	m_szRemoteIp=szIp.data();
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connecting to %s on port %s"),szIp.data(),szPort.data());
	struct sockaddr_in sockAddrIn;
	m_lpFrm->m_lpSock->fillSockAddr(&sockAddrIn,inAddress,iPort);
	m_Sock = ::socket(PF_INET,SOCK_STREAM,0);
	if (m_Sock < 0){
		requestFailed(i18n("Unable to create STREAM socket for connection."));
		return;		
	}
	if(fcntl(m_Sock, F_SETFL, O_NONBLOCK) < 0){
		::close(m_Sock);
		requestFailed(i18n("Unable to create STREAM socket for connection.(fcntl failed)"));
		return;
	};
	m_lpTimer=new QTimer();
	QObject::connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(connectTimedOut()));
	m_lpTimer->start(m_lpFrm->m_lpOpt->iDCCCHATTimeout * 1000);
	if(::connect( m_Sock, (struct sockaddr*)(&sockAddrIn), sizeof(sockAddrIn) )<0){
		if(errno != EINPROGRESS){
			killSock();
			QString szError;
			m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(errno),szError);
			doFmtOutput(KVI_OUT_ERROR,i18n("Unable to connect to %s : %s"),m_szRemoteIp.data(),szError.data());
			return;
		}
	}
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	m_lpWriteNotifier= new QSocketNotifier(m_Sock, QSocketNotifier::Write,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QObject::connect(m_lpWriteNotifier, SIGNAL(activated(int)), this, SLOT(slotConnectedToDaemon(int)));
	_debug_leavetrace("connectToDaemon");
}

//============ acceptConnectedChat ============//

void KviChatWnd::acceptConnectedChat(int sockFd,const char *szIp,const char *szNick)
{
	_debug_entertrace("acceptConnectedChat");
	QString szStat;
	m_lpFrm->setStatusText((szStat.sprintf(i18n("Accepting DCC Chat from %s@%s"),szNick,szIp)).data());
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Incoming DCC Chat from %s@%s"),szNick,szIp);
	m_szRemoteNick=szNick;
	m_szRemoteIp=szIp;
	m_Sock = sockFd;
	if (m_Sock < 0){
		acceptFailed(i18n("Wrong file descriptor."));
		killSock();
		return;		
	}
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QString szN=m_lpFrm->m_lpGlb->szNick.data();
	//send response : 101 myNick
	if(szN.isEmpty() || (!strcmp(szN.data(),KVI_STR_NULL))){
		szN=m_lpFrm->m_lpUsr->szNick.data();
		if(szN.isEmpty() || (!strcmp(szN.data(),KVI_STR_NULL))){
			doFmtOutput(KVI_OUT_INTERNAL,i18n("You have specified no nickname. Presenting myself as UNKNOWN"));
			szN=i18n("UNKNOWN");
		}
	}
	m_bConnected=true;
	szN.prepend("101 ");
	sendData(szN.data());
	doOutput(KVI_OUT_INTERNAL,i18n("Connected."));
	if(m_lpFrm->m_lpOpt->bShowMotdOnInDcc)sendMotd();
	_debug_leavetrace("acceptConnectedChat");
}

void KviChatWnd::acceptChat(QString &szPort,QString &szIp,const char *szNick)
{
	QString szStat;
//	szStat.sprintf(i18n("Accepting DCC CHAT connection from %s"),szNick);
	m_lpFrm->setStatusText((szStat.sprintf(i18n("Accepting DCC Chat from %s"),szNick)).data());
	m_szRemoteNick=QString(szNick);
	bool bOK=false;
	unsigned short int iPort=(unsigned short int)szPort.toUInt(&bOK);
	if(!bOK){
		acceptFailed(i18n("Can't retrieve the port to connect. DCC CHAT failed."));
		return;	
	}
	szIp=szIp.stripWhiteSpace();
	unsigned long iAddr=szIp.toULong(&bOK);
	if(!bOK){
		acceptFailed(i18n("Can't retrieve the address to connect. DCC CHAT failed."));
		return;	
	}
	struct in_addr inAddress;
	inAddress.s_addr=ntohl(iAddr);
	m_szRemoteIp=QString(inet_ntoa(inAddress));
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connecting to %s on port %s"),m_szRemoteIp.data(),szPort.data());
	struct sockaddr_in sockAddrIn;
	m_lpFrm->m_lpSock->fillSockAddr(&sockAddrIn,inAddress,iPort);
	m_Sock = ::socket(PF_INET,SOCK_STREAM,0);
	if (m_Sock < 0){
		acceptFailed(i18n("Unable to create STREAM socket for connection."));
		return;		
	}
	if(fcntl(m_Sock, F_SETFL, O_NONBLOCK) < 0){
		::close(m_Sock);
		acceptFailed(i18n("Unable to create STREAM socket for connection.(fcntl failed)"));
		return;
	};
	m_lpTimer=new QTimer();
	QObject::connect(m_lpTimer,SIGNAL(timeout()),this,SLOT(connectTimedOut()));
	m_lpTimer->start(m_lpFrm->m_lpOpt->iDCCCHATTimeout * 1000);
	if(::connect( m_Sock, (struct sockaddr*)(&sockAddrIn), sizeof(sockAddrIn) )<0){
		if(errno != EINPROGRESS){
			killSock();
//			int errore=m_lpFrm->m_lpSock->translateFromSystemError(errno);
			QString szError;
			m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(errno),szError);
			doFmtOutput(KVI_OUT_ERROR,i18n("Unable to connect to %s : %s"),m_szRemoteIp.data(),szError.data());
			return;
		}
	}
	m_lpReadNotifier = new QSocketNotifier(m_Sock, QSocketNotifier::Read,this);
	m_lpWriteNotifier= new QSocketNotifier(m_Sock, QSocketNotifier::Write,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	QObject::connect(m_lpWriteNotifier, SIGNAL(activated(int)), this, SLOT(slotConnected(int)));
}

//============ slotConnected ============//

void KviChatWnd::slotConnected(int)
{
	_debug_entertrace("slotConnected");
	if(m_lpWriteNotifier){
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	if(m_lpTimer){
		if(m_lpTimer->isActive())m_lpTimer->stop();
		delete m_lpTimer;
		m_lpTimer=0;
	}
	int nSockErr;
	ksize_t iSize=sizeof(int);
	if(getsockopt(m_Sock,SOL_SOCKET,SO_ERROR,(void *)&nSockErr,&iSize)==-1)nSockErr=EBADF;
	if(nSockErr!=0){
		QString szErr;
		m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(nSockErr),szErr);
		doFmtOutput(KVI_OUT_ERROR,i18n("Connect failed : %s"),szErr.data());
		killSock();
		return;
	}
	m_bConnected=true;
	doOutput(KVI_OUT_INTERNAL,i18n("Connected"));
	//Accepted connection
	if(m_lpFrm->m_lpOpt->bShowMotdOnInDcc)sendMotd();
	_debug_leavetrace("slotConnected");
}

//============ slotConnectedToDaemon ============//

void KviChatWnd::slotConnectedToDaemon(int)
{
	_debug_entertrace("slotConnectedToDaemon");
	if(m_lpWriteNotifier){
		delete m_lpWriteNotifier;
		m_lpWriteNotifier=0;
	}
	if(m_lpTimer){
		if(m_lpTimer->isActive())m_lpTimer->stop();
		delete m_lpTimer;
		m_lpTimer=0;
	}
	int nSockErr;
	ksize_t iSize=sizeof(int);
	if(getsockopt(m_Sock,SOL_SOCKET,SO_ERROR,(void *)&nSockErr,&iSize)==-1)nSockErr=EBADF;
	if(nSockErr!=0){
		QString szErr;
		m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(nSockErr),szErr);
		doFmtOutput(KVI_OUT_ERROR,i18n("Connect failed : %s"),szErr.data());
		killSock();
		return;
	}
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connected to daemon (%s)"),m_szRemoteIp.data());
	doOutput(KVI_OUT_INTERNAL,i18n("Logging in..."));
	// Accepted connection send : 100 MyNickname
	// and wait for acknowledgement
	QString szN=m_lpFrm->m_lpGlb->szNick.data();
	if(szN.isEmpty() ||(!strcmp(szN.data(),KVI_STR_NULL))){
		szN=m_lpFrm->m_lpUsr->szNick.data();
		if(szN.isEmpty()||(!strcmp(szN.data(),KVI_STR_NULL))){
			doFmtOutput(KVI_OUT_INTERNAL,i18n("You have specified no nickname; Presenting myself as UNKNOWN"));
			szN=i18n("UNKNOWN");
		}
	}
	szN.prepend("100 ");
	m_bConnected=true;
	sendData(szN.data());
	m_bConnected=false;
	// Now wait for response : 101 ServerNickname
	m_bWaitForAck=true;
	_debug_leavetrace("slotConnectedToDaemon");
}

//============ sendMotd ============//

void KviChatWnd::sendMotd()
{
	_debug_entertrace("sendMotd");
	if(!m_lpFrm->m_lpOpt->lpListDccMotd->isEmpty()){
		const char *szS=0;
		doOutput(KVI_OUT_INTERNAL,i18n("Sending mesage of the day"));
		for(szS=m_lpOpt->lpListDccMotd->first();szS;szS=m_lpOpt->lpListDccMotd->next()){
			QString szString=szS;
			m_lpFrm->m_lpUserParser->m_lpIdentifiers->substitute(this,szString,0);
			sendData(szString.data());
			szString.prepend("[ MOTD >> ]:");
			doOutput(KVI_OUT_MOTD,szString.data());
		}
	}
	_debug_leavetrace("sendMotd");
}

//============ connectedSlot ============//

void KviChatWnd::connectedSlot()
{
	_debug_entertrace("connectedSlot");
	m_szRemoteIp=inet_ntoa(m_lpListeningSocket->connectedAddr.sin_addr);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Connected to %s"),m_szRemoteIp.data());
	m_lpReadNotifier=new QSocketNotifier(m_Sock,QSocketNotifier::Read,this);
	QObject::connect(m_lpReadNotifier, SIGNAL(activated(int)), this, SLOT(slotDataInBuffer(int)));
	m_bConnected=true;
	if(m_lpFrm->m_lpOpt->bShowMotdOnOutDcc)sendMotd();
	_debug_leavetrace("connectedSlot");
}

//============ connectTimedOut ============//

void KviChatWnd::connectTimedOut()
{
	_debug_entertrace("connectTimedOut");
	QString szStat;
	szStat.sprintf(i18n("DCC CHAT to %s timed out"),m_szRemoteNick.data());
	m_lpFrm->setStatusText(szStat.data());
	doFmtOutput(KVI_OUT_ERROR,i18n("DCC CHAT to %s timed out"),m_szRemoteNick.data());
	if(m_lpTimer){
		if(m_lpTimer->isActive())m_lpTimer->stop();
		delete m_lpTimer;
		m_lpTimer=0;
	}
	if(m_Sock != -1)::close(m_Sock);
	m_bConnected=false;
	_debug_leavetrace("connectTimedOut");
}

//============ slotDataInBuffer ============//

void KviChatWnd::slotDataInBuffer(int)
{
	_debug_entertrace("slotDataInBuffer");
	char szIncomingData[1025];
   	bzero(szIncomingData, 1025); //just a space for the terminator
	if(m_lpListeningSocket){
		delete m_lpListeningSocket;
		m_lpListeningSocket=0;
	}
	int readLength = read(m_Sock,szIncomingData, 1024);
	if (readLength <= 0){
		if(readLength==0){
			QString szStat;
			szStat.sprintf(i18n("DCC CHAT with %s terminated"),m_szRemoteNick.data());
			m_lpFrm->setStatusText(szStat.data());
			doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC CHAT with %s terminated (Remote end closed connection)"),m_szRemoteNick.data());
			killSock();
		} else {
			if((errno != EINTR)&&(errno != EAGAIN)){
				QString szStat;
				szStat.sprintf(i18n("DCC CHAT with %s terminated"),m_szRemoteNick.data());
				m_lpFrm->setStatusText(szStat.data());
				if(errno==111)doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC CHAT with %s failed : Connection refused (Remote end closed listening socket)"),
						m_szRemoteNick.data());
				else {
					QString szErr;
					m_lpFrm->m_lpSock->getErr(m_lpFrm->m_lpSock->translateFromSystemError(errno),szErr);
					doFmtOutput(KVI_OUT_INTERNAL,i18n("DCC CHAT with %s terminated by a transmission error (%s)"),m_szRemoteNick.data(),szErr.data());
				}
				killSock();
			}
		}
  	} else {
		QString szIncomingString(szIncomingData,readLength+1);
		szIncomingString.prepend(m_szLastIncompleteMessage);
		m_szLastIncompleteMessage = "";
		int lenToCopy;
		char *lpStrData;
		const char *d;
		int idx;
		//Split lines and remove the CRLF
		//all this gnary hack just to save a couple 
		//(but maybe 50...) of ms each time :)
		while (!szIncomingString.isEmpty()){
			//find '\n' in the string
			//subst: int idx= szIncomingString.find('\n');
			lpStrData=szIncomingString.data();
			d = strchr(lpStrData,'\n' );
			idx = ( d ? (int)(d-lpStrData) : -1 );	
			if (idx > -1){
				if(idx > 1){
					//Subst: szNewDataLine=szIncomingString.left(idx);
					//cut also the CR so (idx-1)=CR index
					//lets say that idx=10=(LF index) so CR index=9 and is the 10th char..
					int noCR=((lpStrData[idx-1]=='\r') ? 0 : 1);
					QString szNewDataLine(idx+noCR); //up to CR (excl.) and space for \0 so here 10 (9+1) chars
					strncpy(szNewDataLine.data(),lpStrData,idx-1+noCR); //copy 9 chars
					*(szNewDataLine.data()+(idx-1+noCR)) = '\0'; //add the 10th
					//Disable the notifier while processing the data...
					m_lpReadNotifier->setEnabled(false);
					if(parseMessage(szNewDataLine))m_lpReadNotifier->setEnabled(true);
					//else the notifier is already dead...
					//Subst:szincomingString.remove(0,idx+2);
					lenToCopy=szIncomingString.length()-idx; //lets say the string is long 100 so lenToCopy=90
					memmove(lpStrData,lpStrData+(idx+1),lenToCopy ); //start from the char after the LF and copy 90 (89+1)
					((QByteArray) (szIncomingString)).resize(lenToCopy); //and resize to 90 (89+ '\0')
				} else {
					//I noticed that we newer run here...but it is possible...
					szIncomingString.remove(0,idx+1); //Just a (CR)LF...
				}
			} else {
				m_szLastIncompleteMessage = szIncomingString.copy();
				szIncomingString = "";
			}
		}
	}
	_debug_leavetrace("slotDataInBuffer");
}

//============ parseMessage ============//

bool KviChatWnd::parseMessage(QString &szMessage)
{
	_debug_entertrace("parseMessage");
	if(m_bWaitForAck){
		//Was waiting for acknowledgement : 101 servernickname
		m_bWaitForAck=false;
		if(szMessage.left(3)=="101"){ //Kewl , acknowledged
			szMessage.remove(0,3);
			szMessage=szMessage.stripWhiteSpace();
			m_szRemoteNick=szMessage.data();
			m_bConnected=true;
			doOutput(KVI_OUT_INTERNAL,i18n("DCC Chat estabilished"));
			doFmtOutput(KVI_OUT_INTERNAL,i18n("Remote nick is %s"),m_szRemoteNick.data());
			if(m_lpFrm->m_lpOpt->bShowMotdOnOutDcc)sendMotd();
			return true;
		}
		if(szMessage.left(3)=="150"){
			doFmtOutput(KVI_OUT_ERROR,i18n("Got negative response : [%s] : Service unavailable"),szMessage.data());
			requestFailed(i18n("Remote end refused the connection"));
			killSock();
			return false;
		}
		if(szMessage.left(3)=="151"){
			doFmtOutput(KVI_OUT_ERROR,i18n("Got negative response : [%s] : Connection rejected"),szMessage.data());
			requestFailed(i18n("Remote end refused the connection"));
			killSock();
			return false;
		}
		doFmtOutput(KVI_OUT_ERROR,i18n("Got negative response : [%s]"),szMessage.data());
		requestFailed(i18n("Got an unrecognized reply from the daemon"));
		killSock();
		return false;
	}
	if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnDCCChatText]->bEnabled){
		QString szPara=m_szRemoteNick+" "+szMessage;
		m_lpFrm->m_lpUserParser->executeEvent(this,m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnDCCChatText],szPara);
	}
	if((*(szMessage.data()))==0x01){
		//ACTION ?
		int idx=szMessage.find("ACTION");
		if(idx != -1){
			szMessage.remove(0,(idx+7));
			szMessage.remove(szMessage.length()-1,1);
			doFmtOutput(KVI_OUT_ACTION,"%s %s",m_szRemoteNick.data(),szMessage.data());
			return true;
		}
	}
	QString szNicko;
	m_lpFrm->formatNickOutputString(szNicko,m_szRemoteNick);
	doFmtOutput(KVI_OUT_NONE,"%s%s",szNicko.data(),szMessage.data());
	_debug_leavetrace("parseMessage");
	return true;
}

//============ closeSlot ============//

void KviChatWnd::closeSlot()
{
	_debug_entertrace("closeSlot");
	killSock();
	hide();
	_macro_kviApplication->processEvents();
	blockSignals(true);
	m_lpMdi->m_lpTaskBar->removeButton(id());
	m_lpMdi->m_lpChildList->setAutoDelete(false);
	m_lpMdi->m_lpChildList->removeRef(this);
	m_lpMdi->focusTopChild();
	m_lpMdi->fillWinListPopup();
	m_lpFrm->queryExternalDestroy(this);
	_debug_leavetrace("closeSlot");
}

void KviChatWnd::applyOptions()
{
	m_lpInput->enableColorBox(m_lpFrm->m_lpOpt->bShowColorBox);
	m_lpInput->setDefaultBackgroundColor(m_lpInt->clr_bk_input);
	m_lpInput->setDefaultForegroundColor(m_lpInt->clr_fr_input);
	m_lpInput->setMarkBackgroundColor(m_lpInt->clr_bk_sel_input);
	m_lpInput->setMarkForegroundColor(m_lpInt->clr_fr_sel_input);
	m_lpInput->setCursorColor(m_lpInt->clr_cursor_input);
	m_lpInput->setFont(m_lpInt->fnt_input);
	m_lpOutput->m_bTimestamp=m_lpFrm->m_lpOpt->bTimestamp;
	m_lpOutput->m_bShowPixmaps=m_lpFrm->m_lpOpt->bShowPixmaps;

	if(!m_lpInt->pix_bk_input.isNull())m_lpInput->setBackgroundPixmap(m_lpInt->pix_bk_input);
	m_lpOutput->setFont(m_lpInt->fnt_output);
	const QSize sz=size();
	QResizeEvent e(sz,sz);
	resizeEvent(&e);
}

void KviChatWnd::resizeEvent(QResizeEvent *)
{
	_debug_entertrace("resizeEvent");
	updateRects();
	QRect rct=viewSizeHint();
	QFontMetrics fnt=m_lpInput->fontMetrics();
	int input_hght=fnt.lineSpacing()+8;
	m_lpInput->setGeometry(rct.left(),rct.top()+rct.height()-input_hght,rct.width(),input_hght);
	m_lpOutput->setGeometry(rct.left(),rct.top(),rct.width(),rct.height()-(input_hght+KVI_MDI_VIEW_SEP));
	_debug_leavetrace("resizeEvent");
}

void KviChatWnd::doFmtOutput(int nType,const char *szFmt,...)
{
	char szText[600]; //It should be big enough... I hope...
	va_list list;
	va_start(list,szFmt);
	if(vsnprintf(szText,600,szFmt,list)==-1)debug("WARNING : Output string truncated"); // Fritz: vsnprintf + realloc
	va_end(list);
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,(const char *) &szText);
}

void KviChatWnd::doOutput(int nType,const char *szText)
{
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,szText);
}

//============ sendData ============//

void KviChatWnd::sendData(const char *szData)
{
	_debug_entertrace("sendData");
	if(!m_bConnected)return;
	if(::write(m_Sock,szData,strlen(szData))>0)::write(m_Sock,"\r\n",2);
	_debug_leavetrace("sendData");
}


#include "m_kvi_chat.moc"
