//   $Id: kvi_cache.cpp,v 1.4 1998/09/20 20:22:06 fritz Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CLASS_NAME_ "KviCache"


#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_cache.h"
#include "kvi_debug.h"
#include "kvi_child.h"
#include "kvi_app.h"

#include <kmsgbox.h>

//============ KviCache ============//

KviCache::KviCache()
{
	_debug_entertrace("KviCache");
	m_lpCache=new QDict<KviSection>(29,false);
	m_lpCache->setAutoDelete(true);
	if(!readCache()){
		KMsgBox::message(0,i18n("File I/O Error"),i18n("Unable to read the variables cache to $HOME/kvirc/Config/kvi.cache.conf"));
	}
	_debug_leavetrace("KviCache");
}
//============ ~KviCache ============//
KviCache::~KviCache()
{
	_debug_entertrace("~KviCache");
	if(!writeCache()){
		KMsgBox::message(0,i18n("File I/O Error"),i18n("Unable to save the variables cache to $HOME/kvirc/Config/kvi.cahce.conf"));
	}
	clearCache();
	delete m_lpCache;
	_debug_leavetrace("~KviCache");
}
//============ clearCache ============//
void KviCache::clearCache()
{
	_debug_entertrace("clearCache");
	QDictIterator<KviSection> it(*m_lpCache);
	while(it.current()){
		KviSection *lpS=it.current();
		lpS->clear(); //delete all internal keys
		++it;
	}
	m_lpCache->clear(); //delete all (empty now) sections
	_debug_leavetrace("clearCache");
}
//============ echoToWindow ============//
void KviCache::echoToWindow(KviMdiChild *lpC)
{
	_debug_entertrace("echoToWindow");
	lpC->doOutput(KVI_OUT_INTERNAL,i18n("Cache file $HOME/kvirc/Config/kvi.cache.conf : memory dump"));
	lpC->doFmtOutput(KVI_OUT_INTERNAL,i18n("Cache contains %d sections"),m_lpCache->count());
	QDictIterator<KviSection> it(*m_lpCache);
	uint total=0;
	while(it.current()){
		KviSection *lpS=it.current();
		lpC->doFmtOutput(KVI_OUT_INTERNAL,i18n("# Dumping section [%s] : %u keys"),it.currentKey(),lpS->count());
		total+=lpS->count();
		QDictIterator<QString> i(*lpS);
		while(i.current()){
			lpC->doFmtOutput(KVI_OUT_INTERNAL,i18n("- Key :[%s] - Value :[%s]"),i.currentKey(),i.current()->data());
			++i;
		}
		++it;
	}
	lpC->doFmtOutput(KVI_OUT_INTERNAL,i18n("Cache dump terminated : total %u keys"),total);
	_debug_leavetrace("echoToWindow");
}
//============ insertKey ============//
void KviCache::insertKey(const char *section,const char *key,const char *value)
{
	_debug_entertrace("insertKey");
	KviSection *lpS=m_lpCache->find(section);
	if(!lpS){ //create it!
		lpS=new KviSection(29,false);
		lpS->setAutoDelete(true);
		m_lpCache->insert(section,lpS);
	}
	if(lpS->find(key))lpS->remove(key);
	QString *pVal=new QString(value);
	lpS->insert(key,pVal);
//	debug("inserting %s:%s",key,value);
	_debug_leavetrace("insertKey");
}
//============ getKey ============//
const char * KviCache::getKey(const char *section,const char *key)
{
	_debug_entertrace("getKey");
	KviSection *lpS=m_lpCache->find(section);
	if(!lpS)return 0;
	QString *pKey=lpS->find(key);
	if(pKey)return pKey->data();
	else return 0;
	_debug_leavetrace("getKey");
}
//============ removeKey ============//
bool KviCache::removeKey(const char *section,const char *key)
{
	_debug_entertrace("removeKey");
	KviSection *lpS=m_lpCache->find(section);
	if(!lpS)return false;
	bool bRet=lpS->remove(key);
	if(lpS->isEmpty())m_lpCache->remove(section);
	return bRet;
	_debug_leavetrace("removeKey");
}
//============ removeSection ============//
bool KviCache::removeSection(const char *section)
{
	_debug_entertrace("removeSection");
	KviSection *lpS=m_lpCache->find(section);
	if(!lpS)return false;
	lpS->clear();
	return m_lpCache->remove(section);
	_debug_leavetrace("removeSection");
}
//============ readSection ============//
bool KviCache::readSection(KviSection *kvisect,QDataStream &stream)
{
	_debug_entertrace("readSection");
	Q_UINT32 numItems;
	stream >> numItems;
	if(stream.device()->status() != IO_Ok)return false;
	QString szKey;
	QString szVal;
	for(Q_UINT32 i=0;i<numItems;i++){
		stream >> szKey;
		stream >> szVal;
		if(stream.device()->status() != IO_Ok)return false;
		if(!kvisect->find(szKey.data()))kvisect->insert(szKey.data(),new QString(szVal.data()));
	}
	_debug_leavetrace("readSection");
	return true;
}
//============ writeSection ============//
bool KviCache::writeSection(KviSection *kvisect,QDataStream &stream)
{
	_debug_entertrace("writeSection");
	Q_UINT32 numItems=kvisect->count();
	stream << numItems;
//	debug("wrote numItemZ");
	if(stream.device()->status() != IO_Ok)return false;
//	debug("ok");
	QDictIterator<QString> it(*kvisect);

	QString szKey;
	QString szVal;

	while(it.current()){
		szKey=it.currentKey();
		szVal=it.current()->data();
//		debug("writing key %s",szKey.data());
		stream << szKey;
		stream << szVal;
		if(stream.device()->status() != IO_Ok)return false;
		++it;
	}
//	debug("suxesso");
	_debug_leavetrace("writeSection");
	return true;
}
//============ readCache ============//
bool KviCache::readCache()
{
	_debug_entertrace("readCache");
	QString szFileName=_macro_getKVircHomeDirectory("Config/kvi.cache.conf");
	QFile f(szFileName.data());
	if(!f.open(IO_ReadOnly))return true; //no file....just skip and wait next time :)
	QDataStream stream(&f);

	Q_UINT32 numItems;
	stream >> numItems;
//	debug("num items %u",numItems);
	if(stream.device()->status() != IO_Ok){
//		debug("state fail!!!");
		f.close();
		return false;
	}
	QString szKey;
	for(Q_UINT32 i=0;i<numItems;i++){
		stream >> szKey;
//		debug("reading key %s",szKey.data());
		if(stream.device()->status() != IO_Ok){
			f.close();
//			debug("IO FAILLLLLLLL");
			return false;
		}
		KviSection *lpS=new KviSection(29,false);
		lpS->setAutoDelete(true);
//		debug("reading section");
		if(!readSection(lpS,stream)){
//			debug("can not rewad section");
			lpS->clear();
			delete lpS;
			f.close();
			return false;
		}
		if(m_lpCache->find(szKey.data())){
			removeSection(szKey.data());
		}
		m_lpCache->insert(szKey.data(),lpS);
	}
//	debug("success");
	f.close();
	_debug_leavetrace("readCache");
	return true;
}
//============ writeCache ============//
bool KviCache::writeCache()
{
	_debug_entertrace("writeCache");
	QString szFileName=_macro_getKVircHomeDirectory("Config/kvi.cache.conf");
	QFile f(szFileName.data());
	if(!f.open(IO_WriteOnly|IO_Truncate))return false;
	QDataStream stream(&f);

//	debug("file opened, stream going");
	Q_UINT32 numItems=m_lpCache->count();
	stream << numItems;
	if(stream.device()->status() != IO_Ok){
//		debug("IO STATE FAIL %d",stream.device()->status());
		f.close();
		return false;
	}
//	debug("written numItems..looping");
	QDictIterator<KviSection> it(*m_lpCache);
	while(it.current()){
		QString szKey=it.currentKey();
		stream << szKey;
//		debug("writing key %s",szKey.data());
		if(stream.device()->status() != IO_Ok){
			f.close();
//			debug("ooops  IO FAILED");
			return false;
		}
		if(!writeSection(it.current(),stream)){
//			debug("ops..could not write a section");
			f.close();
			return false;
		}
		++it;
	}
//	debug("success");
	f.close();
	_debug_leavetrace("writeCache");
	return true;
}

//
// $Log: kvi_cache.cpp,v $
// Revision 1.4  1998/09/20 20:22:06  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.3  1998/09/16 17:15:58  fritz
// Starting i18n.
//
// Revision 1.2  1998/09/16 16:13:08  pragma
// Moving to use a dynamic kvirc home directory.
// Big commit :)
//
//
