
/*

    File: rshproxy/rsh.c

    Copyright (C) 1999  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>

#include <signal.h>
#include <wait.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>

#include "rsh.h"
#include "ip-lib.h"
#include "lib.h"


static char *waitfor = "";
static int waitexit = 0;

static void alarm_handler(int sig)
{
	syslog(LOG_NOTICE, "timeout: %s", waitfor);
	if (waitexit != 0)
		exit (1);
	return;
}

int set_timeout(int secs, char *string, int we)
{
	if (secs == 0) {
		alarm(0);
		return (0);
		}
		
	waitexit = we;
	waitfor  = string;
	alarm(secs);
	signal(SIGALRM, alarm_handler);

	return (0);
}


unsigned int get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "can't get interface info: %m");
		exit (-1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}

unsigned int get_peer_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "can't get peer info: %m");
		exit (-1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);
	if (port == 0) {
		syslog(LOG_NOTICE, "invalid port info, port= 0");
		exit (-1);
		}

	return (port);
}


int get_client_info(rsh_t *x, int pfd)
{
	int	size;
	struct sockaddr_in saddr;
	struct in_addr *addr;
	struct hostent *hostp = NULL;

	*x->client = 0;
	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0)
		return (-1);
		
	copy_string(x->client_ip, (char *) inet_ntoa(saddr.sin_addr), sizeof(x->client_ip));
	if (numeric_only != 0)
		copy_string(x->client, x->client_ip, sizeof(x->client));
	else {
		addr = &saddr.sin_addr;
		hostp = gethostbyaddr((char *) addr,
				sizeof (saddr.sin_addr.s_addr), AF_INET);

		hostp = NULL;
		if (hostp == NULL) {
			strcpy(x->client, x->client_ip);
			return (0);
			}

		strcpy(x->client, hostp->h_name);
		}

	strlwr(x->client);
	return (0);
}




char *readline_fd(rsh_t *x, int fd, char *line, int size, int eol)
{
	unsigned int c;
	int	k;

	set_timeout(10, "input", 0);
	c = k = line[0] = 0;
	size -= 2;
	while (k < size) {
		if (read(fd, &c, sizeof(char)) != 1) {
			line[k] = 0;
			return (NULL);
			}
		else if (c == 0  ||  c == eol)
			break;

		line[k++] = c;
		}

	line[k] = 0;
	set_timeout(0, "", 0);
	
	return (line);
}

char *cfgets(rsh_t *x, char *line, int size)
{
	char	*p;

	*line = 0;
	if ((p = readline_fd(x, 0, line, size, 0)) == NULL)
		return (NULL);
	else if (debug != 0)
		fprintf (stderr, "CLI >>>: %s\n", p);

	return (line);
}

int cfputs(rsh_t *x, char *line)
{
	if (debug)
		fprintf (stderr, ">>> CLI: %s\n", line);
		
	write(1, line, strlen(line));
	write(1, "\r\n", 2);

	return (0);
}

int cfpute(rsh_t *x, char *line)
{
	write(1, "\001", 1);
	cfputs(x, line);
	return (0);
}


char *sfgets(rsh_t *x, char *line, int size)
{
	char *p;

	*line = 0;
	if ((p = readline_fd(x, x->server.fd, line, size, '\n')) == NULL)
		return (NULL);
	else if (debug != 0)
		fprintf (stderr, "SVR >>>: %s\n", p);

	return (line);
}

int sfputs(rsh_t *x, char *format, ...)
{
	char	buffer[300];
	va_list	ap;

	va_start(ap, format);
	vsnprintf (buffer, sizeof(buffer) - 2, format, ap);
	va_end(ap);

	if (debug)
		fprintf (stderr, ">>> SVR: %s\n", buffer);

	write(x->server.fd, buffer, strlen(buffer));
	write(x->server.fd, "\000", 1);

	return (0);
}


int set_variables(rsh_t *x)
{
	char	*vp, var[200], val[200];

	vp = x->config->varname;

	snprintf (var, sizeof(var) - 2, "%sINTERFACE", vp);
	setenv(var, x->interface, 1);
	
	snprintf (val, sizeof(val) - 2, "%u", x->port);
	snprintf (var, sizeof(var) - 2, "%sPORT", vp);
	setenv(var, val, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENT", vp);
	setenv(var, x->client_ip, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENTNAME", vp);
	setenv(var, x->client, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENTLOGIN", vp);
	setenv(var, x->data.client_uname, 1);
	
	snprintf (var, sizeof(var) - 2, "%sSERVER", vp);
	setenv(var, x->server.ipnum, 1);
	
	snprintf (val, sizeof(val) - 2, "%u", x->server.port);
	snprintf (var, sizeof(var) - 2, "%sSERVERPORT", vp);
	setenv(var, val, 1);

	snprintf (var, sizeof(var) - 2, "%sSERVERNAME", vp);
	setenv(var, x->server.name, 1);
	
	snprintf (var, sizeof(var) - 2, "%sSERVERLOGIN", vp);
	setenv(var, x->data.username, 1);
	
	snprintf (var, sizeof(var) - 2, "%sSERVERCMD", vp);
	setenv(var, x->data.command, 1);
	
	snprintf (var, sizeof(var) - 2, "%sUSERNAME", vp);
	setenv(var, x->local.username, 1);
	
	snprintf (var, sizeof(var) - 2, "%sPASSWD", vp);
	setenv(var, x->local.password, 1);
	
	return (0);
}

int run_acp(rsh_t *x)
{
	int	rc, pid, pfd[2];
	char	line[300];
	
	if (*x->config->acp == 0)
		return (0);

	rc = 0;
	if (pipe(pfd) != 0) {
		syslog(LOG_NOTICE, "can't pipe: %m");
		exit (1);
		}
	else if ((pid = fork()) < 0) {
		syslog(LOG_NOTICE, "can't fork acp: %m");
		exit (1);
		}
	else if (pid == 0) {
		int	argc;
		char	*argv[32];

		close(0);		/* Das acp kann nicht vom client lesen. */
		dup2(pfd[1], 2);	/* stderr wird vom parent gelesen. */
		close(pfd[0]);
		set_variables(x);
		
		copy_string(line, x->config->acp, sizeof(line));
		argc = split(line, argv, ' ', 30);
		argv[argc] = NULL;
		execvp(argv[0], argv);

		syslog(LOG_NOTICE, "can't exec acp %s: %m", argv[0]);
		exit (1);
		}
	else {
		int	len;
		char	message[300];

		close(pfd[1]);
		*message = 0;
		if ((len = read(pfd[0], message, sizeof(message) - 2)) < 0)
			len = 0;

		message[len] = 0;
		noctrl(message);
		

		if (waitpid(pid, &rc, 0) < 0) {
			syslog(LOG_NOTICE, "error while waiting for acp: %m");
			exit (1);
			}

		rc = WIFEXITED(rc) != 0? WEXITSTATUS(rc): 1;
		if (*message == 0)
			copy_string(message, rc == 0? "access granted": "access denied", sizeof(message));

		if (*message != 0)
			syslog(LOG_NOTICE, "%s (rc= %d)", message, rc);
		}
		
	return (rc);
}

int apply_vnchack(rsh_t *x, int baseport, char *prglist, int type)
{
	int	i, display;
	char	prg[80], word[80];
	char	*p, *r, buffer[RSH_CMDLEN + 2];

	copy_string(buffer, x->data.command, sizeof(buffer));
	p = buffer;
	get_word(&p, word, sizeof(word));
	
	r = prglist;
	while (*get_quoted(&r, ',', prg, sizeof(prg)) != 0) {
		if (strcmp(word, prg) == 0)
			break;
		}

	if (*prg == 0)
		return (0);	/* Programm nicht auf VNC-Starterliste */

	if ((r = strchr(p, ':')) == NULL)
		return (0);	/* keine display Angabe */

	r++;
	display = atoi(r);
	if (display < 0  ||  display > 20)
		return (0);	/* display nicht im Bereich */

	x->extra.server.port = baseport + display;
	copy_string(x->extra.server.ipnum, x->server.ipnum, sizeof(x->extra.server.ipnum));

	/*
	 * Server Port fuer Client allokieren.
	 */

	for (i=10; i<100; i++) {
		x->extra.client.sock = bind_to_port(x->interface, baseport + i);
		if (x->extra.client.sock > 0)
			break;
		}

	if (x->extra.client.sock < 0) {
		syslog(LOG_NOTICE, "can't get VNC port");
		exit (1);
		}

	x->extra.type  = type;
	x->extra.mode  = RSH_CLIENTFWD;
	x->extra.state = RSH_LISTEN;

	copy_string(x->extra.client.ipnum, x->data.client, sizeof(x->extra.client.ipnum));
	x->extra.client.port = baseport + i;

	return (1);
}

int apply_xhack(rsh_t *x)
{
	int	i;
	unsigned int c;
	char	word[80], prg[80], display[80];
	unsigned char *p, *r, *s, *q, buffer[RSH_CMDLEN + 2];

	copy_string(buffer, x->data.command, sizeof(buffer));
	p = buffer;
	get_word((char **) &p, word, sizeof(word));

	r = x->config->xprg;
	if (*r != 0) {
		while (*get_quoted((char **) &r, ',', prg, sizeof(prg)) != 0) {
			if (strcmp(word, prg) == 0)
				break;
			}

		if (*prg == 0)
			return (0);	/* Programm nicht auf Liste */
		}


	while ((c = *p) != 0) {
		p = skip_ws(p);
		if ((c = *p) == '"'  ||  c == '\''  ||  c == '`')
			break;

		if (*p == '-'  &&  strncmp(p, "-display", 8) == 0) {
			if (p[8] > 0  &&  p[8] <= ' ') {
				r = s = skip_ws(p + 8);
				get_word((char **) &s, display, sizeof(display));
				s = skip_ws(s);

				if ((q = strchr(display, ':')) == NULL)
					goto cont;

				x->extra.client.port = 6000 + atoi(&q[1]);
				copy_string(x->extra.client.ipnum, x->data.client, sizeof(x->extra.client.ipnum));

				for (i=10; i<1000; i++) {
					x->extra.server.sock = bind_to_port(x->server.localinterface, 6000 + i);
					if (x->extra.server.sock > 0)
						break;
					}

				if (x->extra.server.sock < 0) {
					syslog(LOG_NOTICE, "can't get X port");
					exit (1);
					}

				x->extra.type  = RSH_XWIN;
				x->extra.mode  = RSH_SERVERFWD;
				x->extra.state = RSH_LISTEN;

				x->extra.server.port = 6000 + i;
				copy_string(x->extra.server.ipnum, x->server.ipnum, sizeof(x->extra.server.ipnum));

				*r = 0;
				snprintf (x->data.command, sizeof(x->data.command) - 2, "%s%s:%d %s",
						buffer, x->server.localinterface, x->extra.server.port - 6000, s);
				if (debug) {
					fprintf(stderr, "proxy display is %s:%d\n", x->server.localinterface, x->extra.server.port - 6000);
					fprintf(stderr, "command= %s\n", x->data.command);
					}

				return (1);
				}
			}

cont:
		while ((c = *p) != 0  &&  c > ' ') {
			if ((c = *p) == '"'  ||  c == '\''  ||  c == '`')
				break;

			p++;
			}
		}
	
	return (0);
}

int doconnect(rsh_t *x)
{
	char	*p, line[300];
	struct hostent *hostp;
	struct sockaddr_in saddr;

	x->data.port = get_peer_info(0, x->data.client, sizeof(x->data.client));
	if (x->port == PORT_REXEC)
		/* nichts */ ;
	else {
		if (x->data.port < 512  ||  x->data.port >= IPPORT_RESERVED) {
			cfpute(x, "permission denied");
			syslog(LOG_NOTICE, "invalid port %u: %s", x->data.port, x->data.client);
			exit (1);
			}
		}
		
	if (cfgets(x, line, sizeof(line)) == NULL) {
		syslog(LOG_NOTICE, "lost client (port): %s", x->client_ip);
		exit (1);
		}

	x->data.stderrport = atoi(line);
	if (x->data.stderrport > 0) {
		x->data.stderrsock = openip(x->data.client, x->data.stderrport,
					(x->port == PORT_REXEC)? 0: 1);
		if (x->data.stderrsock < 0) {
			syslog(LOG_NOTICE, "can't open stderr to client, port %u: %x", x->data.stderrport, x->data.client);
			exit (1);
			}
		}


	if (x->port == PORT_REXEC) {
		if (cfgets(x, x->data.username, sizeof(x->data.username)) == NULL) {
			syslog(LOG_NOTICE, "lost client (server username): %s", x->client_ip);
			exit (1);
			}

		if (cfgets(x, x->data.password, sizeof(x->data.password)) == NULL) {
			syslog(LOG_NOTICE, "lost client (server password): %s", x->client_ip);
			exit (1);
			}

		strcpy(x->data.client_uname, "unknown-user");
		}
	else {
		if (cfgets(x, x->data.client_uname, sizeof(x->data.client_uname)) == NULL) {
			syslog(LOG_NOTICE, "lost client (client username): %s", x->client_ip);
			exit (1);
			}

		if (cfgets(x, x->data.username, sizeof(x->data.username)) == NULL) {
			syslog(LOG_NOTICE, "lost client (server username): %s", x->client_ip);
			exit (1);
			}

		*x->data.password = 0;
		}


	if (x->port == PORT_RLOGIN)
		strcpy(x->data.command, "login");
	else {
		if (cfgets(x, x->data.command, sizeof(x->data.command)) == NULL) {
			syslog(LOG_NOTICE, "lost client (command): %s", x->client_ip);
			exit (1);
			}
		}

	syslog(LOG_NOTICE, "request: client= %s@%s:%u, user= %s, cmd= %s", x->data.client_uname, x->data.client, x->data.port, x->data.username, x->data.command);
	
	if (x->config->selectserver == 0) {
		if ((p = strchr(x->data.username, '@')) != NULL  &&  (p = strchr(x->data.username, '%')) != NULL) {
			cfpute(x, "service unavailable");
			syslog(LOG_NOTICE, "hostname supplied: %s", p);
			exit (1);
			}

		copy_string(x->server.name, x->config->u.server, sizeof(x->server.name));
		}
	else {

		/*
		 * Es wird das erste Vorkommen des @-Zeichens gesucht, nicht das
		 * letzte, da sonst Proxy-Routing durch den Client ermoeglicht
		 * wird.
		 */

		if ((p = strchr(x->data.username, '@')) == NULL  &&  (p = strchr(x->data.username, '%')) == NULL) {
			cfpute(x, "service unavailable");
			syslog(LOG_NOTICE, "missing hostname");
			exit (1);
			}

		*p++ = 0;
		copy_string(x->server.name, p, sizeof(x->server.name));

		/*
		 * Den Server auf der Serverliste suchen, wenn eine Liste
		 * vorhanden ist.
		 */

		if ((p = x->config->u.serverlist) != NULL  &&  *p != 0) {
			int	permitted;
			char	pattern[80];

			permitted = 0;
			while ((p = skip_ws(p)), *get_quoted(&p, ',', pattern, sizeof(pattern)) != 0) {
				noctrl(pattern);
				if (strpcmp(x->server.name, pattern) == 0) {
					permitted = 1;
					break;
					}
				}

			if (permitted == 0) {
				cfpute(x, "service unavailable");
				syslog(LOG_NOTICE, "hostname not permitted: %s", x->server.name);
				exit (1);
				}
			}
		}




	/*
	 * Port und IP-Nummer des Servers holen.
	 */

	x->server.port = get_port(x->server.name, x->port /* 514 */);
	if ((hostp = gethostbyname(x->server.name)) == NULL) {
		cfpute(x, "service unavailable");
		syslog(LOG_NOTICE, "can't resolve hostname: %s", x->server.name);
		exit (1);
		}

	memcpy(&saddr.sin_addr, hostp->h_addr, hostp->h_length);
	copy_string(x->server.ipnum, inet_ntoa(saddr.sin_addr), sizeof(x->server.ipnum));




	/*
	 * Wenn vorhanden Proxy Login und Passwort auslesen.
	 */

	if ((p = strchr(x->data.username, ':')) != NULL) {
		*p++ = 0;
		copy_string(x->local.username, x->data.username, sizeof(x->local.username));
		copy_string(x->data.username, p, sizeof(x->data.username));
		}

	if ((p = strchr(x->data.password, ':')) != NULL) {
		*p++ = 0;
		copy_string(x->local.password, x->data.password, sizeof(x->local.password));
		copy_string(x->data.password, p, sizeof(x->data.password));
		}

	/*
	 * Wenn kein Proxy Benutzername vorhanden ist wird der Server Login
	 * verwendet.
	 */

	if (*x->local.username == 0)
		copy_string(x->local.username, x->data.username, sizeof(x->local.username));


	/*
	 * Access Control Programm starten
	 */

	if (*x->config->acp != 0) {
		if (run_acp(x) != 0)
			exit (0);
		}




	/*
	 * Verbindung zum Server aufbauen.
	 */

	x->server.fd = openip(x->server.name, x->server.port, (x->port == PORT_REXEC)? 0: 1);
	if (x->server.fd < 0) {
		cfpute(x, "service unavailable");
		syslog(LOG_NOTICE, "can't connect to server: %s", x->server.name);
		exit (1);
		}

	get_interface_info(x->server.fd, x->server.localinterface, sizeof(x->server.localinterface));
	syslog(LOG_NOTICE, "connected to server: %s", x->server.name);


	/*
	 * X-Hack bearbeiten.
	 */

	if (x->port != PORT_RLOGIN) {
		if (x->config->vnchack != 0  &&  apply_vnchack(x, 5900, x->config->vncprg, RSH_VNC) != 0) {
			if (debug != 0)
				fprintf (stderr, "vnc hack ok.\n");
			}
		else if (x->config->xhack != 0  &&  apply_xhack(x) != 0) {
			if (debug != 0)
				fprintf (stderr, "xhack complete\n");
			}
		}
		

	/*
	 * stderr Verbindung aufbauen.
	 */

	if (x->data.stderrport == 0)
		sfputs(x, "");
	else {
		int	sock, adrlen, port;
		char	interface[80];

		x->server.stderrsock = bind_to_port(x->server.localinterface, -1);
		if (x->server.stderrsock < 0) {
			syslog(LOG_NOTICE, "can't allocate stderr socket: %s", x->client_ip);
			exit (1);
			}

		x->server.stderrport = get_interface_info(x->server.stderrsock, interface, sizeof(interface) - 2);
		sfputs(x, "%d", x->server.stderrport);

		set_timeout(10, "stderr accept", 1);
		adrlen = sizeof(struct sockaddr);
		sock = accept(x->server.stderrsock, (struct sockaddr *) &saddr, &adrlen);
		set_timeout(0, "", 0);

		if (sock < 0) {
			syslog(LOG_NOTICE, "accept() error: %m");
			exit (1);
			}

		port = get_peer_info(sock, interface, sizeof(interface));
		if (strcmp(interface, x->server.ipnum) != 0) {
			syslog(LOG_NOTICE, "wrong client on stderr, expected %s, got %s", x->server.ipnum, interface);
			exit (1);
			}
		else if (x->port != PORT_REXEC  &&  port >= IPPORT_RESERVED) {
			syslog(LOG_NOTICE, "invalid peer on stderr: %s:%u", interface, port);
			exit (1);
			}

		close(x->server.stderrsock);
		x->server.stderrsock = sock;
		syslog(LOG_NOTICE, "stderr connected to %s:%u", interface, port);
		}

	if (x->port == PORT_REXEC) {
		sfputs(x, x->data.username);
		sfputs(x, x->data.password);
		}
	else {
		sfputs(x, x->data.client_uname);
		sfputs(x, x->data.username);
		}

	if (x->port == PORT_RLOGIN)
		/* nichts */ ;
	else
		sfputs(x, x->data.command);

	return (0);
}


int add_fd(fd_set *set, int fd, int max)
{
	FD_SET(fd, set);
	if (fd > max)
		max = fd;

	return (max);
}
	
int proxy_request(config_t *config)
{
	int	max, rc, bytes;
	unsigned long started, now, bytes_send, bytes_received;
	char	buffer[4096];
	struct timeval tov;
	fd_set	connection, available;
	rsh_t	*x;

	started = time(NULL); 
	x = allocate(sizeof(rsh_t));
	x->config = config;

	if (get_client_info(x, 0) < 0) {
		syslog(LOG_NOTICE, "can't get client info: %m");
		exit (1);
		}

	x->port = get_interface_info(0, x->interface, sizeof(x->interface));
	syslog(LOG_NOTICE, "port %u connected to client: %s", x->port, x->client);

	if (doconnect(x) != 0)
		return (1);


	if (x->port == PORT_RLOGIN)
		/* nichts */ ;
	else {
		set_timeout(10, "server response", 1);
		rc = 0;
		if (read(x->server.fd, &rc, 1) != 1) {
			syslog(LOG_NOTICE, "status: server error, no response");
			exit (1);
			}
		else if (rc == 0) {
			syslog(LOG_NOTICE, "status: connection accepted");
			write(1, &rc, 1);

			if (x->extra.type == RSH_VNC) {
				char	line[80];
				
				snprintf (line, sizeof(line) - 2, "Proxy display is %d (port %d) on %s.", x->extra.client.port - 5900, x->extra.client.port, x->interface);
				cfputs(x, line);
				}
			}
		else if (rc == 1) {
			char	line[200];

			sfgets(x, line, sizeof(line));
			syslog(LOG_NOTICE, "status: connection rejected: %s", line);
			cfpute(x, line);
			}
		else {
			syslog(LOG_NOTICE, "status: unexpected server response, rc= %02X", rc);
			exit (1);
			}

		set_timeout(0, "", 0);
		}


	FD_ZERO(&connection);
	max = add_fd(&connection, 0, 0);
	max = add_fd(&connection, x->server.fd, max);
	if (x->server.stderrsock > 0)
		max = add_fd(&connection, x->server.stderrsock, max);

	if (x->extra.type != 0) {
		if (x->extra.mode == RSH_SERVERFWD)
			max = add_fd(&connection, x->extra.server.sock, max);
		else if (x->extra.mode == RSH_CLIENTFWD)
			max = add_fd(&connection, x->extra.client.sock, max);
		else {
			syslog(LOG_NOTICE, "program error: rsh forwarding");
			exit (1);
			}
		}

	if (x->extra.type == 0)
		x->config->wait_extra = 0;

	bytes_send = bytes_received = 0;
	while (1) {
		memmove(&available, &connection, sizeof(fd_set));

		if ((x->rshstatus & RSH_STDOUTCLOSED)  &&  x->config->wait_extra > 0  &&
		    x->extra.state == RSH_LISTEN) {

			/*
			 * Die eigentlich rsh Verbindung ist schon geschlossen,
			 * der zusaetzliche Kanal aber noch nicht aufgebaut.
			 * Wir geben Client und Server 60 Sekunden um das zu
			 * erledigen.
			 */

			if (debug) {
				if ((x->rshstatus & RSH_CLOSED) == RSH_CLOSED)
					fprintf (stderr, "waiting %d seconds ...\n", x->config->wait_extra);
				}

			tov.tv_sec  = x->config->wait_extra;
			tov.tv_usec = 0;
			}
		else {
			tov.tv_sec  = x->config->timeout;
			tov.tv_usec = 0;
			}

		rc = select(max + 1, &available, (fd_set *) NULL, (fd_set *) NULL, &tov);
		if (rc < 0) {
			syslog(LOG_NOTICE, "select() error: %m\n");
			break;
			}
		else if (rc == 0) {
			syslog(LOG_NOTICE, "connection timed out: client= %s, server= %s:%u",
				x->client, x->server.name, x->server.port);
			break;
			}

		if (FD_ISSET(0, &available)) {
			if ((bytes = read(0, buffer, sizeof(buffer) - 2)) <= 0) {
				if (debug)
					fprintf (stderr, "stdin closed\n");

				shutdown(x->server.fd, 1);
				FD_CLR(0, &connection);

				x->rshstatus |= RSH_STDINCLOSED;
				}
			else if (write(x->server.fd, buffer, bytes) != bytes)
				break;
				
			bytes_send = bytes_send + bytes;
			}
			
		if (FD_ISSET(x->server.fd, &available)) {
			if ((bytes = read(x->server.fd, buffer, sizeof(buffer) - 2)) <= 0) {
				if (x->extra.mode != RSH_NOEXTRA  &&  x->config->wait_extra > 0) {
					if (debug)
						fprintf (stderr, "stdout closed\n");

					shutdown(x->server.fd, 0);
					shutdown(1, 1);
					FD_CLR(x->server.fd, &connection);
					
					x->rshstatus |= RSH_STDOUTCLOSED;
					}
				else
					break;
				}
			else if (write(1, buffer, bytes) != bytes)
				break;
				
			bytes_received = bytes_received + bytes;
			}

		if (x->server.stderrsock > 0) {
			if (FD_ISSET(x->server.stderrsock, &available)) {
				if ((bytes = read(x->server.stderrsock, buffer, sizeof(buffer) - 2)) <= 0) {
					if (x->extra.mode != RSH_NOEXTRA  &&  x->config->wait_extra > 0) {
						if (debug)
							fprintf (stderr, "stderr closed\n");

						shutdown(x->server.stderrsock, 0);
						shutdown(x->data.stderrsock, 1);
						FD_CLR(x->server.stderrsock, &connection);

						x->rshstatus |= RSH_STDERRCLOSED;
						}
					else
						break;
					}
				else if (write(x->data.stderrsock, buffer, bytes) != bytes)
					break;

				bytes_received = bytes_received + bytes;
				}
			}

		if (x->extra.type != 0) {
			if (x->extra.state == RSH_LISTEN) {
				int	sock, adrlen;
				char	peerip[80];
				struct sockaddr_in saddr;
				
				if (x->extra.mode == RSH_SERVERFWD  &&  FD_ISSET(x->extra.server.sock, &available)) {
					if (debug)
						fprintf (stderr, "server connected\n");

					adrlen = sizeof(struct sockaddr);
					sock = accept(x->extra.server.sock, (struct sockaddr *) &saddr, &adrlen);
					close(x->extra.server.sock);
					x->extra.server.sock = sock;

					get_peer_info(sock, peerip, sizeof(peerip));
					if (strcmp(x->extra.server.ipnum, peerip) != 0) {
						syslog(LOG_NOTICE, "connection from wrong server, expected %s, got %s", x->extra.server.ipnum, peerip);
						exit (1);
						}
						
					if ((x->extra.client.sock = openip(x->data.client, x->extra.client.port, 0)) < 0) {
						syslog(LOG_NOTICE, "can't connect to X client: %s:%u", x->data.client, x->extra.client.port);
						exit (1);
						}
					}
				else if (x->extra.mode == RSH_CLIENTFWD  &&  FD_ISSET(x->extra.client.sock, &available)) {
					if (debug)
						fprintf (stderr, "client connected\n");

					adrlen = sizeof(struct sockaddr);
					sock = accept(x->extra.client.sock, (struct sockaddr *) &saddr, &adrlen);
					close(x->extra.client.sock);
					x->extra.client.sock = sock;

					get_peer_info(sock, peerip, sizeof(peerip));
					if (strcmp(x->extra.client.ipnum, peerip) != 0) {
						syslog(LOG_NOTICE, "connection from wrong client, expected %s, got %s", x->extra.client.ipnum, peerip);
						exit (1);
						}
						
					if ((x->extra.server.sock = openip(x->server.ipnum, x->extra.server.port, 0)) < 0) {
						syslog(LOG_NOTICE, "can't connect to VNC server: %s:%u", x->server.ipnum, x->extra.server.port);
						exit (1);
						}
					}
				else
					continue;

				FD_ZERO(&connection);
				max = add_fd(&connection, 0, 0);
				max = add_fd(&connection, x->server.fd, max);
				if (x->server.stderrsock > 0)
					max = add_fd(&connection, x->server.stderrsock, max);

				max = add_fd(&connection, x->extra.server.sock, max);
				max = add_fd(&connection, x->extra.client.sock, max);

				x->extra.state = RSH_CONNECTED;
				}
			else if (x->extra.state == RSH_CONNECTED) {
				if (FD_ISSET(x->extra.server.sock, &available)) {
					if ((bytes = read(x->extra.server.sock, buffer, sizeof(buffer) - 2)) <= 0)
						break;
					else if (write(x->extra.client.sock, buffer, bytes) != bytes)
						break;
						
					bytes_received = bytes_received + bytes;
					}

				if (FD_ISSET(x->extra.client.sock, &available)) {
					if ((bytes = read(x->extra.client.sock, buffer, sizeof(buffer) - 2)) <= 0)
						break;
					else if (write(x->extra.server.sock, buffer, bytes) != bytes)
						break;
						
					bytes_send = bytes_send + bytes;
					}
				}
			}
		}

	time((time_t *) &now); 
	syslog(LOG_NOTICE, "disconnect: client= %s, server= %s:%u, received= %ld, send= %ld, duration= %ld",
		x->client, x->server.name, x->server.port,
		bytes_received, bytes_send,
		now - started);
		
	return (0);
}

