
/*

    File: rshproxy/rexecd.c

    Copyright (C) 1999,2000  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>

#include <signal.h>
#include <wait.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>

#include <pwd.h>
#include <shadow.h>

#include "rexecd.h"
#include "ip-lib.h"
#include "lib.h"


char	*program =		"";
char	progname[80] =		"";

int	debug =			0;
char	numeric_only =		0;
char	*logname =		"";


static char *waitfor = "";
static int waitexit = 0;

static void alarm_handler(int sig)
{
	syslog(LOG_NOTICE, "timeout: %s", waitfor);
	if (waitexit != 0)
		exit (1);
	return;
}

int set_timeout(int secs, char *string, int we)
{
	if (secs == 0) {
		alarm(0);
		return (0);
		}
		
	waitexit = we;
	waitfor  = string;
	alarm(secs);
	signal(SIGALRM, alarm_handler);

	return (0);
}


unsigned int get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "can't get interface info: %m");
		exit (-1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}

unsigned int get_peer_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "can't get peer info: %m");
		exit (-1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);
	if (port == 0) {
		syslog(LOG_NOTICE, "invalid port info, port= 0");
		exit (-1);
		}

	return (port);
}


int get_client_info(rexec_t *x, int pfd)
{
	int	size;
	struct sockaddr_in saddr;
	struct in_addr *addr;
	struct hostent *hostp = NULL;

	*x->client = 0;
	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0)
		return (-1);
		
	copy_string(x->client_ip, (char *) inet_ntoa(saddr.sin_addr), sizeof(x->client_ip));
	addr = &saddr.sin_addr;
	
	if (numeric_only != 0)
		copy_string(x->client, x->client_ip, sizeof(x->client));
	else {
		hostp = gethostbyaddr((char *) addr,
				sizeof (saddr.sin_addr.s_addr), AF_INET);

		hostp = NULL;
		if (hostp == NULL) {
			strcpy(x->client, x->client_ip);
			return (0);
			}

		strcpy(x->client, hostp->h_name);
		}

	strlwr(x->client);
	return (0);
}




char *readline_fd(rexec_t *x, int fd, char *line, int size, int eol)
{
	unsigned int c;
	int	k;

	set_timeout(10, "input", 0);
	c = k = line[0] = 0;
	size -= 2;
	while (k < size) {
		if (read(fd, &c, sizeof(char)) != 1) {
			line[k] = 0;
			return (NULL);
			}
		else if (c == 0  ||  c == eol)
			break;

		line[k++] = c;
		}

	line[k] = 0;
	set_timeout(0, "", 0);
	
	return (line);
}

char *cfgets(rexec_t *x, char *line, int size)
{
	char	*p;

	*line = 0;
	if ((p = readline_fd(x, 0, line, size, 0)) == NULL)
		return (NULL);
	else if (debug != 0)
		fprintf (stderr, "CLI >>>: %s\n", p);

	return (line);
}

int cfputs(rexec_t *x, char *line)
{
	if (debug)
		fprintf (stderr, ">>> CLI: %s\n", line);
		
	write(1, line, strlen(line));
	write(1, "\r\n", 2);

	return (0);
}

int cfpute(rexec_t *x, char *line)
{
	write(1, "\001", 1);
	cfputs(x, line);
	return (0);
}



int set_variables(rexec_t *x)
{
	char	*vp, var[200], val[200];

	vp = x->config->varname;

	snprintf (var, sizeof(var) - 2, "%sINTERFACE", vp);
	setenv(var, x->interface, 1);
	
	snprintf (val, sizeof(val) - 2, "%u", x->port);
	snprintf (var, sizeof(var) - 2, "%sPORT", vp);
	setenv(var, val, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENT", vp);
	setenv(var, x->client_ip, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENTNAME", vp);
	setenv(var, x->client, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENTLOGIN", vp);
	setenv(var, x->data.client_uname, 1);
	
	snprintf (var, sizeof(var) - 2, "%sCMD", vp);
	setenv(var, x->data.command, 1);
	
	return (0);
}

int run_acp(rexec_t *x)
{
	int	rc, pid, pfd[2];
	char	line[300];
	
	if (*x->config->acp == 0)
		return (0);

	rc = 0;
	if (pipe(pfd) != 0) {
		syslog(LOG_NOTICE, "can't pipe: %m");
		exit (1);
		}
	else if ((pid = fork()) < 0) {
		syslog(LOG_NOTICE, "can't fork acp: %m");
		exit (1);
		}
	else if (pid == 0) {
		int	argc;
		char	*argv[32];

		close(0);		/* Das acp kann nicht vom client lesen. */
		dup2(pfd[1], 2);	/* stderr wird vom parent gelesen. */
		close(pfd[0]);
		set_variables(x);
		
		copy_string(line, x->config->acp, sizeof(line));
		argc = split(line, argv, ' ', 30);
		argv[argc] = NULL;
		execvp(argv[0], argv);

		syslog(LOG_NOTICE, "can't exec acp %s: %m", argv[0]);
		exit (1);
		}
	else {
		int	len;
		char	message[300];

		close(pfd[1]);
		*message = 0;
		if ((len = read(pfd[0], message, sizeof(message) - 2)) < 0)
			len = 0;

		message[len] = 0;
		noctrl(message);
		

		if (waitpid(pid, &rc, 0) < 0) {
			syslog(LOG_NOTICE, "error while waiting for acp: %m");
			exit (1);
			}

		rc = WIFEXITED(rc) != 0? WEXITSTATUS(rc): 1;
		if (*message == 0)
			copy_string(message, rc == 0? "access granted": "access denied", sizeof(message));

		if (*message != 0)
			syslog(LOG_NOTICE, "%s (rc= %d)", message, rc);
		}
		
	return (rc);
}


int doinit(rexec_t *x)
{
	char	line[300];

	x->data.port = get_peer_info(0, x->data.client, sizeof(x->data.client));
	if (x->port == PORT_REXEC)
		/* nichts */ ;
	else {
		if (x->data.port < 512  ||  x->data.port >= IPPORT_RESERVED) {
			cfpute(x, "permission denied");
			syslog(LOG_NOTICE, "invalid port %u: %s", x->data.port, x->data.client);
			exit (1);
			}
		}
		
	if (cfgets(x, line, sizeof(line)) == NULL) {
		syslog(LOG_NOTICE, "lost client (port): %s", x->client_ip);
		exit (1);
		}

	x->data.stderrport = atoi(line);
	if (x->data.stderrport > 0) {
		x->data.stderrsock = openip(x->data.client, x->data.stderrport,
					(x->port == PORT_REXEC)? 0: 1);
		if (x->data.stderrsock < 0) {
			syslog(LOG_NOTICE, "can't open stderr to client, port %u: %x", x->data.stderrport, x->data.client);
			exit (1);
			}

		dup2(x->data.stderrsock, 2);
		}


	if (x->port == PORT_REXEC) {
		if (cfgets(x, x->data.username, sizeof(x->data.username)) == NULL) {
			syslog(LOG_NOTICE, "lost client (server username): %s", x->client_ip);
			exit (1);
			}

		if (cfgets(x, x->data.password, sizeof(x->data.password)) == NULL) {
			syslog(LOG_NOTICE, "lost client (server password): %s", x->client_ip);
			exit (1);
			}

		strcpy(x->data.client_uname, "unknown-user");
		}
	else {
		if (cfgets(x, x->data.client_uname, sizeof(x->data.client_uname)) == NULL) {
			syslog(LOG_NOTICE, "lost client (client username): %s", x->client_ip);
			exit (1);
			}

		if (cfgets(x, x->data.username, sizeof(x->data.username)) == NULL) {
			syslog(LOG_NOTICE, "lost client (server username): %s", x->client_ip);
			exit (1);
			}

		*x->data.password = 0;
		}


	if (x->port == PORT_RLOGIN)
		strcpy(x->data.command, "login");
	else {
		if (cfgets(x, x->data.command, sizeof(x->data.command)) == NULL) {
			syslog(LOG_NOTICE, "lost client (command): %s", x->client_ip);
			exit (1);
			}
		}

	syslog(LOG_NOTICE, "request: client= %s@%s:%u, user= %s, cmd= %s", x->data.client_uname, x->data.client, x->data.port, x->data.username, x->data.command);
	


	/*
	 * Access Control Programm starten
	 */

	if (*x->config->acp != 0) {
		if (run_acp(x) != 0)
			exit (0);
		}


	/*
	 * Verbindung ist soweit vorbereitet, wir sind hier fertig.
	 */

	return (0);
}

int dologin(rexec_t *x)
{
	char	salt[8], *crypted, *cryptpw;
	struct spwd *spw;
	struct passwd *pw;

	cryptpw = "";
	if ((spw = getspnam(x->data.username)) != NULL)
		cryptpw = spw->sp_pwdp;
	else if ((pw = getpwnam(x->data.username)) != NULL)
		cryptpw = pw->pw_passwd;
	else {
		syslog(LOG_NOTICE, "login request for unkown user: %s", x->data.username);
		cfpute(x, "access denied");
		exit (1);
		}

	salt[0] = cryptpw[0];
	salt[1] = cryptpw[1];
	salt[2] = 0;

	crypted = crypt(x->data.password, salt);
	if (strcmp(cryptpw, crypted) != 0) {
		syslog(LOG_NOTICE, "bad login: %s", x->data.username);
		cfpute(x, "bad login");
		exit (1);
		}

	if ((pw = getpwnam(x->data.username)) == NULL) {
		syslog(LOG_NOTICE, "can't get passwd entry");
		cfpute(x, "access denied");
		exit (1);
		}
		
	x->user.uid = pw->pw_uid;
	x->user.gid = pw->pw_gid;
	copy_string(x->user.home, pw->pw_dir, sizeof(x->user.home));
	copy_string(x->user.shell, pw->pw_shell, sizeof(x->user.shell));

	return (0);
}


int rexec_request(config_t *config)
{
	int	pid;
	unsigned long started;
	rexec_t	*x;

	started = time(NULL); 
	x = allocate(sizeof(rexec_t));
	x->config = config;

	if (get_client_info(x, 0) < 0) {
		syslog(LOG_NOTICE, "can't get client info: %m");
		exit (1);
		}

	x->port = get_interface_info(0, x->interface, sizeof(x->interface));
	syslog(LOG_NOTICE, "port %u connected to client: %s", x->port, x->client);

	if (doinit(x) != 0)
		return (1);
	else if (dologin(x) != 0)
		return (1);

	if ((pid = fork()) < 0) {
		syslog(LOG_NOTICE, "can't fork()");
		exit (1);
		}
	else if (pid == 0) {
		if (chdir(x->user.home) != 0)
			syslog(LOG_NOTICE, "can't chdir() to %s", x->user.home);
		else if (setgroups(0, NULL) != 0  ||
			 setregid(x->user.gid, x->user.gid) != 0  ||
			 setreuid(x->user.uid, x->user.uid) != 0)
			syslog(LOG_NOTICE, "can't change uid/gid to %u/%u", x->user.uid, x->user.gid);
		else {
			write(1, "\000", 1);
			execlp(x->user.shell, x->user.shell, "-c", x->data.command, NULL);
			syslog(LOG_NOTICE, "can't exec() shell: %s", x->user.shell);
			}

		exit (1);
		}
	else if (pid > 0) {
		syslog(LOG_NOTICE, "+OK request complete: %s@%s", x->data.username, x->client);
		}
		
	return (0);
}



void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[], char *envp[])
{
	int	c, i, k;
	char	*p, option[80];
	config_t *config;
	extern unsigned int get_interface_info(int fd, char *name, int size);

	if ((p = strrchr(argv[0], '/')) == NULL)
		program = argv[0];
	else {
		copy_string(progname, &p[1], sizeof(progname));
		program = progname;
		}

	logname = program;
	config = allocate(sizeof(config_t));
	config->timeout = 60 * 30;
	strcpy(config->varname, "SERVER_");

	k = 1;
	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'd')
				debug = 1;
			else if (c == 'a') {
				if (k >= argc)
					missing_arg(c, "access control program");

				copy_string(config->acp, argv[k++], sizeof(config->acp));
				}
 			else if (c == 'l') {
				if (k >= argc)
					missing_arg(c, "logname");

				logname = argv[k++];
				}
			else if (c == 'n')
				numeric_only = 1;
			else if (c == 'v') {
				if (k >= argc)
					missing_arg(c, "varname prefix");

				copy_string(config->varname, argv[k++], sizeof(config->varname));
				}
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (-1);
				}
			}
		}


	
	if (k < argc) {
		fprintf (stderr, "%s: extra arguments on command line: %s ...\n", program, argv[k]);
		exit (1);
		}

	openlog(logname, LOG_PID, LOG_MAIL);
	rexec_request(config);

	exit (0);
}


