/*
 *  This program is a user space rarp request program.  It will send
 *  Rarp requests out over an ethernet interface, defined with a
 *  command-line parameter and process the requests coming back 
 *  to the computer through that interface.  It's usage is:
 *
 *	rarpc [-t timeout] [-n retrycount] [device]
 *	ie.  rarpc eth1
 *  
 *  It will use eth0 by default but use another ethernet device if 
 *  the option is given on the command line.
 *
 *  The output of the program will be:
 *
 *  IPADDR=XXX.XXX.XXX.XXX
 *
 *  The program is meant to be used in a shell script with the following
 *  syntax:
 *
 *  ifconfig eth0 0.0.0.0 -arp   # no arp (yet)
 *  eval `rarpc`
 *  ifconfig eth0 $IPADDR up +arp # reconfig with arp enabled
 *  etc.
 *
 *  Maintainer of this code is 
 *  Shawn C. Jury
 *  jurys@hq.22sigbde.army.mil
 *
 *
 */
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketio.h>
#include <sys/ioctl.h>
#include <linux/netdevice.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/if_ether.h>
#include <linux/in.h>
#include "rarpc.h"



/*************************************************************************
/  This function attempts to open a socket on the device that is
/  given to the program, or the default.  If it cannot open the 
/  socket it will give an error and return an error code.  It is
/  called from the main program. 
/************************************************************************/


int Open_Rarp_Socket(struct ifreq ifr, char *my_hw_addr[RARP_HLN]) {

	int s=socket(AF_INET, SOCK_PACKET, htons(ETH_P_RARP)); 

	if(ioctl(s, SIOCGIFHWADDR, &ifr)<0) {
		/* oops */
		fprintf(stderr, "Error opening socket\n");
		return (int ) -1;
	}
	memcpy(my_hw_addr, ifr.ifr_hwaddr.sa_data,6);
        return s;
}

/*************************************************************************
/  This function creates a RARP packet from the information received
/  the device.  It will return the size of the packet once it has 
/  been created.  It is called from the function Send_Rarp_query.
/************************************************************************/


int make_rarp_packet(char *hw_addr[RARP_HLN], struct rarp_type *rarp_pkt) {


	rarp_pkt->hardware_address_space  =	htons(RARP_HRD);
	rarp_pkt->protocol_address_space  =	htons(RARP_PRO);
	rarp_pkt->hardware_address_length =	RARP_HLN;
	rarp_pkt->protocol_address_length =	RARP_PLN;
	rarp_pkt->opcode		  =	htons(RARP_REQUEST_OPCODE);

	memcpy(rarp_pkt->source_hardware_address, hw_addr, RARP_HLN);
	memset(rarp_pkt->source_protocol_address, 0, RARP_PLN);
	memcpy(rarp_pkt->target_hardware_address, hw_addr, RARP_HLN);
	memset(rarp_pkt->target_protocol_address, 0, RARP_PLN);

	return sizeof(*rarp_pkt);
}

/*************************************************************************
/  This function sends the RARP request packet out onto the LAN.  It
/  also calls the make_rarp_packet function.  This function is called
/  from the main program.
/************************************************************************/

int Send_Rarp_query(struct ifreq *ifr, char *my_hw_addr[RARP_HLN], 
		    int s, const int verbose) {

 	struct rarp_type  rarp;
	char buffer[RARP_HEADER_SIZE];
	struct sockaddr sa;
	int len;

	memset(buffer, 0xFF, 6);    			/* To broadcast */
	memcpy(buffer+6, my_hw_addr, 6); 		/* HW Addr of sender */
	buffer[12] = 0x80;
	buffer[13] = 0x35;
	
	len=14+make_rarp_packet(my_hw_addr, (struct rarp_type *) (buffer+14));

	strcpy(sa.sa_data,ifr->ifr_name);
	sendto(s, buffer,len,0,&sa ,sizeof(sa));

	if (verbose) {
	   dump((struct rarp_type *) (buffer+14),len, "sendto:\n"); 
	} /* end if (verbose) */

	return 0;
}


/*************************************************************************
/  This function is the main work horse of the program.  It checks
/  the device to see if it is able to read any input from it.  If it 
/  can, it will read in the packet, gather all information, and check 
/  to see if it a RARP response.  If it is a response, it then checks to
/  see if it is meant for our device by compairing the target hardware
/  address with our hardware address. If it is meant for us, it will
/  print the IP address to standard output and return a successful status.
/  If it is not meant for us, it will return an error.  It also returns an
/  error if there is nothing to be read from the socket prior to the 
/  timeout.  There is a function call in this function which calls
/  the Produce_TPN_Data function.  This is only called if the USER 
/  specifies that they are on the TPN in the rarpc.h file.  This is
/  for use with the US Military's Mobile Subscriber Equipment's Tactical
/  Packet Network.  This function is called from the main program.
/************************************************************************/


int Check_Reply(int s, struct ifreq *ifr, char *my_hw_addr[RARP_HLN],const int verbose, const int timeout) {

	struct sockaddr sa;
	struct rarp_type *rarp;
	struct in_addr inaddr;
	char   tmpha[2*RARP_HLN + 3];
	char *buf;
        char net_id[15];
	fd_set read_template;
	struct timeval wait;
	int sal;
	int len, n;
	int rcvbuf;
	int retval;

	sal=sizeof(sa);
	rcvbuf=1024;

	/*   Set time out */
	wait.tv_sec = 0;
	wait.tv_usec = timeout;

	FD_ZERO(&read_template);
	FD_SET(s, &read_template);
	n = s;
	n++;
	retval = select(n, &read_template, NULL, NULL, &wait);
	if (retval) {
		buf=malloc(rcvbuf);
		len=recvfrom(s,buf,rcvbuf,0,&sa ,&sal);
		if(strcmp(sa.sa_data,ifr->ifr_name)!=0) {
			if (verbose) {
				printf("not from %s\n", ifr->ifr_name);
			}  /* if verbose */
			return (int) -1;	/* Not from device */
		} /* if (strcmp) */
		else {
			if (verbose) {
				printf("Received %d bytes of data\n", len);
			}  /* if verbose */
			rarp = (struct rarp_type *) (buf+14);

			if (rarp->opcode == htons(RARP_RESPONSE_OPCODE) && 
				(!strcmp((char *)inet_nsap_ntoa(RARP_HLN, 
						rarp->target_hardware_address, tmpha),
					(char *)inet_nsap_ntoa(RARP_HLN, 
						my_hw_addr, tmpha)))) {
				if (verbose) {
	       		   		dump((struct rarp_type *) (buf+14), \
					      len, "recv:\n"); 
				} /* if verbose */
 
      				memcpy((char*) &inaddr, rarp->target_protocol_address, RARP_PLN);
				printf("export IPADDR=%s;\n", inet_ntoa(inaddr));

				if (MSE_TPN) {
					Produce_TPN_Data(inet_ntoa(inaddr));
				} /* end if (MSE_TPN) */
				return 0;
			}  /* end  if (rarp->opcode...... */
			else {
				return -1;
	 		} /* end else */
	   	} /* else if (strcmp) */
	} /* if retval */
	else {
		return -1;
	} /* else if retval */
}

/*************************************************************************
/  This function is only called if the user specifies that they are
/  on the TPN.  It will simplify the process of configuring  the machine
/  when the unit moves or you change to another port off of the switch.
/  It parces the IP address received from the switch, extracts the 
/  NETID, and produces all (hopefully) of the needed output for 
/  configuring your switch.  This will allow any unit to use this software
/  on their machines without having to change the code.  This function is
/  called from the Check_Reply routine.  
/************************************************************************/


int Produce_TPN_Data(char *recvd_addr)
{
	char netid[20];

	strcpy(netid,strtok(recvd_addr,"."));
	strcat(netid,".");
	strcat(netid,strtok(NULL, "."));
	printf("export NETWORK=%s.0.0\n",netid);
	printf("export NETMASK=255.255.0.0\n");
	printf("export BROADCAST=%s.255.255\n", netid);
	printf("export GATEWAY=%s.196.1\n", netid);
	printf("export NS1=%s.192.111\n", netid);
}
	

/*************************************************************************
/  This function will help debug the program.  It will show the 
/  information that is leaving the machine and being received 
/  to the machine (RARP packets only).  It is called from the Send
/  and Check functions.  It will only be called if verbose is 
/  an option
/************************************************************************/


int dump(struct rarp_type *rarp, int n, char *title) {

	char   tmpha[2*RARP_HLN + 3];
	int    i;
	struct in_addr inaddr;

	fprintf(stderr, title);

   /**********************
    * format RARP header *
    **********************/

	i = 0;
	if (n >= RARP_HEADER_SIZE - 14) {
		i = RARP_HEADER_SIZE - 14;
		fprintf(stderr,
		    "   hrd=0x%04x  pro=0x%04x  hln=0x%02x  pln=0x%02x  op=0x%04x\n",
		    rarp->hardware_address_space, rarp->protocol_address_space, 
		    rarp->hardware_address_length, rarp->protocol_address_length, 
		    rarp->opcode);

		memcpy((char*)&inaddr, rarp->source_protocol_address, RARP_PLN);
		fprintf(stderr,
		    "   sha=%s  spa=%s\n", inet_nsap_ntoa(RARP_HLN,  
		    rarp->source_hardware_address, tmpha), inet_ntoa(inaddr));

		memcpy((char*)&inaddr, rarp->target_protocol_address, RARP_PLN);
		fprintf(stderr,
		    "   tha=%s  tpa=%s\n", 
		    inet_nsap_ntoa(RARP_HLN,rarp->target_hardware_address, tmpha ),
		    inet_ntoa(inaddr));
	}  /* end if n>=RARP_HEADER_SIZE - 14 */

	/***********************************************
	* format any ancillary data after RARP header *
	***********************************************/

	while (i < n) {
		int j;

      /* print byte offset relative to RARP header */

		fprintf(stderr, "   %03x:", i);

      /* print up to 16 octets of data */

		for (j = 16; j--  &&  i < n; i++)
			fprintf(stderr, " %02x", ((u_char*)rarp)[i]);

		fprintf(stderr, "\n");
	} /* while (i < n ) */

}


/*************************************************************************
/  This function is used to print the help message that is given by the
/  program.  It is called if the user specifies the '-h' switch or if
/  there is something wrong with the options given to the program.
/************************************************************************/

void Print_Usage(void) {

	printf("Usage:\n");
	printf("\trarpc [-h] [-t timeout] [-n retrycount] [device]\n");
	printf("\n\t\t-h \t\t\tPrint this message\n");
	printf("\t\t-t timeout\t\tmilliseconds (per transmission)\n");
	printf("\t\t-n retrycount\t\tnumber of retransmissions\n");
	printf("\t\tdevice\t\t\teth0, eth1, ... (eth0 is default)\n\n");
}


int main (int argc, char **argv) {

	struct ifreq 		interface_request;
	char 			device_name[4]; 
	char			*my_hw_address[RARP_HLN];
	int			socket_fd;
	int			i; 				/* Loop Control */
	int			timeout=TIMEOUT_DFLT;
	int			retry=RETRY_DFLT;
	int			verbose=0; 



/* Collect command line parameters */

	for (i = 1; i < argc  &&  *argv[i] == '-'; i++) {
	char *val;

	switch (argv[i][1]) {
		case 't':
			val = (argv[i][2]) ? &argv[i][2] : argv[++i];
			timeout = atol(val);
			if (timeout <= 0)
			timeout = TIMEOUT_DFLT;
			break;

		case 'n':
			val = (argv[i][2]) ? &argv[i][2] : argv[++i];
			retry = atol(val);
			if (retry < 0)
			    retry = RETRY_DFLT;
			break;

		case 'v':
			/* undocumented feature for debugging */
			verbose = 1;
			break;

		default:
			Print_Usage();
			exit(1);
         } /* end switch */
      } /* end for loop */

/* End Command-line portion */

	if (i >= argc || !argv[i] || !*argv[i]) {
	   strcpy(device_name, "eth0");
	} /* end if (i >= argc */
	else {
	   strcpy(device_name, argv[i]);
	} /* end else */
	if (verbose) {
		printf("Device is: %s\n",device_name);
	}  /* end if (verbose) */

	strcpy(interface_request.ifr_name, device_name);

	socket_fd = Open_Rarp_Socket(interface_request,  my_hw_address);

	for (i = 0; i < retry; i++) {
	
		if (verbose) {
			printf("Sent %d packet[s].\n",i+1);
		} /* if verbose */

		Send_Rarp_query(&interface_request, my_hw_address, socket_fd,
				verbose);

		if (Check_Reply(socket_fd, &interface_request, my_hw_address, 
				verbose, timeout) == 0) {
			break; 
		} /* if (Check_Reply) */
	}
	
	close(socket_fd);
	return 0;
}
