/*
   Module:	NETPROBE
  
   Version:	0.6
   
   Function:	The program will configure itself from a file to
        	indicate the Internet Hosts that are part of the
                network (LAN or WAN). The hosts are "ping"ed to
                determine their status. The output of the status
                is displayed either graphically (in a text screen)
                or as a simple text chart of response times.
                
   Inputs:	In graphical mode, some keyboard commands are
        	available. Poll time may be modified to
        	either stop polling or start it (if none set).
                A rescan is available for non-polled execution.
                p - poll (prompted for poll time)
                r - rescan
                
   Outputs:	The standard output device is the screen. In
        	graphical mode, the output is a "curses" screen.
                In standard mode, the output is "stdout" so
                redirection is possible.
                
   Usage:	Common methods of execution are
   
        	% netprobe
                
                   << standard mode execution to standard output >>
                   
                % netprobe -g
                
                   << 1 shot graphical mode execution >>
                   
                % netprobe -g -p 240
                
                   << polled graphical mode with a 4 minute (240 sec)
                      polling time >>
                      
                % netprobe -g -p 240 -l logfile
                
                   << as above but with a logfile which indicates
                      changes >>
                      
   NOTES on PORTABILITY
   
      This program is tied to the "ping -c 3" command of the operating 
      system and the format for the output of that command.
      
      Future work could make it independent of this.


   NOTES on CONFIGURATION FILE
   
             
      The configuration file is defaulted to the "/etc/netcheck.conf"
      name. If you desire to specify this from the command line, then
      use "-c configfile" option

   	FILE INFO inside Configuration File
        
        # in column 1 indicates comment
        Valid lines should have 3 fields, LAN Location
        Host ID and IP Address.
        
        LAN Location can be     NAMESERVER
                        	LOCAL
                                ROUTER
                                EXTERNAL
                                
        Host ID can be a host name (for lookup in nameserver)
           or * to indicate known by IP Address
           
        IP Address is either * (for don't care...since name was given)
           or actual IP address of host (unused if name was specified)

   	(for example... with ficticious names and numbers)   
   
local 		novell		192.122.21.3
nameserver	cornw		192.122.21.1
nameserver	space		192.122.21.2
local		george		*
local		alien		*
local		starship	*
local		pppstar1	*
local		pppstar2	*
router		*		192.122.21.100
external	*		192.122.40.1
external	*		192.122.41.2


           
   LIMITATIONS
   
      Graphical display limits the number of monitored LAN machines
      to ~18.... remember that the larger the number, the longer the
      working time to display. Our main concern is the nameservers and
      important file servers on the LAN along with the INTERNET router
      box. Only 1 router box can be displayed. A few external hosts
      (in CLOSE proximity) are recommended for connection verification.

      Graphical display assumes BUS Network (not star or token)...
      sorry... although it is just a diagram problem NOT a functionality
      problem.


   COMPILING SOURCE CODE
   
      Since the source code is in one file currently, the compilation
      is performed as follows
      
      % gcc -o netprobe netprobe.c -lncurses
      
      OR
      
      % make
         	(if you have the included makefile)
            


        Copyright (C) 1995 Gordon MacKay
        "netprobe" is free. Anyone may redistribute copies of "netprobe"
        to anyone under the terms stated in the GNU General Public License.
        The author assumes no responsibility for errors or omissions or
        damages resulting from the use of "netprobe".
                            
        If you have questions, hints or you have found an error mail
        it to
                                      
        <mackay@slcorn.c.stlawrencec.on.ca>  or
        <mackay@gemsoft.c.stlawrencec.on.ca>
                                                
                                                

*/


#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <ncurses/ncurses.h>
#include <signal.h>
#include <unistd.h>

FILE *fp;

#define  gotoxy(x,y)	move(y,x)
#define  clrscr()	clear()

time_t	now;
char tmstr[40];

char logname[FILENAME_MAX]="/etc/netprobe.log";
char fname[FILENAME_MAX]="/etc/netcheck.conf";

#define MINPOLLTIME 40
#define KEYDELAY 10

int polltime;
int tpoll;
double pollsec;

/* Graphics Control Variables */

int colour = FALSE;
int graphics;
int pcmode = FALSE;
int midxscr;
int midyscr;
int numoflocal = 0;
int numofext = 0;
int numofrouters = 0;
int numofservers = 0;

/*	Format for Net Check File

   LAN Location		Host ID		IP Address
   
   NA - nameserver	IPnameaddress	IPNumericAddress
   LO - local
   RO - Router
   
*/

enum { ROUTER, LOCALLAN, NAMESERVER, EXTERNAL };

struct netcheck {
   int location;
   char name[60];
   char addr[20];
   char hopcnt;
   char status;
   double response;
   int dx,dy;			/* Display Loc on Screen */
   struct netcheck *link;
   struct netcheck *lanlink;	/* Link for EXT or LAN */
   };
   
typedef struct netcheck *PNET;

int ns;
int seq;
PNET first, current   ;
PNET locallan, curloc;
PNET external, curext;
struct netcheck STLAN,STMAIN,STEXT;
PNET router=NULL;
int alldone=FALSE;

/* SIGNAL Handling */
int (*f)();


/* LOG CHECK... */
int logfile=FALSE;
FILE *flog;
int changes;

char buf[128];

void config()
{
   FILE *fc;
static   char type[80];
static   char hostid[80];
static   char ipaddr[80];
   PNET hold;
   int i;
   int len;
      
   fc = fopen(fname,"r");
   if (!fc)
   {
      	printf("Problems on opening %s --> config file for program\n",fname);
        printf("See if the file exists and make sure there are\n");
        printf("READ privileges for all legitimate users\n");
        exit(1);
   }
   first = &STMAIN;
   first->link = NULL;
   locallan = &STLAN;
   locallan->lanlink=NULL;
   external = &STEXT;
   external->lanlink=NULL;
   current = first;
   curloc = locallan;
   curext = external;
   while (fgets(buf,128,fc)!=NULL)
   {
      buf[strlen(buf)-1]=0;
      if (buf[0]!='#')
      {
      	len = strlen(buf);
        for (i=0;i<len;i++) buf[i] = tolower(buf[i]);
      	ns = sscanf(buf,"%s %s %s",type,hostid,ipaddr);
        if (ns==3)
        {
        	hold = (PNET)malloc(sizeof(struct netcheck));
                strncpy(hold->name,hostid,60);
                if (!strcmp(hostid,"*"))
                   strncpy(hold->name,ipaddr,20);
                strncpy(hold->addr,ipaddr,20);
                current->link = hold;
                hold->link = NULL;
                hold->lanlink = NULL;
                hold->status= 127;  /* Priming for log file... */
                if (!strncmp(type,"lo",2))
                {
                   hold->location = LOCALLAN;
                   curloc->lanlink=hold;
                   curloc = hold;
                   numoflocal++;
                }
                else if (!strncmp(type,"ro",2))
                {
                   hold->location = ROUTER;
 /*                  curloc->lanlink=hold;
                   curloc = hold;
                   numoflocal++;
 */
                   router = hold;
                   numofrouters++;
                }   
                else if (!strncmp(type,"na",2))
                {
                   hold->location = NAMESERVER;
                   curloc->lanlink=hold;
                   curloc = hold;
                   numoflocal++;
                   numofservers++;
                }                   
                else
                {
                   hold->location = EXTERNAL;
                   curext->lanlink=hold;
                   curext = hold;
                   numofext++;
                }                   
                current = hold;
        }
      }
   }                                                  
   fclose(fc);
}          	                
 
void freelist()
{
   PNET next;
   
   	current = first->link;
        while (current)
        {
        	next = current->link;
                free(current);
                current=next;
        }
}
     	
void perform()
{
static char mess[]="Working";
static char dot[]=".";
static	char	cmd[128];
static  char	part[]="ping -c 3 ";
static  char    t1[40];   	
static  char    t2[40];   	
static  char    t3[40];   	
static  char    t4[40];   	
static  char    t5[40];   	
static  char    t6[40];   	
static  char    t7[40];   	
   	char dum;
        int tmpstatus;

   	changes = 0;  /* No status changes yet.... */        
    	current = first->link;
   	now = time(NULL);   
        strcpy(tmstr,ctime(&now));
        tmstr[strlen(tmstr)-1]=0;	/* Remove trailing \n */
    	if (graphics)
    	{
    		gotoxy(2,LINES-2);
    		refresh();
    		printw("%s",mess);
    	}
	else
	         write(1,mess,strlen(mess));
        while (current)
        {
        	strcpy(cmd,part);
                strcat(cmd,current->name);	
                fp = popen(cmd,"r");
                if (graphics)
                {
                	addch(dot[0]);
                	refresh();
                }
                else
	                write(1,dot,1);
                current->response = 0.0;
                if (fp)
                {
                           seq = 0;
                   	   while (fgets(buf,128,fp)!=NULL)
                           {
                                      buf[strlen(buf)-1]=0;
                                      ns = sscanf(buf,"%s %s %s %s %s %s %s",
                                         t1,t2,t3,t4,t5,t6,t7);
                                      if (seq)
                                      {
                                                sscanf(t4,"%[^/]%c%[^/]%c%s"
                                                   ,t1,&dum,t2,&dum,t3);
                                                sscanf(t2,"%lf",&current->response);
                                      }
                                      else
                                      {
                                      	if (!strcmp(t2,"packets"))
                                        {

                                        	t7[strlen(t7)-1] = 0;
                                                tmpstatus = atoi(t7);                                           
                                                if (tmpstatus!=current->status)
                                                {
                                                      if (logfile)
                                                      {
                                                            if (changes==0)
fprintf(flog,"***************************************\n");
                                                            if (tmpstatus==0)
                                                                 fprintf(flog,"%-30s %10s\n",current->name,"OK");
                                                            else if (tmpstatus<100)
                                                                 fprintf(flog,"%-30s %10s\n",current->name,"WARNING");
                                                            else if (tmpstatus==100)
                                                                 fprintf(flog,"%-30s %10s\n ",current->name,"DOWN");
                                                      }
                                                      changes++;
                                                      current->status = tmpstatus;
                                                }
                                                seq++;
                                        }
                                        else if (!strncmp(t6,"ttl",3))
                                        {
                                        	current->hopcnt = 255 - atoi(&t6[4]);
                                        }
                                      }                 	                                          
                        	}
                                pclose(fp);
                }
                current = current->link;
        }
    	if (graphics)
    	{
    		gotoxy(2,LINES-2);
    		printw("                                     ");
    		refresh();
    	}
        if (changes && logfile)
        {
           fprintf(flog,"   at %s\n",tmstr);
           fflush(flog);
        }

}

void screenone(PNET current)
{
               	strncpy(buf,current->name,19);
                buf[19]=0;
                printf("%-20s",buf);
                switch (current->location)
                {
                case ROUTER:
                   	printf("%10s","ROUTER");
                        break;
                case LOCALLAN:
                   	printf("%10s","LOCAL");
                        break;
                case NAMESERVER:
                   	printf("%10s","NAMESEVER");
                        break;
                case EXTERNAL:
                   	printf("%10s","EXTERNAL");
                        break;
                default:
                   	printf("%10s"," ");
                }
                printf("  %4d ",current->hopcnt);
                if (current->status==0)
                   printf("%10s ","OK");
                else if (current->status<100)
                   printf("%10s ","WARNING");
                else if (current->status==100)
                   printf("%10s ","DOWN");
                else
                   printf("%10s ","!!!!");
                printf("%7.1lf\n",current->response);

}
 
void screentitle()
{
   	printf("\n\n");
   	printf("\t***************************************\n");
        printf("\t*          Network Status             *\n");
        printf("\t*   %28s      *\n",tmstr);
   	printf("\t***************************************\n");
        printf("\n   HOSTNAME        CONNECTION   HOPS   STATUS  TIMING(ms)\n");
        printf("   ========        ==========   ====   ======  ==========\n");
        printf("\n");
}

void graphicstitle()
{

   	int numofsections = 0;
        int wtmp;
        int entriesperline;  /* Max of 6 */
        int firstyline;
        int width = COLS-2;
        int height = LINES-2;
        int stxentry;
        int widthentry;
        int j,y;
        int ywidth;        
        int k,i,x,linespot;
 
   	border(0,0,0,0,0,0,0,0);
   	refresh();
        midxscr = COLS/2;
        midyscr = LINES/2;
        gotoxy(midxscr-7,0);
        printw("Network Status");
        gotoxy(midxscr-12,LINES-1);
        printw("%24s",tmstr);
        refresh();                   	
        /*
           	Fix the Graphics Coordinates for ALL display entries
        */
        entriesperline = (numoflocal/3)+1;
        widthentry = (width-16)/(entriesperline+1);
        stxentry = 8+widthentry;
        ywidth = (height-numofext-1)/3;
        if (entriesperline>6)
        {
        	gotoxy(20,LINES-2);
                printw("Too many local entries...");
                refresh();
                return;
        }
        firstyline = 3+numofext;
        /*   Vertical lines */
        gotoxy(COLS-8,3);
        vline(ACS_VLINE,LINES-5);
        y = firstyline;
        for (k=0;k<3;k++)
        {
        j = stxentry+2;
        gotoxy(8,y);
        hline(ACS_HLINE,COLS-16);
        for (i=0;i<entriesperline;i++)
        {
        	if (i % 2)
                {
                   mvaddch(y,j,ACS_BTEE);
                   mvaddch(y-1,j,ACS_VLINE);
                }
                else
                {
                   mvaddch(y,j,ACS_TTEE);
                   mvaddch(y+1,j,ACS_VLINE);
                }
                j += widthentry;
        }
        gotoxy(COLS-8,y);
        addch(ACS_RTEE);
        y += ywidth;
        }
        
        /*   Time to set parameters in the actual entries for display */
        current = external->lanlink;
        y = 2;
        
        while  (current)
        {
        	current->dx = 3;
                current->dy = y;
                y++;
                current = current->lanlink;
        }
        current = locallan->lanlink;
        y = firstyline;
        x = stxentry-1;
        linespot = 0;
        while (current)
        {
        	current->dx = x;
                if (linespot % 2)
                   current->dy = y-1;
                else
                   current->dy = y+1;
                linespot++;
                if (linespot==entriesperline)
                {
                	y += ywidth;
                        linespot=0;
                        x = stxentry-3;
                }
                else
                   	x += widthentry;
                current = current->lanlink;
        }
        gotoxy(3,1);
        printw("EXTERNAL WAN");
        if (router)
        {
        	/* gotoxy(7,firstyline-3);
                vline(ACS_VLINE,3);        	 */
                gotoxy(5,firstyline+3);
                printw("ROUTER");
                router->dx = 3;
                router->dy = firstyline+2;
                gotoxy(8,firstyline);
                addch(ACS_BLOCK);
                gotoxy(8,firstyline+1);
                addch(ACS_BLOCK);
                gotoxy(8,firstyline-1);
                addch(ACS_BLOCK);
        }
        gotoxy(COLS-20,LINES-2);
        printw("                  ");
        if (tpoll)
        {
                   gotoxy(COLS-20,LINES-2);
                   printw("POLLING %6.2lf sec",(double)pollsec);
        }
           

}

void graphicsone(PNET current)
{
	static char bb[80];
        	/* standout(); */
                if (current->status==100)
                {
                	if (colour)
                           attron(COLOR_PAIR(1));
                        else
                	   attron(A_BLINK);
                }	
		else if (!colour)
                   	standout();
                else if (current->status!=0)
                   	attron(COLOR_PAIR(2));
                else
                   	attron(COLOR_PAIR(3));
                   	
                gotoxy(current->dx,current->dy);
		strncpy(bb,current->name,20);
                bb[19]=0;
		printw("%s",bb);
                if (colour)
                   attrset(COLOR_PAIR(4));
                else
                   attrset(A_NORMAL);
}
 
void preparescr()
{

             initscr(); cbreak(); noecho();
             clear();
             nonl();
/*             intrflush(stdscr,FALSE); */
             keypad(stdscr,TRUE);
             if (tpoll)
                  halfdelay(KEYDELAY); 
             if (has_colors())
             {
                colour = TRUE;
                start_color();
                init_pair(4,COLOR_WHITE,COLOR_BLUE);
                init_pair(1,COLOR_WHITE,COLOR_RED);
                init_pair(2,COLOR_BLACK,COLOR_YELLOW);
                init_pair(3,COLOR_BLACK,COLOR_GREEN);
                attrset(COLOR_PAIR(4));
                erase();
             }
               
}

void restorescr()
{
   nocbreak();
   echo();
   nl();
   endwin();

}
 
void display()
{
   char dum;
   int pollingcnt;
   static char buf[80];

   	if (graphics)
           graphicstitle();
        else
           screentitle();
   	current = first->link;
        while (current)
        {
           	if (graphics)
                   graphicsone(current);
                else
                   screenone(current);
                current = current->link;
        }
        pollingcnt = (int)pollsec;
        if (!tpoll) pollingcnt = 1;
        while (pollingcnt)        
        {
           if (graphics)
           {
        	gotoxy(COLS-1,LINES-2);
	           dum = getch();                       
                   if (dum==ERR)
                        alldone = FALSE;
                   else if (dum=='p')
                   {
                	nocbreak();
                   	tpoll = TRUE;
            		gotoxy(2,LINES-2);
                        printw("                       ");
            		gotoxy(2,LINES-2);
                        refresh();
                        printw("Enter Poll Time:");
                        echo();
                        getstr(buf);
                        noecho();
            		gotoxy(2,LINES-2);
                        printw("                       ");
                        polltime = atoi(buf);
                        if (polltime< MINPOLLTIME)
                           polltime = MINPOLLTIME;
                        halfdelay(KEYDELAY);
                        alldone = FALSE;
                        pollsec = (double)polltime*10/KEYDELAY;
                        pollingcnt = 1;
                   }
                   else if (dum=='q')
                   {
                   	if (tpoll)
                        {
                        	tpoll=FALSE;
                                pollingcnt = 1;
                                cbreak();
                        }
                        alldone=TRUE;
                   }
	           else if (dum!='r')	/* RESCAN */
	           	alldone = TRUE;
	   }
	   else
		alldone = TRUE;
           pollingcnt--;
        }
	           
}

void processargs(int argc, char *argv[])
{
   /*	Make to accomodate extra features such as polling
        color settings and state logging via command line
   */
      	int spot=1;
        int i;
        
        tpoll = FALSE;
   	graphics = FALSE;
   	if (argc>1)
        {
        	for (i=1;i<argc;i++)
                {
                        if (!strncmp(argv[i],"-g",2))
                        {
                              if (argv[i][2]=='p')
                                   pcmode = TRUE;
                              graphics = TRUE;
                        }
                        if (!strncmp(argv[i],"-p",2))
                        {
                        	i++;
                                if (i<argc )
                                {
                                	polltime = atoi(argv[i]);
                                        if (polltime<MINPOLLTIME)
                                           polltime=MINPOLLTIME;
                                        pollsec = (double)polltime*10/KEYDELAY;
                                        tpoll = TRUE;
                                }
                        }
                        if (!strncmp(argv[i],"-l",2))
                        {
                        	i++;
                                if (i<argc )
                                {
                                	logfile = TRUE;
                                        strcpy(logname,argv[i]);
                                        flog=fopen(logname,"a");
                                        if (flog==NULL)
                                        {
                                        	printf("No logging\n");
                                                logfile = FALSE;
                                        }
                                }
                        }
                        if (!strncmp(argv[i],"-c",2))
                        {
                        	i++;
                                if (i<argc )
                                {
                                        strcpy(fname,argv[i]);
                                }
                        }
                }
                                
   	}
   }
    	

/*	Algorithm

   	-	Configure with file
        -	Process through list... PINGING to hosts and
           	recording results...
        -	Display Summary.......
        
*/

int main(int argc, char *argv[])
{
	
   	processargs(argc,argv);   	
        if (graphics)
           preparescr();
   	config();
        while (!alldone)
        {
	        perform();                 
        	display();
        }
   	freelist();
        if (graphics)
           restorescr();
        if (logfile)
           fclose(flog);
}
