/************************************************************************\
**  udp_pipe.c - Copy standard input to a UDP port and data received    **
**                 on this port to standard output                      **
**                                                                      **
**  Copyright (c) 2001 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/
	
	#include "config.h"

	#ifdef HAVE_ERRNO_H
	#  include <errno.h>
	#endif
	#ifdef HAVE_FCNTL_H
	#  include <fcntl.h>
	#endif
	#include <signal.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#ifdef HAVE_UNISTD_H
	#  include <unistd.h>
	#endif
	#include <netinet/in.h>
	#include <sys/socket.h>

	#ifdef HAVE_GETOPT_H
	#  include <getopt.h>
	#endif
	
	#include "netpipe.h"

	#define DATA_BUFFER_LEN 4096

	static int	verbose = 0;

	int
main (int argc, char * argv [])
{
	char			data [DATA_BUFFER_LEN];
	int			nb_data;

	int			option;
	char *			local_hostname = NULL;
	char *			local_portname = NULL;
	char *			remote_hostname = NULL;
	char *			remote_portname = NULL;
	
	struct sockaddr_in	local_address;
	struct sockaddr_in	remote_address;
	
	int			stdin_2_socket = 1;
	int			socket_2_stdout = 1;

	int			my_socket;
	int			broadcast = 0;
	int			sock_val;
	socklen_t		sock_len;
	int			multicast = 0;
	struct ip_mreq		multicast_req;

	int			nb_desc;
	fd_set	read_set;
	
	while (1) {
#ifdef HAVE_GETOPT_LONG
		static struct option long_options[] = {
			{ "local-address",	1,	0,	'A' },
			{ "remote-address",	1,	0,	'a' },
			{ "local-port",		1,	0,	'P' },
			{ "remote-port",	1,	0,	'p' },
			{ "input-only",		0,	0,	'i' },
			{ "output-only",	0,	0,	'o' },
			{ "broadcast",		0,	0,	'b' },
			{ "multicast",		0,	0,	'm' },
			{ "verbose",		0,	0,	'v' },
			{ NULL, 		0,	0,	0   },
		};
		option = getopt_long (argc, argv, "A:a:be:imoP:p:v", long_options, NULL);
#else
		option = getopt (argc, argv, "A:a:be:imoP:p:v");
#endif
		if (option == EOF)
			break;
		switch (option) {
			case 'A' :
				local_hostname  = optarg;
				break;
			case 'a' :
				remote_hostname  = optarg;
				break;
			case 'b' :
				broadcast = 1;
				break;
			case 'P' :
				local_portname = optarg;
				break;
			case 'p' :
				remote_portname = optarg;
				break;
			case 'i':
				socket_2_stdout = 0;
				break;
			case 'o':
				stdin_2_socket = 0;
				break;
			case 'm' :
				multicast = 1;
				break;
			case 'v':
				fprintf (stderr, "%s (" VERSION ") Christophe Blaess 1997-2002\n", argv[0]);
				verbose ++;
				break;
			default :
				fprintf (stderr, "Usage: %s [options]\n",argv [0]);
#ifdef HAVE_GETOPT_LONG
				fprintf (stderr, " options: -A, --local-address ADDR   local IP address or host name\n");
				fprintf (stderr, "          -P, --local-port PORT      local UDP port number or service name\n");
				fprintf (stderr, "          -a, --remote-address ADDR  remote IP address or host name\n");
				fprintf (stderr, "          -p, --remote-port PORT     remote UDP port number or service name\n");
				fprintf (stderr, "          -i, --input-only           redirect only standard input\n");
				fprintf (stderr, "          -o, --output-only          redirect only standard output\n");
				fprintf (stderr, "          -b, --broadcast            use UDP broadcast mode\n");
				fprintf (stderr, "          -m, --multicast            join the local multicast group\n");
				fprintf (stderr, "          -v, --verbose\n");
#else
				fprintf (stderr, " options: -A ADDR     local IP address or host name\n");
				fprintf (stderr, "          -P PORT     local UDP port number or service name\n");
				fprintf (stderr, "          -a ADDR     remote IP address or host name\n");
				fprintf (stderr, "          -p PORT     remote UDP port number or service name\n");
				fprintf (stderr, "          -i          redirect only standard input\n");
				fprintf (stderr, "          -o          redirect only standard output\n");
				fprintf (stderr, "          -b          use UDP broadcast mode\n");
				fprintf (stderr, "          -m          join the local multicast group\n");
				fprintf (stderr, "          -v\n");
#endif
				exit (EXIT_FAILURE);
		}
	}
	if ((! socket_2_stdout) && (! stdin_2_socket)) {
		fprintf (stderr, "You can't use both -i and -o options!\n");
		exit (EXIT_FAILURE);
	}
	socket_2_stdout = (socket_2_stdout) && (local_hostname != NULL) && (local_portname != NULL);
	stdin_2_socket =  (stdin_2_socket) && (remote_hostname != NULL) && (remote_portname != NULL);
	if (! (stdin_2_socket | socket_2_stdout)) {
		fprintf (stderr, "You must give a local or a remote address!\n");
		exit (EXIT_FAILURE);
	}
	if (multicast && (! socket_2_stdout)) {
		fprintf (stderr, "Multicast mode does not concern sending to remote address\n");
		exit (EXIT_FAILURE);
	}
	if ((my_socket = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror ("unable to get a UDP socket");
		exit (EXIT_FAILURE);
	}
	if (socket_2_stdout) {
		memset ((char *) & local_address, 0, sizeof (struct sockaddr_in));
		local_address . sin_family = AF_INET;
		if (give_ip_address (local_hostname, & (local_address . sin_addr)) < 0) {
			fprintf (stderr, "unknown local host name\n");
			exit (EXIT_FAILURE);
		}
		if ((local_address . sin_port = give_port_number (local_portname, "udp")) == 0) {
			fprintf (stderr, "unknown local port\n");
			exit (EXIT_FAILURE);
		}

		if (multicast) {
			multicast_req . imr_multiaddr . s_addr = local_address . sin_addr . s_addr;
			multicast_req . imr_interface . s_addr = htons (INADDR_ANY);
			if (setsockopt (my_socket, IPPROTO_IP, IP_ADD_MEMBERSHIP, & multicast_req, sizeof (struct ip_mreq)) < 0) {
				perror ("unable to use multicast group");
				exit (EXIT_FAILURE);
			}
		}
		if (bind (my_socket, (struct sockaddr *) & local_address, sizeof (struct sockaddr_in)) < 0) {
			perror ("unable to use the given local address");
			exit (EXIT_FAILURE);
		}
	}
	
	if (stdin_2_socket) {
		memset ((char *) & remote_address, 0, sizeof (struct sockaddr_in));
		remote_address . sin_family = AF_INET;
		if (give_ip_address (remote_hostname, & (remote_address . sin_addr)) < 0) {
			fprintf (stderr, "unknown remote host name\n");
			exit (EXIT_FAILURE);
		}
		if ((remote_address . sin_port = give_port_number (remote_portname, "udp")) == 0) {
			fprintf (stderr, "unknown remote port\n");
			exit (EXIT_FAILURE);
		}
	}

	if (broadcast) {
		sock_len = sizeof (sock_val);
		if (getsockopt (my_socket, SOL_SOCKET, SO_BROADCAST, (char *) & sock_val, & sock_len) < 0) {
			perror ("unable to read broadcast mode");
			exit (EXIT_FAILURE);
		}
		sock_val = 1;
		if (setsockopt (my_socket, SOL_SOCKET, SO_BROADCAST, (char *) & sock_val, sock_len) < 0) {
			perror ("unable to set broadcast mode");
			exit (EXIT_FAILURE);
		}
	}

	if (signal (SIGPIPE, SIG_IGN) == SIG_ERR)
		perror ("unable to ignore SIGPIPE signal");

#ifdef SETVBUF_REVERSED
	if (setvbuf (stdout, _IONBF, NULL, 0) < 0)
#else
	if (setvbuf (stdout, NULL, _IONBF, 0) < 0)
#endif
		perror ("unable to use unbuffered output mode");

	if (fcntl (STDIN_FILENO, F_SETFL, fcntl (STDIN_FILENO, F_GETFL) | O_NONBLOCK) < 0)
		perror ("unable to use non-blocking mode on stdin");
	
	while (1) {
		nb_desc = 0;
		FD_ZERO (& read_set);
		if (stdin_2_socket) {
			FD_SET (STDIN_FILENO, & read_set);
			if (STDIN_FILENO > nb_desc)
				nb_desc = STDIN_FILENO;
		}
		if (socket_2_stdout) {
			FD_SET (my_socket, & read_set);
			if (my_socket > nb_desc)
				nb_desc = my_socket;
		}
		if (select (nb_desc + 1, & read_set, NULL, NULL, NULL) < 0) {
			if (errno == EINTR)
				continue;
			perror ("error during select() system-call");
			exit (EXIT_FAILURE);
		}
		if ((socket_2_stdout) && (FD_ISSET (my_socket, & read_set))) {
			while ((nb_data = recvfrom (my_socket, data, DATA_BUFFER_LEN, 0, NULL, NULL)) < 0) {
				if (errno == EINTR)
					continue;
				perror ("error during read on socket");
				exit (EXIT_FAILURE);
			}
			if (nb_data == 0) /* EOF */
				exit (EXIT_SUCCESS);
			while (write (STDOUT_FILENO, data, nb_data) < 0) {
				if (errno == EINTR)
					continue;
				perror ("error during write on stdout");
				exit (EXIT_FAILURE);
			}
		}
		if ((stdin_2_socket) && (FD_ISSET (STDIN_FILENO, & read_set))) {
			while ((nb_data = read (STDIN_FILENO, data, DATA_BUFFER_LEN)) < 0) {
				if (errno == EINTR)
					continue;
				perror ("error during read on stdin");
				exit (EXIT_FAILURE);
			}
			if (nb_data == 0) /* EOF */
				exit (EXIT_SUCCESS);
			while (sendto (my_socket, data, nb_data, 0, (struct sockaddr *) & remote_address, sizeof (struct sockaddr_in)) < 0) {
				if (errno == EINTR)
					continue;
				if (errno == EPIPE)
					exit (EXIT_SUCCESS);
				perror ("error during write on socket");
				exit (EXIT_FAILURE);
			}
		}
	}
	exit (EXIT_SUCCESS);
}
