/************************************************************************\
**  tcp_server.c - Copy standard input to a TCP port and data received  **
**                 on this port to standard output                      **
**                                                                      **
**  Copyright (c) 2001 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/
	
	#include "config.h"

	#define _REENTRANT

	/* _XOPEN_SOURCE >= 600 needed to have rwlock features in
	  the LinuxThreads library */
	#define _XOPEN_SOURCE 600 

	#ifdef HAVE_ERRNO_H
	#  include <errno.h>
	#endif
	#ifdef HAVE_FCNTL_H
	#  include <fcntl.h>
	#endif
	#ifdef HAVE_PTHREAD_H
	#  include <pthread.h>
	#endif
	#include <signal.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#ifdef HAVE_UNISTD_H
	#  include <unistd.h>
	#endif
	#include <arpa/inet.h>
	#include <netinet/in.h>
	#include <sys/socket.h>

	#ifdef HAVE_GETOPT_H
	#  include <getopt.h>
	#endif
	
	#ifdef TIME_WITH_SYS_TIME
	#  include <sys/time.h>
	#  include <time.h>
	#else
	#  if HAVE_SYS_TIME_H
	#    include <sys/time.h>
	#  else
	#    include <time.h>
	#  endif
	#endif
	
	
	#include "netpipe.h"

	#define DATA_BUFFER_LEN 4096

	static char	data [DATA_BUFFER_LEN];
	static int	nb_data = 0;

	static int	verbose = 0;
	static int	stdin_2_socket  = 1;
	static int	socket_2_stdout = 1;

	static pthread_rwlock_t	rwlock	= PTHREAD_RWLOCK_INITIALIZER;
	static pthread_cond_t	cond	= PTHREAD_COND_INITIALIZER;
	static pthread_cond_t	ack	= PTHREAD_COND_INITIALIZER;
	static pthread_mutex_t	mutex	= PTHREAD_MUTEX_INITIALIZER;

	typedef struct {
		int		socket;
		pthread_t	thread;
	} thread_arg_t;

	static void *	thread_tcp_server	(void *);
	static void *	thread_stdin_2_buffer	(void *);
	static void *	thread_connection	(void *);
	static void *	thread_buffer_2_socket	(void *);
	static void *	thread_socket_2_stdout	(void *);
	

	static void *
thread_tcp_server (void * arg)
{
	int		my_socket = (int) arg;
	int		connected_socket;
	pthread_t	thread;
	
	while (1) {
		if ((connected_socket = accept (my_socket, NULL, NULL)) < 0)
			continue;
		if (fcntl (connected_socket, F_SETFL, fcntl (connected_socket, F_GETFL) | O_NONBLOCK) < 0)
			perror ("unable to set non-blocking mode on socket");
		if (pthread_create (& thread, NULL, thread_connection, (void *) connected_socket) != 0) {
			perror ("unable to create new thread");
			exit (EXIT_FAILURE);
		}
		pthread_detach (thread);
	}
	return (NULL);
}


	static void *
thread_stdin_2_buffer (void * unused)
{
	fd_set read_set;
	while (nb_data >= 0) {
		/* We need to be sure that data are available on stdin before issuing
		   the read(), so the rwlock is blocked as short as possible */
		FD_ZERO (& read_set);
		FD_SET (STDIN_FILENO, & read_set);
		if (select (STDIN_FILENO + 1, & read_set, NULL, NULL, NULL) < 0) {
			if (errno == EINTR)
				continue;
			perror ("error during select() system call");
			exit (EXIT_FAILURE);
		}
		if (! FD_ISSET (STDIN_FILENO, & read_set))
			continue;
		pthread_rwlock_wrlock (& rwlock);
		while ((nb_data = read (STDIN_FILENO, data, DATA_BUFFER_LEN)) < 0) {
			if (errno == EINTR)
				continue;
			perror ("error during read() system call");
			exit (EXIT_FAILURE);
		}
		if (nb_data == 0) { /* EOF */
			if (verbose)
				fprintf (stderr, "standart input closed\n");
			nb_data = -1; /* signal for others threads */
		}
		pthread_mutex_lock (& mutex);
		pthread_cond_broadcast (& cond);
		pthread_rwlock_unlock (& rwlock);
		pthread_cond_wait (& ack, & mutex);
		pthread_mutex_unlock (& mutex);
	}
	pthread_exit (NULL);
}


	static void *
thread_connection (void * arg)
{
	int 			connected_socket = (int) arg;
	struct sockaddr_in	sockaddr;
	socklen_t		sock_len;
	pthread_t		thr_buffer_2_sock;
	pthread_t		thr_sock_2_stdout;

	sock_len = sizeof (struct sockaddr_in);
	if (getpeername (connected_socket, (struct sockaddr *) & sockaddr, & sock_len) < 0) {
		perror ("unable to obtain peer name");
		pthread_exit (NULL);
	}
	if (verbose)
		fprintf (stderr, "connection from %s (%u)\n",
				inet_ntoa (sockaddr.sin_addr),
				ntohs (sockaddr.sin_port));
	if (socket_2_stdout) {
		if (pthread_create (& thr_sock_2_stdout, NULL, thread_socket_2_stdout, arg) < 0) {
			perror ("unable to create a new thread");
			close (connected_socket);
			pthread_exit (NULL);
		}
	}
	if (stdin_2_socket) {
		if (pthread_create (& thr_buffer_2_sock, NULL, thread_buffer_2_socket, arg) < 0) {
			perror ("unable to create a new thread");
			close (connected_socket);
			pthread_exit (NULL);
		}
	}
	if (socket_2_stdout) {
		pthread_join (thr_sock_2_stdout, NULL);
		if (stdin_2_socket)
			pthread_cancel (thr_buffer_2_sock);
	}
	if (stdin_2_socket) {
		pthread_join (thr_buffer_2_sock, NULL);
		if (socket_2_stdout)
			pthread_cancel (thr_sock_2_stdout);
	}
	if (verbose)
		fprintf (stderr, "end of connection from %s (%u)\n",
				inet_ntoa (sockaddr.sin_addr),
				ntohs (sockaddr.sin_port));
	close (connected_socket);
	return (NULL);
}


	static void *
thread_buffer_2_socket (void * arg)
{
	int 	connected_socket = (int) arg;
	int	nb_to_write;
	int	nb_written;

	while (1) {
		pthread_mutex_lock (& mutex);
		pthread_cleanup_push ((void (*)(void *)) pthread_mutex_unlock, & mutex);
		pthread_cond_signal (& ack); 
		pthread_cond_wait (& cond, & mutex);
		pthread_rwlock_rdlock (& rwlock);
		pthread_cleanup_pop (1); /* pthread_mutex_unlock (& mutex); */
		pthread_cleanup_push ((void (*)(void *)) pthread_rwlock_unlock, & rwlock);
		nb_to_write = nb_data; /* nb_data could be modified after releasing the rwlock */
		nb_written = 0;
		while (nb_to_write > 0) {
			while ((nb_written = write (connected_socket, & data [nb_data - nb_to_write], nb_to_write)) < 0) {
				if (errno == EINTR)
					continue;
				if (errno == EPIPE)
					break;
				perror ("error during write() on socket");
				break;
			}
			if (nb_written < 0)
				break;
			nb_to_write -= nb_written;
		}
		pthread_cleanup_pop (1); /* pthread_rwlock_unlock (& rwlock); */
		if ((nb_to_write < 0) || (nb_written < 0))
			break;
	}
	pthread_mutex_unlock (& mutex);
	return (NULL);
}


	static void *
thread_socket_2_stdout (void * arg)
{
	int 	connected_socket = (int) arg;
	char	read_buffer [DATA_BUFFER_LEN];
	int	nb_read;
	fd_set	read_set;

	while (1) {
		/*  The connected socket is in non-blocking mode, we need select() before read() */
		FD_ZERO (& read_set);
		FD_SET (connected_socket, & read_set);
		if (select (connected_socket + 1, & read_set, NULL, NULL, NULL) < 0) {
			if (errno == EINTR)
				continue;
			perror ("error during select() system-call");
			exit (EXIT_FAILURE);
		}
		while ((nb_read = read (connected_socket, read_buffer, DATA_BUFFER_LEN)) < 0) {
			if (errno == EINTR)
				continue;
			if (errno != ECONNRESET)
				perror ("error during read() system call");
			break;
		}
		if (nb_read <= 0) /* EOF */
			break;
		while (write (STDOUT_FILENO, read_buffer, nb_read) < 0) {
			if (errno == EINTR)
				continue;
			if (errno == EPIPE) {/* broken pipe: normal termination of program */
				if (verbose)
					fprintf (stderr, "standard output closed\n");
				exit (EXIT_SUCCESS);
			}
			perror ("error during write on stdout");
			exit (EXIT_FAILURE);
		}
	}
	return (NULL);
}


	int
main (int argc, char * argv [])
{

	int			option;
	char *			hostname = "localhost";
	char *			portname = "";
	char *			exec_string = NULL;

	int			my_socket;
	int			connected_socket;

	struct sockaddr_in	my_address;
	int			sock_val;
	socklen_t		sock_len;

	pthread_t		thread_stdin;
	pthread_t		thread_server;

	int			ret;
	
	while (1) {
#ifdef HAVE_GETOPT_LONG
		static struct option long_options[] = {
			{ "address",	1,	0,	'a' },
			{ "execute",	1,	0,	'e' },
			{ "port",	1,	0,	'p' },
			{ "verbose",	0,	0,	'v' },
			{ "input-only",	0,	0,	'i' },
			{ "output-only",0,	0,	'o' },
			{ NULL, 	0,	0,	0   },
		};
		option = getopt_long (argc, argv, "a:e:iop:v", long_options, NULL);
#else
		option = getopt (argc, argv, "a:e:iop:v");
#endif
		if (option == EOF)
			break;
		switch (option) {		
			case 'a':
				hostname = optarg;
				break;
			case 'e':
				exec_string = optarg;
				break;
			case 'i':
				socket_2_stdout = 0;
				break;
			case 'o':
				stdin_2_socket = 0;
				break;
			case 'p':
				portname = optarg;
				break;
			case 'v':
				fprintf (stderr, "%s (" VERSION ") Christophe Blaess 1997-2002\n", argv[0]);
				verbose ++;
				break;
			default :
				fprintf (stderr, "Usage: %s [options]\n",argv [0]);
#ifdef HAVE_GETOPT_LONG
				fprintf (stderr, " options: -a, --address ADDRESS  IP address or host name of the server\n");
				fprintf (stderr, "          -p, --port PORT        TCP port number or service name of the server\n");
				fprintf (stderr, "          -e, --execute STRING   string to execute with stdin/stdout redirected\n");
				fprintf (stderr, "          -i, --input-only       redirect only standard input\n");
				fprintf (stderr, "          -o, --output-only      redirect only standard output\n");
				fprintf (stderr, "          -v, --verbose          verbose\n");
#else
				fprintf (stderr, " options: -a ADDRESS  IP address or host name of the server\n");
				fprintf (stderr, "          -p PORT     TCP port number or service name of the server\n");
				fprintf (stderr, "          -e STRING   string to execute with stdin/stdout redirected\n");
				fprintf (stderr, "          -i          redirect only standard input\n");
				fprintf (stderr, "          -o          redirect only standard output\n");
				fprintf (stderr, "          -v          verbose\n");
#endif
				exit (EXIT_FAILURE);
		}
	}
	
	if ((! socket_2_stdout) && (! stdin_2_socket)) {
		fprintf (stderr, "You can't use both -i and -o options!\n");
		exit (EXIT_FAILURE);
	}
	memset ((char *) & my_address, 0, sizeof (my_address));
	my_address . sin_family = AF_INET;
	if (give_ip_address (hostname, & (my_address . sin_addr)) < 0) {
		fprintf (stderr, "%s: unknown host %s\n", argv [0], hostname);
		exit (EXIT_FAILURE);
	}
	if ((my_address . sin_port = give_port_number (portname, "udp")) == 0) {
		fprintf (stderr, "%s: unknown port %s\n", argv [0], portname);
		exit (EXIT_FAILURE);
	}
	if ((my_socket = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
		perror ("unable to get a TCP socket");
		exit (EXIT_FAILURE);
	}
	sock_len = sizeof (sock_val);
	if (getsockopt (my_socket, SOL_SOCKET, SO_REUSEADDR, (char *) & sock_val, & sock_len) < 0) {
		perror ("unable to reuse address");
	} else {
		sock_val = 1;
		if (setsockopt (my_socket, SOL_SOCKET, SO_REUSEADDR, (char *) & sock_val, sock_len) < 0)
			perror ("unable to reuse address");
	}	

	if (bind (my_socket, (struct sockaddr *) & my_address, sizeof (struct sockaddr_in)) <0) {
		perror ("unable to use the given adress/port");
		exit (EXIT_FAILURE);
	}
	
	listen (my_socket, 5);
	
	if (exec_string != NULL) {
		signal (SIGCHLD, SIG_IGN);
		while (1) {
			if ((connected_socket = accept (my_socket, NULL, NULL)) < 0)
				continue;
			if (fork() == 0) {
				if (socket_2_stdout)
					dup2(connected_socket, STDIN_FILENO);
				if (stdin_2_socket)
					dup2(connected_socket, STDOUT_FILENO);
				ret = system (exec_string);
				exit (ret);
			}
			close (connected_socket);
		}
	}
	
	if (signal (SIGPIPE, SIG_IGN) == SIG_ERR)
		perror ("unable to ignore SIGPIPE signal");

	pthread_create (& thread_server, NULL, thread_tcp_server, (void *) my_socket);
	if (stdin_2_socket) {
		pthread_create (& thread_stdin, NULL, thread_stdin_2_buffer, NULL);
		pthread_join (thread_stdin, NULL);
		pthread_cancel (thread_server);
	}
	pthread_exit (NULL);
}
