/************************************************************************\
**  tcp_client.c - Copy standard input to a TCP port and data received  **
**                 on this port to standard output                      **
**                                                                      **
**  Copyright (c) 2001 Christophe Blaess <ccb@club-internet.fr>         **
**    ---------------------------------------------------------------   **
**                                                                      **
** This program is free software; you can redistribute it and/or modify **
** it under the terms of the GNU General Public License as published by **
** the Free Software Foundation.                                        **
**                                                                      **
**  This program is distributed in the hope that it will be useful,     **
** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, write to the Free Software          **
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA             **
** 02111-1307  USA                                                      **
**                                                                      **
**    ---------------------------------------------------------------   **
**                                                                      **
** Ce programme est libre, vous pouvez le redistribuer et/ou le modifier**
** selon les termes de la Licence Publique Gnrale GNU publie par la  **
** Free Software Foundation.                                            **
**                                                                      **
** Ce programme est distribu car potentiellement utile, mais SANS      **
** AUCUNE GARANTIE, ni explicite ni implicite, y compris les garanties  **
** de commercialisation ou d'adaptation dans un but spcifique.         **
** Reportez-vous  la Licence Publique Gnrale GNU pour plus de dtails**
**                                                                      **
** Vous devez avoir reu une copie de la Licence Publique Gnrale GNU  **
** en mme temps que ce programme ; si ce n'est pas le cas, crivez    **
** la Free Software Foundation, Inc, 59 Temple Place, Suite 330, Boston **
** MA 02111-1307, tats-Unis.                                           **
**                                                                      **
\************************************************************************/

	#include "config.h"

	#ifdef HAVE_ERRNO_H
	#  include <errno.h>
	#endif
	#ifdef HAVE_FCNTL_H
	#  include <fcntl.h>
	#endif
	
	#include <signal.h>
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#ifdef HAVE_UNISTD_H
	#  include <unistd.h>
	#endif
	#include <netinet/in.h>
	#include <sys/socket.h>

	#ifdef HAVE_GETOPT_H
	#  include <getopt.h>
	#endif
	
	#include "netpipe.h"

	#define DATA_BUFFER_LEN 4096

	static int verbose = 0;

	int
main (int argc, char * argv [])
{
	char			data [DATA_BUFFER_LEN];
	int			nb_data;

	int			option;
	char *			hostname = "localhost";
	char *			portname = "2002";
	char *			exec_string = NULL;
	int			my_socket;
	struct sockaddr_in	srv_address;
	int			stdin_2_socket  = 1;
	int			socket_2_stdout = 1;

	int			nb_desc;
	fd_set			read_set;
	int			nb_written;
	
	while (1) {
#ifdef HAVE_GETOPT_LONG
		static struct option long_options[] = {
			{ "address",	1,	0,	'a' },
			{ "execute",	1,	0,	'e' },
			{ "port",	1,	0,	'p' },
			{ "input-only",	0,	0,	'i' },
			{ "output-only",0,	0,	'o' },
			{ "verbose",	0,	0,	'v' },
			{ NULL, 	0,	0,	0   },
		};
		option = getopt_long (argc, argv, "a:e:iop:v", long_options, NULL);
#else
		option = getopt (argc, argv, "a:e:iop:v");
#endif
		if (option == EOF)
			break;
		switch (option) {
			case 'a':
				hostname  = optarg;
				break;
			case 'e':
				exec_string = optarg;
				break;
			case 'p':
				portname = optarg;
				break;
			case 'i':
				socket_2_stdout = 0;
				break;
			case 'o':
				stdin_2_socket = 0;
				break;
			case 'v':
				fprintf (stderr, "%s (" VERSION ") Christophe Blaess 1997-2002\n", argv[0]);
				verbose ++;
				break;
			default:
				fprintf (stderr, "Usage: %s [options]\n",argv [0]);
#ifdef HAVE_GETOPT_LONG
				fprintf (stderr, " options: -a, --address ADDRESS  IP address or host name of the server\n");
				fprintf (stderr, "          -p, --port PORT        TCP port number or service name of the server\n");
				fprintf (stderr, "          -e, --execute STRING   string to execute with stdin/stdout redirected\n");
				fprintf (stderr, "          -i, --input-only       redirect only standard input\n");
				fprintf (stderr, "          -o, --output-only      redirect only standard output\n");
				fprintf (stderr, "          -v, --verbose          verbose\n");
#else
				fprintf (stderr, " options: -a ADDRESS  IP address or host name of the server\n");
				fprintf (stderr, "          -p PORT     TCP port number or service name of the server\n");
				fprintf (stderr, "          -e STRING   string to execute with stdin/stdout redirected\n");
				fprintf (stderr, "          -i          redirect only standard input\n");
				fprintf (stderr, "          -o          redirect only standard output\n");
				fprintf (stderr, "          -v          verbose\n");
#endif
				exit (EXIT_FAILURE);
		}
	}
	if ((! socket_2_stdout) && (! stdin_2_socket)) {
		fprintf (stderr, "You can't use both -i and -o options!\n");
		exit (EXIT_FAILURE);
	}
	memset ((char *) & srv_address, 0, sizeof (struct sockaddr_in));
	srv_address . sin_family = AF_INET;
	if (give_ip_address (hostname, & (srv_address . sin_addr)) < 0) {
		perror ("unknown server host name");
		exit (EXIT_FAILURE);
	}
	if ((srv_address . sin_port = give_port_number (portname, "tcp")) == 0) {
		perror ("unknown server port");
		exit (EXIT_FAILURE);
	}
	if ((my_socket = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
		perror ("unable to get a TCP socket");
		exit (EXIT_FAILURE);
	}
	if (connect (my_socket, (struct sockaddr *) & srv_address, sizeof (srv_address)) < 0) {
		perror ("unable to connect to server");
		exit (EXIT_FAILURE);
	}

	if (exec_string != NULL) {
		if (socket_2_stdout)
			dup2(my_socket, STDIN_FILENO);
		if (stdin_2_socket)
			dup2(my_socket, STDOUT_FILENO);
		exit (system (exec_string));
	}

	signal (SIGPIPE, SIG_IGN);
#ifdef SETVBUF_REVERSED
	setvbuf (stdout, _IONBF, NULL, 0);
#else
	setvbuf (stdout, NULL, _IONBF, 0);
#endif
	if (fcntl (my_socket, F_SETFL, fcntl (my_socket, F_GETFL) | O_NONBLOCK) < 0) {
		perror ("unable to use non-blocking mode on socket");
	}
	while (1) {
		FD_ZERO (& read_set);
		nb_desc = 0;
		if (socket_2_stdout) {
			FD_SET (my_socket, & read_set);
			if (my_socket > nb_desc)
				nb_desc = my_socket;
		}
		if (stdin_2_socket) {
			FD_SET (STDIN_FILENO, & read_set);
			if (STDIN_FILENO > nb_desc)
				nb_desc = STDIN_FILENO;
		}
		if (select (nb_desc + 1, & read_set, NULL, NULL, NULL) < 0) {
			if (errno == EINTR)
				continue;
			perror ("error during select() system-call");
			exit (EXIT_FAILURE);
		}
		if ((socket_2_stdout) && (FD_ISSET (my_socket, & read_set))) {
			while ((nb_data = read (my_socket, data, DATA_BUFFER_LEN)) < 0) {
				if (errno == EINTR)
					continue;
				perror ("error during read on socket");
				exit (EXIT_FAILURE);
			}
			if (nb_data == 0) {
				if (verbose)
					fprintf (stderr, "socket closed on input\n");
				/* EOF normal termination */
				exit (EXIT_SUCCESS);
			}
			while (write (STDOUT_FILENO, data, nb_data) < 0) {
				if (errno == EINTR)
					continue;
				if (errno == EPIPE) {
					if (verbose)
						fprintf (stderr, "stdout closed\n");
					exit (EXIT_SUCCESS);
				}
				perror ("error during write on stdout");
				exit (EXIT_FAILURE);
			}
		}
		if ((stdin_2_socket) && (FD_ISSET (STDIN_FILENO, & read_set))) {
			while ((nb_data = read (STDIN_FILENO, data, DATA_BUFFER_LEN)) < 0) {
				if (errno == EINTR)
					continue;
				perror ("error during read on stdin");
				exit (EXIT_FAILURE);
			}
			if (nb_data == 0) {
				if (verbose)
					fprintf (stderr, "stdin closed\n");
				exit (EXIT_SUCCESS);
			}
			fcntl (my_socket, F_SETFL, fcntl (my_socket, F_GETFL) & ~O_NONBLOCK);
			while ((nb_written = write (my_socket, data, nb_data)) < 0) {
				if (errno == EINTR)
					continue;
				if (errno == EPIPE) {
					if (verbose)
						fprintf (stderr, "socket closed on output\n");
					exit (EXIT_SUCCESS);
				}
				perror ("error during write on socket");
				break;
			}
			fcntl (my_socket, F_SETFL, fcntl (my_socket, F_GETFL) | O_NONBLOCK);
		}
	}
}
