/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1991, 1992, 1993, 1995, 1996, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

#include <ac/stdio.h>

#include <arglex.h>
#include <error.h>
#include <mem.h>
#include <reverse.h>
#include <srrf/address.h>
#include <srrf/reader.h>
#include <srrf/origin.h>

int	verbose;
long	net_mask;
long	network;
int	delete_foreign_names;


static long network_to_netmask _((long));

static long
network_to_netmask(n)
	long		n;
{
	int		nn;

	nn = (n >> 24) & 0xFF;
	if (nn < 128)
		return 0xFF000000;
	if (nn < 192)
		return 0xFFFF0000;
	return 0xFFFFFF00;
}


void
reverse(infile, outfile)
	char		*infile;
	char		*outfile;
{
	FILE		*fp;
	srrf_t		*rp;
	srrf_list_ty	*rlp;
	size_t		j;
	static char	stdout_name[] = "(stdout)";
	srrf_class_ty	*in_class;
	srrf_type_ty	*in_soa_type;
	srrf_type_ty	*in_ns_type;
	srrf_type_ty	*in_a_type;
	srrf_type_ty	*in_ptr_type;
	string_ty	*s;
	srrf_t		*rp2;

	/*
	 * find the magic stuff
	 */
	in_class = srrf_class_by_name("in");
	assert(in_class);
	in_soa_type = srrf_type_by_name(in_class, "soa");
	assert(in_soa_type);
	in_ns_type = srrf_type_by_name(in_class, "ns");
	assert(in_ns_type);
	in_a_type = srrf_type_by_name(in_class, "a");
	assert(in_a_type);
	in_ptr_type = srrf_type_by_name(in_class, "ptr");
	assert(in_ptr_type);

	/*
	 * read the input file
	 */
	rlp = srrf_reader(infile, verbose, delete_foreign_names);

	/*
	 * open the output file
	 */
	if (outfile)
	{
		fp = fopen(outfile, "w");
		if (!fp)
			nfatal("creat \"%s\"", outfile);
	}
	else
	{
		outfile = stdout_name;
		fp = stdout;
	}

	/*
	 * scan the resources to determine the network address
	 */
	if (!network)
	{
		for (j = 0; j < rlp->nrecords; ++j)
		{
			long	tmp;
	
			rp = rlp->record[j];
			if (rp->type != in_a_type)
				continue;
			tmp = srrf_address(rp->arg.string[0]->str_text);
			if (!tmp)
				fatal("can't cope with zero address");
			if (!network)
			{
				if (!net_mask)
					net_mask = network_to_netmask(tmp);
				network = tmp & net_mask;
			}
			else
			{
				if ((network & net_mask) != (tmp & net_mask))
					fatal("%s: %d: all addresses must be in the same network", rp->file_name->str_text, rp->line_number);
			}
		}
		if (!network)
			fatal("%s: no addresses", infile);
	}
	if (!net_mask)
		net_mask = network_to_netmask(network);

	/*
	 * print the warning
	 */
	fprintf(fp, "; Do not edit this file.  It is generated\n");
	fprintf
	(
		fp,
		"; by a '%s %s %s' command.\n",
		progname,
		(infile ? infile : "-"),
		((outfile == stdout_name) ? "-" : outfile)
	);

	/*
	 * work out the origin
	 */
	srrf_origin_set((string_ty *)0);
	if (net_mask == 0xFFFFFF00)
	{
		s =
			str_format
			(
				"%d.%d.%d.in-addr.arpa.",
				(unsigned char)(network >> 8),
				(unsigned char)(network >> 16),
				(unsigned char)(network >> 24)
			);
		srrf_origin_set(s);
		str_free(s);
	}
	else if (net_mask == 0xFFFF0000)
	{
		s =
			str_format
			(
				"%d.%d.in-addr.arpa.",
				(unsigned char)(network >> 16),
				(unsigned char)(network >> 24)
			);
		srrf_origin_set(s);
		str_free(s);
	}
	else if (net_mask == 0xFF000000)
	{
		s =
			str_format
			(
				"%d.in-addr.arpa.",
				(unsigned char)(network >> 24)
			);
		srrf_origin_set(s);
		str_free(s);
	}
	else
	{
		fatal
		(
			"net mask %d.%d.%d.%d is bogus",
			(int)((net_mask >> 24) & 0xFF),
			(int)((net_mask >> 16) & 0xFF),
			(int)((net_mask >> 8) & 0xFF),
			(int)(net_mask & 0xFF)
		);
	}
	srrf_origin_print(fp);

	/*
	 * find the soa resource and use bits of it
	 */
	rp = 0;
	for (j = 0; j < rlp->nrecords; ++j)
	{
		rp = rlp->record[j];
		if (rp->type == in_soa_type)
			break;
		rp = 0;
	}
	if (!rp)
		fatal("%s: no 'soa' resource", infile);
	rp2 = srrf_alloc();
	assert(srrf_origin_get());
	rp2->name = str_copy(srrf_origin_get());
	rp2->class = in_class;
	rp2->type = in_soa_type;
	for (j = 0; j < rp->arg.nstrings; ++j)
		strlist_append(&rp2->arg, rp->arg.string[j]);
	if (rp->file_name)
		rp2->file_name = str_copy(rp->file_name);
	rp2->line_number = rp->line_number;
	srrf_print(fp, rp2);
	srrf_free(rp2);

	/*
	 * scan through the resources (again!)
	 * and print the 'ptr' resources
	 */
	for (j = 0; j < rlp->nrecords; ++j)
	{
		rp = rlp->record[j];
		if (rp->type == in_ptr_type)
		{
			fatal
			(
			      "%s: %d: forward map should not have PTR records",
				rp->file_name->str_text,
				rp->line_number
			);
		}
		if (rp->type == in_a_type)
		{
			long	tmp;

			tmp = srrf_address(rp->arg.string[0]->str_text);
			if ((tmp & net_mask) == network)
			{
				rp2 = srrf_alloc();
				rp2->name =
					str_format
					(
						"%d.%d.%d.%d.in-addr.arpa.",
						(int)(tmp & 0xFF),
						(int)((tmp >> 8) & 0xFF),
						(int)((tmp >> 16) & 0xFF),
						(int)((tmp >> 24) & 0xFF)
					);
				rp2->class = in_class;
				rp2->type = in_ptr_type;
				strlist_append(&rp2->arg, rp->name);
				if (rp->file_name)
					rp2->file_name = str_copy(rp->file_name);
				rp2->line_number = rp->line_number;
				srrf_print(fp, rp2);
				srrf_free(rp2);
			}
		}
		else if (rp->type == in_ns_type)
		{
			rp2 = srrf_alloc();
			rp2->name = str_copy(srrf_origin_get());
			rp2->class = in_class;
			rp2->type = in_ns_type;
			strlist_append(&rp2->arg, rp->arg.string[0]);
			if (rp->file_name)
				rp2->file_name = str_copy(rp->file_name);
			rp2->line_number = rp->line_number;
			srrf_print(fp, rp2);
			srrf_free(rp2);
		}
	}
	srrf_list_free(rlp);

	/*
	 * close the output file
	 */
	if (fflush(fp))
		nfatal("write \"%s\"", outfile);
	if (fp != stdout && fclose(fp))
		nfatal("close \"%s\"", outfile);
}
