/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1992, 1993, 1995, 1996, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

#include <ac/ctype.h>
#include <ac/stdio.h>
#include <ac/string.h>

#include <arglex.h>
#include <error.h>
#include <mem.h>
#include <netgroup.h>
#include <regu_expr.h>
#include <srrf/reader.h>
#include <symtab.h>


typedef struct pattern pattern;
struct pattern
{
	char		*name;
	char		*pattern;
	srrf_list_ty	*list;
};

int		verbose;
int		delete_foreign_names;
static size_t	npat;
static pattern	*pat;


static string_ty *first_part _((string_ty *));

static string_ty *
first_part(s)
	string_ty	*s;
{
	char		*dot;

	dot = strchr(s->str_text, '.');
	if (!dot)
		return str_from_c("");
	return str_n_from_c(s->str_text, dot - s->str_text);
}


static string_ty *no_trailing_dot _((string_ty *));

static string_ty *
no_trailing_dot(s)
	string_ty	*s;
{
	if (s->str_length < 2 || s->str_text[s->str_length - 1] != '.')
		return str_copy(s);
	return str_n_from_c(s->str_text, s->str_length - 1);
}


static void emit _((FILE *, char *, size_t, srrf_t **));

static void
emit(fp, name, nres, res)
	FILE	*fp;
	char	*name;
	size_t	nres;
	srrf_t	**res;
{
	if (nres > 10)
	{
		long	nsub;
		long	incr;
		int	j;

		nsub = (nres + 9) / 10;
		if (nsub > 10)
			nsub = 10;
		incr = (nres + nsub - 1) / nsub;
		for (j = 0; j < nsub; ++j)
		{
			char	name2[100];
			long	base;
			long	n;
			long	len;

			strcpy(name2, name);
			len = strlen(name2);
			if (!isdigit(name2[len - 1]))
				name2[len++] = '-';
			name2[len++] = '0' + j;
			name2[len] = 0;
			base = j * incr;
			n = nres - base;
			if (n > incr)
				n = incr;
			emit(fp, name2, n, res + base);
		}

		fprintf(fp, "%s", name);
		for (j = 0; j < nsub; ++j)
		{
			char name2[100];
			long len;

			strcpy(name2, name);
			len = strlen(name2);
			if (!isdigit(name2[len - 1]))
				name2[len++] = '-';
			name2[len++] = '0' + j;
			name2[len] = 0;
			fprintf(fp, " %s", name2);
		}
		fprintf(fp, "\n");
	}
	else
	{
		int	k;

		fprintf(fp, "%s", name);
		for (k = 0; k < nres; ++k)
		{
			string_ty	*tmp;
	
			tmp = first_part(res[k]->name);
			fprintf(fp, " %s", tmp->str_text);
			str_free(tmp);
		}
		fprintf(fp, "\n");
	}
}


static void cname_reap _((void *));

static void
cname_reap(p)
	void		*p;
{
	strlist_ty	*slp;

	slp = p;
	strlist_free(slp);
	mem_free(slp);
}


void
netgroup(infile, outfile)
	char		*infile;
	char		*outfile;
{
	FILE		*fp;
	srrf_t		*rp;
	srrf_list_ty	*rlp;
	size_t		j, k;
	srrf_t		*kp;
	string_ty	*tmp;
	string_ty	*tmp2;
	srrf_class_ty	*in_class;
	srrf_type_ty	*in_a_type;
	srrf_type_ty	*in_cname_type;
	srrf_type_ty	*in_hinfo_type;
	symtab_ty	*cname_stp;
	symtab_ty	*hinfo_stp;
	strlist_ty	*slp;

	/*
	 * find the magic stuff
	 */
	in_class = srrf_class_by_name("in");
	assert(in_class);
	in_a_type = srrf_type_by_name(in_class, "a");
	assert(in_a_type);
	in_cname_type = srrf_type_by_name(in_class, "cname");
	assert(in_cname_type);
	in_hinfo_type = srrf_type_by_name(in_class, "hinfo");
	assert(in_hinfo_type);

	/*
	 * read the input file
	 */
	rlp = srrf_reader(infile, verbose, delete_foreign_names);

	/*
	 * remember cname and hinfo records
	 */
	hinfo_stp = symtab_alloc(5);
	cname_stp = symtab_alloc(5);
	cname_stp->reap = cname_reap;
	for (j = 0; j < rlp->nrecords; ++j)
	{
		rp = rlp->record[j];

		if (rp->type == in_hinfo_type)
			symtab_assign(hinfo_stp, rp->name, rp);
		if (rp->type == in_cname_type)
		{
			slp = symtab_query(cname_stp, rp->arg.string[0]);
			if (!slp)
			{
				slp = mem_alloc(sizeof(strlist_ty));
				strlist_zero(slp);
				symtab_assign(cname_stp, rp->arg.string[0], slp);
			}
			strlist_append_unique(slp, rp->name);
		}
	}

	/*
	 * open the output file
	 */
	if (outfile)
	{
		fp = fopen(outfile, "w");
		if (!fp)
			nfatal("creat \"%s\"", outfile);
	}
	else
	{
		outfile = "(stdout)";
		fp = stdout;
	}

	/*
	 * scan the resources looking for hosts
	 */
	for (j = 0; j < rlp->nrecords; ++j)
	{
		int	tabbed;

		rp = rlp->record[j];
		if (rp->type != in_a_type)
			continue;

		/*
		 * Use the host name as the netgroup name,
		 * include the host in the netgroup,
		 * include the fully qualified domain name in the netgroup.
		 */
		tmp = first_part(rp->name);
		fprintf(fp, "%s (%s,-,)", tmp->str_text, tmp->str_text);
		tmp2 = no_trailing_dot(rp->name);
		if (!str_equal(tmp, tmp2))
		{
			fprintf(fp, " (%s,-,)", tmp2->str_text);
			tabbed = 1;
		}
		str_free(tmp);
		str_free(tmp2);

		/*
		 * search for aliases
		 */
		slp = symtab_query(cname_stp, rp->name);
		if (slp)
		{
			for (k = 0; k < slp->nstrings; ++k)
			{
				string_ty	*name;

				name = slp->string[k];
				tmp = first_part(name);
				fprintf(fp, " (%s,-,)", tmp->str_text);
				tmp2 = no_trailing_dot(name);
				if (!str_equal(tmp, tmp2))
					fprintf(fp, " (%s,-,)", tmp2->str_text);
				str_free(tmp);
				str_free(tmp2);
			}
		}
		fprintf(fp, "\n");
		if (ferror(fp))
			nfatal("write \"%s\"", outfile);

		/*
		 * search for host info
		 *	(only use the first one)
		 */
		kp = symtab_query(hinfo_stp, rp->name);
		if (kp)
		{
			size_t	m;

			/*
			 * found a matching HINFO resource,
			 * now see if it fits any pattern.
			 */
			for (m = 0; m < npat; ++m)
			{
				pattern	*pp;

				pp = &pat[m];
				if
				(
					regular_expression_match
					(
						pp->pattern,
						kp->arg.string[0]->str_text
					)
				||
					regular_expression_match
					(
						pp->pattern,
						kp->arg.string[1]->str_text
					)
				)
				{
					srrf_list_append(pp->list, rp);
					break;
				}
			}
		}
	}

	/*
	 * list any pattern matches
	 *
	 * may need to break into sub-groups
	 * so that the groups never grow too large.
	 */
	for (j = 0; j < npat; ++j)
	{
		pattern	*pp;

		pp = &pat[j];
		if (!pp->list->nrecords)
			continue;
		emit(fp, pp->name, pp->list->nrecords, pp->list->record);
		if (ferror(fp))
			nfatal("write \"%s\"", outfile);
	}

	/*
	 * close the output file
	 */
	if (fflush(fp))
		nfatal("write \"%s\"", outfile);
	if (fp != stdout && fclose(fp))
		nfatal("close \"%s\"", outfile);
}


void
netgroup_pattern(name, re)
	char		*name;
	char		*re;
{
	size_t		nbytes;
	pattern		*pp;

	nbytes = (npat + 1) * sizeof(*pat);
	pat = mem_change_size(pat, nbytes);
	pp = pat + npat++;
	pp->name = name;
	pp->pattern = re;
	pp->list = srrf_list_alloc();
}
