/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1995, 1996, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to filter DNS database tables
 */

#include <ac/stdio.h>
#include <ac/string.h>

#include <error.h>
#include <filter.h>
#include <mem.h>
#include <srrf.h>
#include <srrf/origin.h>

static	int		no_line_numbers;
static	size_t		class_pass_count;
static	srrf_class_ty	**class_pass;
static	size_t		class_stop_count;
static	srrf_class_ty	**class_stop;
static	size_t		type_pass_count;
static	srrf_type_ty	**type_pass;
static	size_t		type_stop_count;
static	srrf_type_ty	**type_stop;
static	int		delete_foreign_names;
static	int		verbose;


static void fix_line_number _((FILE *, string_ty **, int *, string_ty *, int));

static void
fix_line_number(fp, fn1, lino1, fn2, lino2)
	FILE		*fp;
	string_ty	**fn1;
	int		*lino1;
	string_ty	*fn2;
	int		lino2;
{
	if (!*fn1)
	{
		*fn1 = str_copy(fn2);
		*lino1 = lino2;
		goto print;
	}
	if (!str_equal(*fn1, fn2))
	{
		str_free(*fn1);
		*fn1 = str_copy(fn2);
		*lino1 = lino2;
		goto print;
	}
	if (*lino1 == lino2)
		return;
	if (*lino1 > lino2 - 10 && *lino1 < lino2)
	{
		while (*lino1 < lino2)
		{
			fprintf(fp, "\n");
			++*lino1;
		}
		return;
	}
	*lino1 = lino2;
	print:
	fprintf(fp, "$line %d \"%s\"\n", *lino1, (*fn1)->str_text);
}


void
filter(infile, outfile)
	char		*infile;
	char		*outfile;
{
	FILE		*fp;
	string_ty	*file_name;
	int		line_number;
	size_t		j;
	int		ok;
	srrf_t		*rp;
	srrf_class_ty	*in_class;
	srrf_type_ty	*in_soa_type;

	/*
	 * find the magic stuff
	 */
	in_class = srrf_class_by_name("in");
	assert(in_class);
	in_soa_type = srrf_type_by_name(in_class, "soa");
	assert(in_soa_type);

	/*
	 * open the input file
	 */
	srrf_open(infile);

	/*
	 * open the output file
	 */
	if (outfile)
	{
		fp = fopen(outfile, "w");
		if (!fp)
			nfatal("open \"%s\"", outfile);
	}
	else
	{
		outfile = "(stdout)";
		fp = stdout;
	}

	/*
	 * read all of the resources and either print them or drop them
	 *	(comments are lost)
	 */
	file_name = 0;
	line_number = 1;
	for (;;)
	{
		if (delete_foreign_names)
			rp = srrf_read_dfn();
		else
			rp = srrf_read();
		if (!rp)
			break;
		if (verbose)
			srrf_print(stderr, rp);
		
		/*
		 * print the origin when see soa record
		 */
		if (rp->type == in_soa_type)
		{
			srrf_origin_print(fp);
			++line_number;
		}

		/*
		 * see if the record is OK to be printed
		 */
		ok = 1;
		if (class_pass_count)
		{
			for (j = 0; j < class_pass_count; ++j)
				if (rp->class == class_pass[j])
					break;
			if (j >= class_pass_count)
				ok = 0;
		}
		if (ok)
		{
			for (j = 0; j < class_stop_count; ++j)
			{
				if (rp->class == class_stop[j])
				{
					ok = 0;
					break;
				}
			}
		}
		if (ok && type_pass_count)
		{
			for (j = 0; j < type_pass_count; ++j)
				if (rp->type == type_pass[j])
					break;
			if (j >= type_pass_count)
				ok = 0;
		}
		if (ok)
		{
			for (j = 0; j < type_stop_count; ++j)
			{
				if (rp->type == type_stop[j])
				{
					ok = 0;
					break;
				}
			}
		}

		/*
		 * print all acceptable reords
		 */
		if (ok)
		{
			int		nlines;

			if (!no_line_numbers)
			{
				if (rp->type == in_soa_type)
				{
					nlines = srrf_print_config(fp);
					line_number += nlines;
				}
				fix_line_number
				(
					fp,
					&file_name,
					&line_number,
					rp->file_name,
					rp->line_number
				);
			}
			nlines = srrf_print(fp, rp);
			line_number += nlines;
		}
		srrf_free(rp);
	}
	if (file_name)
		str_free(file_name);

	/*
	 * close the input file
	 */
	srrf_close();

	/*
	 * close the output file
	 */
	if (fflush(fp))
		nfatal("write \"%s\"", outfile);
	if (fp != stdout && fclose(fp))
		nfatal("close \"%s\"", outfile);
}


void
filter_delete_foreign_names()
{
	delete_foreign_names = 1;
}


void
filter_verbose()
{
	verbose = 1;
}


void
filter_no_line_numbers()
{
	no_line_numbers = 1;
}


void
filter_allow_class(name)
	char		*name;
{
	srrf_class_ty	*n;
	size_t		nbytes;

	n = srrf_class_by_name(name);
	if (!n)
		fatal("class \"%s\" unknown", name);
	nbytes = (class_pass_count + 1) * sizeof(*class_pass);
	class_pass = mem_change_size(class_pass, nbytes);
	class_pass[class_pass_count++] = n;
}


void
filter_delete_class(name)
	char		*name;
{
	srrf_class_ty	*n;
	size_t		nbytes;

	n = srrf_class_by_name(name);
	if (!n)
		fatal("class \"%s\" unknown", name);
	nbytes = (class_stop_count + 1) * sizeof(*class_stop);
	class_stop = mem_change_size(class_stop, nbytes);
	class_stop[class_stop_count++] = n;
}


static void split_into_class_type_pair _((char *, char **, char **));

static void
split_into_class_type_pair(s, class_p, type_p)
	char		*s;
	char		**class_p;
	char		**type_p;
{
	char		*cp;

	cp = strchr(s, ',');
	if (!cp)
		fatal("types must be specified as \"class,type\" pairs, because types tend to be class specific");
	*cp = 0;
	*class_p = s;
	*type_p = cp + 1;
}


void
filter_allow_type(name)
	char		*name;
{
	char		*class_name;
	srrf_class_ty	*class;
	char		*type_name;
	srrf_type_ty	*type;
	size_t		nbytes;

	split_into_class_type_pair(name, &class_name, &type_name);
	class = srrf_class_by_name(class_name);
	if (!class)
		fatal("class \"%s\" unknown in \"%s\"", class_name, name);
	type = srrf_type_by_name(class, type_name);
	if (!type)
		fatal("type \"%s\" unknown in \"%s\"", type_name, name);
	nbytes = (type_pass_count + 1) * sizeof(*type_pass);
	type_pass = mem_change_size(type_pass, nbytes);
	type_pass[type_pass_count++] = type;
}


void
filter_delete_type(name)
	char		*name;
{
	char		*class_name;
	srrf_class_ty	*class;
	char		*type_name;
	srrf_type_ty	*type;
	size_t		nbytes;

	split_into_class_type_pair(name, &class_name, &type_name);
	class = srrf_class_by_name(class_name);
	if (!class)
		fatal("class \"%s\" unknown in \"%s\"", class_name, name);
	type = srrf_type_by_name(class, type_name);
	if (!type)
		fatal("type \"%s\" unknown in \"%s\"", type_name, name);
	nbytes = (type_stop_count + 1) * sizeof(*type_stop);
	type_stop = mem_change_size(type_stop, nbytes);
	type_stop[type_stop_count++] = type;
}
