/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1996, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to filter bootparam entries
 */

#include <ac/ctype.h>
#include <ac/stdio.h>
#include <ac/string.h>

#include <arglex.h>
#include <error.h>
#include <filter.h>
#include <mem.h>
#include <output.h>
#include <srrf/reader.h>
#include <srrf/bootparams.h>
#include <symtab.h>


int verbose;


static string_ty *first_part _((string_ty *));

static string_ty *
first_part(s)
	string_ty	*s;
{
	char		*dot;

	dot = strchr(s->str_text, '.');
	if (!dot)
		return str_from_c("");
	return str_n_from_c(s->str_text, dot - s->str_text);
}


static void load_the_defaults _((symtab_ty *, string_ty *, void *, void *));

static void
load_the_defaults(stp, key, data, arg)
	symtab_ty	*stp;
	string_ty	*key;
	void		*data;
	void		*arg;
{
	symtab_ty	*stp2;
	void		*data2;

	stp2 = arg;
	data2 = symtab_query(stp2, key);
	if (!data2)
		symtab_assign(stp2, key, data);
}


static srrf_t *hinfo_to_aarch _((srrf_t *));

static srrf_t *
hinfo_to_aarch(rp)
	srrf_t		*rp;
{
	typedef struct map_ty map_ty;
	struct map_ty
	{
		char	*prefix;
		char	*aarch;
	};

	static map_ty map[] =
	{
		{ "SUN-4/", "sparc", },
		{ "IBM-PC", "i386", },
	};

	map_ty		*mp;
	string_ty	*s;
	srrf_t		*rp2;

	/*
	 * look at the hardware hinfo field
	 */
	s = str_upcase(rp->arg.string[0]);
	for (mp = map; mp < ENDOF(map); ++mp)
	{
		if (0 == memcmp(s->str_text, mp->prefix, strlen(mp->prefix)))
			break;
	}
	str_free(s);
	if (mp >= ENDOF(map))
		return 0;

	/*
	 * build a bootparam aarch row
	 */
	rp2 = srrf_alloc();
	rp2->name = str_copy(rp->name);
	rp2->class = srrf_class_by_name("bootparam");
	assert(rp2->class);
	rp2->type = srrf_type_by_name(rp2->class, "aarch");
	assert(rp2->type);
	if (rp->file_name)
		rp2->file_name = str_copy(rp->file_name);
	rp2->line_number = rp->line_number;
	s = str_from_c(mp->aarch);
	strlist_append(&rp2->arg, s);
	str_free(s);

	/*
	 * return the constructed row
	 */
	return rp2;
}


void
filter(infile, outfile)
	char		*infile;
	char		*outfile;
{
	srrf_list_ty	*rlp;
	size_t		j, k;
	srrf_class_ty	*bootparam_class;
	srrf_class_ty	*ether_class;
	srrf_class_ty	*in_class;
	srrf_type_ty	*in_a_type;
	srrf_type_ty	*in_cname_type;
	srrf_type_ty	*in_hinfo_type;
	srrf_type_ty	*ether_a_type;
	symtab_ty	*defaults_stp;
	srrf_bootparam_aux_ty aux;
	static string_ty *Name;

	/*
	 * read the input file
	 */
	rlp = srrf_reader(infile, verbose, 1);

	/*
	 * open the output file
	 */
	output_open(outfile);

	/*
	 * find the necessary type pointer
	 */
	bootparam_class = srrf_class_by_name("bootparam");
	assert(bootparam_class);
	ether_class = srrf_class_by_name("ether");
	assert(ether_class);
	in_class = srrf_class_by_name("in");
	assert(in_class);
	ether_a_type = srrf_type_by_name(ether_class, "a");
	assert(ether_a_type);
	in_a_type = srrf_type_by_name(in_class, "a");
	assert(in_a_type);
	in_cname_type = srrf_type_by_name(in_class, "cname");
	assert(in_cname_type);
	in_hinfo_type = srrf_type_by_name(in_class, "hinfo");
	assert(in_hinfo_type);

	/*
	 * warning comment at the top of the file
	 */
	output_printf("# Do not edit this file.  It is generated\n");
	output_printf
	(
		"# using a '%s %s %s' command.\n",
		progname,
		(infile ? infile : "-"),
		(outfile ? outfile : "-")
	);

	/*
	 * scan the resources looking for hinfo
	 */
	aux.in_hinfo_stp = symtab_alloc(5);
	aux.in_a_stp = symtab_alloc(5);
	aux.name_stp = symtab_alloc(5);
	aux.ether_a_stp = symtab_alloc(5);
	for (j = 0; j < rlp->nrecords; ++j)
	{
		srrf_t		*rp;

		rp = rlp->record[j];
		if (rp->type == in_hinfo_type)
			symtab_assign(aux.in_hinfo_stp, rp->name, rp);
		if (rp->type == in_a_type)
			symtab_assign(aux.in_a_stp, rp->name, rp);
		if (rp->type == ether_a_type)
			symtab_assign(aux.ether_a_stp, rp->name, rp);
		if (rp->type == in_a_type || rp->type == in_cname_type)
			symtab_assign(aux.name_stp, rp->name, rp);
	}

	/*
	 * scan the resources looking for hosts
	 */
	defaults_stp = symtab_alloc(5);
	for (j = 0; j < rlp->nrecords; ++j)
	{
		srrf_t		*rp;
		symtab_ty	*stp;
		int		number_of_params;
		static string_ty *aarch;

		rp = rlp->record[j];
		if
		(
			rp->class == bootparam_class
		&&
			rp->name->str_text[0] == '*'
		&&
			rp->name->str_text[1] == '.'
		)
		{
			string_ty	*name;

			/*
			 * set a defailt
			 */
			name = str_from_c(rp->type->name);
			symtab_assign(defaults_stp, name, rp);
			str_free(name);
		}
		if (rp->type != in_a_type)
			continue;

		/*
		 * Write out all of the associated bootparam records, if any.
		 */
		number_of_params = 0;
		aux.substitute = symtab_alloc(5);
		stp = symtab_alloc(5);
		for (k = 0; k < rlp->nrecords; ++k)
		{
			srrf_t		*rp2;
			string_ty	*name;

			rp2 = rlp->record[k];
			if (!str_equal(rp2->name, rp->name))
				continue;
			if (rp2->class != bootparam_class)
				continue;

			name = str_from_c(rp2->type->name);
			symtab_assign(stp, name, rp2);
			if (rp2->arg.nstrings == 1)
				symtab_assign(aux.substitute, name, rp2);
			str_free(name);
			++number_of_params;
		}

		/*
		 * The hardware architecture can be determined from the
		 * hinfo, sometimes.  This is because bootparams is
		 * unique to Sun, mostly.
		 */
		if (!aarch)
			aarch = str_from_c("aarch");
		if (number_of_params > 0 && !symtab_query(stp, aarch))
		{
			srrf_t		*hi_rp;
			srrf_t		*ha_rp;

			hi_rp = symtab_query(aux.in_hinfo_stp, rp->name);
			if (hi_rp)
			{
				ha_rp = hinfo_to_aarch(hi_rp);
				if (ha_rp)
				{
					string_ty	*name;

					name = str_from_c(ha_rp->type->name);
					symtab_assign(stp, name, ha_rp);
					symtab_assign(aux.substitute, name, ha_rp);
					str_free(name);
					++number_of_params;
				}
			}
		}

		/*
		 * output the records
		 */
		if (number_of_params > 0)
		{
			string_ty	*tmp;
			strlist_ty	key;

			if (!symtab_query(aux.ether_a_stp, rp->name))
			{
				output_error_locn
				(
					rp->file_name->str_text,
					rp->line_number,
				   "host \"%s\" does not have an ether address",
					rp->name->str_text
				);
			}

			symtab_walk(defaults_stp, load_the_defaults, stp);
			tmp = first_part(rp->name);
			output_printf("%s", tmp->str_text);
			symtab_keys(stp, &key);
			strlist_sort(&key);
			if (!Name)
				Name = str_from_c("name");
			symtab_assign(stp, Name, tmp);
			for (k = 0; k < key.nstrings; ++k)
			{
				srrf_t	*rp2;

				rp2 = symtab_query(stp, key.string[k]);
				assert(rp2);
				output_printf(" \\\n\t");
				srrf_invoke_aux1(rp2, &aux);
			}
			strlist_free(&key);
			symtab_free(stp);
			symtab_free(aux.substitute);
			aux.substitute = 0;
			output_printf("\n");
		}
	}
	symtab_free(defaults_stp);
	srrf_list_free(rlp);

	/*
	 * close the output file
	 */
	output_close();
}
