/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1996, 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to filter bootp entries
 */

#include <ac/ctype.h>
#include <ac/string.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <error.h>
#include <filter.h>
#include <mem.h>
#include <output.h>
#include <srrf/reader.h>
#include <srrf/bootp.h>
#include <symtab.h>


int verbose;


static string_ty *no_trailing_dot _((string_ty *));

static string_ty *
no_trailing_dot(s)
	string_ty	*s;
{
	if (s->str_length >= 2 && s->str_text[s->str_length - 1] == '.')
		return str_n_from_c(s->str_text, s->str_length - 1);
	return str_copy(s);
}


static void load_the_defaults _((symtab_ty *, string_ty *, void *, void *));

static void
load_the_defaults(stp, key, data, arg)
	symtab_ty	*stp;
	string_ty	*key;
	void		*data;
	void		*arg;
{
	symtab_ty	*stp2;
	void		*data2;

	stp2 = arg;
	data2 = symtab_query(stp2, key);
	if (!data2)
		symtab_assign(stp2, key, data);
}


static srrf_t *bootp_ds_from_in_ns _((strlist_ty *));

static srrf_t *
bootp_ds_from_in_ns(slp)
	strlist_ty	*slp;
{
	srrf_t		*rp;
	size_t		j;

	rp = srrf_alloc();
	rp->name = str_from_c("*");
	rp->class = srrf_class_by_name("bootp");
	rp->type = srrf_type_by_name(rp->class, "ds");
	for (j = 0; j < slp->nstrings; ++j)
		strlist_append(&rp->arg, slp->string[j]);
	return rp;
}


static srrf_t *bootp_ha_from_ether_a _((srrf_t *));

static srrf_t *
bootp_ha_from_ether_a(rp)
	srrf_t		*rp;
{
	srrf_t		*rp2;

	rp2 = srrf_alloc();
	rp2->name = str_copy(rp->name);
	rp2->class = srrf_class_by_name("bootp");
	rp2->type = srrf_type_by_name(rp2->class, "ha");
	strlist_append(&rp2->arg, rp->name);
	rp2->file_name = str_copy(rp->file_name);
	rp2->line_number = rp->line_number;
	return rp2;
}


static srrf_t *bootp_ip_from_in_a _((srrf_t *));

static srrf_t *
bootp_ip_from_in_a(rp)
	srrf_t		*rp;
{
	srrf_t		*rp2;

	rp2 = srrf_alloc();
	rp2->name = str_copy(rp->name);
	rp2->class = srrf_class_by_name("bootp");
	rp2->type = srrf_type_by_name(rp2->class, "ip");
	strlist_append(&rp2->arg, rp->name);
	rp2->file_name = str_copy(rp->file_name);
	rp2->line_number = rp->line_number;
	return rp2;
}


static int cmp _((const void *, const void *));

static int
cmp(va, vb)
	const void	*va;
	const void	*vb;
{
	typedef struct table_ty table_ty;
	struct table_ty
	{
		char	*name;
		int	collate;
	};

	static table_ty table[] =
	{
		{ "tc",	2, },
		{ "ht",	1, },
		/* everything else = 0 */
	};

	table_ty	*tp;
	string_ty	*a;
	int		wa;
	string_ty	*b;
	int		wb;

	a = *(string_ty **)va;
	b = *(string_ty **)vb;

	wa = 0;
	for (tp = table; tp < ENDOF(table); ++tp)
	{
		if (!strcmp(a->str_text, tp->name))
		{
			wa = tp->collate;
			break;
		}
	}
	if (!wa && isupper(a->str_text[0]))
		wa = -1;

	wb = 0;
	for (tp = table; tp < ENDOF(table); ++tp)
	{
		if (!strcmp(b->str_text, tp->name))
		{
			wb = tp->collate;
			break;
		}
	}
	if (!wb && isupper(b->str_text[0]))
		wb = -1;

	if (wa != wb)
	{
		/* reverse: largest first */
		return (wb - wa);
	}
	return strcmp(a->str_text, b->str_text);
}


void
filter(infile, outfile)
	char		*infile;
	char		*outfile;
{
	srrf_list_ty	*rlp;
	size_t		j, k;
	srrf_class_ty	*bootp_class;
	srrf_class_ty	*ether_class;
	srrf_class_ty	*in_class;
	srrf_type_ty	*ether_a_type;
	srrf_type_ty	*in_a_type;
	srrf_type_ty	*in_cname_type;
	srrf_type_ty	*in_hinfo_type;
	srrf_type_ty	*in_ns_type;
	symtab_ty	*defaults_stp;
	int		number_of_errors;
	bootp_aux_ty	aux;
	strlist_ty	ns;
	srrf_t		*rp;
	srrf_t		*rp2;
	string_ty	*name;

	/*
	 * read the input file
	 */
	rlp = srrf_reader(infile, verbose, 1);

	/*
	 * open the output file
	 */
	output_open(outfile);

	/*
	 * find the necessary type pointer
	 */
	bootp_class = srrf_class_by_name("bootp");
	assert(bootp_class);
	ether_class = srrf_class_by_name("ether");
	assert(ether_class);
	in_class = srrf_class_by_name("in");
	assert(in_class);
	ether_a_type = srrf_type_by_name(ether_class, "a");
	assert(ether_a_type);
	in_a_type = srrf_type_by_name(in_class, "a");
	assert(in_a_type);
	in_hinfo_type = srrf_type_by_name(in_class, "hinfo");
	assert(in_hinfo_type);
	in_cname_type = srrf_type_by_name(in_class, "cname");
	assert(in_cname_type);
	in_ns_type = srrf_type_by_name(in_class, "ns");
	assert(in_ns_type);

	/*
	 * warning comment at the top of the file
	 */
	output_printf("# Do not edit this file.  It is generated\n");
	output_printf
	(
		"# using a '%s %s %s' command.\n",
		progname,
		(infile ? infile : "-"),
		(outfile ? outfile : "-")
	);

	/*
	 * scan the resources looking for ether addresses
	 */
	aux.in_hinfo_stp = symtab_alloc(5);
	aux.in_a_stp = symtab_alloc(5);
	aux.ether_a_stp = symtab_alloc(5);
	strlist_zero(&ns);
	for (j = 0; j < rlp->nrecords; ++j)
	{
		rp = rlp->record[j];
		if (rp->type == ether_a_type)
			symtab_assign(aux.ether_a_stp, rp->name, rp);
		if (rp->type == in_a_type)
			symtab_assign(aux.in_a_stp, rp->name, rp);
		if (rp->type == in_cname_type)
			symtab_assign(aux.in_a_stp, rp->name, rp);
		if (rp->type == in_hinfo_type)
			symtab_assign(aux.in_hinfo_stp, rp->name, rp);
		if (rp->type == in_ns_type)
			strlist_append(&ns, rp->arg.string[0]);
	}

	/*
	 * generate "ds" from the "in ns" entries
	 */
	defaults_stp = symtab_alloc(5);
	if (ns.nstrings)
	{
		rp = bootp_ds_from_in_ns(&ns);
		strlist_free(&ns);
		name = str_from_c(rp->type->name);
		symtab_assign(defaults_stp, name, rp);
		str_free(name);
	}

	/*
	 * scan the resources looking for hosts
	 */
	number_of_errors = 0;
	for (j = 0; j < rlp->nrecords; ++j)
	{
		symtab_ty	*stp;
		int		number_of_params;

		rp = rlp->record[j];
		if
		(
			rp->class == bootp_class
		&&
			rp->name->str_text[0] == '*'
		&&
			rp->name->str_text[1] == '.'
		)
		{
			/*
			 * set a defailt
			 */
			name = str_from_c(rp->type->name);
			symtab_assign(defaults_stp, name, rp);
			str_free(name);
		}
		if (rp->type != in_a_type)
			continue;

		/*
		 * collect all of the associated bootp records
		 */
		number_of_params = 0;
		stp = symtab_alloc(5);
		for (k = 0; k < rlp->nrecords; ++k)
		{
			rp2 = rlp->record[k];
			if (!str_equal(rp2->name, rp->name))
				continue;
			if (rp2->class != bootp_class)
				continue;

			name = str_from_c(rp2->type->name);
			symtab_assign(stp, name, rp2);
			str_free(name);
			++number_of_params;
		}

		/*
		 * print the records
		 */
		if (number_of_params > 0)
		{
			strlist_ty	key;

			symtab_walk(defaults_stp, load_the_defaults, stp);
			name = no_trailing_dot(rp->name);
			output_printf("%s:", name->str_text);
			str_free(name);

			/*
			 * generate an "ip" entry
			 */
			rp2 = bootp_ip_from_in_a(rp);
			name = str_from_c(rp2->type->name);
			symtab_assign(stp, name, rp2);
			str_free(name);
			++number_of_params;

			/*
			 * generate a "ha" entry
			 */
			rp2 = bootp_ha_from_ether_a(rp);
			name = str_from_c(rp2->type->name);
			symtab_assign(stp, name, rp2);
			str_free(name);
			++number_of_params;

			/*
			 * Order they keys alphabetically, except
			 * for a few ordering constraints
			 */
			symtab_keys(stp, &key);
			qsort
			(
				key.string,
				key.nstrings,
				sizeof(key.string[0]),
				cmp
			);

			/*
			 * print the keys out in order
			 */
			for (k = 0; k < key.nstrings; ++k)
			{
				rp2 = symtab_query(stp, key.string[k]);
				assert(rp2);
				output_printf("\\\n\t:");
				srrf_invoke_aux1(rp2, &aux);
			}
			strlist_free(&key);
			symtab_free(stp);
			output_printf("\n");
		}
	}
	symtab_free(defaults_stp);
	srrf_list_free(rlp);

	/*
	 * close the output file
	 */
	output_close();
}
