/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1996, 1999, 2000 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to check for srrf duplicates
 */

#include <error.h>
#include <srrf.h>
#include <srrf/address.h>
#include <symtab.h>


static int number_of_errors;


static void check_for_dangling_cname _((symtab_ty *, string_ty *, void *,
	void *));

static void
check_for_dangling_cname(in_cname_stp, key, data, arg)
	symtab_ty	*in_cname_stp;
	string_ty	*key;
	void		*data;
	void		*arg;
{
	srrf_t		*rp;
	srrf_t		*rp2;
	symtab_ty	*name_stp;

	rp = data;
	name_stp = arg;
	rp2 = symtab_query_downcase(name_stp, rp->arg.string[0]);
	if (!rp2)
	{
		error
		(
		"%s: %d: the alias \"%s\" does not refer to a known IP address",
			rp->file_name->str_text,
			rp->line_number,
			rp->name->str_text
		);
		++number_of_errors;
		return;
	}
	if (rp2->type == rp->type)
	{
		error
		(
"%s: %d: the alias \"%s\" refers to another alias, \
it must refer directly to an IP address",
			rp->file_name->str_text,
			rp->line_number,
			rp->name->str_text
		);
		++number_of_errors;
	}
}


void
srrf_check_for_duplicates(res, nres, filename)
	srrf_t		**res;
	size_t		nres;
	char		*filename;
{
	size_t		j;
	srrf_t		*rp;
	srrf_t		*rp2;
	unsigned long	last_address;
	srrf_class_ty	*in_class;
	srrf_type_ty	*in_a_type;
	srrf_type_ty	*in_cname_type;
	srrf_class_ty	*ether_class;
	srrf_type_ty	*ether_a_type;
	symtab_ty	*name_stp;
	symtab_ty	*in_a_stp;
	symtab_ty	*in_cname_stp;
	symtab_ty	*ether_a_stp;

	name_stp = symtab_alloc(5);
	in_a_stp = symtab_alloc(5);
	in_cname_stp = symtab_alloc(5);
	ether_a_stp = symtab_alloc(5);

	if (!filename)
		filename = "standard input";
	number_of_errors = 0;
	last_address = 0;

	in_class = srrf_class_by_name("in");
	assert(in_class);
	ether_class = srrf_class_by_name("ether");
	assert(ether_class);
	in_a_type = srrf_type_by_name(in_class, "a");
	assert(in_a_type);
	ether_a_type = srrf_type_by_name(ether_class, "a");
	assert(ether_a_type);
	in_cname_type = srrf_type_by_name(in_class, "cname");
	assert(in_cname_type);

	for (j = 0; j < nres; ++j)
	{
		rp = res[j];
		if (rp->type == in_a_type)
		{
			unsigned long	addr;
			string_ty	*key;

			addr = srrf_address(rp->arg.string[0]->str_text);
			if (addr < last_address)
			{
				error
				(
					"%s: %d: warning: address %s out of order",
					rp->file_name->str_text,
					rp->line_number,
					rp->arg.string[0]->str_text
				);
			}
			last_address = addr;

			key = str_format("%ld", addr);
			rp2 = symtab_query_downcase(in_a_stp, key);
			if (rp2)
			{
				error
				(
					"%s: %d: duplicate %s host IP address",
					rp->file_name->str_text,
					rp->line_number,
					rp->arg.string[0]->str_text
				);
				error
				(
					"%s: %d: ...here is the first definition",
					rp2->file_name->str_text,
					rp2->line_number
				);
				++number_of_errors;
			}
			else
				symtab_assign_downcase(in_a_stp, key, rp);
			str_free(key);
		}
		if (rp->type == in_cname_type)
			symtab_assign_downcase(in_cname_stp, rp->name, rp);
		if (rp->type == in_a_type || rp->type == in_cname_type)
		{
			rp2 = symtab_query_downcase(name_stp, rp->name);
			if (rp2)
			{
				error
				(
					"%s: %d: duplicate \"%s\" host name",
					rp->file_name->str_text,
					rp->line_number,
					rp->name->str_text
				);
				error
				(
					"%s: %d: ...here is the first definition",
					rp2->file_name->str_text,
					rp2->line_number
				);
				++number_of_errors;
			}
			else
			{
				assert(rp->name);
				symtab_assign_downcase(name_stp, rp->name, rp);
			}
		}
		if (rp->type == ether_a_type)
		{
			/*
			 * The ether address has been re-written in a
			 * canonical form, so it is suitable for use as
			 * a key.
			 */
			rp2 = symtab_query_downcase(ether_a_stp, rp->arg.string[0]);
			if (rp2)
			{
				error
				(
					"%s: %d: duplicate %s host Ethernet address",
					rp->file_name->str_text,
					rp->line_number,
					rp->arg.string[0]->str_text
				);
				error
				(
					"%s: %d: ...here is the first definition",
					rp2->file_name->str_text,
					rp2->line_number
				);
				++number_of_errors;
			}
			else
				symtab_assign_downcase(ether_a_stp, rp->arg.string[0], rp);
		}
	}

	/*
	 * Now that we have all of the definitions,
	 * make sure the cname records point at a records.
	 */
	symtab_walk(in_cname_stp, check_for_dangling_cname, name_stp);

	/*
	 * free all of the symbol tables
	 */
	symtab_free(name_stp);
	symtab_free(in_a_stp);
	symtab_free(in_cname_stp);
	symtab_free(ether_a_stp);

	if (number_of_errors)
	{
		fatal
		(
			"%s: found %d fatal error%s",
			filename,
			number_of_errors,
			(number_of_errors == 1 ? "" : "s")
		);
	}
}
