/*
 *	dnsutl - utilities to make DNS easier to configure
 *	Copyright (C) 1999 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate host maps
 */

#include <map_hosts.h>

#include <map.h>
#include <map_hosts.h>
#include <mem.h>
#include <srrf/address.h>


map_host_t *
map_host_new()
{
	map_host_t	*this;

	this = mem_alloc(sizeof(map_host_t));
	this->address = 0;
	strlist_zero(&this->name);
	return this;
}


void
map_host_delete(this)
	map_host_t	*this;
{
	if (this->address)
	{
		str_free(this->address);
		this->address = 0;
	}
	strlist_free(&this->name);
	mem_free(this);
}


void
map_host_open(fn)
	const char	*fn;
{
	map_open(fn);
}


void
map_host_close()
{
	map_close();
}


map_host_t *
map_host_read()
{
	strlist_ty	sl;
	map_host_t	*this;
	size_t		j;

	for (;;)
	{
		if (!map_read(&sl))
			return 0;
		if (sl.nstrings < 2)
		{
			map_error("hosts entries require at least 2 strings");
			strlist_free(&sl);
			continue;
		}
		break;
	}

	this = map_host_new();
	this->address = srrf_address_cannonicalize(sl.string[0]);
	if (!this->address)
	{
		map_error("the string \"%s\" is not a legal IP address",
			sl.string[0]);
		this->address = str_copy(sl.string[0]);
	}
	for (j = 1; j < sl.nstrings; ++j)
		strlist_append(&this->name, sl.string[j]);
	strlist_free(&sl);
	return this;
}
