/*
	MultiMouse - bind MULTIple mice into a MOUSE -
	Copyright (C) 1995 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)

	MultiMouse is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published
	by the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	MultiMouse is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
	See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>

#include "mumse.h"

int msFifo;
struct mouseInfo msInfo[MAX_MICE];

/*
   analyze packet and convert into MouseSystems protocol
*/

static
void ConvertProtocol(struct mouseInfo *msi, u_char *packet, int len)
{
    int dx, dy;
    bool released=FALSE;
    u_char stat, now;
    static u_char last;
    /*
       stat: buttons status modified for emu mode
       now:  bare status
       last: high4bits=last now, low4bits=last stat
    */

    switch (msi->type) {
    case MOUSE_MICROSOFT:
	if (len < 3) {
	    /*
	       Logitech First Mouse 3 buttons mode use variable length
	       packet. (normal mode = 3bytes, with 3rd button = 4bytes)
	       In long packet, extra byte (first 1 byte) is stored in
	       packet[3].
	    */
	    stat = packet[3] | (last & 0x5);
	    dx = dy = 0;
	    break;
	}
	stat = ((packet[0] & 0x20) >> 3) | ((packet[0] & 0x10) >> 4)
	    | (msi->noemu ? last & 2: 0);
	dx = (char)(((packet[0] & 0x03) << 6) | (packet[1] & 0x3F));
	dy = (char)(((packet[0] & 0x0C) << 4) | (packet[2] & 0x3F));
	break;
#ifndef	THROUGH_MOUSESYS
    case MOUSE_MOUSESYS:
	stat = (~packet[0]) & 0x07;
	dx = (char)(packet[1]) + (char)(packet[3]);
	dy = - ((char)(packet[2]) + (char)(packet[4]));
	break;
#endif
    case MOUSE_MMSERIES:
    case MOUSE_LOGITECH:
	stat = packet[0] & 0x07;
	dx = (packet[0] & 0x10) ? packet[1]: - packet[1];
	dy = (packet[0] & 0x08) ? - packet[2]: packet[2];
	break;
    case MOUSE_PS2:
	stat = ((packet[0] & 0x01) << 2) | ((packet[0] & 0x06) >> 1);
	dx = (char)(packet[1]);
	dy = -(char)(packet[2]);
	break;
    case MOUSE_BUSMOUSE:
	stat = (~packet[0]) & 0x07;
	dx = (char)packet[1];
	dy = - (char)packet[2];
	break;
    }
    now = stat;
    /*
       3 buttons emulation process
    */
    if (!msi->noemu) {
	if ((!dx && !dy) || (last & 0x2)) {
	    stat &= ~5;
	    if ((now & 5) == 5) stat |= 2;
	}
	if (!dx && !dy && !(stat & 2)) {
	    if ((last & 0x50) && !(now & 5)) {
		stat |= ((last & 2) ? last: (last >> 4)) & 5;
		if (!(last & 2)) released = TRUE;
	    } else {
		last = (now << 4) | stat;
		return;
	    }
	}
    }

#if 0
    printf("(%d, %02X %02X %02X %02X %02X)", len, packet[0],
	   packet[1], packet[2], packet[3], packet[4]);
#endif

    /*
       make MouseSystems protocol packet
    */
    packet[0] = ~(stat | 0xf8) | 0x80;
    /* miceConf[MOUSE_MOUSESYS].mp.hid */
    packet[1] = (char)dx;
    packet[3] = 0;
    packet[2] = (char)-dy;
    packet[4] = 0;

#ifdef	DEBUG
    printf("L:%2d N:%02X L:%02X ", len, now, last);
#endif
#ifdef	DEBUG
    printf("W1:S=%02X:%02X %02X %02X %02X %02X\n",
	   (~packet[0]) & 0x07, packet[0],
	   packet[1], packet[2], packet[3], packet[4]);
#endif
    last = (now << 4) | stat;
    write(msFifo, packet, 5);

    /*
       released == TRUE if and only if button is released in emu mode
    */
    if (released == TRUE) {
	packet[0] |= 0x5;
	write(msFifo, packet, 5);
#ifdef	DEBUG
    printf("L:%2d N:%02X L:%02X ", len, now, last);
#endif
#ifdef	DEBUG
    printf("W2:S=%02X:%02X %02X %02X %02X %02X\n",
	   (~packet[0]) & 0x07, packet[0],
	   packet[1], packet[2], packet[3], packet[4]);
#endif
    }
}

static
void GetPacket(struct mouseInfo *msi)
{
    static u_char packet[MOUSE_PKSIZE];
    int stat = 0, len = 0;
    u_char ch;

#ifdef	THROUGH_MOUSESYS
    if (msi->type == MOUSE_MOUSESYS) {
	if ((len = read(msi->fd, packet, MOUSE_PKSIZE)) > 0)
	    write(msFifo, packet, len);
	return;
    }
#endif
    while (read(msi->fd, &ch, 1) > 0) {
	len ++;
#ifdef	DEBUG
/*	printf("%02X(%02d) ", ch, stat);*/
	fflush(stdout);
#endif
	if (!stat) {
	    if ((ch & msi->mp.hmask) == msi->mp.hid) {
		packet[0] = ch;
		stat = 1;
	    } else if (msi->noemu && msi->type == MOUSE_MICROSOFT) {
		if ((ch & ~0x23) == 0) {
		    packet[3] = (ch & 0x20) >> 4;
		    ConvertProtocol(msi, packet, len);
		    write(msFifo, packet, 5);
		    len = 0;
		    break;
		}
	    }
	    continue;
	}
	if (msi->type != MOUSE_PS2
	    && ((ch & msi->mp.dmask) || ch == 0x80)) {
	    len = stat = 0;
	    continue;
	}
	packet[stat] = ch;
	stat ++;
	if (stat == msi->mp.len) {
	    ConvertProtocol(msi, packet, len);
	    len = stat = 0;
	    break;
	}
    }
}

void Selection(int numMouse)
{
    fd_set readFds, orgReadFds;
    int	i;
    int numFds;

    FD_ZERO(&orgReadFds);
    for (i = 0; i < numMouse; i ++) {
	FD_SET(msInfo[i].fd, &orgReadFds);
	if (numFds < msInfo[i].fd) numFds = msInfo[i].fd;
    }
    numFds ++;
    while (1) {
	int ret;
	readFds = orgReadFds;
	ret = select(numFds, &readFds, NULL, NULL, NULL);
	if (ret < 0 && errno == EINTR) {
#ifdef	DEBUG
	    printf("EINTR\n");
#endif
	    errno = 0;
	    continue;
	}
	for (i = 0; i < numMouse; i ++) {
	    if (FD_ISSET(msInfo[i].fd, &readFds)) {
		GetPacket(&msInfo[i]);
	    }
	}
    }
}
