////////////////////////////////////////////////////////////////////////////////
//
//	Class : mxMailMsgDisplay
//
//	Author : Andy Jefferson
//
//	Description :
//		mxMailMsgDisplay displays the contents of an mxMailMsg
//		object. A mxMailMsg is passed at creation.
//		The display takes the form of a text area showing the message
//		together with From,To,Subject,Date fields.
//
//	Public Methods :
//		(Constructor) requires :
//			name	The name used for resource specifications
//			message	A mxMailMessage object.
//
//		setMessage()	Updates the message.
//
//	Limitations :
//
////////////////////////////////////////////////////////////////////////////////
#ifndef mxMailMsgDisplay_h
#define mxMailMsgDisplay_h
#ifdef __cplusplus

#include <vector.h>

#include "akComponent.h"
#include "akButton.h"
#include "akPixmap.h"
#include "akPromptString.h"
#include "mxMailMsg.h"
#include "mxMailAddressBook.h"
#include "mxMailAddressSelect.h"
#include "mxMailAttachment.h"
#include "mxFileViewer.h"
#include "mxMailMsgViewer.h"

typedef enum
{
	MX_MSG_DISPLAY,
	MX_MSG_COMPOSE
} mxMailMsgDisplayMode;

class mxMailMsgDisplay : public akComponent
{
	public:

		// Constructor and destructor

		mxMailMsgDisplay(Widget parent,
				 char *name,
				 mxMailMsgDisplayMode mode,
				 mxMailMsg msg,
				 mxMailAddressBook *book);
		~mxMailMsgDisplay();

		// Update facilities for both modes

		void		setMessage(mxMailMsg msg);

		// Update facilities for COMPOSE mode

		void		addAttachment(const mxMailAttachment& att);
		bool		deleteSelectedAttachment();
		void		clearMessage();
		bool		setSubject(string);
		bool		setTo(string);
		bool		setCc(string);
		bool		setBcc(string);
		bool		setText(string);
		bool		insertText(string);

		void		highlightAndShowBodyText(int start,int end);
		void		removeBodyTextHighlights();

		// Access functions for both modes

		mxMailMsgDisplayMode	mode() const;
		const mxMailMsg&	msg();
		const mxMailAttachment	*selectedAttachment() const;
		string			to() const;
		string			cc() const;
		string			bcc() const;
		string			subject() const;
		string			text() const;

	private:

		// The message that the display represents

		mxMailMsg		_msg;

		// Mode of operation

		mxMailMsgDisplayMode	_mode;

		// Addressbook for resolving aliases

		mxMailAddressBook	*_addressbook;
		mxMailAddressSelect	*_addressSelector;

		// Panel creation function/data

		akPromptString	*_fromField;
		akPromptString	*_toField;
		akPromptString	*_ccField;
		akPromptString	*_bccField;
		akPromptString	*_subjectField;
		akPromptString	*_dateField;
		akPromptString	*_organisationField;
		akPromptString	*_messageIdField;
		Widget		_messageField;
		Widget		_attachmentFrame;
		Widget		_attachmentContainer;
		akButton	*_fileAttachmentButton;
		akButton	*_fileButton;
		akButton	*_printButton;
		akButton	*_addressButton;

		int		_fileAttachmentNo;

		// Icons representing any attachments

		vector<mxMailAttachmentIcon>	_attachmentIcons;
		static akPixmap			*_fileAttachmentIcon;
		static akPixmap			*_imageAttachmentIcon;
		static akPixmap			*_audioAttachmentIcon;
		static akPixmap			*_videoAttachmentIcon;
		static akPixmap			*_messageAttachmentIcon;
		static akPixmap			*_applicationAttachmentIcon;

		// Utility functions

		void		update();
		string		enteredMessageText() const;

		// Callbacks for view panel options

		static void	fileCB(Widget,XtPointer,XtPointer);
		static void	fileOK(XtPointer,XtPointer);
		static void	fileCANCEL(XtPointer,XtPointer);
		static void	printCB(Widget,XtPointer,XtPointer);
		static void	fileAttachmentCB(Widget,XtPointer,XtPointer);
		static void	fileAttachmentOK(XtPointer,XtPointer);
		static void	containerActionCB(Widget,XtPointer,XtPointer);
		static void	addressCB(Widget,XtPointer,XtPointer);
		static void	addressSelectedCB(XtPointer,mxMailAddressSelection);

		// Viewers

		vector<mxFileViewer*>		_fileViewers;
		vector<mxMailMsgViewer*>	_messageViewers;
		static void	fileViewerClosedCB(Widget,XtPointer,XtPointer);
		static void	messageViewerClosedCB(Widget,XtPointer,XtPointer);
};

inline
mxMailMsgDisplayMode	mxMailMsgDisplay::mode() const
	{return _mode;}
inline
string	mxMailMsgDisplay::to() const
	{return _toField->value();}
inline
string	mxMailMsgDisplay::cc() const
	{return _ccField->value();}
inline
string	mxMailMsgDisplay::bcc() const
	{return _bccField->value();}
inline
string	mxMailMsgDisplay::subject() const
	{return _subjectField->value();}
inline
string	mxMailMsgDisplay::text() const
	{return enteredMessageText();}

#endif
#endif
