/*
 *  The Reker version of cut - like the standard UN*X cut plus add. features.
 *  Copyright (C) 2000  <Frank Reker>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *	This program is written by 
 * 		Frank Reker <frank@reker.net>
 *	at: Tue May  2 03:42:09 CEST 2000
 */

/*
 * compile with: gcc -O3 -o cut cut.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include "manual.h"

#define VERSION "0.92"
#define LAST_MODIFIED_AT "Mon Nov 27 04:25:12 CET 2000"
#define LAST_MODIFIED_BY "Frank Reker <frank@reker.net>"

#define rmalloc(size) calloc ((size), 1)
#define rfree(ptr) { if (ptr) { free ((ptr)); ptr = NULL; } }
#ifdef INLINE
#undef INLINE
#endif
#if defined __GNUC__ && __GNUC__ >= 2
#define INLINE inline
#else
#define INLINE
#endif

#ifndef NO_LONG_PARAM
#define LONG_PARAM
#endif

typedef struct {
	char * buf, *s;
	int size;
	} Tbuf;

static char * delim=NULL;
static int delimlen=0;
static char * new_delim=NULL;
static int * fields=NULL;
static int fieldnum;
static char * fieldstr=NULL;
static FILE * fin; /* = stdin; */
static FILE * fout; /* = stdout; */
static int only_delim = 0;
static int order=0, sorted=0, once=0;
static char * text_to_insert=NULL;
static int text_to_insert_len=0;
static int * linelist=NULL;
static int linenum=0;
static char * linestr = NULL;
static int only_linematch=0;
static int all_fields = 0;
static char * PROG = NULL;

INLINE
char * nindex (char * s, char c, int len) {
	for (; len; len--, s++)
		if (*s==c) return s;
	return NULL;
}

sort () {
	int i, max=-1, *field;
	char * nfield, *s;
	for (i=0, field=fields; i<fieldnum; i++, field++) 
		if (*field > max) max=*field;
	if (max == -1) return;
	if (!(nfield = rmalloc ((unsigned int) max + 1))) {
		fprintf (stderr, "%s: field %u is out of range\n", PROG, max);
		exit (2); }
	for (i=0, field=fields; i<fieldnum; i++, field++) {
		if (*field == -1) continue;
		if (*field < -1) {
			fprintf (stderr, "%s: field %u is out of range\n", PROG, *field);
			exit (2); }
		if (once && i+1 <fieldnum && *(field+1) == -1) {
			nfield[*field]=2;
			i++; field++; }
		else
			nfield[*field]=1; }
	for (i=0,s=nfield,field=fields; i<=max; i++, s++) {
		if (*s) *(field++)=i;
		if (once && *s == 2) *(field++)=-1; }
	fieldnum = field-fields;
	fields=realloc (fields, fieldnum * sizeof (int));
	free (nfield);
}

int eval_fieldlist (int num);

int preprocfl () {
	char *s=fieldstr;
	int bakfieldnum, *bakfields;
	sorted = 0;
	once = 1;
	if (!strcmp (s, "-") || !strcmp (s, "1-") || !strcmp (s, "-~1") ||
						!strcmp (s, "1-~1"))
		all_fields = 1;
	else {
		while (s=index (s, '~')) {
			if (s==fieldstr || *(s-1)!='-' || *(s+1)!='1' || 
												isdigit (*(s+2))) {
				once=0;
				break; }
			s++; }
		if (once) {
			s = fieldstr;
			while (s = index (s, '-'))
				if (*(++s) == '/' || *s=='~' && *(s+1)=='1' && *(s+2)=='/') {
					once = 0;
					break; }}
		if (!once) return 0; }
	eval_fieldlist (-1);
	sorted = order;
	if (all_fields) 
		sorted = 1;
	else if (!order) {
		bakfieldnum = fieldnum;
		if (!(bakfields = rmalloc ((fieldnum + 1) * sizeof (int)))) {
			fprintf (stderr, "%s: out of memory\n", PROG);
			exit (3); }
		memcpy (bakfields, fields, fieldnum * sizeof (int));
		sort ();
		sorted = (fieldnum == bakfieldnum && !memcmp (bakfields, fields, 
									fieldnum * sizeof (int)));
		fieldnum = bakfieldnum;
		rfree (fields);
		fields = bakfields; }
	return 1;
}


int eval_fieldlist (int num) {
	int neg, shift=1;
	int i, min, max,j=0;
	int *field;
	char * s;
	rfree (fields);
	if (!fieldstr) return 0;
	fieldnum = 256;
	field = fields = malloc (256 * sizeof (int));
	for (s=fieldstr; *s; s++) {
		shift = 1;
		if (*s == '~') {
			neg=1;
			s++; }
		else 
			neg=0;
		if (*s == '-') {
			min = 0;
			s++;}
		else if (*s == ',') continue;
		else if (isdigit (*s)) {
			i=0;
			for (; isdigit (*s); s++)
				i = i*10 + *s - '0';
			if (*s == ',' || !*s) {
				if (!i || num!=-1 && i>num) continue;
				*field = neg ? num - i : i - 1;
				if ((j=++field-fields) == fieldnum) {
					fieldnum *=2;
					if (!(fields = realloc (fields, fieldnum * sizeof (int)))){
						fprintf (stderr, "%s: out of memory\n", PROG);
						exit (3); }
					field = fields + j; }
				if (!*s) s--;
				continue; }
			else {
				min = neg ? num - i : i -1; 
				s++; }}
		else for (; *s && *s!=','; s++);
		if (!*s || *s == ',' || *s=='/' && s!=fieldstr && *(s-1) == '-') 
			max = num;
		else {
			if (*s == '~') {
				neg=1;
				s++; }
			else 
				neg=0;
			i=0;
			for (; isdigit (*s); s++)
				i = i*10 + *s - '0';
			max = neg ? num - i : i - 1; }
		if (*s == '/') {
			i=0;
			for (s++; isdigit (*s); s++)
				i = i*10 + *s - '0';
			shift = i ? i : 1; }
		for (; *s!=',' && *s; s++);
		if (num == -1) {
			if (max < 0) {
				*field = min;
				if ((j=++field-fields) == fieldnum) {
					fieldnum *=2;
					if (!(fields = realloc (fields, fieldnum * sizeof (int)))){
						fprintf (stderr, "%s: out of memory\n", PROG);
						exit (3); }
					field = fields + j; }
				*field = -1;
				if ((j=++field-fields) == fieldnum) {
					fieldnum *=2;
					if (!(fields = realloc (fields, fieldnum * sizeof (int)))){
						fprintf (stderr, "%s: out of memory\n", PROG);
						exit (3); }
					field = fields + j; }
				if (!*s) s--;
				continue; }
			if (min < 0) min = 0; }
		if (max >= min) {
			if (num != -1) {
				if (max < 0 || min >= num) continue;
				if (max >= num) max = num - 1;
				if (min <= 0) min = 0; }
			for (i=min; i<=max; i+=shift) {
				*field = i;
				if ((j=++field-fields) == fieldnum) {
					fieldnum *=2;
					if (!(fields = realloc (fields, fieldnum * sizeof (int)))){
						fprintf (stderr, "%s: field %u is out of range\n",
													PROG, max);
						exit (2); }
					field = fields + j; }}}
		else {
			if (num != -1) {
				if (min < 0 || max >= num) continue;
				if (min >= num) min = num - 1;
				if (max <= 0) max = 0; }
			for (i=min; i>=max; i-=shift) {
				*field = i;
				if ((j=++field-fields) == fieldnum) {
					fieldnum *=2;
					if (!(fields = realloc (fields, fieldnum * sizeof (int)))){
						fprintf (stderr, "%s: field %u is out of range\n",
													PROG, min);
						exit (2); }
					field = fields + j; }}}
		if (!*s) s--; }
	fieldnum = field - fields;
	if (order) sort();
	return 1;
}

#define SWAP(a,b) { a^=b; b^=a; a^=b; }

int eval_linelist () {
	char *s = linestr;
	int * field, *s2, *new_list;
	int min, max,i, shift, range, j;
	volatile int nmax;
	rfree (linelist);
	if (!linestr || !*linestr) return 0;
	if (!(field = linelist = malloc ((linenum = 256) * sizeof (int)))) {
		fprintf (stderr, "%s: out of memory\n", PROG);
		exit (3); }
	for (; *s; s++) {
		shift = 1;
		range = 0;
		if (*s==',') continue;
		if (*s=='-') {
			min = 0;
			range = 1;
			if (isdigit (*++s)) {
				for (i=0; isdigit (*s); s++)
					i = 10*i + *s - '0'; 
				max = i ? i-1 : 0; }
			else max = -1; }
		else if (isdigit (*s)) {
			for (i=0; isdigit (*s); s++)
				i = 10*i + *s - '0'; 
			min = i ? i-1 : 0; 
			if (*s == '-') {
				s++;
				range = 1; }
			if (isdigit (*s)) {
				for (i=0; isdigit (*s); s++)
					i = 10*i + *s - '0'; 
				max = i ? i-1 : 0;
				if (max < min) SWAP (min, max); }
			else max = -1; }
		if (*s=='/') {
			for (i=0, s++; isdigit (*s); s++)
				i = 10*i + *s - '0'; 
			shift = i ? i : 1; }
		for (; *s && *s!=','; s++);
		if (!*s) s--;
		if (!range) max = min;
		if (max < 0) {
			max = -1 * shift;
			if ((j = field - linelist) + 2 > linenum) {
				linenum *= 2;
				if (!(linelist = realloc (linelist, linenum * sizeof (int)))) {
					fprintf (stderr, "%s: out of memory\n", PROG);
					exit (3); }
				field = linelist + j; }
			*field++ = min;
			*field++ = max; }
		else {
			for (i=min; i <= max; i+= shift) {
				if ((j = field - linelist) >= linenum) {
					linenum *= 2;
					if (!(linelist = realloc (linelist, linenum * 
														sizeof (int)))) {
						fprintf (stderr, "%s: line %u is out of range\n",
													PROG, max);
						exit (2); }
					field = linelist + j; }
				*field++ = i; }}}
	linenum = field - linelist;
	/* now sort */
	max = -1;
	for (i=0, field=linelist; i<linenum; i++, field++)
		if (*field > max) max = *field;
	if (max == -1) {
		rfree (linelist);
		return 0; }
	if ((nmax = (max + 1) * sizeof (int)) < 0) {
		fprintf (stderr, "%s: line %u is out of range\n", PROG, max);
		exit (2); }
	if (!(new_list = rmalloc ((unsigned)nmax))) {
		fprintf (stderr, "%s: line %u is out of range\n", PROG, max);
		exit (2); }
	nmax /= sizeof (int);
	for (i=0, field=linelist; i<linenum; i++, field++) {
		if (*field < 0) continue;
		if (*field > nmax) {
			fprintf (stderr, "%s: line %u is out of range\n", PROG, max);
			exit (2); }
		if (*(field + 1) < 0)
			new_list[*field] = *(field + 1);
		else new_list[*field] = 1; }
	field = linelist;
	for (s2=new_list, i=0; i<=max; i++, *s2++) {
		if (*s2 == 1) 
			*field++ = i;
		else if (*s2 < 0) {
			*field++ = i;
			*field++ = *s2; 
			break; }}
	linenum = field-linelist;
	return 1;
}
	
int isline () {
	static int lineno=0;
	static int * field = NULL;
	if (!field) field = linelist;
	if (lineno == *field) {
		if (field - linelist + 1 < linenum) field++;
		lineno++;
		return 1; }
	if (*field < 0) 
		return !((lineno++ - *(field-1)) % ((-1) * *field));
	lineno++;
	return 0;
}

static char outbuf[1025];
static char * outpoint=outbuf;

INLINE
int PUTC (char ch) {
	if (outpoint - outbuf == 1024) {
		fwrite (outbuf, sizeof (char), 1024, fout);
		outpoint = outbuf; }
	*outpoint = ch;
	outpoint++; 
}

INLINE
int PRINT (char * str, int len) {
	static char * s;
	static int i;
	s=str;
	while ((i = 1024 - (outpoint - outbuf)) < len) {
		memcpy (outpoint, s, i);
		s+=i;
		len -= i;
		fwrite (outbuf, sizeof (char), 1024, fout);
		outpoint = outbuf; }
	memcpy (outpoint, s, len);	
	outpoint += len;
}
		
INLINE
int FLUSH () {
	fwrite (outbuf, sizeof (char), outpoint-outbuf, fout);
	outpoint = outbuf; 
}

INLINE
int GETC (int mode) {
	static char buf[1025];
	static char * s=NULL;
	static int num;
	if (!mode || s-buf==1024) {
		s=buf;
		num=fread (buf, 1, 1024, fin);
		if (num != 1024) 
			buf[num] = EOF; }
	if (mode == 1 && *s!=EOF) return *s++;
	else return *s;
}

INLINE
int eat_line () {
	static int c;
	while ((c=GETC (1)) != EOF && c!='\n');
	return c!=EOF; 
}

int print_line () {
	static int c;
	if (GETC (2) == EOF) return 0;
	while ((c=GETC (1)) != EOF && c!='\n')
		PUTC (c);
	PUTC ('\n');
	return c!=EOF; 
}

int line () {
	int bufsize=256;
	Tbuf ** buf;
	Tbuf ** cbuf;
	int c, i, j;
	int * field, num;
	int delim_bak, first=1;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		if (!only_delim) PUTC (c);
		GETC (1);
		return 1; }
	cbuf = buf = rmalloc (bufsize * sizeof (void *));
	*cbuf = malloc (sizeof (Tbuf));
	(*cbuf)->size = 256;
	(*cbuf)->s = (*cbuf)->buf = malloc ((*cbuf)->size * sizeof (char));
	while ((c=GETC (1)) != EOF && c != '\n') {
		if ((i=(*cbuf)->s - (*cbuf)->buf) == (*cbuf)->size) {
			(*cbuf)->size *= 2;
			(*cbuf)->buf = realloc ((*cbuf)->buf, (*cbuf)->size * 
											sizeof (char));
			(*cbuf)->s = (*cbuf)->buf + i; }
		*(*cbuf)->s = c;
		(*cbuf)->s++;
		if (nindex (delim, c, delimlen)) {
			(*cbuf)->buf = realloc((*cbuf)->buf, ((*cbuf)->size = (*cbuf)->s -
					(*cbuf)->buf + 1) * sizeof (char));
			(*cbuf)->s = (*cbuf)->buf + (*cbuf)->size - 2;
			if (new_delim) *(*cbuf)->s = *new_delim;
			cbuf++;
			if ((i = cbuf - buf) == bufsize) {
				bufsize *= 2;
				buf = realloc (buf, bufsize * sizeof (void *));
				cbuf = buf + i; }
			*cbuf = malloc (sizeof (Tbuf));
			(*cbuf)->size = 256;
			(*cbuf)->s = (*cbuf)->buf = malloc ((*cbuf)->size * 
												sizeof (char)); }}
	(*cbuf)->buf = realloc((*cbuf)->buf, ((*cbuf)->size = (*cbuf)->s -
					(*cbuf)->buf + 2) * sizeof (char));
	(*cbuf)->s = (*cbuf)->buf + (*cbuf)->size - 2;
	*(*cbuf)->s = new_delim ? *new_delim : *delim;
	if (!(only_delim && cbuf == buf)) {
		num = cbuf - buf + 1;
		if (!once) eval_fieldlist (num);
		for (i=1,field=fields; i<=fieldnum; i++,field++) {
			if (*field >= num) continue;
			if (*field == -1) {
				if (i==1) continue;
				for (j=*(field-1)+1; j<num; j++) {
					PUTC (delim_bak);
					delim_bak = *buf[j]->s;
					PRINT (buf[j]->buf, buf[j]->s - buf[j]->buf); }}
			else {
				if (!first) 
					PUTC (delim_bak);
				else first = 0;
				delim_bak = *buf[*field]->s;
				PRINT (buf[*field]->buf, buf[*field]->s - buf[*field]->buf); }}
		PUTC ('\n'); }
	for (cbuf=buf; *cbuf; cbuf++) {
		free ((*cbuf)->buf);
		free (*cbuf); }
	free (buf);
	return c != EOF;
}

int tline () {
	int c;
	int field=0, in=0, i;
	int *afield = fields;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if (all_fields) return print_line ();
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		if (!only_delim) PUTC (c);
		GETC(1);
		return 1; }
	if (field == *afield) in=1;
	while ((c=GETC (1)) != EOF && c != '\n') {
		if (nindex (delim, c, delimlen)) {
			field++;
			if (in && *afield != -1 && afield-fields+1 < fieldnum) afield++;
			in = field == *afield || *afield == -1;
			if (in && afield != fields)
				if (new_delim) PUTC(*new_delim);
				else PUTC (c); }
		else if (in) PUTC (c); }
	PUTC ('\n');
	return c != EOF;
}

int iline () {
	int bufsize=256;
	Tbuf ** buf;
	Tbuf ** cbuf;
	int c, i, j;
	int * field, num;
	int delim_bak;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		if (!only_delim) PRINT (text_to_insert, text_to_insert_len);
		PUTC (c);
		GETC (1);
		return 1; }
	cbuf = buf = rmalloc (bufsize * sizeof (void *));
	*cbuf = malloc (sizeof (Tbuf));
	(*cbuf)->size = 256;
	(*cbuf)->s = (*cbuf)->buf = malloc ((*cbuf)->size * sizeof (char));
	while ((c=GETC (1)) != EOF && c != '\n') {
		if ((i=(*cbuf)->s - (*cbuf)->buf) == (*cbuf)->size) {
			(*cbuf)->size *= 2;
			(*cbuf)->buf = realloc ((*cbuf)->buf, (*cbuf)->size * 
											sizeof (char));
			(*cbuf)->s = (*cbuf)->buf + i; }
		*(*cbuf)->s = c;
		(*cbuf)->s++;
		if (nindex (delim, c, delimlen)) {
			(*cbuf)->buf = realloc((*cbuf)->buf, ((*cbuf)->size = (*cbuf)->s -
					(*cbuf)->buf + 1) * sizeof (char));
			(*cbuf)->s = (*cbuf)->buf + (*cbuf)->size - 2;
			if (new_delim) *(*cbuf)->s = *new_delim;
			cbuf++;
			if ((i = cbuf - buf) == bufsize) {
				bufsize *= 2;
				buf = realloc (buf, bufsize * sizeof (void *));
				cbuf = buf + i; }
			*cbuf = malloc (sizeof (Tbuf));
			(*cbuf)->size = 256;
			(*cbuf)->s = (*cbuf)->buf = malloc ((*cbuf)->size * 
												sizeof (char)); }}
	(*cbuf)->buf = realloc((*cbuf)->buf, ((*cbuf)->size = (*cbuf)->s -
					(*cbuf)->buf + 2) * sizeof (char));
	(*cbuf)->s = (*cbuf)->buf + (*cbuf)->size - 2;
	*(*cbuf)->s = new_delim ? *new_delim : *delim;
	if (!(only_delim && cbuf == buf)) {
		num = cbuf - buf + 1;
		eval_fieldlist (num + 1);
		field = fields;
		for (i=0; i<num; i++) {
			if (i) PUTC (delim_bak);
			delim_bak = *buf[i]->s;
			if (i==*field) {
				PRINT (text_to_insert, text_to_insert_len);
				if (new_delim) PUTC(*new_delim);
				else PUTC (*delim);
				if (field - fields + 1 < fieldnum) field++; }
			PRINT (buf[i]->buf, buf[i]->s - buf[i]->buf); }
		if (*field == num) {
			if (i) PUTC (delim_bak);
			PRINT (text_to_insert, text_to_insert_len); }
		PUTC ('\n'); }
	for (cbuf=buf; *cbuf; cbuf++) {
		free ((*cbuf)->buf);
		free (*cbuf); }
	free (buf);
	return c != EOF;
}

int tiline () {
	int c;
	int field=0, in=0, i;
	int *afield = fields;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		if (!only_delim) PRINT (text_to_insert, text_to_insert_len);
		PUTC (c);
		GETC(1);
		return 1; }
	if (field == *afield) {
		in=1;
		PRINT (text_to_insert, text_to_insert_len);
		if (new_delim) PUTC(*new_delim);
		else PUTC (*delim); }
	while ((c=GETC (1)) != EOF && c != '\n') {
		if (nindex (delim, c, delimlen)) {
			field++;
			if (in && *afield != -1 && afield-fields+1 < fieldnum) afield++;
			in = field == *afield || *afield == -1;
			if (new_delim) PUTC(*new_delim);
			else PUTC (c); 
			if (in) {
				PRINT (text_to_insert, text_to_insert_len);
				if (new_delim) PUTC(*new_delim);
				else PUTC (*delim); }}
		else PUTC (c); }
	if (*afield == -1 || *afield > field || 
					afield-fields+1 < fieldnum && *(afield+1) > field) {
		if (new_delim) PUTC(*new_delim);
		else PUTC (*delim); 
		PRINT (text_to_insert, text_to_insert_len); }
	PUTC ('\n');
	return c != EOF;
}

int sline () {
	int bufsize=256;
	Tbuf ** buf;
	Tbuf ** cbuf;
	int c, i, j;
	int * field, num;
	int delim_bak;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		PUTC (c);
		GETC (1);
		return 1; }
	cbuf = buf = rmalloc (bufsize * sizeof (void *));
	*cbuf = malloc (sizeof (Tbuf));
	(*cbuf)->size = 256;
	(*cbuf)->s = (*cbuf)->buf = malloc ((*cbuf)->size * sizeof (char));
	while ((c=GETC (1)) != EOF && c != '\n') {
		if ((i=(*cbuf)->s - (*cbuf)->buf) == (*cbuf)->size) {
			(*cbuf)->size *= 2;
			(*cbuf)->buf = realloc ((*cbuf)->buf, (*cbuf)->size * 
											sizeof (char));
			(*cbuf)->s = (*cbuf)->buf + i; }
		*(*cbuf)->s = c;
		(*cbuf)->s++;
		if (nindex (delim, c, delimlen)) {
			(*cbuf)->buf = realloc((*cbuf)->buf, ((*cbuf)->size = (*cbuf)->s -
					(*cbuf)->buf + 1) * sizeof (char));
			(*cbuf)->s = (*cbuf)->buf + (*cbuf)->size - 2;
			if (new_delim) *(*cbuf)->s = *new_delim;
			cbuf++;
			if ((i = cbuf - buf) == bufsize) {
				bufsize *= 2;
				buf = realloc (buf, bufsize * sizeof (void *));
				cbuf = buf + i; }
			*cbuf = malloc (sizeof (Tbuf));
			(*cbuf)->size = 256;
			(*cbuf)->s = (*cbuf)->buf = malloc ((*cbuf)->size * 
												sizeof (char)); }}
	(*cbuf)->buf = realloc((*cbuf)->buf, ((*cbuf)->size = (*cbuf)->s -
					(*cbuf)->buf + 2) * sizeof (char));
	(*cbuf)->s = (*cbuf)->buf + (*cbuf)->size - 2;
	*(*cbuf)->s = new_delim ? *new_delim : *delim;
	if (!(only_delim && cbuf == buf)) {
		num = cbuf - buf + 1;
		eval_fieldlist (num);
		field = fields;
		for (i=0; i<num; i++) {
			if (i) PUTC (delim_bak);
			delim_bak = *buf[i]->s;
			if (i==*field) {
				PRINT (text_to_insert, text_to_insert_len);
				if (field - fields + 1 < fieldnum) field++; }
			else PRINT (buf[i]->buf, buf[i]->s - buf[i]->buf); }
		PUTC ('\n'); }
	for (cbuf=buf; *cbuf; cbuf++) {
		free ((*cbuf)->buf);
		free (*cbuf); }
	free (buf);
	return c != EOF;
}

int tsline () {
	int c;
	int field=0, in=0, i;
	int *afield = fields;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		PUTC (c);
		GETC(1);
		return 1; }
	if (field == *afield) {
		in=1;
		PRINT (text_to_insert, text_to_insert_len); }
	while ((c=GETC (1)) != EOF && c != '\n') {
		if (nindex (delim, c, delimlen)) {
			field++;
			if (in && *afield != -1 && afield-fields+1 < fieldnum) afield++;
			in = field == *afield || *afield == -1;
			if (new_delim) PUTC(*new_delim);
			else PUTC (c); 
			if (in) PRINT (text_to_insert, text_to_insert_len); }
		else if (!in) PUTC (c); }
	PUTC ('\n');
	return c != EOF;
}

int bline () {
	char * buf, * s;
	int buflen = 256;
	int i, j, c, num;
	int * field;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		PUTC (c);
		GETC (1);
		return 1; }
	s = buf = malloc (buflen * sizeof (char));
	while ((c=GETC (1)) != EOF && c != '\n') {
		if ((i=s-buf) == buflen) {
			buflen *= 2;
			buf = realloc (buf, buflen);
			s = buf + i; }
		*s = c;
		s++; }
	num = s - buf;
	if (!once) eval_fieldlist (num);
	for (i=1,field=fields; i<=fieldnum; i++,field++) {
		if (*field >= num) continue;
		if (*field == -1) {
			if (i==1) continue;
			for (j=*(field-1)+1; j<num; j++) 
				PUTC (buf[j]); }
		else
			PUTC (buf[*field]); }
	PUTC ('\n');
	free (buf);
	return c != EOF;
}

int tbline () {
	int c, i=0;
	int * field = fields;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if (all_fields) return print_line ();
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		PUTC (c);
		GETC (1);
		return 1; }
	while ((c=GETC (1)) != EOF && c != '\n') {
		if (i==*field) {
			PUTC (c);
			field++; }
		else if (*field == -1 && field != fields)
			PUTC (c); 
		i++; }
	PUTC ('\n');
	return c != EOF;
}

int ibline () {
	char * buf, * s, *s2;
	int buflen = 256;
	int i, j, c, num;
	int * field;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		if (!only_delim) PRINT (text_to_insert, text_to_insert_len);
		PUTC (c);
		GETC (1);
		return 1; }
	s = buf = malloc (buflen * sizeof (char));
	while ((c=GETC (1)) != EOF && c != '\n') {
		if ((i=s-buf) == buflen) {
			buflen *= 2;
			buf = realloc (buf, buflen);
			s = buf + i; }
		*s = c;
		s++; }
	num = s - buf;
	eval_fieldlist (num + 1);
	field=fields;
	for (s2=buf, i=0; s2<s; s2++, i++) {
		if (*field == i) {
			PRINT (text_to_insert, text_to_insert_len);
			if (field - fields + 1 < fieldnum) field++; }
		PUTC (*s2); }
	if (*field == num) 
		PRINT (text_to_insert, text_to_insert_len);
	PUTC ('\n');
	free (buf);
	return c != EOF;
}

int tibline () {
	int c, i=0;
	int * field = fields;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		if (!only_delim) PRINT (text_to_insert, text_to_insert_len);
		PUTC (c);
		GETC (1);
		return 1; }
	while ((c=GETC (1)) != EOF && c != '\n') {
		if (i==*field) {
			PRINT (text_to_insert, text_to_insert_len);
			if (field - fields + 1 < fieldnum) field++; }
		else if (*field == -1 && field != fields)
			PRINT (text_to_insert, text_to_insert_len);
		PUTC (c); 
		i++; }
	if (*field == i || *field == -1) 
		PRINT (text_to_insert, text_to_insert_len);
	PUTC ('\n');
	return c != EOF;
}

int sbline () {
	char * buf, * s, *s2;
	int buflen = 256;
	int i, j, c, num;
	int * field;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		PUTC (c);
		GETC (1);
		return 1; }
	s = buf = malloc (buflen * sizeof (char));
	while ((c=GETC (1)) != EOF && c != '\n') {
		if ((i=s-buf) == buflen) {
			buflen *= 2;
			buf = realloc (buf, buflen);
			s = buf + i; }
		*s = c;
		s++; }
	num = s - buf;
	eval_fieldlist (num);
	field=fields;
	for (s2=buf, i=0; s2<s; s2++, i++) {
		if (*field == i) {
			PRINT (text_to_insert, text_to_insert_len);
			if (field - fields + 1 < fieldnum) field++; 
			s2 += text_to_insert_len - (text_to_insert_len ? 1 : 0);
			i += text_to_insert_len;
			if (s2 < s) 
				while (*field < i && field - fields + 1 < fieldnum) 
					field++;
			i -= text_to_insert_len ? 1 : 0; }
		else PUTC (*s2); }
	PUTC ('\n');
	free (buf);
	return c != EOF;
}

int tsbline () {
	int c, i=0, j;
	int * field = fields;
	if (linelist && !isline()) {
		if (!only_linematch)
			return print_line ();
		else return eat_line (); }
	if ((c=GETC(2)) == EOF) return 0;
	else if (c=='\n') {
		PUTC (c);
		GETC (1);
		return 1; }
	while ((c=GETC (1)) != EOF && c != '\n') {
		if (i==*field) {
			PRINT (text_to_insert, text_to_insert_len);
			for (j=0; j+1<text_to_insert_len && c!=EOF && c!='\n'; j++)
				c = GETC (1);
			if (c==EOF || c=='\n') break;
			i += text_to_insert_len;
			while (*field < i && field - fields + 1 < fieldnum && *field!=-1) 
				field++;
			i -= text_to_insert_len ? 1 : 0; }
		else if (*field == -1 && field != fields) {
			PRINT (text_to_insert, text_to_insert_len);
			for (j=0; j+1<text_to_insert_len && c!=EOF && c!='\n'; j++)
				c = GETC (1);
			if (c==EOF || c=='\n') break;
			i += text_to_insert_len - (text_to_insert_len ? 1 : 0); }
		else PUTC (c); 
		i++; }
	PUTC ('\n');
	return c != EOF;
}

typedef struct _infile {
	char * file;
	struct _infile * next;
	} Tinfile;



usage (int mode) {
	switch (mode) {
	case 0:
		fprintf (stderr, "%s: ", PROG);
		fprintf (stderr, "usage: cut [<options>...] [<files>...]\n"
			"This is the Reker version of cut. \nIt has some additional "
			"features to traditional cut.\n\n"
			"It prints selected parts of each line from each input file"
			" to standard output.\n"
			"If no file is given or file is ``-'' read from stdin.\n"
			"For a list of available options call ``cut --usage''.\n"
			"For more information on the options read the man page.\n"
			"\nReport bugs to: Frank Reker <frank@reker.net>\n\n");
		break;
	case 1:
		fprintf (stderr, "%s: ", PROG);
		fprintf (stderr, "usage: cut [<options>...] [<files>...]\n"
			"options:\n"
			"\t -f, --fields <field_list>\n"
			"\t -d, --delimiter <delimiter_list>\n"
			"\t -b, --bytes <byte_list>\n"
			"\t -c, --characters <character_list>\n"
			"\t -l, --lines <line_list>\n"
			"\t -u, --new-delimiter <new_delimiter>\n"
			"\t -i, --insert <text_to_insert>\n"
			"\t -e, --substitute <text_to_insert>\n"
			"\t -n (ignored for now)\n"
			"\t -o, --order\n"
			"\t -r, --reorder\n"
			"\t -s, --only-delimited\n"
			"\t -m, --only-matched-lines\n"
			"\t -V, --ver, --version\n"
			"\t --copyright, --copyleft\n"
			"\t -h, --help\n"
			"\t --usage\n"
			"\t --man, --manual (prints a manual page to standard out)\n"
			"For more information on the options read the man page.\n"
			"\nReport bugs to: Frank Reker <frank@reker.net>\n\n");
		break; 
	case 2:
		printf ("%s", manual_string);
		break; }
	exit (0);
}

print_version (int mode) {
	switch (mode) {
	case 0:
		printf ("%s\n", VERSION);
		break;
	case 1:
		printf ("%s: version %s\n", PROG, VERSION);
		printf ("\twritten by Frank Reker <frank@reker.net>\n");
		break;
	case 2:
		printf ("%s: version %s\n", PROG, VERSION);
		printf ("\twritten by Frank Reker <frank@reker.net>\n");
		printf ("\tlast modified at: %s\n", LAST_MODIFIED_AT);
		printf ("\t              by  %s\n", LAST_MODIFIED_BY);
		printf ("\tcompiled at: %s  %s\n", __DATE__, __TIME__);
		break; }
	exit (0);
}

print_copyleft () {
	printf ("%s: version %s\n", PROG, VERSION);
	printf ("\twritten by Frank Reker <frank@reker.net>\n");
	printf ("\tlast modified at: %s\n", LAST_MODIFIED_AT);
	printf ("\t              by  %s\n", LAST_MODIFIED_BY);
	printf ("\tcompiled at: %s  %s\n", __DATE__, __TIME__);
	printf ("
    The Reker version of cut - like the standard UN*X cut plus add. features.
    Copyright (C) 2000  <Frank Reker>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
	\n");
	exit (0);
}

static const char * default_fieldstr = "-";
static const char * default_delim = " \t";
static const char * default_posix_delim = "\t";
	
int main (int argc, char ** argv) {	
	int i, j, bl=0, ins=0, subst=0, c;
	int posixly_correct=0;
	char * s, *s2;
	Tinfile * infiles = NULL, *p;
	fin = stdin;
	fout = stdout;
	PROG = (s=rindex (*argv, '/')) ? strdup (s+1) : strdup (*argv);
#ifndef REKER_STYLE
	if (!strcmp (PROG, "cut")) {
		order=1;
		posixly_correct = 1; }
#endif
	if (getenv ("POSIXLY_CORRECT")) {
		order=1;
		posixly_correct = 1; }
	else order=0;
	if (getenv ("REKER_STYLE")) {
		order=0;
		posixly_correct = 0; }
	for (i=1; i<argc; i++) {
		if (!strncmp (argv[i], "-d", 2)
#ifdef LONG_PARAM
					|| !strncmp (argv[i], "--delimiter", 11)
#endif
					) {
#ifdef LONG_PARAM
			if (argv[i][1] == '-') {
				if (argv[i][11] == '=' && argv[i][12])
					s = argv[i]+12;
				else if (*argv[++i] != '=')
					s = argv[i];
				else if (argv[i][1])
					s = argv[i]+1;
				else
					s = argv[++i]; }
			else
#endif
				if (!*(s = argv[i]+2) && ++i != argc)
					s = argv[i]; 
			if (index (s, '\\')) {
				delim = malloc (strlen (s));
				s2=delim;
				for (; *s; s++) {
					if (*s == '\\') {
						switch (*++s) {
						case 0: 
						case '\\': *s2++ = '\\'; break;
						case 'b': *s2++ = '\b'; break;
						case 'f': *s2++ = '\f'; break;
						case 'n': *s2++ = '\n'; break;
						case 'r': *s2++ = '\r'; break;
						case 't': *s2++ = '\t'; break;
						case '\'': *s2++ = '\''; break;
						case '\"': *s2++ = '\"'; break;
						case 'x': case 'X':
							if (*(s+1) && *(s+2)) {
								if (*++s >= 'a') j=16 * (*s - 'a' + 10);
								else if (*s >= 'A') j=16 * (*s - 'A' + 10);
								else j=16 * (*s - '0');
								if (*++s >= 'a') j += *s - 'a' + 10;
								else if (*s >= 'A') j += *s - 'A' + 10;
								else j += *s - '0';
								*s2++ = j; }
							break;
						default:
							if (*(s+1) && *(s+2) && isdigit (*s) &&
									isdigit (*(s+1)) && isdigit (*(s+2))) {
								j=64*(*s - '0') + 8*(*(s+1) - '0') + 
															*(s+2) - '0';
								s+=2;
								*s2++ = j; }
							else if (*s == '0')
								*s2++ = 0; 
							break; }}
					else *s2++ = *s; }
				delimlen = s2 - delim; }
			else {
				delim = s; 
				delimlen = strlen (s); }}
		else if (!strncmp (argv[i], "-u", 2)
#ifdef LONG_PARAM
					|| !strncmp (argv[i], "--new-delimiter", 15)
#endif
					) {
#ifdef LONG_PARAM
			if (argv[i][1] == '-') {
				if (argv[i][15] == '=' && argv[i][16])
					s = argv[i]+16;
				else if (*argv[++i] != '=')
					s = argv[i];
				else if (argv[i][1])
					s = argv[i]+1;
				else
					s = argv[++i]; }
			else
#endif
				if (!*(s = argv[i]+2) && ++i != argc)
					s = argv[i]; 
			if (index (s, '\\')) {
				new_delim = malloc (strlen (s));
				s2=new_delim;
				for (; *s; s++) {
					if (*s == '\\') {
						switch (*++s) {
						case 0: 
						case '\\': *s2++ = '\\'; break;
						case 'b': *s2++ = '\b'; break;
						case 'f': *s2++ = '\f'; break;
						case 'n': *s2++ = '\n'; break;
						case 'r': *s2++ = '\r'; break;
						case 't': *s2++ = '\t'; break;
						case '\'': *s2++ = '\''; break;
						case '\"': *s2++ = '\"'; break;
						case 'x': case 'X':
							if (*(s+1) && *(s+2)) {
								if (*++s >= 'a') j=16 * (*s - 'a' + 10);
								else if (*s >= 'A') j=16 * (*s - 'A' + 10);
								else j=16 * (*s - '0');
								if (*++s >= 'a') j += *s - 'a' + 10;
								else if (*s >= 'A') j += *s - 'A' + 10;
								else j += *s - '0';
								*s2++ = j; }
							break;
						default:
							if (*(s+1) && *(s+2) && isdigit (*s) &&
									isdigit (*(s+1)) && isdigit (*(s+2))) {
								j=64*(*s - '0') + 8*(*(s+1) - '0') + 
															*(s+2) - '0';
								s+=2;
								*s2++ = j; }
							else if (*s == '0')
								*s2++ = 0; 
							break; }}
					else *s2++ = *s; }}
			else new_delim = s;  }
		else if (!strncmp (argv[i], "-f", 2)) {
			if (!*(fieldstr=argv[i]+2) && ++i != argc)
				fieldstr=argv[i]; }
#ifdef LONG_PARAM
		else if (!strncmp (argv[i], "--fields", 8)) {
			if (argv[i][8] == '=' && argv[i][9])
				fieldstr = argv[i]+9;
			else if (*argv[++i] != '=')
				fieldstr = argv[i];
			else if (argv[i][1])
				fieldstr = argv[i]+1;
			else
				fieldstr = argv[++i]; }
#endif
		else if (!strncmp (argv[i], "-b", 2)
					|| !strncmp (argv[i], "-c", 2)) {
			if (!*(fieldstr=argv[i]+2) && ++i != argc)
				fieldstr=argv[i]; 
			bl = 1; }
#ifdef LONG_PARAM
		else if (!strncmp (argv[i], "--bytes", 7)) {
			if (argv[i][7] == '=' && argv[i][8])
				fieldstr = argv[i]+8;
			else if (*argv[++i] != '=')
				fieldstr = argv[i];
			else if (argv[i][1])
				fieldstr = argv[i]+1;
			else
				fieldstr = argv[++i];
			bl = 1; }
		else if (!strncmp (argv[i], "--characters", 12)) {
			if (argv[i][12] == '=' && argv[i][13])
				fieldstr = argv[i]+13;
			else if (*argv[++i] != '=')
				fieldstr = argv[i];
			else if (argv[i][1])
				fieldstr = argv[i]+1;
			else
				fieldstr = argv[++i];
			bl = 1; }
#endif
		else if (!strncmp (argv[i], "-i", 2)) {
			if (!*(text_to_insert=argv[i]+2) && ++i != argc)
				text_to_insert=argv[i]; 
			ins = 1; }
#ifdef LONG_PARAM
		else if (!strncmp (argv[i], "--insert", 8)) {
			if (argv[i][8] == '=' && argv[i][9])
				text_to_insert = argv[i]+9;
			else if (*argv[++i] != '=')
				text_to_insert = argv[i];
			else if (argv[i][1])
				text_to_insert = argv[i]+1;
			else
				text_to_insert = argv[++i];
			ins = 1; }
#endif
		else if (!strncmp (argv[i], "-e", 2)) {
			if (!*(text_to_insert=argv[i]+2) && ++i != argc)
				text_to_insert=argv[i]; 
			subst = 1; }
#ifdef LONG_PARAM
		else if (!strncmp (argv[i], "--substitute", 12)) {
			if (argv[i][12] == '=' && argv[i][13])
				text_to_insert = argv[i]+13;
			else if (*argv[++i] != '=')
				text_to_insert = argv[i];
			else if (argv[i][1])
				text_to_insert = argv[i]+1;
			else
				text_to_insert = argv[++i];
			subst = 1; }
#endif
		else if (!strncmp (argv[i], "-l", 2)) {
			if (!*(linestr=argv[i]+2) && ++i != argc)
				linestr=argv[i]; 
			eval_linelist (); }
#ifdef LONG_PARAM
		else if (!strncmp (argv[i], "--lines", 7)) {
			if (argv[i][7] == '=' && argv[i][8])
				linestr = argv[i]+8;
			else if (*argv[++i] != '=')
				linestr = argv[i];
			else if (argv[i][1])
				linestr = argv[i]+1;
			else
				linestr = argv[++i];
			eval_linelist (); }
#endif
		else if (!strncmp (argv[i], "-n", 2)) {
			/* ignore for now */ }
		else if (!strncmp (argv[i], "-s", 2)) {
			only_delim = 1; }
		else if (!strncmp (argv[i], "-o", 2)) {
			order = 1; }
		else if (!strncmp (argv[i], "-r", 2)) {
			order = 0; }
		else if (!strncmp (argv[i], "-m", 2)) {
			only_linematch = 1; }
		else if (argv[i][0] == '-' && isalpha (argv[i][1]) && 
									isalpha (argv[i][2])) {
			for (s=&argv[i][1]; *s; s++) {
				if (*s == 'n') {
					/* ignore for now */ }
				else if (*s == 's')
					only_delim = 1;
				else if (*s == 'o')
					order = 1;
				else if (*s == 'r')
					order = 0;
				else if (*s == 'm')
					only_linematch = 1;
				else if (*s == 'V')
					print_version (0);
				else if (*s == 'h')
					usage (0);
				else {
					fprintf (stderr, "%s: Unknown parameter -%c\n",
												PROG, s);
					exit (1); }}}
#ifdef LONG_PARAM
		else if	(!strcmp (argv[i], "--only-delimited")) {
			only_delim = 1; }
		else if	(!strcmp (argv[i], "--only-matched-lines")) {
			only_linematch = 1; }
		else if (!strcmp (argv[i], "--order")) {
			order = 1; }
		else if (!strcmp (argv[i], "--reorder")) {
			order = 0; }
#endif
		else if (!strcmp (argv[i], "-h") ||
					!strcmp (argv[i], "--help")) {
			usage (0); }
		else if (!strcmp (argv[i], "--usage")) {
			usage (1); }
		else if (!strcmp (argv[i], "--man") ||
					!strcmp (argv[i], "--manual")) {
			usage (2); }
		else if (!strcmp (argv[i], "-V")) {
			print_version (0); }
		else if (!strcmp (argv[i], "--ver")) {
			print_version (1); }
		else if (!strcmp (argv[i], "--version")) {
			print_version (2); }
		else if (!strcmp (argv[i], "--copyright") ||
					!strcmp (argv[i], "--copyleft")) {
			print_copyleft (); }
		else if (!strcmp (argv[i], "--")) {
			for (i++; i<argc; i++) {
				if (!infiles) 
					p = infiles = rmalloc (sizeof (Tinfile));
				else 
					p = p->next = rmalloc (sizeof (Tinfile));
				p->file = argv[i]; }}
		else {
			if (argv[i][0] == '-' && argv[i][1] != 0) {
				fprintf (stderr, "%s: illegal option ``%s''\n", PROG,
										argv[i]);
				exit (1); }
			if (!infiles) 
				p = infiles = rmalloc (sizeof (Tinfile));
			else 
				p = p->next = rmalloc (sizeof (Tinfile));
			p->file = argv[i]; }}
	if (!fieldstr || !*fieldstr) fieldstr = (char*) default_fieldstr;
	if (!delim || !delimlen) {
		if (posixly_correct) {
			delim = (char*) default_posix_delim;
			delimlen = 1; }
		else {
			delim = (char*) default_delim;
			delimlen = 2; }}
	if (posixly_correct) 
		delimlen = 1;		/* only the first delimiter is interpreted */
	if (ins || subst) {
		order = 1;
		text_to_insert_len = strlen (text_to_insert); }
	preprocfl ();
	if (!infiles) {
		fin = stdin;
		GETC (0);
		if (bl)	{
			if (ins) {
				if (sorted)	while (tibline()); 
				else		while (ibline()); }
			else if (subst) {
				if (sorted)	while (tsbline()); 
				else		while (sbline()); }
			else if (all_fields && !linelist)
								while ((c=GETC (1))!=EOF) PUTC (c);
			else if (sorted)	while (tbline()); 
			else				while (bline()); }
		else if (ins) {
			if (sorted)	while (tiline()); 
			else		while (iline()); }
		else if (subst) {
			if (sorted)	while (tsline()); 
			else		while (sline()); }
		else {
			if (all_fields && !linelist)
								while ((c=GETC (1))!=EOF) PUTC (c);
			else if (sorted)	while (tline()); 
			else				while (line()); }
		FLUSH (); }
	else {
		for (p=infiles; p; p=p->next) {
			if (!strcmp (p->file, "-")) 
				fin = stdin;
			else if (!(fin=fopen (p->file, "r"))) {
				fprintf (stderr, "%s: cannot open file ``%s\'\'",
							PROG, p->file);
				switch (errno) {
				case ETXTBSY:
					fprintf (stderr, ": file is busy\n"); break;
				case EACCES:
					fprintf (stderr, ": permission denied\n"); break;
				case ENAMETOOLONG:
					fprintf (stderr, ": filename is too long\n"); break;
				case ENOENT:
				case ENOTDIR:
				case ELOOP:
					fprintf (stderr, ": file does not exist\n"); break;
				case EMFILE:
				case ENFILE:
					fprintf (stderr, ": too many files open\n"); break;
				default:
					fprintf (stderr, "\n"); break; }
				continue; }
			GETC (0);
			if (bl)	{
				if (ins) {
					if (sorted)	while (tibline()); 
					else		while (ibline()); }
				else if (subst) {
					if (sorted)	while (tsbline()); 
					else		while (sbline()); }
				else if (all_fields && !linelist)
									while ((c=GETC (1))!=EOF) PUTC (c);
				else if (sorted)	while (tbline()); 
				else				while (bline()); }
			else if (ins) {
				if (sorted)	while (tiline()); 
				else		while (iline()); }
			else if (subst) {
				if (sorted)	while (tsline()); 
				else		while (sline()); }
			else {
				if (all_fields && !linelist)
									while ((c=GETC (1))!=EOF) PUTC (c);
				else if (sorted)	while (tline()); 
				else				while (line()); }
			if (fin != stdin) fclose (fin); 
			FLUSH (); }}
	return 0;
}


















/*
 * Overrides for XEmacs and vim so that we get a uniform tabbing style.
 * XEmacs/vim will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 4
 * c-basic-offset: 4
 * tab-width: 4
 * End:
 * vim:tw=0:ts=4:wm=0:
 */

