
/*

    File: pop3/pop3.c
  
    Copyright (C) 1999  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <signal.h>
#include <netdb.h>
#include <netinet/in.h>
#include <syslog.h>
#include <sys/time.h>
#include <wait.h>

#include "pop3.h"
#include "ip-lib.h"
#include "lib.h"


#define	CLIENT_INPUT		1
#define	SERVER_INPUT		2

int	waitingfor		= 0;



unsigned get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "can't get interface info: %m");

		copy_string(ip, "local.host", max);
		port = 0;

		return (port);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}



void sigalrm_handler(int sig)
{
	char	*p;

	if (waitingfor == CLIENT_INPUT)
		p = "while waiting for client";
	else if (waitingfor == SERVER_INPUT)
		p = "while waiting for server";
	else
		p = "in unknown state";

	syslog(LOG_NOTICE, "-ERR: timeout %s", p);
	exit (-1);
}

int waitfor(int what, int seconds)
{
	waitingfor = what;
	alarm(seconds);
	signal(SIGALRM, sigalrm_handler);

	return (0);
}


int open_connection(pop3_t *x, char *server)
{
	if (x->config->type == IP_SERVER) {
		snprintf (x->server.name, sizeof(x->server.name) - 2, "%s:%u", x->server.u.ip.hostname, x->server.u.ip.port);
		x->server.u.ip.fp = ip_open(x->server.u.ip.hostname, x->server.u.ip.port);
		if (x->server.u.ip.fp == NULL) {
			syslog(LOG_NOTICE, "-ERR: can't connect to server: %s, user= %s", x->server.name, x->client.username);
			return (1);
			}

		x->server.type = IP_SERVER;
		}
	else {
		syslog(LOG_NOTICE, "-ERR: local server not supported, user= %s", x->client.username);
		return (1);
		}

	if (verbose)
		syslog(LOG_NOTICE, "proxy connected to server: %s", x->server.name);

	return (0);
}

int close_connection(pop3_t *x)
{
	if (x->server.type == IP_SERVER) {
		fclose(x->server.u.ip.fp);
		}
	else {
		syslog(LOG_NOTICE, "-ERR: local server not supported");
		exit (1);
		}

	return (0);
}



int sfputs(pop3_t *x, char *line)
{
	int	len;

	if (debug != 0)
		fprintf (stderr, ">>> SVR: %s\n", line);

	if (x->server.type == IP_SERVER) {
/*		rewind(x->server.u.ip.fp); */
		len = fprintf (x->server.u.ip.fp, "%s\r\n", line);
		fflush(x->server.u.ip.fp);
		}
	else {
		syslog(LOG_NOTICE, "-ERR: local server not supported");
		exit (1);
		}

	if (len != strlen(line) + 2) {
		syslog(LOG_NOTICE, "-ERR: lost server while writing: %s", x->server.name);
		exit (-1);
		}
		
	return (0);
}

char *sfgets(pop3_t *x, char *line, int size)
{
	waitfor(SERVER_INPUT, x->config->timeout);
	*line = 0;
	if (x->server.type == IP_SERVER) {
/*		rewind(x->server.u.ip.fp); */
		if (fgets(line, size, x->server.u.ip.fp) == NULL) {
			syslog(LOG_NOTICE, "-ERR: server closed connection: %s", x->server.name);
			exit (1);
			}

		noctrl(line);
		}
	else {
		syslog(LOG_NOTICE, "-ERR: local server not supported");
		exit (1);
		}

	waitfor(0, 0);
	if (debug != 0)
		fprintf (stderr, "<<< SVR: %s\n", line);

	return (line);
}

int sfget_response(pop3_t *x, char *line, int size)
{
	char	*p, word[80];

	p = sfgets(x, line, size);
	get_word(&p, word, sizeof(word));
	if (strcasecmp(word, "-ERR") == 0)
		return (1);
	else if (strcasecmp(word, "+OK") != 0) {
		syslog(LOG_NOTICE, "-ERR: invalid response from server: %s", word);
		exit (1);
		}

	return (0);
}

int sfputcmd(pop3_t *x, char *command, char *par, char *line, int size, char **here)
{
	int	rc;
	char	cmdline[200];

	if (par == NULL  ||  *par == 0)
		copy_string(cmdline, command, sizeof(cmdline));
	else
		snprintf (cmdline, sizeof(cmdline) - 2, "%s %s", command, par);

	sfputs(x, cmdline);
	rc = sfget_response(x, line, size);

	if (here != NULL)
		*here = line;

	return (rc);
}


char *cfgets(pop3_t *x, char *line, int size)
{
	waitfor(CLIENT_INPUT, 60);
	if (fgets(line, size, stdin) == NULL) {
		syslog(LOG_NOTICE, "-ERR: lost client while reading: %s", x->client.name);
		exit (-1);
		}

	waitfor(0, 0);
	noctrl(line);

	if (size >= 100)
		line[100] = 0;			/* RFC 1939 */

	if (debug != 0)
		fprintf (stderr, "<<< CLI: %s\n", line);

	return (line);
}

int cfputs(pop3_t *x, char *line)
{
	int	len;

	if (debug != 0)
		fprintf (stderr, ">>> CLI: %s\n", line);

	len = printf ("%s\r\n", line);
	fflush(stdout);
	if (len != strlen(line) + 2) {
		syslog(LOG_NOTICE, "-ERR: lost client while writing: %s", x->client.name);
		exit (-1);
		}
		
	return (0);
}


int log_clientsuccess(pop3_t *x)
{
	unsigned long now;
	FILE	*fp;

	if (*x->logfile == 0)
		return (0);

	if ((fp = fopen(x->logfile, "w")) == NULL)
		syslog(LOG_NOTICE, "can't open file: %s", x->logfile);
	else {
		now = time(NULL);
		fprintf (fp, "%lu %s %s %s\n", now, x->client.ipnum, x->client.username, x->server.name);
		fclose (fp);
		}

	return (0);
}

int search_allowlist(char *server, char *list)
{
	char	*p, pattern[200];

	if (list == NULL  ||  *list == 0) {
		
		/*
		 * Kann eigentlich auch nicht vorkommen, wird vorher
		 * getestet.
		 */

		return (0);
		}

	p = list;
	while ((p = skip_ws(p)), *get_quoted(&p, ',', pattern, sizeof(pattern)) != 0) {
		noctrl(pattern);
		if (strpcmp(server, pattern) == 0)
			return (1);
		}
	
	return (0);
}

void doquit(pop3_t *x)
{
	cfputs(x, "+OK closing the connection");
	syslog(LOG_NOTICE, "+OK: closing connection to %s", x->client.name);
	close_connection(x);

	exit (0);
}


int set_variables(pop3_t *x)
{
	char	*vp, var[200], val[200];

	vp = x->config->varname;

	snprintf (var, sizeof(var) - 2, "%sINTERFACE", vp);
	setenv(var, x->interface, 1);
	
	snprintf (val, sizeof(val) - 2, "%u", x->port);
	snprintf (var, sizeof(var) - 2, "%sPORT", vp);
	setenv(var, val, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENT", vp);
	setenv(var, x->client.ipnum, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENTNAME", vp);
	setenv(var, x->client.name, 1);

	snprintf (var, sizeof(var) - 2, "%sSERVER", vp);
	setenv(var, x->server.u.ip.hostname, 1);
	
	snprintf (val, sizeof(val) - 2, "%u", x->server.u.ip.port);
	snprintf (var, sizeof(var) - 2, "%sSERVERPORT", vp);
	setenv(var, val, 1);

	snprintf (var, sizeof(var) - 2, "%sSERVERLOGIN", vp);
	setenv(var, x->client.username, 1);
	
	snprintf (var, sizeof(var) - 2, "%sUSERNAME", vp);
	setenv(var, x->local.username, 1);
	
	snprintf (var, sizeof(var) - 2, "%sPASSWD", vp);
	setenv(var, x->local.password, 1);
	
	return (0);
}

int run_acp(pop3_t *x)
{
	int	rc, pid, pfd[2];
	char	line[300];
	
	if (*x->config->acp == 0)
		return (0);

	rc = 0;
	if (pipe(pfd) != 0) {
		syslog(LOG_NOTICE, "-ERR: can't pipe: %m");
		exit (1);
		}
	else if ((pid = fork()) < 0) {
		syslog(LOG_NOTICE, "-ERR: can't fork acp: %m");
		exit (1);
		}
	else if (pid == 0) {
		int	argc;
		char	*argv[32];

		close(0);		/* Das acp kann nicht vom client lesen. */
		dup2(pfd[1], 2);	/* stderr wird vom parent gelesen. */
		close(pfd[0]);
		set_variables(x);
		
		copy_string(line, x->config->acp, sizeof(line));
		argc = split(line, argv, ' ', 30);
		argv[argc] = NULL;
		execvp(argv[0], argv);

		syslog(LOG_NOTICE, "can't exec acp %s: %m", argv[0]);
		exit (1);
		}
	else {
		int	len;
		char	message[300];

		close(pfd[1]);
		*message = 0;
		if ((len = read(pfd[0], message, sizeof(message) - 2)) < 0)
			len = 0;

		message[len] = 0;
		noctrl(message);
		

		if (waitpid(pid, &rc, 0) < 0) {
			syslog(LOG_NOTICE, "-ERR: error while waiting for acp: %m");
			exit (1);
			}

		rc = WIFEXITED(rc) != 0? WEXITSTATUS(rc): 1;
		if (*message == 0)
			copy_string(message, rc == 0? "access granted": "access denied", sizeof(message));

		if (*message != 0)
			syslog(LOG_NOTICE, "%s (rc= %d)", message, rc);
		}
		
	return (rc);
}


int dologin(pop3_t *x)
{
	int	k;
	char	*p, word[80], line[300];

	/*
	 * Wir lesen zunaechst den Benutzernamen und das Passwort vom
	 * Client.
	 */

	k = 0;
	while (1) {
		if (k >= 6) {
			syslog(LOG_NOTICE, "-ERR: too many login failures: %s", x->client.name);
			exit (1);
			}

		k++;
		p = cfgets(x, line, sizeof(line));
		get_word(&p, word, sizeof(word));
		strupr(word);

		if (strcmp(word, "USER") == 0) {
			get_word(&p, x->client.username, sizeof(x->client.username));
			cfputs(x, "+OK password required");
			}
		else if (strcmp(word, "PASS") == 0) {
			if (*x->client.username == 0) {
				cfputs(x, "-ERR give USER first");
				continue;
				}

			/*
			 * Der Statusreply kommt weiter unten, wenn wir wissen,
			 * ob der Server existiert und der User-Login ok ist.
			 */

			get_word(&p, x->client.password, sizeof(x->client.password));
			break;
			}
		else if (strcmp(word, "QUIT") == 0) {
			doquit(x);
			}
		else {
			cfputs(x, "-ERR login first");
			}
		}


	/*
	 * Im selectserver Modus den Servernamen holen.
	 */

	if (x->config->selectserver != 0) {
		if (*x->config->serverdelim == 0)
			strcpy(x->config->serverdelim, "@");

		if ((k = strcspn(x->client.username, x->config->serverdelim)) > 0) {
			x->client.username[k++] = 0;
			copy_string(x->server.name, &x->client.username[k], sizeof(x->server.name));
			}

		if (*x->server.name == 0) {

			/*
			 * Kein Server ausgewaehlt.
			 */

			copy_string(x->server.name, x->config->server, sizeof(x->server.name));
			}


		/*
		 * Nochmal pruefen, ob jetzt der Server gesetzt ist.
		 */

		if (*x->server.name == 0  ||  strcmp(x->server.name, "-") == 0) {
			cfputs(x, "-ERR bad login");
			syslog(LOG_NOTICE, "-ERR: no server selected, client= %s, user= %s", x->client.name, x->client.username);
			exit (1);
			}

		/*
		 * Wenn eine Serverliste vorhanden ist, den Servername suchen.
		 */

		if (x->config->serverlist != NULL  &&  *x->config->serverlist != 0) {
			if (search_allowlist(x->server.name, x->config->serverlist) == 0) {
				cfputs(x, "-ERR bad login");
				syslog(LOG_NOTICE, "-ERR: server not on list: %s, user= %s", x->server.name, x->client.username);
				exit (1);
				}
			}
		}
	else
		copy_string(x->server.name, x->config->server, sizeof(x->server.name));


	/*
	 * Servernamen und Port holen
	 */

	if (*x->server.name == 0  ||  strcmp(x->server.name, "-") == 0) {
		syslog(LOG_NOTICE, "-ERR: server empty or unset, user= %s", x->client.username);
		return (1);
		}

	copy_string(x->server.u.ip.hostname, x->server.name, sizeof(x->server.u.ip.hostname));
	x->server.u.ip.port = get_port(x->server.u.ip.hostname, 110);



	/*
	 * Wenn vorhanden, Proxy Benutzernamen und Passwort holen.
	 */

	if ((p = strchr(x->client.username, ':')) != NULL) {
		*p++ = 0;
		copy_string(x->local.username, x->client.username, sizeof(x->local.username));
		copy_string(x->client.username, p, sizeof(x->client.username));
		}

	if ((p = strchr(x->client.password, ':')) != NULL) {
		*p++ = 0;
		copy_string(x->local.password, x->client.password, sizeof(x->local.password));
		copy_string(x->client.password, p, sizeof(x->client.password));
		}


	/*
	 * Ggf. access control programm starten
	 */

	if (*x->config->acp != 0) {
		if (run_acp(x) != 0)
			exit (0);
		}

	/*
	 * Ok, Verbindung zum Server aufbauen ...
	 */

	x->server.connected = 0;
	if (open_connection(x, x->server.name) != 0) {
		cfputs(x, "-ERR bad login");
		exit (1);
		}

	/*
	 * ... und Server-Greeting testen ...
	 */

	p = sfgets(x, line, sizeof(line));
	get_word(&p, word, sizeof(word));
	strupr(word);
	if (strcmp(word, "+OK") != 0) {
		cfputs(x, "-ERR bad login");
		syslog(LOG_NOTICE, "-ERR: unexpected response from server during open: %s, user= %s", word, x->client.username);

		exit (1);
		}


	/*
	 * ... und login senden.
	 */

	if (sfputcmd(x, "USER", x->client.username, line, sizeof(line), &p) != 0) {
		cfputs(x, "-ERR bad login");
		syslog(LOG_NOTICE, "-ERR: server error on USER: %s, user= %s", p, x->client.username);
		exit (1);
		}
	else if (sfputcmd(x, "PASS", x->client.password, line, sizeof(line), &p) != 0) {
		cfputs(x, "-ERR bad login");
		syslog(LOG_NOTICE, "-ERR: login failure: client= %s, user= %s, reason: %s", x->client.name, x->client.username, p);
		exit (1);
		}

	x->server.connected = 1;
	cfputs(x, "+OK maildrop ready");
	if (verbose)
		syslog(LOG_NOTICE, "user logged in: %s", x->client.username);

	return (0);
}


unsigned int checknumber(char *string)
{
	char	*p;
	unsigned int num;

	num = strtoul(string, &p, 10);
	if (*p != 0)
		return (0);
	
	return (num);
}

int spoolresponse(pop3_t *x)
{
	char	line[800];

	while (sfgets(x, line, sizeof(line)) != NULL) {
		cfputs(x, line);
		if (strcmp(line, ".") == 0)
			break;
		}
		
	return (0);
}

int proxy_request(config_t *config)
{
	int	errcount;
	char	*p, word[80], command[10], line[600];
	pop3_t	*x;

	x = allocate(sizeof(pop3_t));
	x->config = config;

	get_clientinfo(0, x->client.ipnum, x->client.name);
	syslog(LOG_NOTICE, "proxy connected to client: %s", x->client.name);
	x->port = get_interface_info(0, x->interface, sizeof(x->interface));

	if (*x->config->clientdir != 0) {
		struct stat sbuf;

		snprintf (x->logfile, sizeof(x->logfile) - 2, "%s/%s", x->config->clientdir, x->client.ipnum);
		if (stat(x->logfile, &sbuf) == 0) {
			if (unlink(x->logfile) != 0)
				syslog(LOG_NOTICE, "can't remove logfile: %s", x->logfile);
			}
		}


	/*
	 * Server-Greeting an Client schicken, Login-Daten lesen, Verbindung
	 * zum Server aufbauen und dort einloggen.
	 */

	cfputs(x, "+OK server ready");
	dologin(x);
	log_clientsuccess(x);

	errcount = 0;
	while (cfgets(x, line, sizeof(line)) != NULL) {
		p = line;
		get_word(&p, command, sizeof(command));
		strupr(command);

		if (strcmp(command, "NOOP") == 0) {
			if (sfputcmd(x, "NOOP", "", line, sizeof(line), &p) != 0) {
				p = line;
				get_word(&p, word, sizeof(word));
				syslog(LOG_NOTICE, "-ERR: server returned error response to NOOP", word);

				cfputs(x, "-ERR server error");
				close_connection(x);
				exit (-1);
				}

			cfputs(x, "+OK");
			}
		if (strcmp(command, "RSET") == 0) {
			if (sfputcmd(x, "RSET", "", line, sizeof(line), &p) != 0) {
				p = line;
				get_word(&p, word, sizeof(word));
				syslog(LOG_NOTICE, "-ERR: server returned error response to RSET: %s", word);

				cfputs(x, "-ERR server error");
				close_connection(x);
				exit (-1);
				}

			cfputs(x, "+OK");
			}
		else if (strcmp(command, "QUIT") == 0) {
			if (sfputcmd(x, "QUIT", "", line, sizeof(line), &p) != 0) {
				get_word(&p, word, sizeof(word));
				syslog(LOG_NOTICE, "server returned error response to QUIT: %s", word);
				break;
				}

			log_clientsuccess(x);
			doquit(x);
			}
		else if (strcmp(command, "DELE") == 0) {
			get_word(&p, word, sizeof(word));
			if (*word == 0) {
				syslog(LOG_NOTICE, "missing DELE parameter, client= %s", x->client.name);
				cfputs(x, "-ERR missing parameter");
				}
			else if (strtoul(word, &p, 10) == 0  ||  *p != 0) {
				syslog(LOG_NOTICE, "invalid DELE parameter: %s", word);
				cfputs(x, "-ERR bad parameter");
				}
			else if (sfputcmd(x, "DELE", word, line, sizeof(line), &p) == 0)
				cfputs(x, "+OK message marked for deletition");
			else
				cfputs(x, "-ERR");
			}
		else if (strcmp(command, "LIST") == 0  ||  strcmp(command, "UIDL") == 0) {
			get_word(&p, word, sizeof(word));
			if (*word == 0) {
				if (sfputcmd(x, command, "", line, sizeof(line), NULL) != 0)
					cfputs(x, line);
				else {
					cfputs(x, line);
					spoolresponse(x);
					}
				}
			else {
				if (*(p = skip_ws(p)) != 0) {
					syslog(LOG_NOTICE, "too many arguments to %s: %s", command, p);
					cfputs(x, "-ERR too many arguments");
					}
				else if (checknumber(word) == 0) {
					syslog(LOG_NOTICE, "bad parameter to %s: %s", command, word);
					cfputs(x, "-ERR bad argument");
					}
				else {
					sfputcmd(x, command, word, line, sizeof(line), NULL);
					cfputs(x, line);
					}
				}
			}
		else if (strcmp(command, "RETR") == 0) {
			get_word(&p, word, sizeof(word));
			if (*(p = skip_ws(p)) != 0) {
				syslog(LOG_NOTICE, "too many parameters to RETR");
				cfputs(x, "-ERR too many parameters");
				}
			else if (*word == 0) {
				syslog(LOG_NOTICE, "missing RETR parameter");
				cfputs(x, "-ERR missing parameter");
				}
			else if (checknumber(word) == 0) {
				syslog(LOG_NOTICE, "invalid RETR parameter: %s", word);
				cfputs(x, "-ERR bad parameter");
				}
			else if (sfputcmd(x, "RETR", word, line, sizeof(line), &p) != 0)
				cfputs(x, "-ERR no such message");
			else {
				cfputs(x, "+OK");
				spoolresponse(x);
				}
			}
		else if (strcmp(command, "STAT") == 0) {
			if (*(p = skip_ws(p)) != 0) {
				syslog(LOG_NOTICE, "too many arguments to STAT: %s", p);
				cfputs(x, "-ERR");
				}
			else if (sfputcmd(x, "STAT", "", line, sizeof(line), &p) != 0)
				cfputs(x, "-ERR");
			else {
				unsigned long msg, bytes;
				
				get_word(&p, word, sizeof(word));
				msg = strtoul(p, &p, 10);
				bytes = strtoul(p, &p, 10);

				snprintf (line, sizeof(line) - 2, "+OK %lu %lu", msg, bytes);
				cfputs(x, line);
				}
			}
		else if (strcmp(command, "TOP") == 0) {
			unsigned int msgno, lines;

			get_word(&p, word, sizeof(word));
			if ((msgno = checknumber(word)) == 0) {
				syslog(LOG_NOTICE, "bad msgno to TOP: %s", word);
				cfputs(x, "-ERR bad argument");
				}
			else if (get_word(&p, word, sizeof(word)), *(p = skip_ws(p)) != 0) {
				syslog(LOG_NOTICE, "too many arguments to TOP: %s", p);
				cfputs(x, "-ERR too many arguments");
				}
			else if (*word == 0) {
				syslog(LOG_NOTICE, "missing linecount to TOP");
				cfputs(x, "-ERR missing argument");
				}
			else if (lines = strtoul(word, &p, 10), *p != 0) {
				syslog(LOG_NOTICE, "bad linecount to TOP: %s", word);
				cfputs(x, "-ERR bad argument");
				}
			else {
				char	param[80];

				snprintf (param, sizeof(param) - 2, "%lu %lu", msgno, lines);
				if (sfputcmd(x, "TOP", param, line, sizeof(line), NULL) != 0)
					cfputs(x, "-ERR no such message");
				else {
					cfputs(x, line);
					spoolresponse(x);
					}
				}
			}
		else if (strcmp(command, "LAST") == 0) {
			if (*(p = skip_ws(p)) != 0) {
				syslog(LOG_NOTICE, "too many arguments to LAST: %s", p);
				cfputs(x, "-ERR");
				}
			else if (sfputcmd(x, "LAST", "", line, sizeof(line), &p) != 0)
				cfputs(x, "-ERR");
			else {
				unsigned long msg;
				
				get_word(&p, word, sizeof(word));
				msg = strtoul(p, &p, 10);

				snprintf (line, sizeof(line) - 2, "+OK %lu", msg);
				cfputs(x, line);
				}
			}
		else {
			syslog(LOG_NOTICE, "unknown command: %s", line);
			cfputs(x, "-ERR unkown command");
			errcount++;
			if (errcount > 5) {
				syslog(LOG_NOTICE, "too many command errors: %s", x->client.name);
				break;
				}
			}
		}

	syslog(LOG_NOTICE, "-ERR: closing connection to %s", x->client.name);
	return (0);
}

