// DSTART
//                      getpop3 - a POP3 client for Linux.
// 
//                    Copyright 1996, Double Precision, Inc.
// 
// THE SOFTWARE IS PROVIDED FREE OF CHARGE, "AS IS", WITHOUT WARRANTY OF ANY
// KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
// EVENT SHALL DOUBLE PRECISION, INC BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, ARISING FROM, OUT OF, OR IN CONNECTION WITH, OR USE OF THIS
// SOFTWARE.
// 
// Permission is granted to copy and distribute this software provided:
// 
//   1. The original getpop3 *.tgz is distributed by itself, or as a component
//      of another distribution, without any modifications, or changes.
//      You may not charge for copying or distributing the original, unaltered,
//      program, except for a nominal fee to cover duplication costs only.
// 
//   2. If you distribute a modified version of getpop3 by itself, or as a
//      component of another distribution, you must clearly identify your
//      changes or alterations, and acknowledge that you are responsible for
//      supporting any changes, and, identify the author of the original version
//      of this product.
// 
// Please see the included documentation for further information.
// 
// DEND
#include	"afx.h"
#include	<iostream.h>
#include	<strstream.h>
#include	<memory.h>
#include	<ctype.h>
#include	<pwd.h>
#include	<unistd.h>
#include	<stdlib.h>
#include	<signal.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/file.h>
#include	<sys/stat.h>
#include	<sys/vfs.h>
#include	<syslog.h>

#include	"debug.h"
#include	"config.h"
#include	"ask.h"
#include	"suck.h"
#include	"asocket.h"

#ifdef	DEBUG
int	debug_level=0;
#endif
int	log_open;

#define	BANNER	VERSION "  Copyright 1997 Double Precision, Inc.\n" \
		"See README.getpop3 for distribution and installation notes.\n"

static const char rcsid[]="$Id: main.C,v 1.10 1997/03/28 16:49:11 mrsam Exp $";
static const char ConfigFilename[]=DEFAULT_CONFIG;

static const char *progname;

void usage()
{
TRACE_FUNCTION("Usage")

	cout << "Usage: " << progname <<
#ifdef	DEBUG
		" [-d level]"
#endif
			" [-a account] [-p password] [-h server] [-l lockfile]\n"
			"\t[-t #] [-V] [-k] [-K] [-o] [-D] [-u uidl] [-m mailer]\n"
			"\t[-s fsys:nblk, fsys:nblk ... ] [-v] [-q] [-A] userid\n"
		"\n"
#ifdef	DEBUG
		"\t-d level       Set debug level (0 - none)\n"
#endif
		"\t-V             Just show the program version.\n"
		"\t-u account     Account (userid) on the POP3 server\n"
		"\t-p password    Password to give to the POP3 server\n"
		"\t-h server      Name of the POP3 server\n"
                "\t-l file        Try to lock this file, if already locked,\n"
                "\t               exit gracefully.\n"
		"\t-A             Automatic mode.  Get mail for every user\n"
		"\t               listed in " << ConfigFilename << ".\n"
		"\t-U uidl        Name of file or directory to save UIDL data\n"
		"\t-k             Keep messages on the POP3 server, don't delete them\n"
		"\t               Also, doesn't download previously-kept messages.\n"
                "\t-K             Erase all record of kept messages (downloads them\n"
                "\t               again.\n"
                "\t-o             If the server doesn't support the UIDL command, use\n"
		"\t               older, alternate, less reliable method for\n"
		"\t               tracking messages already seen.\n"
                "\t-D             Delete messages kept on the server.\n"
		"\n"
		"\t-m mailer      Use this instead of sendmail to deliver mail.\n"
		"\t-v             Verbose mode.  I will trace POP3 dialogue even\n"
		"\t               if my output is not going to a terminal.\n"
		"\t-q             Quiet mode.  I will not complain about errors\n"
		"\t               that are probably caused by PPP being inactive,\n"
		"\t               I'll just terminate quietly.  I certainly WILL\n"
		"\t               complain about anything else.\n"
		"\t-Q             Do not log activity to syslog.\n"
		"\t-s fsys:#, ... Flame guard.  Do not run if filesystem has\n"
		"\t               fewer than # blocks free.\n"
		"\t-t #           Specify timeout interval, # is in seconds\n"
		"\t               (default " << CASOCKET_DEFAULT_TIMEOUT <<
							").\n"
		"\tuserid         Local user who gets the mail.\n"
		;
}

static int dosuck(const char *, const char *, const char *, const char *,
	const CString &, const CString &);

static int prepsuck(BOOL autoMode,
		const char *user, const char *acct, const char *pwd,
		const char *host, const CString &mailerStr, const CString &uidlStr)
{
TRACE_FUNCTION("prepsuck")

	TRACE(1, "Searching " << ConfigFilename << ", user=" << user << "\n");

Config	cf;

	cf.Open(ConfigFilename);

CStringArray opts;
BOOL	found=FALSE;

	while ((cf >> opts) == 0)
	{
		if (opts.GetSize() == 0)	continue;
		opts[0].MakeLower();
		if (_tcscmp(opts[0], "user"))	continue;

		if (opts.GetSize() < 5)
		{
			cerr << "Bad user line in " << ConfigFilename
				<< "\n";
			return (10);
		}
		TRACE(1, "Found " << opts[1] << "\n");
		if (!autoMode && _tcscmp(opts[1], user))	continue;

		if (host)	opts[2]=host;
		if (acct)	opts[3]=acct;
		if (pwd)	opts[4]=pwd;

	CString	uidlPath=uidlStr + '/' + opts[1];

		if (opts.GetSize() >= 6)	uidlPath=opts[5];

		found=TRUE;
	int	rc=dosuck(opts[1], opts[3], opts[4], opts[2], mailerStr, uidlPath);

		if (rc)	return (rc);
	}
	cf.Close();
	if (!found)
		return (dosuck(user, acct, pwd, host, mailerStr, uidlStr));

	return (0);
}

static int dosuck(const char *user, const char *acct, const char *pwd,
		const char *host, const CString &mailerStr, const CString &uidlStr)
{
TRACE_FUNCTION("dosuck")

//	Try to read the rest of the information from the user.

CAsk	AskAcct;

	if (!acct || !*acct)
	{
		if (!AskAcct.Prompt((CString)user + "'s POP3 userid: "))
		{
			usage();
			return (10);
		}
		acct=AskAcct;
	}

	if (_tcslen(acct) == 0)
	{
		usage();
		return (10);
	}

CAsk	AskPwd;

	if (!pwd || !*pwd)
	{
		if (!AskPwd.Prompt((CString)user + "'s POP3 password: ", TRUE))
		{
			usage();
			return (10);
		}
		pwd=AskPwd;
	}

	if (_tcslen(pwd) == 0)
	{
		usage();
		return (10);
	}

CAsk	AskHost;

	if (!host || !*host)
	{
		if (!AskHost.Prompt((CString)user + "'s POP3 server: "))
		{
			usage();
			return (10);
		}
		host=AskHost;
	}

	if (_tcslen(host) == 0)
	{
		usage();
		return (10);
	}

struct stat stat_buf;

CString	file=uidlStr;

	if (stat(uidlStr, &stat_buf) == 0 &&
		S_ISDIR(stat_buf.st_mode))
	{
		file += '/';
		file += user;
	}

	TRACE(1, "Local user: " << user << '\n')
	TRACE(1, "POP3 userid: " << acct << '\n')
	TRACE(1, "POP3 password: " << pwd << '\n')
	TRACE(1, "POP3 server: " << host << '\n')
	TRACE(1, "Mailer: " << mailerStr << '\n')
	TRACE(1, "UidlPath: " << file << '\n')
	return (CSuck().Suck(host, acct, pwd, mailerStr, user, file));
}

static int main2(int, char *[]);

int main(int argc, char *argv[])
{
int rc;

	progname=argv[0];
	log_open=FALSE;
	try
	{
		rc=main2(argc, argv);
	}
	catch (CException *ce)
	{
		ce->ReportError();
		ce->Delete();
		rc= 99;
	}
	if (log_open)
	{
		openlog(VERSION, LOG_PID, LOG_MAIL);
		syslog(LOG_INFO, "Terminated.");
	}
	return (rc);
}

//  New V1.0 - check for free space

static int check_free_space_sys(CString &);

static int check_free_space(CString &s)
{
TRACE_FUNCTION("check_free_space");
int i;

	while ((i=s.Find(',')) >= 0)
	{
	CString cfile=s.Left(i);

		s=s.Mid(i+1);
		if (check_free_space_sys(cfile))
			return (-1);
	}
	return (check_free_space_sys(s));
}

static int check_free_space_sys(CString &s)
{
TRACE_FUNCTION("check_free_space_sys");

long	nBlocks=2048;
int	i;

	if ((i=s.Find(':')) >= 0)
	{
	CString num_blocks_s=s.Mid(i+1);

		s=s.Left(i);

	istrstream parse_num(num_blocks_s);

		parse_num >> nBlocks;
	}
	TRACE(2, "File system is " << s << ", # blocks is " << nBlocks << "\n");
	if (_tcscmp(s, "") == 0)
		return (0);

struct statfs sfs;
int	rc=statfs(s, &sfs);

	TRACE(2, "statfs() returned " << rc << "\n");
	TRACE(2, s << ": # of free blocks is " << sfs.f_bfree << "\n");
	if (sfs.f_bfree >= nBlocks)
		return (0);

	if (isatty(2))
		cerr << "ERROR: " << s << " filesystem has only "
			<< sfs.f_bfree << " free blocks.\n";
	return (1);
}

int main2(int argc, char *argv[])
{
int optc;
char *popacct=NULL, *pophost=NULL;
CString poppwd, popuser;
BOOL	okUser=FALSE, okMailer=FALSE;
CString mailerStr=_T(DEFAULT_SENDMAIL);
CString uidlStr=_T(DEFAULT_UIDL);
char	*lockfile=NULL;
BOOL	automode=FALSE;
CString	fscheck=_T("");
int	version=0;

	if (isatty(2))
		cerr << BANNER << "\n";

	if (getuid() == 0)	// For root, this is OK
	{
		okUser=TRUE;
		okMailer=TRUE;
	}
	signal(SIGPIPE, SIG_IGN);

	{
	Config	configfile;
	CStringArray configs;
	struct stat  stat_buf;

		// Security check.

		if (stat(ConfigFilename, &stat_buf) == 0 &&
			(stat_buf.st_mode & 077) != 0)
		{
			cerr << "!!! " << ConfigFilename
				<< " is readable by others.\n"
				"!!! I refuse to run until you chmod it to 600\n";
			return (10);
		}

		configfile.Open(ConfigFilename);
		while ((configfile >> configs) == 0)
		{
		int n=configs.GetSize();

			if (n == 0)	continue;

			configs[0].MakeLower();
			if (_tcscmp(configs[0], "okuser") == 0)
				okUser=TRUE;
			if (_tcscmp(configs[0], "okmailer") == 0)
				okMailer=TRUE;
			if (_tcscmp(configs[0], "nodots") == 0)
				CSuck::optEscapeDots=FALSE;

			if (_tcscmp(configs[0], "mailer") == 0)
			{
				if (n == 1)
				{
					continue;
				}
				mailerStr=configs[1];
			}

			if (_tcscmp(configs[0], "defaultuidl") == 0)
			{
				if (n == 1)
				{
					usage();
					return (10);
				}
				uidlStr=configs[1];
			}
		}
		configfile.Close();
	}

	log_open=TRUE;
	while ((optc=getopt(argc, argv,
#ifdef	DEBUG
		"d:"
#endif
		"?m:U:u:p:h:kKoDl:qQvAs:Vt:"
			)) >= 0)
	{
		switch (optc)	{
		case '?':
			usage();
			return (0);
#ifdef	DEBUG
		case 'd':
			debug_level=atoi(optarg);
			break;
#endif
		case 'u':
			popacct=optarg;
			break;
		case 'p':
			poppwd=optarg;
			memset(argv[optind-1], ' ', strlen(argv[optind-1]));
			strcpy(argv[optind-1], "");
			break;
		case 'h':
			pophost=optarg;
			break;
		case 'm':
			if (!okMailer)
			{
				cerr << "You are not allowed to use the -m flag.\n";
				return (10);
			}
			mailerStr=optarg;
			break;
		case 'U':
			uidlStr=optarg;
			break;
		case 'k':
			CSuck::optKeep=1;
			break;
		case 'K':
			CSuck::optKeep=1;	// Save user from a bad mistake.
			CSuck::optDelKeep=1;
			break;
		case 'o':
			CSuck::optOlderOk=1;
			break;
		case 'D':
			CSuck::optDelKept=1;
			break;
		case 'l':
			lockfile=optarg;
			break;
		case 'q':
			CSuck::optQuiet=1;
			break;
		case 'Q':
			log_open=FALSE;
			break;
		case 'v':
			CSuck::optVerbose=1;
			break;
		case 't':
			CSuck::optTimeout=atoi(optarg);
			break;
		case 'A':
			if (!okUser)
			{
				cerr << "You are not allowed to specify"
					" other users"
					" using the -A flag.\n";
				return (10);
			}
			automode=1;
			break;
		case 's':
			fscheck=optarg;
			break;
		case 'V':
			version=1;
			break;
		default:
			usage();
			return (10);
		}
	}
	if (version)
	{
		return (0);	// We already printed the banner.
	}
	if (log_open)
	{
		openlog(VERSION, LOG_PID, LOG_MAIL);
		syslog(LOG_INFO, "Started.");
	}

	if (lockfile)
	{
	int	fd=open(lockfile, O_RDWR | O_CREAT, 0777);

		if (fd < 0)
		{
			cerr << "Can't open the lock file.\n";
			return (10);
		}

		if (flock(fd, LOCK_EX | LOCK_NB))
		{
			TRACE(1, "Lock file already locked.\n");
			return (0);
		}
	}
	if (check_free_space(fscheck))
		return (5);

	TRACE(1, "Processed options in " << ConfigFilename << "\n");
	TRACE(1, "   okuser = " << !!okUser << "\n");
	TRACE(1, "   okmailer = " << !!okMailer << "\n");
	TRACE(1, "   mailer = " << mailerStr << "\n");
	TRACE(1, "   uidldir = " << uidlStr << "\n");

	if (optind == argc)
	{
	struct passwd *p=getpwuid(getuid());

		if (!p)
		{
			cerr << "You don't have an entry in passwd!\n";
			return (10);
		}
		popuser=p->pw_name;
		if (automode)
		{
			if (popacct || poppwd || pophost)
			{
				cerr << "Can't specify -u -p -h with -A option.\n";
				return (10);
			}
		}
		return (prepsuck(automode,
			popuser, popacct, poppwd, pophost, mailerStr, uidlStr));
	}
	else if (automode)
	{
		cerr << "With -A option, you don't specify users!\n";
		return (10);
	}
	else if (!okUser)
	{
		cerr << "You are not allowed to specify other users.\n";
		return (10);
	}
	else if (argc - optind > 1)
	{
		if (popacct || poppwd || pophost)
		{
			cerr << "Can't specify -u -p -h with multiple users.\n";
			return (10);
		}

	int	i;

		for (i=optind; i<argc; i++)
		{
		int rc=prepsuck(FALSE,
			 argv[i], NULL, NULL, NULL, mailerStr, uidlStr);

			if (rc)	return (rc);
		}
	}
	else
	{
		return (prepsuck(FALSE, argv[optind],
			popacct, poppwd, pophost, mailerStr, uidlStr));
	}
	return (0);
}
