// DSTART
//                      getpop3 - a POP3 client for Linux.
// 
//                    Copyright 1996-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// 
// DEND
#if	HAVE_CONFIG_H
#include	"autoconfig.h"
#endif
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<netdb.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<errno.h>
#if HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	<strstream.h>
#include	<ctype.h>
#include	<signal.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/wait.h>
#if HAVE_SYS_FILE_H
#include	<sys/file.h>
#endif
#if HAVE_FCNTL_H
#include	<fcntl.h>
#endif
#include	<pwd.h>
#if HAVE_SYSLOG_H
#include	<syslog.h>
#endif
#include	"suck.h"
#include	"debug.h"
#include	"sigtrap.h"

static const char rcsid[]="$Id: suck.C,v 1.21 1998/05/27 01:26:45 mrsam Exp $";

#define	POP3_PORT	110

#define	SHOW(x)	if (isatty(1) || optVerbose) { cout << x << "\n"; }

#define	CRLF	"\r\n"

#define	ISDOT(p)	(*(p) == '.' && ((p)[1] == '\r' || (p)[1] == '\0'))

int	CSuck::optKeep=0, CSuck::optDelKeep=0,
	CSuck::optOlderOk=0, CSuck::optDelKept=0,
	CSuck::optQuiet=0, CSuck::optVerbose=0,
	CSuck::optTimeout=CASOCKET_DEFAULT_TIMEOUT,
	CSuck::optEscapeDots=TRUE;

extern int	log_open;

//  SIGTERM timeout via SIGALRM

static int alarm_caught;

static void alarm_sig(int)
{
	alarm_caught=1;
	alarm(2);
}

int CSuck::read_line()
{
int	s=80;
LPSTR	p=line.GetBuffer(s);
int	i=0;
int	ch;

	for (ch=asyncbuf.get(); ch != '\n' && ch != EOF; ch=asyncbuf.get())
	{
		if (i >= s)
		{
			line.ReleaseBuffer(i);
			p=line.GetBuffer(s += 80);
		}
		p[i++]=ch;
	}
	line.ReleaseBuffer(i);
	if (ch == EOF)
	{
		sockioerr();
		return (-1);
	}
	return (0);
}

int CSuck::Suck(LPCTSTR server, LPCTSTR user, LPCTSTR pwd, const CString &mailer,
	LPCTSTR mailbox,
	const CString &uidlFileName, uid_t uid, gid_t gid)
{
int rc;
TRACE_FUNCTION("Suck");

	if (sockfs >= 0)
		close(sockfs);
	if (lockfs >= 0)
		close(lockfs);

	uidlfs.open(uidlFileName, ios::in | ios::out);
	if (uidlfs.bad() || (lockfs=open(uidlFileName, O_RDWR)) < 0)
	{
		cerr << "Cannot open uidl/lock file '" << uidlFileName
			<< "'\n";
		return (5);
	}

	fchown(uidlfs.rdbuf()->fd(), uid, gid);
	fchmod(uidlfs.rdbuf()->fd(), 0600);

#if HAS_LOCKF
	if (lockf(lockfs, F_TLOCK, 0))
	{
		TRACE(1, "UIDL lock file already locked.\n");
		uidlfs.close();
		close(lockfs);
		lockfs= -1;
		return (0);
	}
#else
#if HAS_FLOCK
	if (flock(lockfs, LOCK_EX | LOCK_NB))
	{
		TRACE(1, "UIDL lock file already locked.\n");
		uidlfs.close();
		close(lockfs);
		lockfs= -1;
		return (0);
	}
#endif
#endif

	sockfs=socket(AF_INET, SOCK_STREAM, 0);

	if (sockfs < 0)
	{
		perror("socket");
		return (1);
	}

	if (optTimeout < 60)	optTimeout=60;
	asyncbuf.manage(sockfs, optTimeout);
	SHOW("Looking up " << server);

#if HAVE_SYSLOG_H
	if (log_open)
		syslog(LOG_INFO, "Connecting to %s for %s", server, user);
#endif

struct hostent *h=gethostbyname(server);

	if (!h)
	{
		SOFT_ERROR perror("gethostbyname");
		return (1);
	}

	if (h->h_addrtype != AF_INET ||
		(size_t)h->h_length < sizeof(struct in_addr))
	{
		cerr << "gethostbyname:  unknown address type.\n";
		return (1);
	}

	SHOW("Connecting...");
long i;

	for (i=0; h->h_addr_list[i]; i++)
	{
	struct sockaddr_in sin;

		sin.sin_family=AF_INET;
		sin.sin_port=htons(POP3_PORT);
		sin.sin_addr=*(const struct in_addr *)h->h_addr_list[i];

	int	rc;

		while ((rc=connect(sockfs, (const struct sockaddr *)&sin,
			sizeof(sin))) && (errno == EWOULDBLOCK
				|| errno == EINPROGRESS))
		{
			if (asyncbuf.waitForRead())
				break;
		}
		if (rc && errno == EISCONN)	rc=0;	// AIX peculiarity
		if (rc == 0)
			break;
	}
	if (!h->h_addr_list[i])
	{
		SOFT_ERROR perror("connect");
		return (1);
	}
	SHOW("Connected.");

	if ((rc=get_ok_err()) != 0)
		return (rc);

	if ((rc=command (_T("USER "), user)) != 0)
		return (rc);

	if ((rc=get_ok_err()) != 0)
		return (rc);

	rc=command(_T("PASS "), pwd, FALSE);
	SHOW(_T("PASS ****"));
	if (rc)	return (rc);

	if ((rc=get_ok_err()) != 0)
		return (rc);

	rc=command(_T("STAT"));

	if ((rc=get_ok_err()) != 0)
		return (rc);

long	msgCnt=0;

	{
	istrstream parse_stat(line);

		parse_stat >> msgCnt;
	}

	TRACE(2, "STAT line tells me I have " << msgCnt << " messages.\n");

	if (msgCnt)
	{
		if ((rc=FigureOutWhatToDo(msgCnt)) != 0)
			return (rc);
	}

// Now, begin rewriting the UIDL file.  First, write out the IDs
// for the messages that we won't get.

	if (!optOlderOk)
	{
		uidlfs.seekg(0);
		uidlfs.seekp(0);

	SigTrap	trap_signals;

		if (truncate(uidlFileName, 0))
		{
			perror("ftruncate");
			uidlfs.close();
			close(sockfs);
			sockfs= -1;
			close(lockfs);
			lockfs= -1;
			return (10);
		}
		uidlfs.close();
		uidlfs.open(uidlFileName, ios::in | ios::out);
		for (i=0; i<msgCnt; i++)
			if (!get_messages[i])
				uidlfs << uidl[i] << '\n';
		uidlfs << flush;
		if (trap_signals)
			return (5);
	}

	for (i=0; i<msgCnt; i++)
	{
		if (get_messages[i])
		{
			rc=SuckMessage(i+1, mailer, mailbox);
			if (rc)
			{
				if (rc >= 10)
					return (rc);
				break;
			}
			if (!optOlderOk)
			{
				TRACE(2, "SaveUIDL " << uidl[i] << '\n');
				uidlfs << uidl[i] << '\n' << flush;
			}
		}

		if (del_messages[i])
		{
			if ((rc=command(_T("DELE "), (i+1))) != 0)
				return (rc);

			if ((rc=get_ok_err()) != 0)
				break;
		}
	}

	if ((rc=command(_T("QUIT"))) != 0)
		return (rc);

int rc2;

	if ((rc2=get_ok_err()) != 0)
		return (rc2);
	close(sockfs);
	sockfs= -1;
	close(lockfs);
	lockfs= -1;
	if (!optOlderOk)
		uidlfs.close();
	return (rc);
}

int CSuck::command(CString cmd, BOOL doEcho)
{
int rc;
CString wcmd=cmd + CRLF;

	rc=asyncbuf.write(wcmd, wcmd.GetLength());

	if (doEcho)
	{
		SHOW( cmd );
	}

	if (rc)
	{
		sockioerr();
		return (5);
	}
	return (0);
}

int CSuck::FigureOutWhatToDo(long cnt)
{
TRACE_FUNCTION("Suck::FigureOutWhatToDo");
int rc;
long i;

	get_messages.SetSize(cnt);
	del_messages.SetSize(cnt);
	uidl.SetSize(cnt);
	if (cnt == 0)	return (0);

	for (i=0; i<cnt; i++)
	{
		get_messages[i]=1;
		del_messages[i]= optKeep ? 0:1;
		uidl[i]=_T("");
	}

	if (optOlderOk)
	{
		rc=OlderMethod();
	}
	else
	{
		if ((rc=command(_T("UIDL"))) != 0)
			return (rc);

		if ((rc=get_ok_err()) != 0)
			return (rc);
		rc=UidlMethod();
	}

	for (i=0; i<cnt; i++)
	{
		TRACE(1, "Get[" << i << "]=" << get_messages[i] << "\n");
		TRACE(1, "Del[" << i << "]=" << del_messages[i] << "\n");
	}
	if (rc)	return (rc);
	return (0);
}

int CSuck::OlderMethod()
{
long	cnt=get_messages.GetSize();
long	i;
int	rc=0;

	for (i=0; i < cnt; i++)
	{
	BOOL seenit=FALSE;
	char buf[40];

		sprintf(buf, "%ld", i+1);

		if ((rc=command(_T("TOP "), (CString)buf + " 0")) != 0)
			return (rc);

		if ((rc=get_ok_err()) != 0)
			return (rc);

		while ((rc=get_multiline()) == 0)
		{
			line.TrimLeft();
			line.MakeLower();
			if (_tcsncmp(line, _T("status:"), 7) == 0)
			{
				if (_tcschr(((LPCTSTR)line)+7, 'r'))
					seenit=TRUE;
			}
		}
		TRACE(2,"Seen " << i+1 << " = " << seenit << '\n');
		if (seenit)
		{
			if (!optDelKeep)
				get_messages[i]=0;
			if (optDelKept)
				del_messages[i]=1;
		}
	}
	if (rc > 1)	return (rc);
	return (0);
}

int CSuck::UidlMethod()
{
TRACE_FUNCTION("CSuck::UidlMethod");
CString	uidl_buf;
int	rc;
long	cnt=get_messages.GetSize();

	while (!(rc=get_multiline()))
	{
	istrstream parse_uidl(line);
	long msgNum=0;

		parse_uidl >> msgNum;
		TRACE(2, "UIDL string " << msgNum << '\n');

		while (parse_uidl.peek() && isspace(parse_uidl.peek()))
			parse_uidl.get();
		if (msgNum > 0 && msgNum <= cnt)
		{
		int	i;

			uidl_buf << parse_uidl;
			if ((i=uidl_buf.Find('\r')) >= 0)
			{
				uidl_buf.GetBuffer();
				uidl_buf.ReleaseBuffer(i);
			}
			TRACE(2, "UIDL string='" << uidl_buf << "'\n");
			uidl[msgNum-1]=uidl_buf;
		}
	}

	uidlfs.seekg(0);
	if (uidlfs.bad())
	{
		perror("seek");
		return (10);
	}

	{
	CString	str;
	long n;

		while ((str << uidlfs) == 0)
		{
			for (n=0; n<cnt; n++)
			{
				if (_tcscmp(str, uidl[n]) == 0 && *uidl[n])
				{
					if (!optDelKeep)
						get_messages[n]=0;
					if (optDelKept)
						del_messages[n]=1;
				}
			}
		}
	}
	if (rc > 1)	return (rc);
	return (0);
}

int CSuck::SuckMessage(long msgnum, LPCTSTR mailerProg, LPCTSTR mailbox)
{
TRACE_FUNCTION("CSuck::SuckMessage");
int rc;
BOOL inHeader=TRUE;
LPCTSTR p;
int	fildes[2];

	TRACE(1, "Sucking message " << msgnum << " to " << mailerProg
		<< " " << mailbox << "\n");

	if (pipe(fildes))
	{
		perror("pipe");
		return (10);
	}

	if ((rc=command(_T("RETR "), msgnum)) != 0)
		return (rc);

	if ((rc=get_ok_err()) != 0)
	{
		close(fildes[0]);
		close(fildes[1]);
		return (rc);
	}

pid_t	pid=fork();

	if (pid < 0)
	{
		perror("fork");
		close(fildes[0]);
		close(fildes[1]);
		return (10);
	}

	if (pid == 0)
	{
		close(0);
		dup(fildes[0]);

	int i;

		for (i=3; i<63; i++)
			close(i);
		if (chdir("/"))
			exit(10);
		setuid(getuid());
		execl(mailerProg, mailerProg, mailbox, NULL);
		exit (255);
	}

	close(fildes[0]);

#if HAVE_SYSLOG_H
	if (log_open)
		syslog(LOG_INFO, "Forked %s, pid %d", mailerProg, pid);
#endif
ofstream ofs;

	ofs.attach(fildes[1]);


	SHOW("Retrieving message...");

long	msgcnt=0;

	for (;;)
	{
		if (read_line())
		{
			alarm_caught=0;
			signal(SIGALRM, alarm_sig);
			alarm(5);
			kill (pid, SIGTERM);
			ofs.close();

		pid_t	wait_pid;

			while ((wait_pid=wait(NULL)) != pid)
			{
				if (alarm_caught)
					kill(pid, SIGKILL);
			}
			alarm(0);
			signal(SIGALRM, SIG_DFL);
#if HAVE_SYSLOG_H
			if (log_open)
				syslog(LOG_ERR, "Socket reset, aborted partial message.");
#endif
			return (5);
		}
		p=line;
		if (ISDOT(p))
			break;
		if (inHeader && (*p == '\r' || *p == '\0'))
			inHeader=FALSE;
		// Nuke Status: header to make all mail appear as new in pine.
		if (inHeader && _tcsnicmp(p, _T("Status:"), 7) == 0)
			continue;

		if (!optEscapeDots && *p == '.')
			++p;

		ofs << p << '\n';
		msgcnt += line.GetLength();
	}

	if (optEscapeDots)
		ofs << ".\r\n";
	ofs.close();

int wait_stat;
pid_t wait_pid;

	while ((wait_pid=wait(&wait_stat)) != pid)
		;

	if (!WIFEXITED(wait_stat) || WEXITSTATUS(wait_stat))
	{
		cerr << mailerProg << " terminated with ";
		if (WIFSIGNALED(wait_stat))
			cerr << " signal " << WTERMSIG(wait_stat);
		else
			cerr << " status " << WEXITSTATUS(wait_stat);

		cerr << ".\n";
#if HAVE_SYSLOG_H
		if (log_open)
			syslog(LOG_ERR, "Failed delivery.");
#endif
		return (5);
	}
#if HAVE_SYSLOG_H
	if (log_open)
		syslog(LOG_INFO, "Message delivered (%ld bytes)", msgcnt);
#endif
	return (0);
}

int CSuck::get_multiline()
{
	if (read_line())	return (5);

	SHOW(line);

LPCTSTR p=line;

	if (ISDOT(p))
	{
		return (1);
	}
	return (0);
}

int CSuck::get_ok_err()
{
	if (read_line())
		return (5);
	if ( *line != '+')
	{
		cerr << line << "\n";
		return (1);
	}

int	i=line.Find(' ');

	if (i < 0)
		i=line.Find('\r');
	if (i < 0)
		i=line.GetLength();

	if (line.Left(i) != "+OK")
	{
		cerr << line << "\n";
		return (1);
	}
	SHOW(line);
	line=line.Mid(i);
	return (0);
}

CSuck::~CSuck()
{
	if (sockfs >= 0)	close(sockfs);
	if (lockfs >= 0)	close(lockfs);
}
