/*

    listen.c  -  a toolbox for dynamically allocated lists

    Copyright (C) 1999  by  Florian Paetz & Christian Garbs

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


*/

//***********************************************************
//* File      : listen.h                                    *
//* Autor     : Florian Paetz, Christian Garbs              *
//* Datum     : 03.05.1999                                  *
//* Funktion  : Liefert Funktionen zur Verwaltung           *
//*             von dynamischen Listen.                     *
//* nderungen: 06.03.1999   Christian Garbs                *
//*            -Es wird immer ein Zeiger auf das Element    *
//*             vor dem eigentlich aktuellen Element, damit *
//*             sich eine Lschroutine fr Elemente benutzt *
//*             geschrieben werden kann.                    *
//*            -Erweiterung um folgende Funktionen          *
//*               - tlist_next                              *  
//*               - tlist_length                            *  
//*               - tlist_destruct                          *
//*             07.04.1999   Christian Garbs                *
//*            -Einbau der Debug-Funktion                   *
//*             09.04.1999   Christian Garbs                *
//*            -tlist_put() funktioniert jetzt              *
//*            -tlist_dispose() lscht jetzt auch den mit   *
//*             pInhalt bezeichneten Speicherbereich        *
//*            -Liste jetzt simultan vorwrts und rckwrts *
//*             verkettet.                                  *
//*            -es wird wieder auf das wirkliche Element    *
//*             gezeigt.                                    *
//*            -tlist_destruct() pat auf doppelt referen-  *
//*             zierte Speicherbereiche auf                 *
//*            -tlist_dump() zeigt die ganze Liste an       *
//*             12.04.1999   Christian Garbs                *
//*            -Fehler in tlist_construct() behoben         *
//*             03.05.1999   Christian Garbs                *
//*            -Ungenauigkeiten entfernt (da waren Warnungen*
//*             beim Kompilieren mit Optimierung)           *
//***********************************************************

#include <strings.h>
#include <stdio.h>
#include <malloc.h>

//#define DEBUG_LISTEN

//************************************************
//* Struktur  : TLISTENELEMENT                   *
//* Listenelement                                *
//************************************************
typedef struct _TLISTENELEMENT
{
	void *pInhalt;
	struct _TLISTENELEMENT *pNext;
	struct _TLISTENELEMENT *pPrev;
} TLISTENELEMENT;

typedef TLISTENELEMENT *PLISTENELEMENT;

//************************************************
//* Struktur : TLISTE                            *
//* Wie der Name schon sagt                      *
//************************************************
typedef struct _TLISTE
{
	PLISTENELEMENT pKopf;
	PLISTENELEMENT pFuss;
	PLISTENELEMENT pAktuell;
} TLISTE;

typedef TLISTE *PLISTE;

//************************************************
//* Funktion  : tlist_go_begin                   *
//* Input     : PLISTE => Pointer auf eine Liste *
//* Setzt pAktuell auf den Listenanfang          *
//************************************************
void tlist_go_begin (PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_go_begin() begin\n");
#endif

	pliste->pAktuell=pliste->pKopf;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_go_begin() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_go_end                     *
//* Input     : PLISTE => Pointer auf eine Liste *
//* Setzt pAktuell auf das Listenende            *
//************************************************
void tlist_go_end (PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_go_end() begin\n");
#endif

	pliste->pAktuell=pliste->pFuss;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_go_end() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_construct                  *
//* Output    : PLISTE => Pointer auf eine leere *
//*                       Liste                  *
//************************************************
PLISTE tlist_construct(void)
{
	PLISTE pliste;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_construct() begin\n");
#endif

	pliste=(TLISTE *) malloc(sizeof(PLISTE));
	pliste->pKopf=NULL;
	pliste->pAktuell=NULL;
	pliste->pFuss=NULL;
	return pliste;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_construct() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_is_end                     *
//* Input     : PLISTE                           *
//* Output    : char der als BOOLEAN interpre-   *
//*             tiert werden kann: 0(FALSE) Ende *
//*             noch nicht erreicht.             *
//*             1(TRUE) Ende erreicht            *
//* Prft, ob pAktuell das letzte Element der    *
//* Liste ist.                                   *
//************************************************
unsigned char tlist_is_end(PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_is_end() begin\n");
#endif

	if ((pliste->pAktuell)==(pliste->pFuss)) {
		return 1;
	} else {
		return 0;
	};

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_is_end() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_is_begin                   *
//* Input     : PLISTE                           *
//* Output    : char der als BOOLEAN interpre-   *
//*             tiert werden kann: 0(FALSE)      *
//*             Anfang nicht erreicht.           *
//*             1(TRUE) Anfang erreicht          *
//* Prft, ob pAktuell das erste Element der     *
//* Liste ist.                                   *
//************************************************
unsigned char tlist_is_begin(PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_is_begin() begin\n");
#endif

	if ((pliste->pAktuell)==(pliste->pKopf)) {
		return 1;
	} else {
		return 0;
	};

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_is_begin() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_put                        *
//* Input     : PLISTE                           *
//*             void* => Pointer auf den Inhalt  *
//*                      der in pAktuell gesetzt *
//*                      wird.                   *
//************************************************
void tlist_put(PLISTE pliste, void *pInhalt)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_put() begin\n");
#endif
	if (pliste->pAktuell!=NULL) {
		pliste->pAktuell->pInhalt=pInhalt;
	};

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_put() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_get                        *
//* Input     : PLISTE                           *
//* Output    : void * => pInhalt von pAktuell   *
//************************************************
void *tlist_get(PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_get() begin\n");
#endif

	return pliste->pAktuell->pInhalt;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_get() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_new                        *
//* Input     : PLISTE                           *
//* Fgt nach pAktuell ein neues Element ein.    *
//************************************************
void tlist_new(PLISTE pliste)
{
	PLISTENELEMENT pNewElem;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_new() begin\n");
#endif

	pNewElem=(TLISTENELEMENT *) malloc(sizeof(PLISTENELEMENT));
	pNewElem->pInhalt=NULL;
	pNewElem->pNext=NULL;
	pNewElem->pPrev=NULL;
	if (pliste->pAktuell==NULL) {
		pliste->pKopf=pNewElem;
		pliste->pAktuell=pNewElem;
		pliste->pFuss=pNewElem;
	} else {
		pNewElem->pNext=pliste->pAktuell->pNext;
		pNewElem->pPrev=pliste->pAktuell;
		if (pNewElem->pPrev!=NULL) {
			pNewElem->pPrev->pNext=pNewElem;
		} else {
			pliste->pKopf=pNewElem;
		};
		if (pNewElem->pNext!=NULL) {
			pNewElem->pNext->pPrev=pNewElem;
		} else {
			pliste->pFuss=pNewElem;
		};
	};
	pliste->pAktuell=pNewElem;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_new() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_next                       *
//* Input     : PLISTE                           *
//* Rckt ein Element in der Liste vor           *
//************************************************
void tlist_next(PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_next() begin\n");
#endif

	if (!tlist_is_end(pliste)) {
		pliste->pAktuell=pliste->pAktuell->pNext;
	};

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_next() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_prev                       *
//* Input     : PLISTE                           *
//* Geht ein Element in der Liste zurck         *
//************************************************
void tlist_prev(PLISTE pliste)
{

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_prev() begin\n");
#endif

	if (!tlist_is_begin(pliste)) {
		pliste->pAktuell=pliste->pAktuell->pPrev;
	};

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_prev() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_dispose                    *
//* Input     : PLISTE                           *
//* Lscht das aktuelle Listenelement und springt*
//* auf das vorhergehende Element                *
//************************************************
void tlist_dispose(PLISTE pliste)
{
	PLISTENELEMENT pDispElem;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_dispose() begin\n");
#endif

	pDispElem=pliste->pAktuell;
	if (pDispElem->pNext!=NULL) {
		pDispElem->pNext->pPrev=pDispElem->pPrev;
	} else {
		pliste->pFuss=pDispElem->pPrev;
	};
	if (pDispElem->pPrev!=NULL) {
		pDispElem->pPrev->pNext=pDispElem->pNext;
		pliste->pAktuell=pDispElem->pPrev;
	} else {
		pliste->pKopf=pDispElem->pNext;
		pliste->pAktuell=pDispElem->pNext;
	};

	free(pDispElem->pInhalt);
	free(pDispElem);

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_dispose() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_length                     *
//* Input     : PLISTE                           *
//* Output    : long int: Anzahl                 *
//* Ermittelt die Anzahl der Listenelemente      *
//************************************************
long int tlist_length(PLISTE pliste)
{
	long int zaehler;
	PLISTENELEMENT pAlterZeiger;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_length() begin\n");
#endif

	pAlterZeiger=pliste->pAktuell;
	tlist_go_begin(pliste);
	for(zaehler=0;!tlist_is_end(pliste);zaehler++) {
		tlist_next(pliste);
	};
	if (pliste->pKopf!=NULL) {
		zaehler++;
	};
	pliste->pAktuell=pAlterZeiger;
	return zaehler;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_length() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_destruct                   *
//* Input     : PLISTE                           *
//* Lscht eine komplette Liste                  *
//************************************************
void tlist_destruct(PLISTE pliste)
{

	PLISTENELEMENT pZeiger;
	void *inhalt;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_destruct() begin\n");
#endif

	pliste->pAktuell=pliste->pKopf;
	while(pliste->pAktuell!=NULL) {
		inhalt=pliste->pAktuell->pInhalt;
		if(inhalt!=NULL) {
			pZeiger=pliste->pAktuell;
			while (pZeiger->pNext!=NULL) {
				if (pZeiger->pInhalt==inhalt) {
					pZeiger->pInhalt=NULL;
				};
				pZeiger=pZeiger->pNext;
			};
		};
		tlist_dispose(pliste);
	};
	free(pliste);

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_destruct() end\n");
#endif

};

//************************************************
//* Funktion  : tlist_dump                       *
//* Input     : PLISTE                           *
//* Gibt die Liste am Bildschirm aus             *
//************************************************
void tlist_dump(PLISTE pliste)
{
	PLISTENELEMENT pZeiger;

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_dump() begin\n");
#endif

	fprintf(stderr,"\n __________________ TLISTDUMP BEGINN __________________");
	fprintf(stderr,"\n/                                                      \\\n");
	fprintf(stderr,"  Liste       :%11d    Listenzeiger:%11d\n",(int)pliste,(int)pliste->pAktuell);
	fprintf(stderr,"  Listenkopf  :%11d    Listenfuss  :%11d\n",(int)pliste->pKopf,(int)pliste->pFuss);
	fprintf(stderr,"  is_end      :%11d    is_begin    :%11d\n",tlist_is_end(pliste),tlist_is_begin(pliste));
	fprintf(stderr,"  tlist_length:%11d\n\n",(int)tlist_length(pliste));
	fprintf(stderr,"      Element          Next          Prev        Inhalt\n");
	
	pZeiger=pliste->pKopf;
	while (pZeiger!=NULL) {
		fprintf(stderr,"  %11d   %11d   %11d   %11d\n",(int)pZeiger,(int)pZeiger->pNext,(int)pZeiger->pPrev,(int)pZeiger->pInhalt);
		pZeiger=pZeiger->pNext;
	};
	fprintf(stderr,"\n\\___________________ TLISTDUMP ENDE ___________________/\n");

#ifdef DEBUG_LISTEN
	fprintf(stderr,"tlist_dump() end\n");
#endif

};

