/* Copyright (C) 1998 Trent Piepho  <xyzzy@u.washington.edu>
 *           (C) 1999 Trent Piepho  <xyzzy@speakeasy.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>
#include "defaults.h"

/* X related globals */
Display *dpy;
Window win;
GC gc;
XFontStruct *font;
int Ascent=33,Descent=0;
int Width=55,Height=33;
XColor FgXColor,HiXColor;
Pixmap ShapeMask;			/* shape stuff */
GC ShapeGC;					/* shape stuff */
int MaskWidth,MaskHeight;	/* shape stuff */
Atom DeleteWindow;			/* Atom of delete window message */
Atom KWMDockWindow;			/* Atom for KWM hint */

/* app-default related globals */
char *FgColor="Black",*BgColor="White",*HiColor="Red";
char *FontName=FONTNAME;
char *DisplayName=NULL;
char *AppName;
int Number;
int Interval=INTERVAL_SPOOL;
char SpoolFile[256];
char Command[256];
char Username[32];
char Password[32];
char *Geometry=NULL;
int Options=0;
#define BELL_MODE		0x01
#define SHAPED_WINDOW	0x02
#define HILIGHT_MAIL	0x04
#define OFFLINE			0x08
#define USE_POP3		0x10
#define USE_APOP3		0x20
#define USE_IMAP		0x40
#define WM_MODE			0x80
#define KDE_MODE		0x100
#define NOKDE			0x200
#define	NOWM			0x400

void usage();
void update();
void handler(int);
void parse_options(int argc,char *argv[]);
void init(int argc,char *argv[]);
int count_mail();
int count_mail_spool();

void update()
{
	static int old_number=-1;
	char str[32];
	static int oldw=-1,oldh=-1;
	int w,h;

	if(Options&OFFLINE && Number==-1)  {
		strcpy(str,"X");
	} else {
		sprintf(str,"%d",Number);
	};

	w = (Width-XTextWidth(font,str,strlen(str)))/2;
	h = (Height+Ascent-Descent)/2;

	if(Options&SHAPED_WINDOW)  {
		if(Number!=old_number || oldw!=w || oldh!=h)  {
			old_number=Number; oldw=w; oldh=h;
			/* these next 3 lines clear the pixmap, is there a cleaner way? */
			XSetFunction(dpy,ShapeGC,GXclear);
			XFillRectangle(dpy,ShapeMask,ShapeGC,0,0,MaskWidth,MaskHeight);
			XSetFunction(dpy,ShapeGC,GXset);
			XDrawString(dpy,ShapeMask,ShapeGC,0,Ascent,str,strlen(str));
			XShapeCombineMask(dpy, win, ShapeBounding, 
				w, h-Ascent, ShapeMask, ShapeSet);
		};
	} else {
		XClearWindow(dpy,win);
	};

	if(Options&HILIGHT_MAIL)  {
		XSetForeground(dpy,gc,Number?HiXColor.pixel:FgXColor.pixel);
	};

	XDrawString(dpy,win,gc,w,h,str,strlen(str));
}

void handler(int nothing)
{
	int old;

	old = Number;
	count_mail();
	if(old==Number) return;
	update();
	if(Options&BELL_MODE && Number>old) XBell(dpy,100);
	XFlush(dpy);
}

void font_height(void)
{
	int foo,bar,baz;
	XCharStruct extents;

	XTextExtents(font,"0123456789",10,&foo,&bar,&baz,&extents);
	Ascent=extents.ascent;
	Descent=extents.descent;
}

void usage(void)
{
	fprintf(stderr,"XLassie v1.4\tby Trent Piepho <xyzzy@u.washington.edu>\n\n");
	fprintf(stderr,"Usage: xlassie [-h] | [-options]\n");
	fprintf(stderr,"where options include:\n");
	fprintf(stderr,"    -h                      print usage\n");
	fprintf(stderr,"    -name <name>            Set app name\n");
	fprintf(stderr,"    -fg <color>             foreground color [%s]\n",FgColor);
	fprintf(stderr,"    -bg <color>             background color [%s]\n",BgColor);
	fprintf(stderr,"    -update <number>        update count every number seconds\n");
	fprintf(stderr,"    -bell                   ring bell when count increases from zero\n");
	fprintf(stderr,"    -fn <fontname>          font for number\n");
	fprintf(stderr,"    -display <displayname>  X server to connect to\n");
	fprintf(stderr,"    -shape                  Use a shaped window\n");
	fprintf(stderr,"    -file <filename>        File to use for mail spool\n");
	fprintf(stderr,"    -command <command>      Command to execute when clicked on\n");
    fprintf(stderr,"    -hilight [color]        Use a different foreground color when\n"
	               "                            there is non-zero mail [%s]\n",HiColor);
#ifdef POP3
	fprintf(stderr,"    -pop3 <server>          Connect to pop3 server rather than local mail spool\n");
	fprintf(stderr,"    -apop3 <server>         Like -pop3, but uses a diferent method.\n");
	fprintf(stderr,"                                Use when -pop3 doesn't find the correct number\n");
	fprintf(stderr,"    -imap <server>          Use the IMAP protocol instead of pop3\n");
	fprintf(stderr,"    -username <name>        Username for pop3/imap server\n"
	               "                                if different from local username\n");
	fprintf(stderr,"    -password <word>        Password to use on pop3/imap server\n");
	fprintf(stderr,"                                Use the password 'ask' to get prompted for\n"
	               "                                the password on stdin.\n");
    fprintf(stderr,"    -offline                Don't exit when the server is unreachable\n");
#endif
    fprintf(stderr,"    -wmaker                 Do stuff to get swallowed into the WindowMaker dock\n");
	fprintf(stderr,"    -kde                    Do stuff to get swallowed into the KDE dock\n");
	fprintf(stderr,"                                Normally KDE or WindowMaker is autodetected\n"
	               "                                and you don't need to use the above options\n");
	fprintf(stderr,"    -nokde, -nowmaker       Don't do the stuff to get swallowed\n");
}

void ask_password()
{
	int i;

	printf("POP3 Password: ");
	fgets(Password,32,stdin);
	i=strlen(Password)-1;
	if(Password[i]=='\n') Password[i]='\0';
}

void parse_options(int argc,char *argv[])
{
	int i;
	int intervalused=0;

	for(i=1;i<argc;i++)  {
		if(!strcmp(argv[i],"-fg"))  {
			if(++i==argc)  { usage(); exit(2); };
			FgColor=argv[i];
		} else if(!strcmp(argv[i],"-bg"))  {
			if(++i==argc)  { usage(); exit(2); };
			BgColor=argv[i];
		} else if(!strcmp(argv[i],"-update"))  {
			if(++i==argc)  { usage(); exit(2); };
			Interval=atoi(argv[i]);
			intervalused=1;
		} else if(!strcmp(argv[i],"-bell"))  {
			Options|=BELL_MODE;
		} else if(!strcmp(argv[i],"-shape"))  {
			Options|=SHAPED_WINDOW;
		} else if(!strcmp(argv[i],"-fn") || !strcmp(argv[i],"-font"))  {
			if(++i==argc)  { usage(); exit(2); };
			FontName=argv[i];
		} else if(!strcmp(argv[i],"-display"))  {
			if(++i==argc)  { usage(); exit(2); };
			DisplayName=argv[i];
		} else if(!strcmp(argv[i],"-file"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
		} else if(!strcmp(argv[i],"-hilight"))  {
			Options|=HILIGHT_MAIL;
			if(i+1!=argc && argv[i+1][0]!='-') HiColor=argv[++i];
		} else if(!strcmp(argv[i],"-command"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(Command,argv[i]);
		} else if(!strcmp(argv[i],"-geometry"))  {
			if(++i==argc)  { usage(); exit(2); };
			Geometry=argv[i];
		} else if(!strcmp(argv[i],"-name")) {
			if(++i==argc)  { usage(); exit(2); };
			AppName=argv[i];
#ifdef POP3
		} else if(!strcmp(argv[i],"-pop3"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
			Options|=USE_POP3;
			if(!intervalused) Interval=INTERVAL_POP3;
		} else if(!strcmp(argv[i],"-apop3"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
			Options|=USE_APOP3;
			if(!intervalused) Interval=INTERVAL_POP3;
		} else if(!strcmp(argv[i],"-imap"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
			Options|=USE_IMAP;
			if(!intervalused) Interval=INTERVAL_POP3;
		} else if(!strcmp(argv[i],"-username"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(Username,argv[i]);
		} else if(!strcmp(argv[i],"-password"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(Password,argv[i]);
			memset(argv[i],0,strlen(argv[i]));
			if(!strcmp(Password,"ask"))  ask_password();
		} else if(!strcmp(argv[i],"-offline"))  {
			Options|=OFFLINE;
		} else if(!strcmp(argv[i],"-wmaker"))  {
			Options|=WM_MODE;
		} else if(!strcmp(argv[i],"-kde")) {
			Options|=KDE_MODE;
		} else if(!strcmp(argv[i],"-nokde")) {
			Options|=NOKDE;
		} else if(!strcmp(argv[i],"-nowmaker")) {
			Options|=NOWM;
#endif
		} else if(!strcmp(argv[i],"-h") || !strcmp(argv[i],"-help"))  {
			usage(); exit(0);
		} else {
			fprintf(stderr,"Unknown option %s\n",argv[i]);
			fprintf(stderr,"Use -h for help\n");
			exit(2);
		};
	};
}

void init(int argc,char *argv[])
{
	int screen;
	XWMHints *wmh;
	XSizeHints *xsh;
	XClassHint *classh;
	XColor color,tmp;
	Window icon;
	int x,y,g;

	parse_options(argc,argv);

	dpy=XOpenDisplay(DisplayName);
	if(dpy==NULL)  {
		fprintf(stderr,"Error: Can't open display: %s\n",DisplayName);
		exit(1);
	};
	screen=DefaultScreen(dpy);
	DeleteWindow = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

	if(!(Options&NOKDE)) {
		/* Turn on KDE mode if we find the KWM docking hint atom */
		KWMDockWindow = XInternAtom(dpy, "KWM_DOCKWINDOW", True);
		if(KWMDockWindow!=None) { 
			Options |= KDE_MODE;
			FontName = KDEFONT;
		};
	}
	if(!(Options&NOWM)) {
		/* Find a window maker atom... */
		Atom a = XInternAtom(dpy, "_WINDOWMAKER_WM_PROTOCOLS", True);
		if(a!=None) {
			Options |= WM_MODE;
			Options |= SHAPED_WINDOW;
			FontName = WMFONT;
		};
	}


	xsh=XAllocSizeHints();
	wmh=XAllocWMHints();
	classh=XAllocClassHint();

	font=XLoadQueryFont(dpy,FontName);
	if(font==NULL)  {
		fprintf(stderr,"%s:  unable to open font \"%s\"\n",argv[0],FontName);
		exit(1);
	};
	font_height();
	Height=Ascent+Descent;
	Width=XTextWidth(font,"88",2);
	xsh->flags = PSize; xsh->width = Width; xsh->height = Height;
/*	printf("%d x %d\n", Width, Height);*/

	g=XWMGeometry(dpy,screen,Geometry,NULL,0,xsh,&x,&y,&Width,&Height,&g);
	if(g&XValue)  { xsh->x=x; xsh->flags|=USPosition; };
	if(g&YValue)  { xsh->y=y; xsh->flags|=USPosition; };
	if(g&WidthValue)  {xsh->width=Width; xsh->flags|=USSize; 
	} else            {Width = xsh->width; };
	if(g&HeightValue)  {xsh->height=Height; xsh->flags|=USSize;
	} else            {Height = xsh->height; };

/*	printf("%dx%d+%d+%d\n",Width,Height,x,y);
	printf("%s%s%s%s\n",
		(xsh->flags&USPosition)?"USPosition ":"",
		(xsh->flags&PPosition)?"PPosition ":"",
		(xsh->flags&USSize)?"USSize ":"",
		(xsh->flags&PSize)?"PSize ":""); */

	win=XCreateSimpleWindow(dpy,RootWindow(dpy,screen),x,y,Width,Height,0,
		BlackPixel(dpy,screen),WhitePixel(dpy,screen));

	wmh->initial_state=NormalState;
	wmh->input=False;
	wmh->window_group = win;
	wmh->flags = StateHint | InputHint | WindowGroupHint;
	classh->res_name = (AppName==NULL)?"xlassie":AppName;
	classh->res_class = "XBiff";

	if(Options&WM_MODE) {
		/* WindowMaker stuff */
		icon=XCreateSimpleWindow(dpy,RootWindow(dpy,screen),x,y,Width,Height,
				 0, BlackPixel(dpy,screen), WhitePixel(dpy,screen));
		wmh->initial_state=WithdrawnState;
		wmh->icon_window=icon;
		wmh->flags |= IconWindowHint;
		XSetClassHint(dpy, icon, classh);
		XSelectInput(dpy, icon, ExposureMask|ButtonPressMask|StructureNotifyMask);
	}
	XmbSetWMProperties(dpy, win, "xlassie", "xlassie", argv, argc,
					   xsh, wmh, classh);
	XSetWMProtocols(dpy, win, &DeleteWindow, 1);
	XSelectInput(dpy, win, ExposureMask|ButtonPressMask|StructureNotifyMask);

	if(Options&KDE_MODE) {
		unsigned char data = 1;
		XChangeProperty(dpy, win, KWMDockWindow, KWMDockWindow, 8, 
		                PropModeReplace, &data, 1);
	}

	XMapWindow(dpy, win);
	if(Options&WM_MODE) win = icon;		/* Use the icon window from now on */

	gc=XCreateGC(dpy,win,0,NULL);

	XAllocNamedColor(dpy,DefaultColormap(dpy,screen),FgColor,&color,&tmp);
	XSetForeground(dpy,gc,color.pixel); FgXColor=color;
	XAllocNamedColor(dpy,DefaultColormap(dpy,screen),BgColor,&color,&tmp);
	XSetBackground(dpy,gc,color.pixel);
	XSetWindowBackground(dpy,win,color.pixel);
	if(Options&HILIGHT_MAIL)  XAllocNamedColor(dpy,DefaultColormap(dpy,screen),
												HiColor,&HiXColor,&tmp);
	XSetFont(dpy,gc,font->fid);

	if(Options&SHAPED_WINDOW)  {
		MaskWidth = Width; MaskHeight = Height;
		ShapeMask = XCreatePixmap(dpy,win,MaskWidth,MaskHeight,1);
		ShapeGC = XCreateGC(dpy,ShapeMask,0,NULL);
		XSetFont(dpy,ShapeGC,font->fid);
	};
}

int main(int argc,char *argv[])
{
	XEvent xev;
	Atom WM_PROTOCOLS;
	struct itimerval itv;
	struct sigaction sig;
	struct passwd* pwd;
	char *mail;

	pwd = getpwuid(getuid());
	mail=getenv("MAIL");
	if(mail==NULL) {
		sprintf(SpoolFile,"/var/spool/mail/%s",pwd->pw_name);
	} else {
		strcpy(SpoolFile,mail);
	};
	strcpy(Username,pwd->pw_name);
	strcpy(Command,COMMAND);

	init(argc,argv);

	WM_PROTOCOLS = XInternAtom(dpy, "WM_PROTOCOLS", False);

#ifdef OFFLINE_DEFAULT
	Options|=OFFLINE;
#endif

	count_mail();

	if(Options&BELL_MODE && Number) XBell(dpy,100);

	sig.sa_handler=handler;
	sigemptyset(&sig.sa_mask);
	sig.sa_flags=SA_RESTART;
	sigaction(SIGALRM,&sig,NULL);
	itv.it_interval.tv_sec=Interval; itv.it_interval.tv_usec=0;
	itv.it_value = itv.it_interval;
	setitimer(ITIMER_REAL,&itv,NULL);

	for(;;)  {
		XNextEvent(dpy,&xev);
		switch(xev.type)  {
		case Expose:
			while(XCheckTypedEvent(dpy,Expose,&xev));
			update();
			break;
		case ButtonPress:
			system(Command);
			break;
		case ConfigureNotify:
			Width = xev.xconfigure.width;
			Height = xev.xconfigure.height;
			update();
			break;
		case ClientMessage:
			if(xev.xclient.message_type == WM_PROTOCOLS)  {
				if(xev.xclient.data.l[0] == DeleteWindow) {
					XCloseDisplay(dpy);
					exit(0);
				};
			};
			break;
		case DestroyNotify:
			XCloseDisplay(dpy);
			exit(0);
		default:
		};
	};
}

#ifdef POP3
int sock_connect(char *,int);

FILE *pop_login()
{
	int fd;
	FILE *f;
	char buf[256];

	fd=sock_connect(SpoolFile,110);
	if(fd==-1)  {
		if(Options&OFFLINE)  {
			Number=-1; return NULL;
		} else {
			fprintf(stderr,"Error connecting to POP3 server %s\n",SpoolFile);
			exit(1);
		};
	};

	f=fdopen(fd,"r+");

	fgets(buf,256,f);

	fflush(f); fprintf(f,"USER %s\r\n",Username);
	fflush(f); fgets(buf,256,f);

	if(buf[0]!='+')  {
		fprintf(stderr,"Server doesn't like your name!\n");
		fprintf(f,"QUIT\r\n");fclose(f);
		return NULL;
	};
	fflush(f); fprintf(f,"PASS %s\r\n",Password);
	fflush(f); fgets(buf,256,f);
	if(buf[0]!='+')  {
		fprintf(stderr,"Server doesn't like your password!\n");
		fprintf(stderr,"It said %s",buf);
		fprintf(f,"QUIT\r\n");fclose(f);
		return NULL;
	};

	return(f);
}

FILE *imap_login()
{
	int fd;
	FILE *f;
	char buf[128];

	fd=sock_connect(SpoolFile, 143);
	if(fd==-1)  {
		if(Options&OFFLINE)  {
			Number=-1; return NULL;
		} else {
			fprintf(stderr, "Error connecting to IMAP server %s\n", SpoolFile);
			exit(1);
		};
	};

	f=fdopen(fd, "r+");

	fgets(buf, 127, f);
	fflush(f); fprintf(f, "a001 LOGIN %s %s\r\n", Username, Password);
	fflush(f); fgets(buf, 127, f);

	if(buf[5]!='O')  {       /* Looking for "a001 OK" */
		fprintf(stderr, "Server doesn't like your name and/or password!\n");
		fprintf(stderr, "a002 LOGOUT\r\n"); fclose(f);
		return NULL;
	};

	return f;
}

int count_mail_pop()
{
	FILE *f;
	char buf[256];
	int total,read;

	f = pop_login();
	if(f==NULL) return -1;

	fflush(f); fprintf(f,"STAT\r\n");
	fflush(f); fgets(buf,256,f);
	sscanf(buf,"+OK %d",&total);
	fflush(f); fprintf(f,"LAST\r\n");
	fflush(f); fgets(buf,256,f);
	sscanf(buf,"+OK %d",&read);
	Number=total-read;

	fprintf(f,"QUIT\r\n");
	fclose(f);

	return 1;
}

struct msg {
	int size;
	int read;
};

struct msg *get_list(FILE *f,int *nout)
{
	char buf[256];
	struct msg *l;
	int i,n;

	fflush(f); fprintf(f,"STAT\r\n");
	fflush(f); fgets(buf,256,f);
	sscanf(buf,"+OK %d",&n);

	l=malloc(sizeof(struct msg)*n);
	fflush(f); fprintf(f,"LIST\r\n");
	fflush(f); fgets(buf,256,f);
	if(strncmp(buf,"+OK",3))  {
		fprintf(stderr,"Can't get list?\n");
		return NULL;
	};

	for(i=0;i<n;i++)  {
		fgets(buf,256,f);
		sscanf(buf,"%*d %d",&l[i].size);
	};
	fgets(buf,256,f);
	if(buf[0]!='.')  {
		fprintf(stderr,"To many messages?\n");
		return NULL;
	};

	*nout=n;
	return l;
}

void get_header(FILE *f,struct msg *l,int i)
{
	char buf[256];
	
	fflush(f); fprintf(f,"TOP %d 1\r\n",i+1);
	fflush(f); fgets(buf,256,f);
	if(strncmp(buf,"+OK",3))  {
		fprintf(stderr,"TOP failed!\n");
		fprintf(stderr,"Returned %s\n",buf);
		return;
	};
	l->read=0;
	for(;;)  {
		fgets(buf,256,f);
		if(!strcmp(buf,".\r\n"))  break;
		if(!strncmp(buf,"Status: R",9))
			l->read=1;
	};
	/* fprintf(stderr,"msg %d %s\n",i+1,l->read?"read":"unread"); */
}

int count_mail_pop_2()
{
	FILE *f;
	int i,j,n2;
	int total=0;
	static struct msg *l1=NULL,*l2;
	static int n;

	f = pop_login();

	if(l1==NULL)  {  /* get initial list */
		l1=get_list(f,&n);
		if(l1==NULL) return -1;
		for(i=0;i<n;i++) {
			get_header(f,l1+i,i);
			if(!l1[i].read) total++;
		};
		Number=total; 
		return 1;
	};

	l2=get_list(f,&n2);

	/* compare lists, retreive new messages */
	for(i=0,j=0;i<n && j<n2;i++,j++)  {
/*		fprintf(stderr,"Old list %d size %d %c, new list %d size %d\n",
		               i+1,l1[i].size,l1[i].read?'R':'U', j+1,l2[j].size); */
		if(l1[i].size != l2[j].size) break;
		if(!l1[i].read)  {
			get_header(f,l2+j,j);
		} else {
			l2[j].read=1;
		};
	};
	for(;j<n2;j++) get_header(f,l2+j,j);

	fclose(f);

	free(l1); l1=l2; l2=NULL; n=n2;
	for(i=0;i<n;i++)  if(!l1[i].read) total++;
	Number=total;

	return 1;
}

int count_mail_imap()
{
	FILE *f;
	char buf[128];
	int total=0;

	f = imap_login();
	if(f==NULL) return -1;

	fflush(f); fprintf(f, "a003 STATUS INBOX (UNSEEN)\r\n");
	fflush(f); fgets(buf, 127, f);
	sscanf(buf, "* STATUS INBOX (UNSEEN %d)", &total);

	Number = total;

	fclose(f);

	return 1;
}
int count_mail()
{
	if(Options&USE_APOP3)  {
		return count_mail_pop_2();
	} else if(Options&USE_POP3)  {
		return count_mail_pop();
	} else if(Options&USE_IMAP) {
		return count_mail_imap();
	} else {
		return count_mail_spool();
	};
}

int count_mail_spool()
#else
int count_mail()
#endif
{
	static time_t mtime=0;
	static off_t size=0;
	struct stat stat;
	int isread=0;
	FILE *spool;
	char buf[256];

	spool = fopen(SpoolFile,"r");
	if(spool==NULL)  {
		fprintf(stderr,"Trying to open spoolfile %s\n",SpoolFile);
		perror("xlassie");
		XCloseDisplay(dpy);
		exit(errno);
	};

	if(fstat(fileno(spool),&stat)) {
		perror("");
		XCloseDisplay(dpy);
		exit(errno);
	};
	/* check to see if file was changed */
	if(stat.st_mtime != mtime || stat.st_size != size)  {  
		mtime = stat.st_mtime;
		size = stat.st_size;
		Number = 0;
		while(fgets(buf, 256, spool)) {
			if(!strncmp(buf, "From ", 5))  {
				Number++; isread=0;
			} else if(!strncmp(buf, "Status: R", 9))  {
				if(!isread) { Number--; isread=1; };
			};
		};
	}

	fclose(spool);

	return 1;
}
