#define _GNU_SOURCE
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <utmp.h>
#include "hwclock.h"


static void
write_wtmp(const char wtmp_file[], 
           const struct utmp oldutmp,
           const struct utmp newutmp) {
/*----------------------------------------------------------------------------
   Write both utmp records to the wtmp log.


   GNU C Library Version 2 contains a function updwtmp() to add a record to the
   wtmp log, and we should use that.  But older C libraries with which we may
   be linked do not have updwtmp(), so we have to use a more primitive method
   to write the wtmp log.  The natural way to write this code to account for
   both situtation (updwtmp() present and updwtmp() not present) is with the
   use of a weak external reference or weak symbol.  
  
   (That declaration might look like:

     extern void __attribute__ ((weak)) updwtmp(const char *wtmp_file,
                                                const struct utmp *utmp);
   )

   But older compilers (gcc before release 2.6 at least) don't
   understand weak declarations, and we've even seen one system where
   gcc accepted the declaration, but spit out an "unresolved external
   reference" link error nonetheless.
   
   So instead, we check explicitly for a modern version of the C library
   in conditional compilation.
-----------------------------------------------------------------------------*/

#if __GLIBC__ >= 2

    if (debug) 
        printf("Updating wtmp file '%s' with updwtmp().\n", wtmp_file);
    updwtmp(wtmp_file, &oldutmp);
    updwtmp(wtmp_file, &newutmp);

#else
    FILE *wtmp;

    if (debug) 
        printf("Updating wtmp file '%s' with fwrite().\n", wtmp_file);
    
    wtmp = fopen(wtmp_file, "a");
    if (!wtmp) {
        fprintf(stderr, "%s: Warning: unable to log record to wtmp file "
                "'%s'.\n"
                "fopen() failed with errno = %s (%d).\n",
                MYNAME, wtmp_file, strerror(errno), errno);
    } else {
        fwrite(&oldutmp, sizeof(oldutmp), 1, wtmp);
        fwrite(&newutmp, sizeof(newutmp), 1, wtmp);
        
        fclose(wtmp);
    }
#endif
}


void 
update_utmp(const struct timeval oldtime,
            const struct timeval newtime) {
/*----------------------------------------------------------------------------
   Put an entry in the utmp database and the wtmp log showing the
   clock setting.

   Note that this particular form of utmp record is forced into the
   utmp format with a sledge hammer and virtually undocumented, so
   things may get a little crazy here.
-----------------------------------------------------------------------------*/
    const char *utmp_file;
    const char *wtmp_file;
    struct utmp oldutmp, newutmp;

    /* To be clean, we put zeroes in all the undefined fields */
    bzero(&oldutmp, sizeof(oldutmp));
    bzero(&newutmp, sizeof(newutmp));

    oldutmp.ut_type = OLD_TIME;
    newutmp.ut_type = NEW_TIME;
#ifdef _HAVE_UT_TV
    oldutmp.ut_tv = oldtime;
    newutmp.ut_tv = newtime;
#else
    oldutmp.ut_time = oldtime.tv_sec;
    newutmp.ut_time = newtime.tv_sec;
#endif

    /* ut_host is undefined in these records.  We put the program name in
       there to assist in debugging.
       */
    strncpy(oldutmp.ut_host, MYNAME, sizeof(oldutmp.ut_host));
    strncpy(newutmp.ut_host, MYNAME, sizeof(newutmp.ut_host));

    utmp_file = getenv("UTMP_FILE");
    if (!utmp_file) 
        utmp_file = UTMP_FILE;

    if (debug) 
        printf("Updating utmp file '%s' with pututline().\n", utmp_file);
    
    utmpname(utmp_file);
    setutent();
    pututline(&oldutmp);
    setutent();
    pututline(&newutmp);
    endutent();

    wtmp_file = getenv("WTMP_FILE");
    if (!wtmp_file) 
        wtmp_file = WTMP_FILE;

    write_wtmp(wtmp_file, oldutmp, newutmp);
}

 

