/**************************************************************************

  This is a component of the hwclock program.

  This file contains the code for accessing the hardware clock via
  the rtc device driver (usually hooked up to the /dev/rtc device
  special file).

  That device driver is usually drivers/char/rtc.c in the Linux source
  tree.  For non-PCI (older) Sun Sparc systems, it is usually
  drivers/sbus/char/rtc.c.

****************************************************************************/

#define _GNU_SOURCE
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <sys/stat.h>

#include "hwclock.h"

/*
  Get defines for rtc stuff.
 
  Getting the rtc defines is nontrivial.  The obvious way is by
  including <linux/mc146818rtc.h> but that again includes <asm/io.h>
  which again includes ...  and on sparc and alpha this gives
  compilation errors for many kernel versions. So, we give the defines
  ourselves here.  Moreover, some Sparc person decided to be
  incompatible, and used a struct rtc_time different from that used in
  mc146818rtc.h.  
*/

/* On Sparcs, there is a <asm/rtc.h> that defines different ioctls
   (that are required on my machine). However, this include file
   does not exist on other architectures. */
/* One might do:
#if defined(__sparc__) || defined(__sparc64__)
#include <asm/rtc.h>
#endif
 */

/* The following is roughly equivalent */
struct sparc_rtc_time
{
        int     sec;    /* Seconds (0-59) */
        int     min;    /* Minutes (0-59) */
        int     hour;   /* Hour (0-23) */
        int     dow;    /* Day of the week (1-7) */
        int     dom;    /* Day of the month (1-31) */
        int     month;  /* Month of year (1-12) */
        int     year;   /* Year (0-99) */
};


#define RTCGET _IOR('p', 0x14, struct sparc_rtc_time)
#define RTCSET _IOW('p', 0x15, struct sparc_rtc_time)


#include <linux/version.h>
/* Check if the rtc device driver is available in this version of
   the system headers.  131072 is linux 2.0.0.  Might need to make
   it conditional on i386 or something too -janl */
#if LINUX_VERSION_CODE >= 131072
#include <linux/kd.h>
static const bool got_rtc = TRUE;
#else
static const bool got_rtc = FALSE;
#endif

/* struct linux_rtc_time is present since Linux 1.3.99 */
/* Earlier (since 1.3.89), a struct tm was used. */
struct linux_rtc_time {
        int tm_sec;
        int tm_min;
        int tm_hour;
        int tm_mday;
        int tm_mon;
        int tm_year;
        int tm_wday;
        int tm_yday;
        int tm_isdst;
};

/* RTC_RD_TIME etc have this definition since Linux 1.99.9 (pre2.0-9) */
#ifndef RTC_RD_TIME
#define RTC_RD_TIME       _IOR('p', 0x09, struct linux_rtc_time)
#define RTC_SET_TIME      _IOW('p', 0x0a, struct linux_rtc_time)
#define RTC_UIE_ON        _IO('p', 0x03)	/* Update int. enable on */
#define RTC_UIE_OFF       _IO('p', 0x04)	/* Update int. enable off */
#endif
/* RTC_EPOCH_READ and RTC_EPOCH_SET ioctls are in kernels since 
   Linux 2.0.34 and 2.1.89 
   */
#ifndef RTC_EPOCH_READ
#define RTC_EPOCH_READ	_IOR('p', 0x0d, unsigned long)	 /* Read epoch */
#define RTC_EPOCH_SET	_IOW('p', 0x0e, unsigned long)	 /* Set epoch */
#endif

/* rtc_read_val is the type of the data returned by a read operation
   (not to be confused with the ioctl operation that actually reads
   the clock!) to the rtc device.
*/
#if defined(__sparc__) || defined(__sparc64__)
/* This only gets used on UltraSparc where it is a 64 bit value */
/* Do not be confused by the fact that the rtc driver code declares
   this as type "long", while we define it as "long long".  The kernel
   (including the rtc driver) is compiled with a 64 bit compiler that
   defines "long" as 64 bits.  The file you are reading is normally
   compiled with a compiler that defines "long" as 32 bits and 
   only "long long" as 64 bits.
*/
typedef unsigned long long rtc_read_val;
#else
typedef unsigned long rtc_read_val;
#endif


static void
do_rtc_read_ioctl(int rtc_fd, struct tm *tm_p, int *retcode_p) {
/*----------------------------------------------------------------------------
   Do the ioctl to read the time.  This is, unfortunately, a slightly
   different function for Sparc than for everything else.  But we
   return the standard 'tm' structure result in spite of the fact that
   the Sparc ioctl returns something else.
   
   If the ioctl fails, issue message to stderr and return rc = 1;
   else, no message and rc = 0.
-----------------------------------------------------------------------------*/
  struct sparc_rtc_time stm;
  int rc;

  /* Non-PCI (older) Sparc systems use RTCGET but PCI ones, and
     everybody else, use RTC_RD_TIME.  The only way we can tell if the
     system wants RTCGET or not is to try it. 
  */
  rc = ioctl(rtc_fd, RTCGET, &stm);
  if (rc != -1) {
    /* It liked the RTCGET, now make it look like we did RTC_RD_TIME */
    tm_p->tm_sec = stm.sec;
    tm_p->tm_min = stm.min;
    tm_p->tm_hour = stm.hour;
    tm_p->tm_mday = stm.dom;
    tm_p->tm_mon = stm.month - 1;
    tm_p->tm_year = stm.year - 1900;
    tm_p->tm_wday = stm.dow - 1;
    tm_p->tm_yday = -1;		/* day in the year */
  } else {
    /* The rtc driver doesn't know RTCGET; it must know RTC_RD_TIME */
    rc = ioctl(rtc_fd, RTC_RD_TIME, tm_p);
  }
  if (rc == -1) {
    fprintf(stderr, "%s: ioctl() to rtc device driver to read the time "
            "failed.  "
            "errno = %s (%d)\n", MYNAME, strerror(errno), errno);
    *retcode_p = 1;
  } else *retcode_p = 0;
  tm_p->tm_isdst = -1;          /* don't know whether it's daylight */
  return;
}  



void
get_epoch(const char rtc_filespec[], unsigned long *epoch_p, char **reason_p){
/*----------------------------------------------------------------------------
  Get the Hardware Clock epoch setting from the kernel.
  
  If we succeed, return the setting (number of year A.D.) as 
  *epoch_p and *reason_p == NULL.

  If we fail, return and English description of what went wrong as a
  null-terminated string in newly malloc'ed storage and the pointer to
  it as *reason_p.
----------------------------------------------------------------------------*/
  int rtc_fd;

  rtc_fd = open(rtc_filespec, O_RDONLY);
  if (rtc_fd < 0) {
    if (errno == ENOENT) 
        asprintf(reason_p,
                 "To manipulate the epoch value in the kernel, we must "
                 "access the Linux 'rtc' device driver via the device special "
                 "file %s.  This file does not exist on this system.\n", 
                 rtc_filespec);
    else {
        asprintf(reason_p, "Unable to open %s, open() errno = %s (%d)\n",
                 rtc_filespec, strerror(errno), errno);
    }
  } else {
    int rc;  /* return code from ioctl */
    rc = ioctl(rtc_fd, RTC_EPOCH_READ, epoch_p);
    if (rc == -1) {
      *reason_p = malloc(200);
      sprintf(*reason_p, "ioctl(RTC_EPOCH_READ) to %s failed, "
              "errno = %s (%d).\n", rtc_filespec, strerror(errno), errno);
    } else {
      *reason_p = NULL;
      if (debug) printf("we have read epoch %ld from %s "
                        "with RTC_EPOCH_READ ioctl.\n", 
                        *epoch_p, rtc_filespec);
    }
    close(rtc_fd);
  }
  return;
}



static void
busywait_for_rtc_clock_tick(const int rtc_fd, int *retcode_p) {
/*----------------------------------------------------------------------------
   Wait for the top of a clock tick by reading from rtc device in a
   busy loop until we see it.
-----------------------------------------------------------------------------*/
  struct tm start_time;
    /* The time when we were called (and started waiting) */
  int rc;

  if (debug)
    printf("Waiting in loop for time from rtc device driver to change\n");

  do_rtc_read_ioctl(rtc_fd, &start_time, &rc);
  if (rc != 0) *retcode_p = 1;
  else {
    /* Wait for change.  Should be within a second, but in case something
       weird happens, we have a limit on this loop to reduce the impact
       of this failure.
       */
    struct tm nowtime;
    time_t giveup_systime;  
        /* System time at which we should give up waiting for the hardware
           clock to change.
        */
    int rc;  /* Return code from do_rtc_read_ioctl */

    giveup_systime = time(NULL) + 3;
    do {
      do_rtc_read_ioctl(rtc_fd, &nowtime, &rc);
    } while (rc == 0 && start_time.tm_sec == nowtime.tm_sec 
             && time(NULL) < giveup_systime);

    if (time(NULL) >= giveup_systime) {
      fprintf(stderr, "%s: Hw clock time did not change for 2+ seconds.\n", 
              MYNAME);
      *retcode_p = 2;
    } else if (rc != 0) *retcode_p = 3;
    else *retcode_p = 0;
  }
}



static int alarm_happened;
    /* An alarm signal has occurred */

static void
remember_alarm(const int dummy) {
/*----------------------------------------------------------------------------
   This is a signal handler.
-----------------------------------------------------------------------------*/
    alarm_happened = TRUE;
}



static void
read_rtc(const int rtc_fd, int * const timed_out_p, int * const error_p) {
/*----------------------------------------------------------------------------
   Read from rtc, but time out after 2 seconds.

   The read blocks until an interrupt which has been enabled occurs.
   We sometimes see the case that we successfully enable an interrupt,
   but the interrupt never occurs.  This is arguably a bug in the rtc
   driver, but its effect is to hang Hwclock and, consequently, startup
   of a system.  That's why we have the timeout.
-----------------------------------------------------------------------------*/

    struct sigaction action;
    struct sigaction oldaction;
    rtc_read_val dummy;
    int rc;

    action.sa_handler = &remember_alarm;
    sigemptyset(&action.sa_mask);
    action.sa_flags = 0;
    
    rc = sigaction(SIGALRM, &action, &oldaction);
      /* Install our signal handler */
    if (rc != 0) {
        fprintf(stderr, "%s: " 
                "sigaction() failed to install handler for SIGALRM.\n",
                MYNAME);
        *error_p = TRUE;
    } else {
        alarm_happened = FALSE;  /* No alarm has happened */

        alarm(2);   /* generate SIGALARM 2 seconds from now */

        rc = read(rtc_fd, &dummy, sizeof(dummy));
        if (rc == -1) {
            if (errno == EINTR && alarm_happened) {
                /* the read timed out */
                *error_p = FALSE;
                *timed_out_p = TRUE;
            } else {
                fprintf(stderr, 
                        "%s: read() to rtc device to wait for clock tick "
                        "failed, errno = %s (%d).\n", 
                        MYNAME, strerror(errno), errno);
                *error_p = TRUE;
                *timed_out_p = FALSE;
            }
        } else {
            *error_p = FALSE;
            *timed_out_p = FALSE;
        }
        alarm(0);   /* cancel alarm, if it hasn't sprung yet */

        sigaction(SIGALRM, &oldaction, NULL);  /* Remove our signal handler */
    }
}



void
synchronize_to_clock_tick_RTC(int * const retcode_p,
                              char rtc_filespec[]) {
/*----------------------------------------------------------------------------
  Wait until the moment the Hardware Clock updates to the next second,
  so we know the exact time.

  The clock only has 1 second precision, so it gives the exact time only
  once per second.

  Return *retcode_p == 0 if it worked, nonzero if it didn't.
-----------------------------------------------------------------------------*/
  int rtc_fd;  /* File descriptor of rtc device driver device */

  rtc_fd = open(rtc_filespec, O_RDONLY);

  if (rtc_fd == -1) {
    fprintf(stderr, "%s: open() of %s failed, errno = %s (%d).\n",
            MYNAME, rtc_filespec, strerror(errno), errno);
    *retcode_p = 1;
  } else {
    int rc;  /* Return code from ioctl */
    /* Turn on update interrupts (one per second) */
    rc = ioctl(rtc_fd, RTC_UIE_ON, 0);
    if (rc == -1 && errno == EINVAL) {
      /* This rtc device doesn't have interrupt functions.  This is typical
         on an Alpha, where the Hardware Clock interrupts are used by the
         kernel for the system clock, so aren't at the user's disposal.
         */
      if (debug) printf("%s does not have interrupt functions. \n", 
                        rtc_filespec);
      busywait_for_rtc_clock_tick(rtc_fd, retcode_p);
    } else if (rc != -1) {
      int rc;  /* return code from ioctl */
      int timed_out;
      int read_error;

      read_rtc(rtc_fd, &timed_out, &read_error);

      /* Turn off update interrupts */
      rc = ioctl(rtc_fd, RTC_UIE_OFF, 0);
      if (rc == -1) {
        fprintf(stderr, 
                "%s: ioctl() to %s to turn off update interrupts "
                "failed, errno = %s (%d).\n", 
                MYNAME, rtc_filespec, strerror(errno), errno);
      }

      if (read_error)
          *retcode_p = 1;
      else if (timed_out) {
          /* This shouldn't happen -- it means the rtc driver is broken.
             But we've seen it numerous times.  Since the update interrupts
             aren't being delivered, we must busywait instead:
             */
          if (debug) printf("%s accepted the ioctl to enable update "
                            "interrupts, but we timed out waiting for one.\n"
                            "busy waiting for clock tick instead...\n",
                            rtc_filespec);
          busywait_for_rtc_clock_tick(rtc_fd, retcode_p);
      } else
          *retcode_p = 0;
    } else {
      fprintf(stderr, "%s: ioctl() to %s to turn on update interrupts "
              "failed unexpectedly, errno = %s (%d).\n", 
              MYNAME, rtc_filespec, strerror(errno), errno);
      *retcode_p = 1;
    } 
    close(rtc_fd);
  }
}



void
read_hardware_clock_rtc_ioctl(const char rtc_filespec[],
                              struct tm * const tm_p) {
/*----------------------------------------------------------------------------
  Read the hardware clock and return the current time as *tm_p.
  Use ioctls to Linux rtc device driver.
-----------------------------------------------------------------------------*/
  int rc;   /* Local return code */
  int rtc_fd;  /* File descriptor of rtc device driver device */
  
  rtc_fd = open(rtc_filespec, O_RDONLY);
  if (rtc_fd == -1) {
      fprintf(stderr, "%s: open() of %s failed, errno = %s (%d).\n",
              MYNAME, rtc_filespec, strerror(errno), errno);
      fprintf(stderr, "An earlier open() of the same file worked fine, "
              "though, so this is\n"
              "no ordinary problem.\n"
          );
      exit(5);
  } else {
    /* Read the RTC time/date */

    do_rtc_read_ioctl(rtc_fd, tm_p, &rc);
    if (rc != 0) {
      fprintf(stderr, "%s: unable to read the time from rtc device.",
              MYNAME);
      exit(5);
    }
    close(rtc_fd);
  }
  tm_p->tm_isdst = -1;          /* don't know whether it's daylight */
}



void
set_hardware_clock_rtc_ioctl(const struct tm new_broken_time, 
                             const char rtc_filespec[],
                             const bool testing) {
/*----------------------------------------------------------------------------
  Set the Hardware Clock to the broken down time <new_broken_time>.
  Use ioctls to "rtc" device rtc_filespec[].
----------------------------------------------------------------------------*/
  int rc;
  int rtc_fd;

  rtc_fd = open(rtc_filespec, O_RDONLY);
  if (rtc_fd < 0) {
    fprintf(stderr, "%s: Unable to open %s, open() errno = %s (%d)\n",
            MYNAME, rtc_filespec, strerror(errno), errno);
    exit(5);
  } else {
    if (testing) 
      printf("Not setting Hardware Clock because running in test mode.\n");
    else {
      rc = ioctl(rtc_fd, RTC_SET_TIME, &new_broken_time);
      if (rc == -1) {
        fprintf(stderr, 
                "%s: ioctl() (RTC_SET_TIME) to %s to set time failed, "
                "errno = %s (%d).\n", 
                MYNAME, rtc_filespec, strerror(errno), errno);
        exit(5);
      } else {
        if (debug)
          printf("ioctl(RTC_SET_TIME) was successful.\n");
      }
    }
    close(rtc_fd);
  }
}



void
set_epoch(const char rtc_filespec[], 
          unsigned long epoch, const bool testing, int *retcode_p) {
/*----------------------------------------------------------------------------
  Set the Hardware Clock epoch in the kernel.
----------------------------------------------------------------------------*/
  if (epoch < 1900)
    /* kernel would not accept this epoch value */
    fprintf(stderr, "%s: The epoch value may not be less than 1900.  "
            "You requested %ld\n", MYNAME, epoch);
  else {
    int rtc_fd;
    
    rtc_fd = open(rtc_filespec, O_RDONLY);
    if (rtc_fd < 0) {
      if (errno == ENOENT) 
        fprintf(stderr, 
                "%s: To manipulate the epoch value in the kernel, we must "
                "access the Linux 'rtc' device driver via the device special "
                "file '%s'.  This file does not exist on this system.\n",
                MYNAME, rtc_filespec);
      fprintf(stderr, "%s: Unable to open %s, open() errno = %s (%d)\n",
              MYNAME, rtc_filespec, strerror(errno), errno);
      *retcode_p = 1;
    } else {
      if (debug) printf("setting epoch to %ld "
                        "with RTC_EPOCH_SET ioctl to %s.\n", 
                        epoch, rtc_filespec);
      if (testing) {
        printf("Not setting epoch because running in test mode.\n");
        *retcode_p = 0;
      } else {
        int rc;                 /* return code from ioctl */
        rc = ioctl(rtc_fd, RTC_EPOCH_SET, epoch);
        if (rc == -1) {
          if (errno == EINVAL)
            fprintf(stderr, "%s: The kernel (specifically, the device driver "
                    "for %s) does not have the RTC_EPOCH_SET ioctl.  "
                    "Get a newer driver.\n", MYNAME, rtc_filespec);
          else 
            fprintf(stderr, "%s: ioctl(RTC_EPOCH_SET) to %s failed, "
                    "errno = %s (%d).\n", 
                    MYNAME, rtc_filespec, strerror(errno), errno);
          *retcode_p = 1;
        } else *retcode_p = 0;
      }
      close(rtc_fd);
    }
  }
}


static void
validate_rtc_file(const char filespec[], char ** const error_p) {
/*----------------------------------------------------------------------------
  See if the file with filespec filespec[] is a usable device special file
  for accessing the Linux rtc device driver.
  
  If so, return *error_p == NULL.  If not, return a text description why
  as a null terminated string in newly malloc'ed storage and return its
  address as *error_p.
-----------------------------------------------------------------------------*/
    /* TODO: We should make sure it's a device special file, because
       otherwise there's no way our ioctls will work on it!
       */
    int rc;
    struct stat statbuf;

    rc = stat(filespec, &statbuf);

    if (rc != 0) {
        asprintf(error_p, "You don't have a file named '%s'.\n"
                 "To access "
                 "the Linux rtc device driver, this would ordinarily be \n"
                 "a character device "
                 "special file with major number 10, minor number 135.\n"
                 "You can use 'mknod' to create this,\n"
                 "but you will just get a different error "
                 "unless you actually\n"
                 "have the rtc device driver installed and "
                 "registered to serve "
                 "that device.\n"
                 "(specific errno from stat() is %d (%s))",
                 filespec, errno, strerror(errno)
            );
    } else {
        /* It exists, but is is a character device special file? */
        if (!S_ISCHR(statbuf.st_mode)) {
            asprintf(error_p, "The file '%s' is not a character device "
                     "special file.  In order to access the Linux rtc "
                     "device driver, it would have to be.  Use ls -l "
                     "to find out what this file really is.", 
                     filespec
                     );
        } else {
            /* There's a character device file there.  But can we open it? */

            int rtc_fd = open(filespec, O_RDONLY);
            
            if (rtc_fd > 0) {
                *error_p = NULL;
                close(rtc_fd);
            } else {
                switch (errno) {
                case ENOENT:
                    asprintf(error_p,
                             "Internal error in %s: stat() succeeded, "
                             "but open says "
                             "'no such file' (ENOENT).", __FUNCTION__);
                    break;
                case ENODEV:
                    asprintf(error_p,
                             "The device special file '%s' exists, but the "
                             "device driver for it is\n"
                             "not in your kernel "
                             "(and it did not get autoloaded either).  "
                             "See the \n"
                             "Hwclock man page (section \"Linux rtc "
                             "device driver \") for details.\n", filespec
                        );
                    break;
                case EBUSY:
                    asprintf(error_p, 
                             "The rtc device is already open "
                             "(maybe by another "
                             "process).  The rtc \n"
                             "device may be open only once.  "
                             "The 'lsof' program will tell you who has \n"
                             "it open, except that you don't necessarily know "
                             "by what device special \n"
                             "file it is open.  It may be accessible via "
                             "other device special \n"
                             "files than '%s'.", filespec
                        );
                    break;
                default:
                    asprintf(error_p,
                             "There was a general file access error "
                             "(nothing specific \n"
                             "to the Linux rtc device driver) "
                             "in opening the file '%s'.\n"
                             "errno was %d (%s).", 
                             filespec, errno, strerror(errno)
                        );
                }
            }
        }  
    }
}


static bool
file_exists(const char filespec[]) {

    bool retval;
    int rc;
    struct stat statbuf;

    rc = stat(filespec, &statbuf);
    
    if (rc == 0)
        retval = TRUE;
    else {
        if (errno == ENOENT)
            retval = FALSE;
        else {
            fprintf(stderr, "Unable to determine if file '%s' exists.  "
                "stat() errno = %s (%d).  Assuming it doesn't exist.\n",
                    filespec, strerror(errno), errno);
            retval = FALSE;
        }
    }
    return retval;
}



void
find_default_rtc(char ** const filespec_p,
                 char ** const error_p) {
/*----------------------------------------------------------------------------
  Find a device special file by which we can access the Linux rtc device
  driver, if any, assuming the user has no preference.

  If we can, return *error_p == NULL.  Else, return *error_p
  pointing to a null-terminated string explanation of why in newly malloc'ed
  storage.

  In the case of success, also return the filespec of the device
  special file that we found can access the hardware clock, in newly
  malloc'ed storage pointed to by *filespec_p.
-----------------------------------------------------------------------------*/
    if (got_rtc) {
        if (file_exists("/dev/misc/rtc"))
            *filespec_p = strdup("/dev/misc/rtc");
        else if (file_exists("/dev/rtc"))
            *filespec_p = strdup("/dev/rtc");
        else
            *filespec_p = NULL;

        if (*filespec_p) {
            char * error;
            validate_rtc_file(*filespec_p, &error);
            if (error) {
                asprintf(error_p, "Found file %s, but %s", 
                         *filespec_p, error);
                free(error);
            } else
                *error_p = NULL;
        } else {
            asprintf(error_p, "Neither /dev/misc/rtc nor /dev/rtc exists.");
        }
    } else {
        asprintf(error_p, "The Linux kernel for which this copy "
                 "of hwclock() was "
                 "built is too old to have the rtc device driver\n");
    }
}    



void
see_if_rtc_works(const char specified_filespec[], 
                 char ** const error_p) {
/*----------------------------------------------------------------------------
   Find out if we are capable of accessing the Hardware Clock via the Linux
   rtc device driver via device special file 'specified_filespec'.  

   If the answer is yes, return *error_p == NULL.  Else, return *error_p
   pointing to a null-terminated string explanation of why in newly malloc'ed
   storage.
-----------------------------------------------------------------------------*/
    if (got_rtc) {
        validate_rtc_file(specified_filespec, error_p);
    } else {
        asprintf(error_p, "The Linux kernel for which this copy "
                 "of hwclock() was "
                 "built is too old to have the rtc device driver\n");
    }
}
 



