/**************************************************************************
                                Hwclock
***************************************************************************

  This is a program for reading and setting the Hardware Clock on an ISA
  family computer.  This is the clock that is also known as the RTC,
  real time clock, or, unfortunately, the CMOS clock.

  See man page for details.

  By Bryan Henderson, 96.09.19.  bryanh@giraffe-data.com

  Based on work by others; see history in file HISTORY

**************************************************************************/
/**************************************************************************
  Maintenance notes

  To compile this, you must use GNU compiler optimization (-O option)
  in order to make the "extern inline" functions from asm/io.h (inb(),
  etc.)  compile.  If you don't optimize, which means the compiler
  will generate no inline functions, the references to these functions
  in this program will be compiled as external references.  Since you
  probably won't be linking with any functions by these names, you will
  have unresolved external references when you link.
  
  In the past, this program was designed to run setuid superuser so
  that a non-privileged user could do read-only functions using the
  direct ISA I/O method.  Originally, direct ISA was the only method
  available.  But now we have the rtc device driver and besides, a
  nonprivileged user has very little need to query the hardware clock.
  In those days, the program refused to do setting operations unless
  the real uid was that of the superuser.  Today, this program does
  not check the real uid at all.  If it is setuid superuser, anyone
  who can run the program can read and set the hardware clock and
  system clock and the adjtime file.

 
  Here's some info on how we must deal with the time that elapses while
  this program runs: There are two major delays as we run:

    1) Waiting up to 1 second for a transition of the Hardware Clock so
       we are synchronized to the Hardware Clock.

    2) Running the "date" program to interpret the value of our --date
       option.

  Reading the /etc/adjtime file is the next biggest source of delay and
  uncertainty.

  The user wants to know what time it was at the moment he invoked us,
  not some arbitrary time later.  And in setting the clock, he is
  giving us the time at the moment we are invoked, so if we set the
  clock some time later, we have to add some time to that.

  So we check the system time as soon as we start up, then run "date"
  and do file I/O if necessary, then wait to synchronize with a
  Hardware Clock edge, then check the system time again to see how
  much time we spent.  We immediately read the clock then and (if 
  appropriate) report that time, and additionally, the delay we measured.

  If we're setting the clock to a time given by the user, we wait some
  more so that the total delay is an integral number of seconds, then
  set the Hardware Clock to the time the user requested plus that
  integral number of seconds.  N.B. The Hardware Clock can only be set
  in integral seconds.

  If we're setting the clock to the system clock value, we wait for
  the system clock to reach the top of a second, and then set the
  Hardware Clock to the system clock's value.

  
  About synchronizing to the Hardware Clock when reading the time: The
  precision of the Hardware Clock counters themselves is one second.
  You can't read the counters and find out that is 12:01:02.5.  But if
  you consider the location in time of the counter's ticks as part of
  its value, then its precision is as infinite as time is continuous!
  What I'm saying is this: To find out the _exact_ time in the
  hardware clock, we wait until the next clock tick (the next time the
  second counter changes) and measure how long we had to wait.  We
  then read the value of the clock counters and subtract the wait time
  and we know precisely what time it was when we set out to query the
  time.

  Hwclock uses this method, and considers the Hardware Clock to have
  infinite precision.

  Waiting: When we wait for a certain point in time to arrive, we do
  it with busy waiting.  Preemption by other processes could seriously
  throw off our settings.  You can avoid such preemption by giving the
  Hwclock process a high priority (see sched_set_scheduler() and the
  Rt program).  In that case, you cause the entire system to stop and
  wait for a second or so for the required time to come around.  If we
  changed Hwclock to use nanosleep, we could avoid that with hardly
  any loss of accuracy.

  Definition of century:  In this program, a century is a 100 year 
  period in which all the years' numbers in the Gregorian calendar
  differ only in their last two decimal digits.  E.g. 1900-1999 is
  a century.  The 20th Century (1901-2000), however, is not.


  About the unusual situation of the Jensen variety of Alpha:

  Martin Ostermann writes: 

  The problem with the Jensen is twofold: First, it has the clock at a
  different address. Secondly, it has a distinction beween "local" and
  normal bus addresses. The local ones pertain to the hardware integrated
  into the chipset, like serial/parallel ports and of course, the RTC.
  Those need to be addressed differently. This is handled fine in the kernel,
  and it's not a problem, since this usually gets totally optimized by the
  compile. But the i/o routines of (g)libc lack this support so far.
  The result of this is, that the old clock program worked only on the
  Jensen when USE_DEV_PORT was defined, but not with the normal inb/outb
  functions.
  
****************************************************************************/ 

#define _GNU_SOURCE
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/timex.h>

#include "shhopt.h"
#include "../version.h"  /* Defines UTIL_LINUX, among other things */
#include "util.h"
#include "adjtime.h"
#include "clock_functions.h"
#include "clock_access.h"
#include "directio.h"
#include "adjust.h"
#include "hwclock.h"

/* Note that we must define the boolean type as int because we use the
   shhopt option processing library which, unfortunately, returns flag
   options as integers.  It is customary to define bool as char, but 
   then we would have to do a lot of conversion in order to interface
   with shhopt.
*/

bool debug;
  /* We are running in debug mode, wherein we put a lot of information about
     what we're doing to standard output.  Because of the pervasive and yet
     background nature of this value, this is a global variable.  */



/* We're going to assume that if the CPU is in the Intel x86 family, 
   this is an ISA family machine.  For all practical purposes, this is 
   the case at the time of this writing, especially after we assume a
   Linux kernel is running on it.
   */
const bool isa_machine = 
#ifdef __i386__
TRUE
#else
FALSE;
#endif
;

const bool alpha_machine = 
#ifdef __alpha__
TRUE
#else
FALSE;
#endif
;

struct cmdline_info {
    enum hwclock_operation operation;
    time_t set_time;
    bool utc;
    bool local;
    bool badyear;
    char *rtc;       /* --rtc option value; NULL if none */
    bool directisa;
    bool fast;
    bool debug;
    bool testing;
    bool arc;
    bool jensen;
    bool srm;
    bool funky;
    double correct;
    int epoch;      /* --epoch option; -1 if none */
    bool nodrift;
    bool slew;
};



static void
interpret_date_string(const char *date_opt, time_t * const time_p,
                      char ** const error_p) {
/*----------------------------------------------------------------------------
  Interpret the value of the --date option, which is something like
  "13:05:01".  In fact, it can be any of the myriad ASCII strings that specify
  a time which the "date" program can understand.  The date option value in
  question is our "dateopt" argument.  

  The specified time is in the local time zone.

  Our output, "*time_p", is a seconds-since-epoch time.

  We use the "date" program to interpret the date string.  "date" must be
  runnable by issuing the command "date" to the /bin/sh shell.  That means
  in must be in the current PATH.

  If anything goes wrong (and many things can), we return return code
  10 and arbitrary *time_p.  Otherwise, return code is 0 and *time_p
  is valid.
----------------------------------------------------------------------------*/
    FILE *date_child_fp;
    char date_resp[100];
    const char magic[]="seconds-since-epoch=";
    char date_command[100];  
    int rc;  /* local return code */
  
    if (date_opt == NULL) {
        asprintf(error_p, "No --date option specified.");
    } else if (strchr(date_opt, '"') != NULL) {
        /* Quotation marks in date_opt would ruin the date command we
           construct.  
        */
        asprintf(error_p, "The value of the --date option "
                 "is not a valid date.  "
                 "In particular, it contains quotation marks.");
    } else {
        sprintf(date_command, "date --date=\"%s\" +seconds-since-epoch=%%s", 
                date_opt);
        if (debug) printf("Issuing date command: %s", date_command);
        
        date_child_fp = popen(date_command, "r");
        if (date_child_fp == NULL) {
            asprintf(error_p, "Unable to run 'date' program "
                     "in /bin/sh shell.  popen() failed with errno=%s (%d)", 
                     strerror(errno), errno);
        } else {
            date_resp[0] = '\0';  /* in case fgets fails */
            fgets(date_resp, sizeof(date_resp), date_child_fp);
            if (debug) printf("response from date command = %s\n", date_resp);
            if (strncmp(date_resp, magic, sizeof(magic)-1) != 0) {
                asprintf(error_p, "The date command issued by %s " 
                         "returned unexpected results.  "
                         "The command was:\n  %s\nThe response was:\n  %s", 
                         MYNAME, date_command, date_resp);
            } else {
                int seconds_since_epoch;
                rc = sscanf(date_resp + sizeof(magic)-1, "%d", 
                            &seconds_since_epoch);
                if (rc < 1) {
                    asprintf(error_p, "The date command issued by %s "
                             "returned something other than an integer "
                             "where the converted"
                             "time value was expected."
                             "The command was:\n  %s\n"
                             "The response was:\n %s",
                             MYNAME, date_command, date_resp);
                } else {
                    *error_p = NULL;
                    *time_p = seconds_since_epoch;
                    if (debug) 
                        printf("date string %s equates to "
                               "%d seconds since 1969 UTC.\n",
                               date_opt, (int) *time_p);
                }
            }
            fclose(date_child_fp);
        }
    }
}


static void
parse_command_line(const int argc, char ** const argv, 
                   struct cmdline_info * const cmdline_p) {
    /* option_def is the control table for the option parser.  These other
       variables are the results of parsing the options and their meanings
       are given by the option_def. 
    */
    bool show, set, systohc, hctosys, adjust, getepoch, setepoch, version;
    char *date_opt, *correct_opt;
    
    optStruct option_def[] = {
        { 'r', (char *) "show",      OPT_FLAG,   &show,         0 },
        { 0,   (char *) "set",       OPT_FLAG,   &set,          0 },
        { 'w', (char *) "systohc",   OPT_FLAG,   &systohc,      0 },
        { 's', (char *) "hctosys",   OPT_FLAG,   &hctosys,      0 },
        { 0,   (char *) "getepoch",  OPT_FLAG,   &getepoch,     0 },
        { 0,   (char *) "setepoch",  OPT_FLAG,   &setepoch,     0 },
        { 'a', (char *) "adjust",    OPT_FLAG,   &adjust,       0 },
        { 'v', (char *) "version",   OPT_FLAG,   &version,      0 },
        { 0,   (char *) "date",      OPT_STRING, &date_opt,     0 },
        { 0,   (char *) "correct",   OPT_STRING, &correct_opt,  0 },
        { 0,   (char *) "epoch",     OPT_UINT,   &cmdline_p->epoch,        0 },
        { 'u', (char *) "utc",       OPT_FLAG,   &cmdline_p->utc,          0 },
        { 0,   (char *) "localtime", OPT_FLAG,   &cmdline_p->local,        0 },
        { 0,   (char *) "badyear",   OPT_FLAG,   &cmdline_p->badyear,      0 },
        { 0,   (char *) "rtc",       OPT_STRING, &cmdline_p->rtc,          0 },
        { 0,   (char *) "directisa", OPT_FLAG,   &cmdline_p->directisa,    0 },
        { 0,   (char *) "fast",      OPT_FLAG,   &cmdline_p->fast,         0 },
        { 0,   (char *) "test",      OPT_FLAG,   &cmdline_p->testing,      0 },
        { 'D', (char *) "debug",     OPT_FLAG,   &cmdline_p->debug,        0 },
        { 'A', (char *) "arc",       OPT_FLAG,   &cmdline_p->arc,          0 },
        { 'J', (char *) "jensen",    OPT_FLAG,   &cmdline_p->jensen,       0 },
        { 'S', (char *) "srm",       OPT_FLAG,   &cmdline_p->srm,          0 },
        { 'F', (char *) "funky-toy", OPT_FLAG,   &cmdline_p->funky,        0 },
        { 0,   (char *) "nodrift",   OPT_FLAG,   &cmdline_p->nodrift,      0 },
        { 0,   (char *) "slew",      OPT_FLAG,   &cmdline_p->slew,         0 },
        { 0,   (char *) NULL,        OPT_END,    NULL,                     0 }
    };
    int argc_parse;       /* argc, except we modify it as we parse */
    char **argv_parse;    /* argv, except we modify it as we parse */
    
    /* set option defaults */
    show = set = systohc = hctosys = adjust = getepoch = setepoch = 
        version = cmdline_p->utc = cmdline_p->local = cmdline_p->badyear =
        cmdline_p->directisa = cmdline_p->fast = cmdline_p->testing = 
        cmdline_p->debug = cmdline_p->jensen = cmdline_p->arc =
        cmdline_p->srm = cmdline_p->funky = cmdline_p->nodrift =
        cmdline_p->slew = FALSE;
    date_opt = correct_opt = cmdline_p->rtc = NULL;
    cmdline_p->epoch = -1; 
    
    argc_parse = argc; argv_parse = argv;
    optParseOptions(&argc_parse, argv_parse, option_def, 0);
        /* Uses and sets argc_parse, argv_parse. 
           Sets show, set, systohc, hctosys, adjust, getepoch, setepoch,
           version, *cmdline_p, date_opt, correct_opt,
        */
  
    if (argc_parse - 1 > 0) {
        fprintf(stderr, MYNAME " takes no non-option arguments.  "
                "You supplied %d.  See man page for complete syntax.\n",
                argc_parse - 1);
        exit(100);
    }

    if (show + set + systohc + hctosys + adjust + 
        getepoch + setepoch + version > 1) {
        fprintf(stderr, 
                "You have specified multiple function options to hwclock.\n"
                "You can only perform one function at a time.\n");
        exit(100);
    }

    if (cmdline_p->directisa && cmdline_p->rtc != NULL) {
        fprintf(stderr, "%s: "
                "Cannot specify both --directisa and --rtc options.\n",
                MYNAME);
        exit(100);
    }
    if (show) cmdline_p->operation = OP_SHOW;
    else if (set) cmdline_p->operation = OP_SET;
    else if (systohc) cmdline_p->operation = OP_SYSTOHC;
    else if (hctosys) cmdline_p->operation = OP_HCTOSYS;
    else if (adjust) cmdline_p->operation = OP_ADJUST;
    else if (getepoch) cmdline_p->operation = OP_GETEPOCH;
    else if (setepoch) cmdline_p->operation = OP_SETEPOCH;
    else if (version) cmdline_p->operation = OP_VERSION;
    else cmdline_p->operation = OP_SHOW;  /* default */
    
    if (cmdline_p->operation == OP_SET) {
        char *error;
        interpret_date_string(date_opt, &cmdline_p->set_time, &error);  
            /* (time-consuming) */
        if (error) {
            fprintf(stderr, 
                    "%s: Cannot set clock because "
                    "no usable set-to time given.  %s\n",
                    MYNAME, error);
            freeString(error);
            exit(100);
        }
    }

    if (correct_opt) {
        char * error;
        cmdline_p->correct = strtod(correct_opt, &error);
        if (*error != '\0') {
            fprintf(stderr, 
                    "%s: Invalid number given as --correct value: %s\n",
                    MYNAME, correct_opt);
            exit(100);
        }
    } else 
        cmdline_p->correct = 0.0;
}



static bool
eleven_minute_mode(void) {
/*----------------------------------------------------------------------------
   Return whether the kernel is in "11 minute mode," which is the mode
   where every 11 minutes, the kernel synchronizes the hardware clock to
   the system clock.  See Hwclock user documentation for details on this
   mode.
-----------------------------------------------------------------------------*/
    int const elevenMinuteModeStatus = 0x40;
        /* Mask for the bit in timex.status that tells whether the kernel
           is in 11 minute mode.  Off means 11 minute mode.
        */

    bool retval;
    struct timex timex;
    int rc;

    timex.modes = 0;  /* Just reading; not setting anything */

    rc = adjtimex(&timex);
    if (rc < 0) {
        fprintf(stderr, "Unable to determine if kernel is in 11 minute mode. "
                "adjtimex() returns errno %d (%s).  "
                "Assuming not in 11 minute mode", 
                errno, strerror(errno));
        retval = FALSE;
    } else {
        retval = !(timex.status & elevenMinuteModeStatus);
    }
    if (debug)
        printf("Kernel %s in 11 minute mode\n", retval ? "IS" : "IS NOT");
    return retval;
}



static void
find_default_clock_access_method(
    bool const jensen,
    enum clock_access_method * const clock_access_p,
    char ** const rtc_file_p
    ) {
/*----------------------------------------------------------------------------
  Figure out how we're going to access the hardware clock, by seeing
  what facilities are available, and using compile-time constants,
  assuming the user has not expressed a preference.

  Return our decision as *clock_access_p.  If that is RTC_IOCTL, then
  also return the filespec through which we can issue the rtc ioctls
  as a null-terminated string in newly malloc'ed storage pointed to by
  *rtc_file_p.
-----------------------------------------------------------------------------*/
    char *error;

    find_default_rtc(rtc_file_p, &error); 
    if (!error) 
        *clock_access_p = RTC_IOCTL;
    else {
        if (debug)
            printf("Not selecting rtc method because:  %s\n", error);
        freeString(error);

        {
            bool kdghwclk_works;
            /* The KDHWCLK method is available and seems to work on 
               this machine. */

            see_if_kdghwclk_works(&kdghwclk_works);  
                /* May issue error messages */

            if (debug)
                if (geteuid() != 0)
                    printf("Cannot select direct ISA method because "
                           "effective uid is not zero\n");

            if (kdghwclk_works) *clock_access_p = KD;
            else if (isa_machine && geteuid() == 0) *clock_access_p = ISA;
            else if (jensen) *clock_access_p = DEV_PORT;
            else if (alpha_machine && geteuid() == 0) *clock_access_p = ISA;
            else *clock_access_p = NOCLOCK;
        }
    }
}



static void
determine_clock_access_method(char * const rtc_filespec,
                              const bool user_requests_isa, 
                              const bool user_says_jensen,
                              enum clock_access_method * const clock_access_p,
                              char ** const rtc_file_p) {
/*----------------------------------------------------------------------------
  Figure out how we're going to access the hardware clock, by seeing
  what facilities are available, looking at invocation options, and 
  using compile-time constants.

  'user_requests_ISA' means the user explicitly asked for the ISA method,
  so we'll use that (even if we know it will fail because the machine
  is incapable!).
  
  rtc_filespec[] is the file specification specified by the user for
  the device special file through which to access the Linux rtc device
  driver.  Null means the user did not specify.  Non-null means the user
  doesn't want to consider any other means to access the clock.
-----------------------------------------------------------------------------*/
    if (rtc_filespec) {
        char *error;
        if (debug)
            printf("User requested rtc device driver clock access method\n");
        see_if_rtc_works(rtc_filespec, &error);
        if (error) {
            fprintf(stderr, "Cannot use rtc device special file '%s'.\n%s\n",
                    rtc_filespec, error);
            freeString(error);
            *clock_access_p = NOCLOCK;
        } else {
            *clock_access_p = RTC_IOCTL;
            *rtc_file_p = strdup(rtc_filespec);
        }
    } else if (user_requests_isa) {
        if (debug)
            printf("User requested direct ISA clock access method.\n");
        if (geteuid() == 0)
            *clock_access_p = ISA;
        else {
            *clock_access_p = NOCLOCK;
            fprintf(stderr, "Cannot use direct ISA I/O to access the clock "
                "because this program is not running with superuser "
                "effective uid.  It must either be run by the superuser or " 
                "be installed setuid superuser in order for --directisa to "
                "work.\n");
        }
    } else {
        const bool jensen = 
            user_says_jensen || 
            (alpha_machine && is_in_cpuinfo("system type", "Jensen"));
        /* See comments at top of program for how Jensen 
           is a special case. */

        if (debug)
            printf("User did not specify a clock access method.  Searching "
                   "for one...\n");
        find_default_clock_access_method(jensen, clock_access_p, rtc_file_p);
    }
    if (debug) {
        switch (*clock_access_p) {
        case ISA: 
            printf("Using direct I/O instructions to ISA clock.\n"); break;
        case KD: printf("Using KDGHWCLK interface to m68k clock.\n"); break;
        case RTC_IOCTL: 
            printf("Using rtc ioctl interface to clock via file '%s'.\n",
                   *rtc_file_p); 
            break;
        case DEV_PORT: printf("Using /dev/port interface to clock.\n"); break;
        case NOCLOCK: 
            printf("Unable to find a usable clock access method.\n"); break;
        default:  
            printf("determine_clock_access_method() returned "
                   "invalid value: %d.\n",
                   *clock_access_p);
        }
    }
}



static void
report_version(void) {

  char *additional_version;   /* malloc'ed */
    /* Stuff to add on to the version report, after the basic version. 
       If this is hwclock packaged with util-linux, this is the 
       util-linux version.  Otherwise, it's nothing.
       */

#ifdef UTIL_LINUX
  additional_version = malloc(strlen(util_linux_version) + 5);
  sprintf(additional_version, "/%s", util_linux_version);
#else
  additional_version = strdup("");
#endif
  printf(MYNAME " " VERSION "%s\n", additional_version);
  free(additional_version);
}



static void
warnElevenMinuteMode(enum hwclock_operation const operation) {

    if (operation == OP_SET ||
        operation == OP_SYSTOHC ||
        operation == OP_ADJUST)
        fprintf(stderr, "WARNING from %s: The clock was in 11 minute mode, "
                "which normally means that something else, possibly "
                "an NTP daemon, has been setting the hardware clock.  "
                "Setting the clock with %s turns off 11 minute "
                "mode.  See %s documentation.\n", MYNAME, MYNAME, MYNAME);
}



static void
manipulate_epoch(const enum hwclock_operation operation, 
                 const int epoch_opt, const bool testing) {
/*----------------------------------------------------------------------------
   Get or set the Hardware Clock epoch value in the kernel, as appropriate.
   'operation' is either OP_SETEPOCH or OP_GETEPOCH.

   'epoch' is the value of the --epoch option, or -1 if the user did
   not specify it.
-----------------------------------------------------------------------------*/
    /* Maintenance note: This should work on non-Alpha machines, but
      the evidence today (98.03.04) indicates that the kernel keeps
      the epoch value only on Alphas.  If that is ever fixed, this
      function should be changed.  
    */

    if (!alpha_machine)
        fprintf(stderr, 
                "%s: The kernel keeps an epoch value for the Hardware Clock "
                "only on an Alpha machine.\n"
                "This copy of hwclock was built for "
                "a machine other than Alpha\n"
                "(and thus is presumably not running "
                "on an Alpha now).  No action taken.\n", MYNAME);
    else {
        switch (operation) {
        case OP_GETEPOCH: {
            unsigned long epoch;
            char *reason;  /* malloc'ed */
            
            get_epoch("/dev/rtc", &epoch, &reason);
            if (reason != NULL) {
                printf("Unable to get the epoch value from the kernel.  %s\n",
                       reason);
                freeString(reason);
            } else 
                printf("Kernel is assuming an epoch value of %lu\n", epoch);
        }
        break;
        case OP_SETEPOCH: 
            if (epoch_opt == -1)
              fprintf(stderr, 
                      "%s: To set the epoch value, you must use the 'epoch' "
                      "option to tell to what value to set it.\n", MYNAME);
            else {
                int rc;
                set_epoch("dev/rtc", epoch_opt, testing, &rc);
              if (rc != 0)
                  printf("Unable to set the epoch value in the kernel.\n");
            }
            break;
        default:
            fprintf(stderr, "%s: Internal error.  Impossible value for "
                    "'operation' in manipulate_epoch()", MYNAME);
            exit(150);
        }
    }
}



int 
main(int argc, char **argv, char **envp) {
/*----------------------------------------------------------------------------
                                   MAIN
-----------------------------------------------------------------------------*/
    struct cmdline_info cmdline;
    /* All the information the user specified on the command line */
    struct timeval startup_time;
    /* The time we started up, in seconds since the epoch, including fractions.
       */

    enum clock_access_method clock_access;
    /* The method that we determine is best for accessing Hardware Clock 
       on this system. 
       */
    char *rtc_filespec;
    /* The filespec of the device special file through which we will access
       the Linux rtc device driver.  Undefined if we are not using the 
       driver.  Typically "dev/rtc".
       */

    bool nodrift;    /* We are assuming hw clock has not been just drifting */
    int retcode;   /* Our eventual return code */

    /* Remember what time we were invoked */
    gettimeofday(&startup_time, NULL);  

    assume_interrupts_enabled();  /* Since we haven't messed with them yet */

    parse_command_line(argc, argv, &cmdline);

    debug = cmdline.debug;

    if (cmdline.jensen && !alpha_machine) {
        fprintf(stderr, 
                "%s: Your options indicate that this is a Jensen model of "
                "DEC Alpha, but this is not an Alpha machine!\n", MYNAME);
        exit(100);
    }

    if (cmdline.srm && !alpha_machine) {
        fprintf(stderr, 
                "%s: Your options indicate that this machine keeps SRM "
                "console time, but only DEC Alphas have such a clock and "
                "this is not an Alpha!\n", MYNAME);
        exit(100);
    }
    if (cmdline.arc && !alpha_machine) {
        fprintf(stderr, "%s: Your options indicate that this machine's clock"
                "keeps ARC console time, "
                "but only DEC Alphas have such a clock and this is "
                "not an Alpha!\n", MYNAME);
        exit(100);
    }

    if (cmdline.directisa && !(isa_machine || alpha_machine)) {
        fprintf(stderr, 
                "%s: You have requested direct access to the ISA Hardware "
                "Clock using machine instructions from the user process.  "
                "But this method only works on an ISA machine with an x86 "
                "CPU, or a similar machine such as DEC Alpha.  "
                "This is not one.\n", MYNAME);
        exit(100);
    }

    if (cmdline.utc && cmdline.local) {
        fprintf(stderr, "%s: The --utc and --localtime options are mutually "
                "exclusive.  You specified both.\n", MYNAME);
        exit(100);
    }

    if (debug) report_version();
  
    if (eleven_minute_mode()) {
        nodrift = TRUE;
        warnElevenMinuteMode(cmdline.operation);
    } else if (cmdline.nodrift)
        nodrift = TRUE;
    else
        nodrift = FALSE;
    
    tzset(); /* init timezone, daylight from TZ or ...zoneinfo/localtime */
    /* An undocumented function of tzset() is to set global variables
       'timezone' and 'daylight'.  Those aren't actually useful, though.
       'daylight' just means you have to adjust 'timezone' for Daylight
       Saving Time, but you don't know how - whether DST is in effect and
       how many minutes of offset it implies if so.

       We do tzset() here, even though C library functions that need
       it will do it implicitly, because this may involve file I/O and
       take a long time, so we don't want to do it where the delay
       could mess up times.  
    */
    switch (cmdline.operation) {
    case OP_VERSION:
        report_version();
        retcode = 0;
        break;
    case OP_GETEPOCH:
    case OP_SETEPOCH:
        manipulate_epoch(cmdline.operation, cmdline.epoch, 
                         cmdline.testing);
        retcode = 0;
    default:
        determine_clock_access_method(cmdline.rtc, cmdline.directisa, 
                                      cmdline.jensen, &clock_access,
                                      &rtc_filespec);
        if (clock_access == NOCLOCK) {
            fprintf(stderr, 
                    "%s: Cannot access the Hardware Clock via any known "
                    "method.\n%s", MYNAME, debug ? "" :
                    "Use --debug option to see the details of "
                    "our search for an access method.\n");
            retcode = 10;
        } else
            manipulate_clock(cmdline.operation, cmdline.set_time, 
                             startup_time, clock_access, rtc_filespec, 
                             cmdline.utc, cmdline.local, cmdline.epoch,
                             cmdline.badyear, 
                             cmdline.arc, cmdline.srm, cmdline.funky,
                             cmdline.correct, cmdline.testing, 
                             cmdline.fast, nodrift, cmdline.slew,
                             &retcode);
    }
    exit(retcode);
}
