#define _GNU_SOURCE
#include <sys/errno.h>
#include <sys/time.h>
#include <sys/timex.h>
#include <limits.h>

#include "hwclock.h"
#include "adjtime.h"
#include "adjust.h"
#include "directio.h"
#include "clock_functions.h"

/* The following are times, in unix standard format (seconds since 1969 UTC) */
#define START_OF_1994 757411200
#define END_OF_1995 820396800

static bool
hw_clock_is_utc(bool           const utc, 
                bool           const local_opt, 
                struct adjtime const adjtime) {
/*----------------------------------------------------------------------------
   Return true iff the hardware clock keeps Coordinated Universal Time
   rather than local time.

   'utc' means the user told us in the invocation options that the
   hardware clock is kept in UTC.
-----------------------------------------------------------------------------*/
  
    bool retval;  /* our return value */

    if (utc) retval = TRUE;
    else if (local_opt) retval = FALSE;
    else retval = (adjtime.local_utc == UTC);
    if (debug) printf("Assuming hardware clock is kept in %s time.\n",
                      retval ? "UTC" : "LOCAL");
    return retval;
}



static void
display_time(const bool hclock_valid, const time_t systime, 
             const float sync_duration, const bool badyear_warn) {
/*----------------------------------------------------------------------------
  Put the time 'systime' on standard output in display format.
  Except if hclock_valid == false, just tell standard output that we don't
  know what time it is.

  Include in the output the adjustment 'sync_duration'.

  If the year is 1994 or 1995 and 'badyear_warn' is true, warn the
  user that he has a brain-damaged clock and needs to use --badyear.
  Since we didn't exist in 1994 and 1995, we know the clock isn't
  correct.

-----------------------------------------------------------------------------*/
  if (!hclock_valid)
    fprintf(stderr, "%s: The Hardware Clock registers contain values that are "
            "either invalid (e.g. 50th day of month) or beyond the range "
            "we can handle (e.g. Year 2095).\n", MYNAME);
  else {
    struct tm *broken_timeP;
    if (badyear_warn && (systime > START_OF_1994 && systime < END_OF_1995)) {
      printf("WARNING:  The Hardware Clock shows a time in 1994 "
             "or 1995.  This probably means you have a Hardware Clock "
             "that is incapable of tracking years after 1999, and you "
             "must use the --badyear option to make hwclock work for "
             "you.  See hwclock documentation for details.\n");
    }
    broken_timeP = localtime(&systime);  /* Sets 'tzname[]' */
    printf("%s  %.6f seconds %s\n", 
           ctime2(systime), -(sync_duration), 
           broken_timeP->tm_isdst ? tzname[1] : tzname[0]);
  }
}



static void
start_system_clock_slew(struct timeval const newtime, 
                        struct timeval const reference_time,
                        int *          const rc_p) {
    
    long const maxAdjtimexOffset = LONG_MAX / 1E6;
        /* The maximum amount of whole seconds of offset that one can
           pass to adjtimex().  
        */

    struct timeval nowtime;
    int rc;

    rc = gettimeofday(&nowtime, NULL);
    if (rc != 0) {
        fprintf(stderr, "gettimeofday() failed, errno=%d (%s)\n",
                errno, strerror(errno));
        *rc_p = 5;
    } else {
        struct timeval const settotime =
            time_inc(newtime, time_diff(nowtime, reference_time));

        if (abs(settotime.tv_sec - nowtime.tv_sec) + 1 > maxAdjtimexOffset) {
            fprintf(stderr, 
                    "%s: System time is too far off to slew.  It is "
                    "about %d seconds wrong, max slew offset is %ld",
                    MYNAME, abs(settotime.tv_sec - nowtime.tv_sec) + 1,
                    maxAdjtimexOffset);
            *rc_p = 1;
        } else {
            struct timex adjtimex_arg;
            int rc;

            adjtimex_arg.modes = ADJ_OFFSET_SINGLESHOT;
            adjtimex_arg.offset = (settotime.tv_sec - nowtime.tv_sec) * 1E6 +
                (settotime.tv_usec - nowtime.tv_usec);

            if (debug)
                printf("  Slewing System Time by %06ld microseconds\n",
                       adjtimex_arg.offset);

            rc = adjtimex(&adjtimex_arg);
            if (rc < 0) {
                if (errno == EPERM)
                    fprintf(stderr, 
                            "%s: Slewing of system clock failed because "
                            "you are not superuser.\n", 
                            MYNAME);
                else
                    fprintf(stderr,
                            "%s: adjtimex() failed, errno=%s (%d)\n", 
                            MYNAME, strerror(errno), errno);
                *rc_p = 1;
            } else
            *rc_p = 0;
        }
    }
} 



static void
step_system_clock(struct timeval const newtime,
                  struct timeval const reference_time,
                  int *          const rc_p) {

    /* For documentation of settimeofday() see, in addition to its man page,
       kernel/time.c in the Linux source code.  
    */
    struct timezone tz;
    struct timeval nowtime;
    struct timeval settotime;
    int rc;  /* local return code */

    /* put daylight saving time offset in minuteswest rather than dsttime,
       since the latter is mostly ignored ... 
    */
    tz.tz_minuteswest = local_time_offset(settotime.tv_sec)/60;
    tz.tz_dsttime = 0;

    rc = gettimeofday(&nowtime, NULL);
    settotime = time_inc(newtime, time_diff(nowtime, reference_time));
    if (debug)
        printf("  Setting System Time to %ld.%06ld, "
               "local time offset to %d minutes\n",
               (long) settotime.tv_sec, (long) settotime.tv_usec,
               tz.tz_minuteswest);
    rc = settimeofday(&settotime, &tz);
    if (rc != 0) {
        if (errno == EPERM)
            fprintf(stderr, "%s: Setting of system clock failed because.\n"
                    "you are not superuser.\n", 
                    MYNAME);
        else
            fprintf(stderr,
                    "%s: settimeofday() failed, errno=%s (%d)\n", 
                    MYNAME, strerror(errno), errno);
        *rc_p = 1;
    } else {
        update_utmp(nowtime, settotime);
        *rc_p = 0;
    }
}



static void
set_system_clock(bool           const hclock_valid, 
                 struct timeval const newtime, 
                 struct timeval const reference_time, 
                 bool           const testing, 
                 bool           const slew, 
                 int *          const rc_p) {
/*----------------------------------------------------------------------------
   Set the System Time to time 'newtime' plus the elapsed time since
   time 'reference_time', according to the current System Time.

   Also set the kernel time zone value to the value indicated by the 
   TZ environment variable and/or /usr/lib/zoneinfo/, interpreted as
   tzset() would interpret them.  Except: do not consider Daylight
   Savings Time to be a separate component of the time zone.  Include
   any effect of DST in the basic timezone value and set the kernel
   DST value to 0.

   And record the time setting in utmp/wtmp.

   If 'testing' is true, don't actually update anything -- just say we 
   would have.
-----------------------------------------------------------------------------*/
    if (debug) {
        printf("Setting system clock.\n");
            /* Note: tv_sec and tv_usec are declared variously on different
               systems: int, long, time_t.  Casting to long below makes it 
               compile everywhere.
            */
            printf("  newtime = %ld.%06ld reference_time=%ld.%06ld\n",
                   (long) newtime.tv_sec, (long) newtime.tv_usec,
                   (long) reference_time.tv_sec, 
                   (long) reference_time.tv_usec);
    }
    
    if (testing) {
        printf("Not setting system clock because running in test mode.\n");
        *rc_p = 0;
    } else {
        if (slew) {
            if (debug)
                printf("  Slewing system clock\n");
            start_system_clock_slew(newtime, reference_time, rc_p);
        } else {
            if (debug)
                printf("  Stepping system clock\n");
            step_system_clock(newtime, reference_time, rc_p);
        }
    }
}



static void
do_hctosys(bool           const hclock_valid, 
           time_t         const hclocktime, 
           struct timeval const read_time, 
           struct adjtime const adjtime,
           bool           const nodrift,
           bool           const testing,
       bool       const slew,
           int *          const rc_p) {
/*----------------------------------------------------------------------------
   Perform the "--hctosys" function: set the System Time to the time
   we read from the Hardware Clock ('hclocktime'), adjusted for
   accumulated drift of the Hardware Clock, as of time 'read_time',
   which is in the past.  If 'nodrift' is true, assume there is no 
   accumulated drift.

   EXCEPT: if hclock_valid is false, just issue an error message
   saying there is no valid time in the Hardware Clock to which to set
   the system time.

   Also set the kernel time zone value to the value indicated by the 
   TZ environment variable and/or /usr/lib/zoneinfo/, interpreted as
   tzset() would interpret them.  Except: do not consider Daylight
   Savings Time to be a separate component of the time zone.  Include
   any effect of DST in the basic timezone value and set the kernel
   DST value to 0.

   And write a record into utmp and wtmp indicating the setting of
   the system time.

   If 'testing' is true, don't actually update anything -- just say we 
   would have.
-----------------------------------------------------------------------------*/

  if (!hclock_valid) {
    fprintf(stderr, "%s: The Hardware Clock does not contain a valid time, so "
            "we cannot set the System Time from it.\n", MYNAME);
    *rc_p = 1;
  } else {
    float accumulated_drift;  
      /* The amount we expect the hardware clock to have drifted since the
         last time it was adjusted for drift or calibrated
      */

    calculate_adjustment(adjtime, t2tv(hclocktime), 
                         NULL, NULL, &accumulated_drift, nodrift, debug);
    set_system_clock(hclock_valid, 
                     time_inc(t2tv(hclocktime), accumulated_drift), 
                     read_time, testing, slew, rc_p);
    *rc_p = 0;
  }
}



static void
do_adjustment(struct adjtime *        const adjtime_p,
              bool                    const hclock_valid, 
              time_t                  const hclocktime, 
              struct timeval          const read_time,
              struct hwclock_config * const hwclock_config_p,
              double                  const correct,
              bool                    const nodrift,
              bool                    const testing) {
/*---------------------------------------------------------------------------
  Do the adjustment for drift requested, by 1) setting the Hardware
  Clock (if necessary), and 2) updating the last-adjusted time in the
  adjtime structure.

  Do not update anything if the Hardware Clock does not currently present
  a valid time.

  Do not update anything if 'nodrift' is true, meaning that simple drift
  has not been occurring.

  arguments 'factor' and 'last_time' are current values from the adjtime
  file.

  'hclock_valid' means the Hardware Clock contains a valid time, and that
  time is 'hclocktime'.

  'read_time' is the current system time (to be precise, it is the system 
  time at the time 'hclocktime' was read, which due to computational delay
  could be a short time ago).

  *hwclock_config_p is the description of the Hardware Clock that tells
  us how to set it.  We update it if we change the clock configuration in the
  process of setting it (e.g. the timezone offset).

  'testing':  We are running in test mode (no updating of clock).

  We do not bother to update the clock if the adjustment would be less than
  one second.  This is to avoid cumulative error and needless CPU hogging
  (remember we use an infinite loop for some timing) if the user runs us
  frequently.

----------------------------------------------------------------------------*/
    if (!hclock_valid) {
        fprintf(stderr, "%s: The Hardware Clock does not contain a valid "
                "time, so we cannot adjust it.\n", MYNAME);
        /* Any previous calibration had to be before the clock got hosed, so
           wipe out the record of it so it won't be used in the future.
        */
        adjtime_p->last_calib_time = 0;  
        adjtime_p->last_adj_time = 0;
        adjtime_p->last_adj_frac = 0;
        adjtime_p->not_adjusted = 0;
        adjtime_p->dirty = TRUE;
    } else if (nodrift) {
        /* Clock has not been simply drifting, so no adjustment for the time
           between previous adjustment and now makes sense either now or in 
           the future.
        */
        if (debug)
            printf("Due to nodrift, resetting drift interval to start now "
                   "(%u), but not adjusting clock.\n", (unsigned)hclocktime);
        adjtime_p->last_adj_time = hclocktime;
        adjtime_p->last_adj_frac = 0;
        adjtime_p->not_adjusted = 0;
        adjtime_p->dirty = TRUE;
    } else if (adjtime_p->last_adj_time == 0) {
        if (debug)
            printf(
                "Not adjusting clock because we have no information about \n"
                "the previous calibration (i.e. the adjtime file is \n"
                "nonexistent or contains zero in the last calibrated time \n"
                "field).\n");
    } else {
        float adjustment; 
        time_t i_adjustment;
        /* Number of seconds we must advance the Hardware Clock */

        calculate_adjustment(*adjtime_p, t2tv(hclocktime),
                             NULL, NULL, &adjustment, nodrift,
                             debug );
        if (adjustment > 1.0 || adjustment < -1.0) {
            set_hardware_clock_exact(time_inc(t2tv(hclocktime), 
                                              adjustment+correct), 
                                     read_time, hwclock_config_p, testing);
            i_adjustment = adjustment;
            adjtime_p->last_adj_time = hclocktime + i_adjustment;
            adjtime_p->last_adj_frac = (adjustment - (float) i_adjustment);
            adjtime_p->not_adjusted = 0;
            adjtime_p->dirty = TRUE;
        } else 
            if (debug) 
                printf("Needed adjustment is less than one second, "
                       "so not setting clock.\n");
    }
}



static void
set_tz_offset_in_adjtime(struct adjtime * const adjtime_p,
                         struct hwclock_config const hwclock_config) {

    int tzOffset;

    if (hwclock_config.universal)
        tzOffset = 0;
    else
        tzOffset = hwclock_config.tzOffset;

    if (debug)
        printf("new timezone offset for hardware clock is %d\n", tzOffset);

    if (tzOffset != adjtime_p->tzOffset) {
        adjtime_p->tzOffset = tzOffset;
        adjtime_p->dirty = TRUE;
    }
}



static void
do_set(time_t                const set_time, 
       double                const correct, 
       struct timeval          const startup_time,
       struct hwclock_config * const hwclock_config_p, 
       bool                    const hclock_valid,
       time_t                  const hclocktime,
       struct timeval          const read_time,
       bool                    const testing, 
       bool                    const fast,
       bool                    const nodrift,
       struct adjtime *        const adjtime_p) {
/*----------------------------------------------------------------------------
   Perform the --set function.
-----------------------------------------------------------------------------*/
    const struct timeval set_to_time = time_inc(t2tv(set_time), correct);

    if (fast)
        set_hardware_clock(hwclock_config_p, set_time, testing);
    else {
        set_hardware_clock_exact(set_to_time, startup_time, hwclock_config_p, 
                                 testing);
        adjust_drift_factor(adjtime_p, set_time, hclock_valid, 
                            time_inc(t2tv(hclocktime), - 
                                     time_diff(read_time, startup_time)),
                            nodrift);
    }
}



static void
do_systohc(struct hwclock_config * const hwclock_config_p, 
           double                  const correct,
           time_t                  const hclocktime, 
           bool                    const hclock_valid,
           struct timeval          const read_time,
           bool                    const fast, 
           bool                    const nodrift,
           bool                    const testing,
           struct adjtime *        const adjtime_p) {
/*----------------------------------------------------------------------------
   Perform the specifics of the hwclock --systohc function.
-----------------------------------------------------------------------------*/
    if (fast)
        set_hardware_clock(hwclock_config_p, time(NULL), testing);
    else {
        struct timeval nowtime;
  
        gettimeofday(&nowtime, NULL);
          
        set_hardware_clock_exact(time_inc(nowtime, correct), nowtime,
                                 hwclock_config_p, testing);
        adjust_drift_factor(adjtime_p, (time_t) nowtime.tv_sec, hclock_valid, 
                            time_inc(t2tv(hclocktime), 
                                     time_diff(nowtime, read_time)),
                            nodrift);
    }
}


static void
perform_function(enum hwclock_operation   const operation,
                 struct hwclock_config  * const hwclock_config_p, 
                 struct adjtime *         const adjtime_p,
                 struct timeval           const read_time,
                 struct timeval           const startup_time,
                 time_t                   const set_time, 
                 double                   const correct,
                 bool                     const testing, 
                 bool                     const fast, 
                 bool                     const nodrift,
                 bool                     const slew,
                 int *                    const retcode_p)
{
    bool hclock_valid;
        /* The Hardware Clock gives us a valid time, or at
           least something close enough to fool mktime().  
        */
    
    time_t hclocktime;
        /* The time the hardware clock had just after we synchronized to
           its next clock tick when we started up.  Defined only if
           hclock_valid is true.  
        */

    read_hardware_clock(*hwclock_config_p, 
                        adjtime_p->last_calib_time,
                        &hclock_valid, &hclocktime); 

    switch (operation) {
    case OP_SHOW:
        display_time(hclock_valid, hclocktime, 
                     time_diff(read_time, startup_time), 
                     !hwclock_config_p->badyear);
        *retcode_p = 0;
        break;
    case OP_SET:
        do_set(set_time, correct, startup_time, 
               hwclock_config_p, hclock_valid, 
               hclocktime,  read_time, testing, fast, nodrift, adjtime_p);
        *retcode_p = 0;
        break;
    case OP_ADJUST:
        do_adjustment(adjtime_p, hclock_valid, hclocktime, read_time, 
                      hwclock_config_p, correct, nodrift, testing);
        *retcode_p = 0;
        break;
    case OP_SYSTOHC: {
        do_systohc(hwclock_config_p, correct,
                   hclocktime, hclock_valid, read_time, 
                   fast, nodrift, testing, adjtime_p);
        *retcode_p = 0;
        }   
        break;
    case OP_HCTOSYS: {
        int rc;
        do_hctosys(hclock_valid, hclocktime, read_time, 
                   *adjtime_p, nodrift, testing, slew, &rc);
        if (rc != 0) {
            printf("Unable to set system clock.\n");
            *retcode_p = 1;
        } else *retcode_p = 0;
        }
        break;
    default:
        fprintf(stderr, "Internal error: impossible value for operation "
                "in perform_function()");
        *retcode_p = 100;
    }
    set_tz_offset_in_adjtime(adjtime_p, *hwclock_config_p);
}



static void
update_adjtime_to_hwclock_config(struct adjtime * const adjtimeP,
                                 struct hwclock_config const hwclock_config,
                                 enum hwclock_operation const operation) {
/*----------------------------------------------------------------------------
   Update the adjtime file buffer *adjtimeP to reflect the hardware clock
   configuration described by 'hwclock_config', if appropriate.
-----------------------------------------------------------------------------*/
    if ((operation == OP_SET || operation == OP_SYSTOHC || 
         operation == OP_ADJUST)) {
        if ((adjtimeP->local_utc == UTC) != 
            hwclock_config.universal) {
            adjtimeP->local_utc = 
                hwclock_config.universal ? UTC : LOCAL;
            adjtimeP->dirty = TRUE;
        }
        if (adjtimeP->epoch != hwclock_config.hc_zero_year) {
            adjtimeP->epoch = hwclock_config.hc_zero_year;
            adjtimeP->dirty = TRUE;
        }
    }
}



void
manipulate_clock(enum hwclock_operation   const operation,
                 time_t                   const set_time,
                 struct timeval           const startup_time, 
                 enum clock_access_method const clock_access,
                 char *                   const rtc_filespec,
                 bool                     const utc, 
                 bool                     const local_opt, 
                 int                      const epoch_opt,
                 bool                     const badyear, 
                 bool                     const arc_opt, 
                 bool                     const srm_opt,
                 bool                     const user_wants_uf, 
                 double                   const correct,
                 bool                     const testing, 
                 bool                     const fast, 
                 bool                     const nodrift,
                 bool                     const slew,
                 int *                    const retcode_p
    ) {
/*---------------------------------------------------------------------------
  Do all the normal work of hwclock - read, set clock, etc.

  Issue output to stdout and error message to stderr where appropriate.

  Return rc == 0 if everything went OK, rc != 0 if not.

  'correct' is the correction (in seconds) user said to add to the
  time when we set the hardware clock to correct for setting
  inaccuracy.
----------------------------------------------------------------------------*/
    int dev_port;
    const char *error; 

    if (epoch_opt != -1 && clock_access != ISA) 
        printf("WARNING: ignoring --epoch option because the clock access "
               "method is not direct ISA.  "
               "Consider the --directisa option.\n");

    get_clock_access(clock_access, &dev_port, &error);

    if (error) {
        fprintf(stderr, "%s: error\n", MYNAME);
        *retcode_p = 1;
        freeString(error);
    } else {
        struct hwclock_config hwclock_config;
        const char * error;
        struct adjtime adjtime;
            /* Contents of the adjtime file, or what they should be. */

        hwclock_config.clock_access = clock_access;
        hwclock_config.rtc_filespec = rtc_filespec;
        hwclock_config.use_uf_bit   = uf_bit_needed(user_wants_uf);
        hwclock_config.badyear      = badyear;
        hwclock_config.dev_port     = dev_port;
        
        read_adjtime(&adjtime, &error, debug);
        if (error) {
            fprintf(stderr, "%s: Unable to read/interpret adjtime file.  "
                    "%s\n", MYNAME, error);
            *retcode_p = 2;
            freeString(error);
        } else {
            struct timeval read_time; 
            /* The time at which we read the Hardware Clock */

            hwclock_config.hc_zero_year = 
                zero_year(epoch_opt, arc_opt, srm_opt, adjtime.epoch);
            hwclock_config.universal = 
                hw_clock_is_utc(utc, local_opt, adjtime);
            update_adjtime_to_hwclock_config(&adjtime, hwclock_config,
                                             operation);
            hwclock_config.tzOffset = adjtime.tzOffset;
            if (fast) {
                *retcode_p = 0;
                gettimeofday(&read_time, NULL);
            } else           
                synchronize_to_clock_tick(hwclock_config, 
                                          &read_time, retcode_p);  
            /* this takes up to 1 second */
            if (*retcode_p == 0) 
                perform_function(operation, &hwclock_config, &adjtime,
                                 read_time, startup_time, set_time,
                                 correct, testing, fast, nodrift, slew,
                                 retcode_p);
            else 
                fprintf(stderr, "Unable to synchronize to a clock tick, "
                        "which means we cannot get better than one second "
                        "precision from the hardware clock.  If this is "
                        "OK with you, run with the --fast option.\n");
                
            save_adjtime(adjtime, testing, debug);
        }
        if (clock_access == DEV_PORT)
            release_clock_access(clock_access, dev_port);
    }
}
