#ifndef CLOCK_ACCESS_H
#define CLOCK_ACCESS_H
#include <time.h>

#include "util.h"

enum clock_access_method {ISA, RTC_IOCTL, KD, DEV_PORT, NOCLOCK};
  /* A method for accessing (reading, writing) the hardware clock:
     
     ISA: 
       via direct CPU I/O instructions that work on an ISA family
       machine (IBM PC compatible) or most DEC Alphas, which implement
       enough of ISA to get by.

     RTC_IOCTL: 
       via the rtc device driver, using device special file /dev/rtc.

     KD:
       via the console driver, using device special file /dev/tty1.
       This is the m68k ioctl interface, known as KDGHWCLK.

     DEV_PORT:
       via the /dev/port device, which is almost the same thing as 
       direct I/O instructions via the ISA method, but works on a Jensen 
       model of Alpha, whereas ISA does not.  Also, use I/O addresses
       0x170 and 0x171 instead of the ISA 0x70 and 0x71.

     NO_CLOCK:
       Unable to determine a usable access method for the system clock.
   */
       


struct hwclock_config {
    /* This is information about the Hardware Clock on this system --
       how to access it and use it, how to interpret its value, etc.
       */
    enum clock_access_method clock_access;
      /* The basic method we use to access the clock */
    char * rtc_filespec;
      /* The file specification of the device special file through which
         we access the Linux rtc device driver.  Meaningless if 
         'clock_access' is not RTC_IOCTL.
      */
    bool universal;
      /* clock is to be kept in absolute time (to wit, Coordinated Universal
         time), as opposed to some local time.
      */
    int tzOffset;
      /* If Hardware Clock is in local time format, this is the 
         number of minutes west of Prime Meridian the timezone its value is
         based on is.  If it is in absolute time, this is zero.
      */
    bool use_uf_bit;
      /* we must synchronize to a clock tick using the clock's UF bit
         instead of the more normal UIP bit.
      */
    int hc_zero_year;
      /* The Gregorian year number to which a value of zero corresponds in the 
         Hardware Clock's year register.
         */
    bool badyear;
      /* clock has the Y2K bug, so we can't trust the year value at all. */
    int dev_port;
      /* File descriptor for /dev/port, if we're using it.  -1 if we
         couldn't open it.  0 if we aren't using it.
      */

};

void
synchronize_to_clock_tick(const struct hwclock_config hwclock_config,
                          struct timeval *tick_time_p, int *retcode_p);

void
read_hardware_clock(const struct hwclock_config hwclock_config,
                    const time_t last_known_time,
                    bool *valid_p, time_t *systime_p);

void
set_hardware_clock(struct hwclock_config * const hwclock_config_p,
                   time_t                  const newtime, 
                   bool                    const testing);

void
set_hardware_clock_exact(struct timeval          const settime, 
                         struct timeval          const ref_time,
                         struct hwclock_config * const hwclock_config_p,
                         bool                    const testing);
void
get_clock_access(enum clock_access_method const clock_access,
                 int *                    const dev_port_p,
                 const char **            const errorP);

void
release_clock_access(enum clock_access_method const clock_access,
                     int                      const dev_port);

#endif
