#define _GNU_SOURCE

#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include "clock_access.h"
#include "directio.h"
#include "hwclock.h"

/* EARLIER(A,B) means timeval A is before timeval B */
#define EARLIER(A,B) \
      ((A).tv_sec < (B).tv_sec \
          || ((A).tv_sec == (B).tv_sec && (A).tv_usec < (B).tv_usec))


static struct tm
make_within_one_year(const struct tm base_tm, const time_t last_known_time) {
/*----------------------------------------------------------------------------
  Compute a time that is the same as the input base_tm, except for a
  different year.  The year shall be whatever year it takes to make the
  output time within one year after last_known_time.

  The timezone for both the input and output values is the value of
  the TZ environment variable.
-----------------------------------------------------------------------------*/
  struct tm broken_last_known_time;
    /* The input time last_known_time, in broken down format */
  struct tm test_time;

  if (debug)
    printf("Ignoring clock year and assuming "
           "it's within 1 year after %s\n",
           ctime2(last_known_time));

  broken_last_known_time = *localtime(&last_known_time);
  
  test_time = base_tm;
  test_time.tm_year = broken_last_known_time.tm_year;

  if (mktime(&test_time) < last_known_time)
    test_time.tm_year += 1;

  return(test_time);
}



static struct tm
localtime_tz(time_t const systime,
             int    const hwClockTzOffset,
             bool   const badyear) {
/*----------------------------------------------------------------------------
  Convert a time in standard unix time (seconds since epoch) into
  broken down format (hours, minutes, etc.) as would be written to the
  Hardware Clock.

  The return value is the broken down time.  This broken down time is
  local time 'hwClockTzOffset' seconds west of the prime meridian.

  Exception: iff 'badyear' is true, the year value is fictionalized to make
  it work in an old hardware clock that is incapable of storing year values
  less than 94 or 95.
-----------------------------------------------------------------------------*/
    struct tm broken_time;

    char *old_TZ;       /* existing value of 'TZ' environment variable */

    /* We use the C library function mktime(), but since it only works on 
       local time zone input, we may have to fake it out by temporarily 
       changing the local time zone to UTC.
    */
    old_TZ = (char *) getenv("TZ");	/* remember original time zone */
    if (debug) {
        if (old_TZ) 
            printf("%s: TZ environment variable value is '%s'.\n",
                   __FUNCTION__, old_TZ);
        else 
            printf("%s: TZ environment variable is not set.\n", __FUNCTION__);
    }

    /* Set the timezone to whatever timezone the hardware clock uses, so
       that C library time functions will do the proper conversions.
    */
    {
        const char * tz;
        casprintf(&tz, "HWCLOCK_TEMP %c%02u:%02u", 
                  hwClockTzOffset < 0 ? '-' : '+',
                  ABS_VALUE(hwClockTzOffset)/3600,
                  (ABS_VALUE(hwClockTzOffset)/60) % 60
            );
        if (debug)
            printf("mktime_tz: temporarily setting TZ to %s\n", tz);
        setenv("TZ", tz, TRUE);
        freeString(tz);
    }
    /* Note: tzset() gets called implicitly by the time code, but only the
       first time.  When changing the environment variable, better call
       tzset() explicitly.
    */
    tzset();


    /* Now have C library to the conversion with the selected timezone
       offset 
    */
    broken_time = *localtime(&systime);

    /* now put back the original zone info.  */
    if (old_TZ) setenv("TZ", old_TZ, TRUE);
    else unsetenv("TZ");
    tzset();

    /* If the clock is incapable of storing the true year value, change
       the year to a fictional stand-in year as described in the prolog.
    */
    if (badyear)
        broken_time.tm_year = 95 + ((broken_time.tm_year + 1) % 4);

    return broken_time;
}



static void
mktime_tz(struct tm    const hw_tm, 
          int          const hwClockTzOffset,
          bool         const badyear,
          time_t       const last_known_time,
          bool *       const valid_p, 
          time_t *     const systime_p) {
/*-----------------------------------------------------------------------------
  Convert a time in broken down format (hours, minutes, etc.)  as read
  from the Hardware Clock into standard unix time (seconds since
  epoch).  Return it as *systime_p.

  The broken down time is argument 'tm'.  This broken down time is
  local time 'hwClockTzOffset' seconds west of the prime meridian.

  Argument 'badyear' true means the input time is from one of those
  machines with the Award BIOS that is incapable of storing a year
  value less than 94 or 95, which means we can't use the year value
  from the clock (see documentation of hwclock's --badyear option).
  In this case, we instead determine the year by assuming that it's
  less than a year since the time <last_known_time>.


  If the argument contains values that do not constitute a valid time,
  and mktime() recognizes this, return *valid_p == false and
  *systime_p undefined.  However, mktime() sometimes goes ahead and
  computes a fictional time "as if" the input values were valid,
  e.g. if they indicate the 31st day of April, mktime() may compute
  the time of May 1.  In such a case, we return the same fictional
  value mktime() does as *systime_p and return *valid_p == true.

-----------------------------------------------------------------------------*/
  time_t mktime_result;  /* The value returned by our mktime() call */
  struct tm adjusted_tm;
    /* The same as the value from our argument, except if we determine
       the year in the argument is garbage, this value contains the year 
       computed from the ADJTIME file instead.
       */
  char *old_TZ;       /* existing value of 'TZ' environment variable */

  /* We use the C library function mktime(), but since it only works on 
     local time zone input, we may have to fake it out by temporarily 
     changing the local time zone to UTC.
     */
  old_TZ = (char *) getenv("TZ");	/* remember original time zone */
  if (debug) {
    if (old_TZ) printf("mktime_tz: TZ environment variable value is '%s'.\n",
                       old_TZ);
    else printf("mktime_tz: TZ environment variable is not set.\n");
  }

  /* Set the timezone to whatever timezone the hardware clock uses, so
     that C library time functions will do the proper conversions.
  */
  {
      const char * tz;
      casprintf(&tz, "HWCLOCK_TEMP %c%02u:%02u", 
                hwClockTzOffset < 0 ? '-' : '+',
                ABS_VALUE(hwClockTzOffset)/3600,
                (ABS_VALUE(hwClockTzOffset)/60) % 60
          );
      if (debug)
          printf("mktime_tz: temporarily setting TZ to %s\n", tz);
      setenv("TZ", tz, TRUE);
      freeString(tz);
  }
  /* Note: tzset() gets called implicitly by the time code, but only the
     first time.  When changing the environment variable, better call
     tzset() explicitly.
  */
  tzset();

  if (badyear) 
    adjusted_tm = make_within_one_year(hw_tm, last_known_time);
  else adjusted_tm = hw_tm; 

  mktime_result = mktime(&adjusted_tm);
  if (mktime_result == -1) {
    /* This apparently (not specified in mktime() documentation) means
       the 'adjusted_tm' structure does not contain valid values (however, not
       containing valid values does _not_ imply mktime() returns -1).
       */
    /* Note that we are assuming here that the invalidity came from the
       hardware values and was not introduced by our adjustments!
       */
    *valid_p = FALSE;
    *systime_p = 0;
    if (debug)
      printf("Invalid values in hardware clock: "
             "%2d/%.2d/%.2d %.2d:%.2d:%.2d\n",
             hw_tm.tm_year, hw_tm.tm_mon+1, hw_tm.tm_mday,
             hw_tm.tm_hour, hw_tm.tm_min, hw_tm.tm_sec
             );
  } else {
    *valid_p = TRUE;
    *systime_p = mktime_result;
    if (debug) 
      printf("Hw clock time : %s = %d seconds since 1969 UTC\n", 
             ctime2(*systime_p), (int) *systime_p);
  }
  /* now put back the original zone info.  */
  if (old_TZ) setenv("TZ", old_TZ, TRUE);
  else unsetenv("TZ");
  tzset();
}



void
synchronize_to_clock_tick(const struct hwclock_config hwclock_config,
                          struct timeval *tick_time_p, int *retcode_p) {
/*-----------------------------------------------------------------------------
  Wait until the moment the Hardware Clock updates to the next second,
  so we know the exact time.

  The clock has only 1 second precision, so it gives the exact time only
  once per second.

  Return the system time that the tick occurred as *tick_time_p.

  Return *retcode_p == 0 if it worked, nonzero if it didn't.
-----------------------------------------------------------------------------*/
  if (debug) printf("Waiting for clock tick...\n");

  switch (hwclock_config.clock_access) {
  case ISA: 
      synchronize_to_clock_tick_ISA(retcode_p, -1, hwclock_config.use_uf_bit); 
      break;
  case DEV_PORT: 
      synchronize_to_clock_tick_ISA(retcode_p, hwclock_config.dev_port,
                                               hwclock_config.use_uf_bit); 
      break;
  case RTC_IOCTL: 
      synchronize_to_clock_tick_RTC(retcode_p, hwclock_config.rtc_filespec); 
      break;
  case KD: 
      synchronize_to_clock_tick_KD(retcode_p); 
      break;
  default:
    fprintf(stderr, "%s: Internal error in synchronize_to_clock_tick.  "
            "Invalid value for clock_access argument: %d.\n",
            MYNAME, hwclock_config.clock_access);
    *retcode_p = 1;
  }
  /* We could make *tick_time_p even more accurate by pushing it down into
     synchronize_to_clock_tick_RTC(), etc.
  */
  gettimeofday(tick_time_p, NULL);

  if (debug) {
      if (*retcode_p == 0)
          printf("...got clock tick\n");
      else
          printf("...failed to get clock tick.\n");
  }
  return;
}


void
read_hardware_clock(const struct hwclock_config hwclock_config,
                    const time_t last_known_time,
                    bool *valid_p, time_t *systime_p) {
/*----------------------------------------------------------------------------
  Read the hardware clock and return the current time via *systime_p
  argument.  

  If the hardware clock fails to tell us a time, return *valid_p == false
  and undefined value as *systime_p.  Otherwise *valid_p == true.

  Consider the hardware clock to be as described by 'hwclock_config'.

  Recognize that the present time is after 'last_known_time', which
  information may be necessary to interpret the value of some hardware
  clocks.
-----------------------------------------------------------------------------*/
  struct tm tm;

  switch (hwclock_config.clock_access) {
  case RTC_IOCTL:
    read_hardware_clock_rtc_ioctl(hwclock_config.rtc_filespec, &tm);
    break;
  case ISA:
    read_hardware_clock_isa(&tm, -1, hwclock_config.hc_zero_year);
    break;
  case DEV_PORT:
    read_hardware_clock_isa(&tm, hwclock_config.dev_port, 
                            hwclock_config.hc_zero_year);
    break;
  case KD:
    read_hardware_clock_kd(&tm);
    break;
  default:
    fprintf(stderr, 
            "%s: Internal error: invalid value (%d) "
            "for clock access method.\n",
            MYNAME, hwclock_config.clock_access);
    exit(5);
  }
  if (debug)
    printf ("Raw time read from Hardware Clock: "
            "Y=%d M=%d D=%d %02d:%02d:%02d\n",
            1900+tm.tm_year, tm.tm_mon+1, tm.tm_mday,
            tm.tm_hour, tm.tm_min, tm.tm_sec);
  mktime_tz(tm, hwclock_config.tzOffset, hwclock_config.badyear, 
            last_known_time, valid_p, systime_p);
}



void
set_hardware_clock(struct hwclock_config * const hwclock_config_p,
                   time_t                  const newtime, 
                   bool                    const testing) {
/*----------------------------------------------------------------------------
  Set the Hardware Clock to the time 'newtime'.

  Assume the Hardware Clock is as described by '*hwclock_config_p' and
  update it to reflect the time zone offset with which we set the time
  (This could be different from what it is already because the "local time"
  clock configuration attribute means to set to whatever is local time at
  the time of the setting, which is not necessarily what it was the last
  time the hardware clock was set).

  hwclock_config_p->badyear true means the clock is incapable of storing
  the proper year value, so we instead store 95, 96, 97, or 98 so that
  it is at least in the right place in the leap year cycle (and will
  remain so for at least the next year).
----------------------------------------------------------------------------*/
  struct tm new_broken_time;  
    /* Time to which we will set Hardware Clock, in broken down format, in
       the time zone of caller's choice
    */
  
  if (hwclock_config_p->universal)
      hwclock_config_p->tzOffset = 0;
  else
      hwclock_config_p->tzOffset = local_time_offset(newtime);

  new_broken_time = localtime_tz(newtime, hwclock_config_p->tzOffset,
                                 hwclock_config_p->badyear);

  if (debug) 
    printf("Setting Hardware Clock to %02d:%02d:%02d(+.5) "
           "with timezone offset %d seconds, "
           "which is %d (+.5) seconds since 1969 UTC\n", 
           new_broken_time.tm_hour, new_broken_time.tm_min, 
           new_broken_time.tm_sec, 
           hwclock_config_p->tzOffset, (unsigned int) newtime);

  switch (hwclock_config_p->clock_access) {
  case RTC_IOCTL:
    set_hardware_clock_rtc_ioctl(new_broken_time, 
                                 hwclock_config_p->rtc_filespec,
                                 testing);
    break;
  case ISA:
    set_hardware_clock_isa(new_broken_time, hwclock_config_p->hc_zero_year, 
                           -1, testing);
    break;
  case DEV_PORT:
    set_hardware_clock_isa(new_broken_time, hwclock_config_p->hc_zero_year, 
                           hwclock_config_p->dev_port, testing);
    break;
  case KD:
    set_hardware_clock_kd(new_broken_time, testing);
    break;
  default:
    fprintf(stderr, 
            "%s: Internal error: invalid value (%d) "
            "for clock access method.\n",
            MYNAME, hwclock_config_p->clock_access);
    exit(5);
  }
}



void
set_hardware_clock_exact(struct timeval          const settime, 
                         struct timeval          const ref_time,
                         struct hwclock_config * const hwclock_config_p,
                         bool                    const testing) {
/*----------------------------------------------------------------------------
  Set the Hardware Clock to the time 'settime'.

  Assume the Hardware Clock is as described by '*hwclock_config_p' and
  update it as described for set_hardware_clock() above.

  But correct 'settime' and wait for a fraction of a second so that
  'settime' is the value of the Hardware Clock as of system time
  'ref_time', which is in the past.  For example, if 'settime' is
  14:03:05 and 'ref_time' is 12:10:04.5 and the current system
  time is 12:10:06.0: Wait .5 seconds (to make exactly 2 seconds since
  'ref_time') and then set the Hardware Clock to 14:03:07, thus
  getting a precise and retroactive setting of the clock.

  (Don't be confused by the fact that the system clock and the Hardware
  Clock differ by two hours in the above example.  That's just to remind 
  you that there are two independent time scales here).

-----------------------------------------------------------------------------*/
  time_t newtime;  /* Time to which we will set Hardware Clock */

  if (debug)
      printf("Reference time for clock setting is %d.%6.6d\n",
             (int) ref_time.tv_sec, (int) ref_time.tv_usec);
  {
      struct timeval now_time;  
      gettimeofday(&now_time, NULL);
      if (debug) 
          printf("Time elapsed since reference time has been %.6f seconds.\n",
                 time_diff(now_time, ref_time));
  
      newtime = time_inc(settime, time_diff(now_time, ref_time)).tv_sec + 1;
  }
  { 
      /* Now delay some more until Hardware Clock time 'newtime' plus 500
         ms arrives.  The +500 ms is because the Hardware Clock always
         sets to your set time plus 500 ms (because it is designed to
         update to the next second precisely 500 ms after you finish the
         setting).  
      */

      const struct timeval target_system_time = 
          time_inc(ref_time, time_diff(time_inc(t2tv(newtime), 0.5), settime));
      struct timeval now_system_time;

      if (debug) 
          printf("Waiting for Time %d.%06d to cause a whole number +.5 "
                 "elapsed seconds.\n",
                 (int) target_system_time.tv_sec, 
                 (int) target_system_time.tv_usec);
  
      do gettimeofday(&now_system_time, NULL);
      while (EARLIER(now_system_time, target_system_time));

  }  

  set_hardware_clock(hwclock_config_p, newtime, testing);

  {
      int retcode;
      bool valid;
      time_t hwclock_time;
      float error;
      struct timeval now_time;
      
      if (debug)
          printf("Now wait for next clock tick (should be 500 ms).\n");

      synchronize_to_clock_tick(*hwclock_config_p, &now_time, &retcode);

      /* And now the Hardware Clock should read newtime + 1 second and
         system time should be ref_time + (newtime - settime) + 1
         second.  I.e. the difference between Hardware Clock time and
         system time should be settime - reftime if we did our job
         right.  If it isn't close, issue warning.  On the author's
         system, it is 1.5 ms behind.
         
         If it's exactly .5 second behind, that could be because this clock
         doesn't do the weird .5 second jump.  If such clocks exist, we need
         to do some more work on this function.  

         Perhaps the --correct value should be allowed for here?
      */

      read_hardware_clock(*hwclock_config_p, newtime, &valid, &hwclock_time);

      error = time_diff(t2tv(hwclock_time), settime) 
          - time_diff(now_time, ref_time);
      if (debug) {
          printf("It is now %.6f seconds since reference time.\n",
                 time_diff(now_time, ref_time));
          printf("Hardware clock ended up %.6f seconds from intended "
                 "set time.\n", error);
      }
      if (ABS_VALUE(error) > .1 && !testing)
          fprintf(stderr, "%s: Warning: Hardware clock ended up %.6f seconds "
                  "from intended set time.\n"
                  "This could be because of a Hwclock bug, a "
                  "Hardware Clock bug, or\n"
                  "very slow program execution.\n", MYNAME, error);
  }
}


void
get_clock_access(enum clock_access_method const clock_access,
                 int *                    const dev_port_p,
                 const char **            const errorP) {

    switch(clock_access) {
    case ISA:
        get_inb_outb_privilege(errorP);
        break;
    case DEV_PORT:
        get_dev_port_access(dev_port_p, errorP);
        break;
    default:
        *errorP = NULL;
    }
}



void
release_clock_access(enum clock_access_method const clock_access,
                     int                      const dev_port) {

    if (clock_access == DEV_PORT)
        close(dev_port);
}
