#define _GNU_SOURCE
#include <time.h>

#include "hwclock.h"
#include "adjust.h"


void
calculate_adjustment(struct adjtime const adjtime,
                     struct timeval const currentTime,
                     int *          const intAdjustmentP, 
                     float *        const retroP,
                     float *        const adjustmentP,
                     bool           const nodrift,
                     bool           const debug ) {
/*----------------------------------------------------------------------------
  Do the drift adjustment calculation.  Estimate out how far the
  hardware clock has drifted since it was last set.

  'adjtime' is the contents of the adjtime file before this adjustment.
  It contains information about past drift and settings.

  'currentTime' is the actual time now.

  We return the result two ways:

    1) A simple floating point number of seconds, as *adjustmentP.

    2) The combination of the following two numbers, which is handy
       because of the way one has to set the Hardware Clock:

       A) an integer number of seconds, as *int_adjustmentP
       
       B) a positive fraction of a second (less than 1), as *retroP

       The sum of these two values is the adjustment needed.  

    Positive means to advance the clock or insert seconds.  Negative
    means to retard the clock or remove seconds.

    Where a return pointer (e.g. adjustmentP) is NULL, we don't return
    the corresponding value.

 ----------------------------------------------------------------------------*/
    float const secPerDay = 24.0 * 60.0 * 60.0;

    float exactAdjustment;

    if (nodrift)
        exactAdjustment = 0.0;
    else {
        float const adjDays = 
            (time_diff(currentTime, t2tv(adjtime.last_adj_time))
             - adjtime.last_adj_frac) / secPerDay;

        exactAdjustment = 
            adjDays * adjtime.drift_factor + adjtime.not_adjusted;
    }  
    if (adjustmentP != NULL)
        *adjustmentP = exactAdjustment;
    
    if (intAdjustmentP != NULL) 
        *intAdjustmentP = FLOOR(exactAdjustment);
  
    if (retroP != NULL)
        *retroP = exactAdjustment - (float) *intAdjustmentP;
    if (debug) {
        printf ("Time since last adjustment is %f seconds\n",
                time_diff(currentTime, t2tv(adjtime.last_adj_time)));
        if (intAdjustmentP != NULL && retroP != NULL)
            printf ("Need to insert %d seconds and refer time back "
                    "%.6f seconds ago\n",
                    *intAdjustmentP, *retroP);
        if (adjustmentP != NULL)
            printf("Need to add %f seconds to Hardware Clock time.\n", 
                   *adjustmentP);
    }
}



void
adjust_drift_factor(struct adjtime * const adjtimeP,
                    time_t           const actualTime, 
                    bool             const hclockValid, 
                    struct timeval   const hclocktime,
                    bool             const nodrift) {
/*---------------------------------------------------------------------------
  Update the drift factor and calibration parameters in '*adjtimeP'
  to reflect the fact that at some recent instant when the actual time
  was 'actualTime', the Hardware Clock said the time was
  'hclocktime', and that we have corrected the Hardware Clock
  accordingly.  Note that 'hclocktime' is a fractional time, taking
  into consideration the Hardware Clock register contents and how long
  those contents had been that.

  We assume that the only cause of error in the Hardware Clock is
  systematic drift and that the user has been doing regular drift
  adjustments using the drift factor in the adjtime file.  Therefore,
  if 'actualTime' and 'hclocktime' are different, that means the drift
  factor isn't quite right. 

  EXCEPT: if 'hclockValid' is false, assume Hardware Clock was not set
  before to anything meaningful and regular adjustments have not been
  done, so don't adjust the drift factor.

  Also, don't adjust if the error is more than 30 minutes, because that
  kind of error probably isn't drift.

  And don't adjust if 'nodrift' is TRUE, because Caller is explicitly 
  telling us that the clock has not been simply drifting.

----------------------------------------------------------------------------*/
    if (nodrift){
        if (debug)
            printf("Not adjusting drift factor because of -nodrift option\n");
    } else if (!hclockValid) {
        if (debug)
            printf("Not adjusting drift factor because the Hardware Clock "
                   "previously contained garbage.\n");
    } else if (adjtimeP->last_calib_time == 0) {
        if (debug)
            printf("Not adjusting drift factor because there is no \n"
                   "previous calibration information (i.e. adjtime file is \n"
                   "nonexistent or has 0 in last calibration time field).\n");
    } else if (time_diff(hclocktime, t2tv(adjtimeP->last_calib_time)) 
               < 23.0 * 60.0 * 60.0) {
        if (debug) 
            printf("Not adjusting drift factor because it has been less "
                   "than a day since the last calibration.\n");
    } else {
        float const secPerDay = 24.0 * 60.0 * 60.0;

        float atimePerHtime; 
            /* adjusted time units per hardware time unit */
        float calDays; 
            /* days since last calibration (in hardware clock time) */
        float expDrift;  
            /* expected drift (sec) since last adjustment */
        float uncDrift;  
            /* uncorrected drift (sec) since last calibration */
        float factorAdjust; 
            /* amount to add to previous drift factor */

        atimePerHtime = 1.0 + adjtimeP->drift_factor / secPerDay;
        calculate_adjustment(*adjtimeP, hclocktime, NULL, NULL,
                             &expDrift, FALSE, debug);
        uncDrift = time_diff(t2tv(actualTime), hclocktime) - expDrift;
        calDays = 
            ((float)(actualTime - adjtimeP->last_calib_time) - uncDrift)
            / (secPerDay * atimePerHtime);
        factorAdjust = uncDrift / calDays;
  
        if (ABS_VALUE(uncDrift) > 30*60.0) {
            if (debug)
                printf("Not adjusting drift factor because we "
                       "calculated the \n"
                       "uncorrected drift as %.0f seconds, "
                       "which is so large that \n"
                       "it probably is not drift at all, but rather some \n"
                       "clock setting anomaly.\n\n", uncDrift);
        } else if (ABS_VALUE(factorAdjust) > 5*60.0) {
            if (debug)
                printf("Not adjusting drift factor because we "
                       "calculated the \n"
                       "uncorrected drift as %.0f seconds per day, "
                       "which is so large that \n"
                       "it probably is not drift at all, but rather some \n"
                       "clock setting anomaly.\n\n", factorAdjust);
        } else {
            if (debug)
                printf("Clock drifted %.1f seconds in the past %f days\n"
                       "in spite of a drift factor of %f seconds/day.\n"
                       "Adjusting drift factor by %f seconds/day\n",
                       uncDrift,
                       (float) (actualTime - adjtimeP->last_calib_time) 
                       / secPerDay,
                       adjtimeP->drift_factor,
                       factorAdjust  );
      
            adjtimeP->drift_factor += factorAdjust;
        }
    }
    adjtimeP->last_calib_time = actualTime;
  
    adjtimeP->last_adj_time = actualTime;

    adjtimeP->last_adj_frac = 0;
  
    adjtimeP->not_adjusted = 0;
    
    adjtimeP->dirty = TRUE;
}



