#define _GNU_SOURCE
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <sys/stat.h>

#include "util.h"
#include "adjtime.h"


static int
tzOffsetGuess(time_t const lastSetTime) {
/*----------------------------------------------------------------------------
  Assuming the hardware clock is in local time, guess at which local time
  it is -- i.e. the time zone offset.

  Base the guess on the fact that the hardware clock was last set
  at time 'lastSetTime'.  Guess that the time zone offset for the hardware
  clock is the same as the timezone offset that the process that is running
  us uses for the time that the hardware clock was last set.

  E.g. if the current process uses US Pacific Time and the clock was last
  set in the summer, assume Pacific Daylight Time.

  'lastSetTime' == 0 means it is unknown when the hardware clock was last
  set.  In that case, we use the time zone offset that the process running
  us uses for the present time.
-----------------------------------------------------------------------------*/
    if (lastSetTime)
        return local_time_offset(lastSetTime);
    else
        return local_time_offset(time(NULL));
}



static void
interpretAdjtimeLine3(char                   line3[], 
                      struct adjtime * const adjtimeP,
                      time_t           const lastAdjTime,
                      const char **    const errorP) {
/*----------------------------------------------------------------------------
  Interpret line3[] as the third line of an adjtime file and set
  relevant members of *adjtimeP according to its contents.
-----------------------------------------------------------------------------*/
    char localUtcString[strlen(line3)+1];
    int matched;

    matched = sscanf(line3, "%s %u %d", 
                     localUtcString, &adjtimeP->epoch, &adjtimeP->tzOffset);

    *errorP = NULL;  /* Initial assumption */

    if (matched < 1)
        adjtimeP->local_utc = LOCAL;
    else {
        if (strcmp(localUtcString, "UTC") == 0) 
            adjtimeP->local_utc = UTC;
        else if (strcmp(localUtcString, "LOCAL") == 0) 
            adjtimeP->local_utc = LOCAL;
        else 
            casprintf(errorP, "The first token "
                      "is invalid.  It must be LOCAL or UTC, indicating "
                      "to which time zone the hardware clock is set.  Its "
                      "present value is '%s'.", localUtcString);
    }
    if (!*errorP) {
        if (matched < 2)
            adjtimeP->epoch = -1;
        else {
            if (adjtimeP->epoch < 1900) {
                casprintf(errorP, "The second token is invalid.  It is the "
                          "Gregorian year number to which a value of zero "
                          "in the hardware clock's year "
                          "register corresponds.  "
                          "It must be an integer >= 1900");
            } 
        }
    }
    if (!*errorP) {
        if (matched < 3) {
            if (adjtimeP->local_utc == UTC)
                adjtimeP->tzOffset = 0;
            else 
                /* The hardware clock is in local time format and the adjtime
                   file doesn't tell which local time (what timezone offset)
                   it is.  So we guess.
                */
                adjtimeP->tzOffset = tzOffsetGuess(lastAdjTime);
        } else
            if (adjtimeP->tzOffset > 43200 || adjtimeP->tzOffset < -43200)
                casprintf(errorP, "Timezone offset value %d is invalid.  "
                          "It is seconds west of the prime meridian, "
                          "which means an integer between -43200 and 43200, "
                          "inclusive.", adjtimeP->tzOffset);
    }
}



static const char *
adjtime_pathX(void) {

    const char * retval;
    struct stat statbuf;

    if (getenv("ADJTIME_PATH"))
        retval = getenv("ADJTIME_PATH");
    else {
        if (stat("/var/state/adjtime", &statbuf) == 0)
            retval = "/var/state/adjtime";
        else if (stat("/etc/adjtime", &statbuf) == 0)
            retval = "/etc/adjtime";
        else
            retval = "/var/state/adjtime";
    }
    return(retval);
}



void
read_adjtime(struct adjtime * const adjtime_p, 
             const char **    const errorP,
             bool             const debug) {
/*----------------------------------------------------------------------------
  Read the adjustment parameters and other persistent variables out of
  the adjtime file.

  Return them as the adjtime structure <*adjtime_p>.

  The adjtime file is the one named by the ADJTIME_PATH environment variable.
  If that is not defined, and /var/state/adjtime exists, that is it.  If that
  doesn't exist but /etc/adjtime does, that is it.  If even that does not
  exist, then return defaults.

  If values are missing from the file, return defaults for them.
  
  return *errorP == NULL if all OK, otherwise place a text description of
  why in a newly malloc'ed storage and return its address as *errorP.

  Note: The default is LOCAL rather than UTC for historical reasons.

-----------------------------------------------------------------------------*/
    const char * adjtime_path = adjtime_pathX();
    FILE *adjfile;
    int rc;  /* local return code */
    struct stat statbuf;  /* We don't even use the contents of this. */

    rc = stat(adjtime_path, &statbuf);
    if (rc < 0 && errno == ENOENT) {
        /* He doesn't have a adjtime file, so we'll use defaults. */
        adjtime_p->drift_factor = 0;
        adjtime_p->last_adj_time = 0;
        adjtime_p->last_adj_frac = 0;
        adjtime_p->not_adjusted = 0;
        adjtime_p->last_calib_time = 0;
        adjtime_p->local_utc = LOCAL;
        adjtime_p->tzOffset = tzOffsetGuess(0);
        adjtime_p->epoch = 1900;

        *errorP = NULL;
    } else { 
        adjfile = fopen(adjtime_path, "r");   /* open file for reading */
        if (adjfile == NULL) {
            const int fopen_errno = errno;
            casprintf(errorP, 
                      "Unable to open file '%s'.  fopen() errno=%d:%s", 
                     adjtime_path, fopen_errno, strerror(fopen_errno));
        } else {
            char line1[81];           /* String: first line of adjtime file */
            char line2[81];           /* String: second line of adjtime file */
            char line3[81];           /* String: third line of adjtime file */
            char line4[81];           /* String: fourth line of adjtime file */
            const char * error;
      
            line1[0] = '\0';          /* In case fgets fails */
            fgets(line1, sizeof(line1), adjfile);
            line2[0] = '\0';          /* In case fgets fails */
            fgets(line2, sizeof(line2), adjfile);
            line3[0] = '\0';          /* In case fgets fails */
            fgets(line3, sizeof(line3), adjfile);
            line4[0] = '\0';          /* In case fgets fails */
            fgets(line4, sizeof(line4), adjfile);
      
            fclose(adjfile);
      
            /* Set defaults in case values are missing from file */
            adjtime_p->drift_factor = 0;
            adjtime_p->last_adj_time = 0;
            adjtime_p->last_adj_frac = 0;
            adjtime_p->not_adjusted = 0;
            adjtime_p->last_calib_time = 0;
      
            sscanf(line1, "%f %d %f", 
                   &adjtime_p->drift_factor,
                   (int *) &adjtime_p->last_adj_time, 
                   &adjtime_p->not_adjusted);
      
            sscanf(line2, "%d", (int *) &adjtime_p->last_calib_time);

            interpretAdjtimeLine3(line3, adjtime_p, adjtime_p->last_adj_time,
                                  &error);

            if (error) {
                casprintf(errorP, 
                          "Third line of adjtime file '%s' is invalid.  %s",
                          adjtime_path, error);
                freeString(error);
            } else
                *errorP = NULL;

            sscanf(line4, "%f", 
                   &adjtime_p->last_adj_frac);
        }
        adjtime_p->dirty = FALSE;
    }
    if (debug) {
        printf("adjtime file:  drift factor = %f, "
               "not adjusted = %f, %s (%d seconds west of GMT), epoch = %d\n",
               adjtime_p->drift_factor, adjtime_p->not_adjusted,
               adjtime_p->local_utc == UTC ? "UTC" : "LOCAL",
               adjtime_p->tzOffset,
               adjtime_p->epoch);
        printf("Last drift adjustment done %s (Time %d) + %f secs\n", 
               ctime2(adjtime_p->last_adj_time),
               (int) adjtime_p->last_adj_time,
               adjtime_p->last_adj_frac);
        printf("Last calibration done %s (Time %d)\n",
               ctime2(adjtime_p->last_calib_time), 
               (int) adjtime_p->last_calib_time);
    }
}



void
save_adjtime(struct adjtime const adjtime, 
             bool           const testing,
             bool           const debug) {
/*-----------------------------------------------------------------------------
  Write the contents of the 'adjtime' structure to its disk file.

  But if the contents are clean (unchanged since read from disk), don't
  bother.
-----------------------------------------------------------------------------*/
    const char * const adjtime_path = adjtime_pathX();

    const char *newContents;   /* Disk file image.  malloc'ed */

    int rc;   /* locally used: return code from a function */

    if (adjtime.dirty) {
        casprintf(&newContents, "%f %ld %f\n%ld\n%s %d %d\n%f\n",
                  adjtime.drift_factor,
                  (long) adjtime.last_adj_time,
                  adjtime.not_adjusted,
                  (long) adjtime.last_calib_time,
                  (adjtime.local_utc == UTC) ? "UTC" : "LOCAL",
                  adjtime.epoch,
                  adjtime.tzOffset,
                  adjtime.last_adj_frac
            );
        if (!newContents)
            fprintf(stderr, "Out of memory for 'newContents' variable");
        else {
            if (testing) {
                printf("Not updating adjtime file because of testing mode.\n");
                printf("Would have written the following to %s:\n%s", 
                       adjtime_path, newContents);
            } else {
                FILE *adjfile;

                adjfile = fopen(adjtime_path, "w");
                if (adjfile == NULL) {
                    const int fopen_errno = errno;
                    printf("Could not open adjtime file %s for output.  "
                           "fopen() returned errno %d: %s.\n"
                           "Drift adjustment parameters not updated.\n", 
                           adjtime_path, fopen_errno, strerror(errno));
                } else {
                    rc = fwrite(newContents, 1, strlen(newContents), adjfile);
                    if (rc < 0) {
                        const int fwrite_errno = errno;
                        printf("Could not update file %s.  "
                               "fwrite() returned errno %d: %s.\n"
                               "Drift adjustment parameters not updated.\n",
                               adjtime_path, fwrite_errno, strerror(errno));
                    }
                    rc = fclose(adjfile);
                    if (rc < 0) {
                        const int fclose_errno = errno;
                        printf("Could not update file %s.  "
                               "fclose() returned errno %d: %s.\n"
                               "Drift adjustment parameters not updated.\n",
                               adjtime_path, fclose_errno, strerror(errno));
                    }
                }
            }
            freeString(newContents);
        }
    } else {
        if (debug)
            printf("Skipping update of adjtime file because "
                   "nothing has changed.\n");
    }
}
