/******************************************************************************
 ******************************************************************************
 **
 ** FILE: ovl_fs.h
 **
 ** DESCRIPTION: This file contains definitions of constants, prototypes, and
 **              macros used by the overlay filesystem source files.
 **
 ** REVISION HISTORY
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 02/27/1998	ARTHUR N.	Initial checkin.
 ** 03/01/1998	ARTHUR N.	Added magic name options.
 ** 03/03/1998	ARTHUR N.	Added storage file inode reference to the
 **				 super block information and the
 **				 OVLFS_ALLOW_XDEV_RENAME compile option.
 ** 03/03/1998	ARTHUR N.	Added support for the noxmnt option.
 ** 03/04/1998	ARTHUR N.	Added the option to store mapping information
 **				 in the storage file.
 ** 03/05/1998	ARTHUR N.	Allocate all memory as KERNEL priority mem.
 ** 03/08/1998	ARTHUR N.	Added support for the VERIFY_MALLOC option.
 ** 03/10/1998	ARTHUR N.	Added support for the "updmntonly" option and
 **				 corrected a mistake in copy_inode_info().
 ** 03/10/1998	ARTHUR N.	Added use of the ovlfs_lock_inode in the
 **				 ovlfs_chown() function.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ifndef __ovl_STDC_INCLUDE
#define __ovl_STDC_INCLUDE	1

#ident "@(#) ovl_fs.h 1.6"

#ifdef __KERNEL__
# include "kernel_lists.h"
#endif

/******************************************************************************
 ******************************************************************************
 **
 ** CONSTANTS:
 **/

	/* handy shortcuts: */
#define INODE_NULL		(struct inode *) NULL


/**
 ** COMPILE-OPTION CONSTANTS
 **/

	/* OVLFS_DEFINE_BMP: if non-zero implement the bmp() inode op */
#ifndef OVLFS_DEFINE_BMP
# define OVLFS_DEFINE_BMP	0
#endif

	/* Needed methods (stolen from kernel source): */
#define OVLFS_NEED_GET_SUPER		1
#define OVLFS_NEED_LOCK_INODE		1
#define OVLFS_NEED_UNLOCK_INODE		1
#define OVLFS_NEED_WAIT_ON_INODE	1
#define OVLFS_NEED_DEF_PERM		1
#define OVLFS_NEED_OPEN_INODE		1

	/* OVLFS_ALLOW_XDEV_RENAME: if a file is in the storage fs and the */
	/*  user renames it across a mount point (mv doesn't see the mount */
	/*  point since it is looking at the ovlfs), the pseudo fs will    */
	/*  allow the rename internally if this is set.                    */
	/* Regardless of the setting, the file in the storage fs will NOT  */
	/*  be moved across a mount point by the rename() operation.       */

#ifndef OVLFS_ALLOW_XDEV_RENAME
# define OVLFS_ALLOW_XDEV_RENAME	1
#endif

/**
 ** FS CONSTANTS
 **
 **	note that the *_INO constants needed to be distinct from those in the
 **	real fs', but that is NO LONGER so.
 **/

#define OVLFS_ROOT_INO		1
#define OVLFS_DIR_INO		0x01000001
#define OVLFS_FILE_INO		0x01000002
#define OVLFS_GENERIC_INO	0x01000003
#define OVLFS_SUPER_MAGIC	0xFE0F		/* Completely made up */

/**
 ** RUNTIME OPTION CONSTANTS
 **/

/* storage options: */
#define OVLFS_NO_STORAGE	0
#define OVLFS_USE_STORAGE	1
#define OVLFS_STG_CK_MEM	2	/* check available mem on alloc */
#define OVLFS_CROSS_MNT_PTS	4
#define OVLFS_STORE_BASE_MAP	8	/* save base fs mappings in stg file? */
#define OVLFS_STORE_STG_MAP	16	/* save stg fs mappings in stg file? */
#define OVLFS_UPD_UMOUNT_ONLY	32	/* only update storage at umount? */

#define OVLFS_STORE_ALL_MAPS	( OVLFS_STORE_BASE_MAP | OVLFS_STORE_STG_MAP )
#define OVLFS_STG_DEF_OPTS	( OVLFS_USE_STORAGE | OVLFS_CROSS_MNT_PTS | \
				  OVLFS_STORE_ALL_MAPS )


	/***
	 *** MAGIC options: set USE_MAGIC to non-zero int on the compile
         ***  line to compile in the source that allows the use of the magic
         ***  directories.  If compiled in, a mount option can be used to
	 ***  indicate whether to turn the magic directories on or off.
	 ***/

#if USE_MAGIC
	/* magic options */
# define OVLFS_NO_MAGIC		0
# define OVLFS_BASE_MAGIC	1
# define OVLFS_OVL_MAGIC	2
# define OVLFS_HIDE_MAGIC	4
# define OVLFS_ALL_MAGIC	( OVLFS_BASE_MAGIC | OVLFS_OVL_MAGIC | \
				  OVLFS_HIDE_MAGIC )

# define OVLFS_MAGIC_NAME	"magic"
# define OVLFS_MAGIC_NAME_LEN	5

# define OVLFS_MAGICR_NAME	"magicr"
# define OVLFS_MAGICR_NAME_LEN	6
#endif

/**
 ** OVLFS Storage CONSTANTS.
 **/

	/* flags for ovlfs_inode_struct */
#define OVLFS_INODE_READ	1	/* Has inode been "read" yet? */

	/* flags for ovlfs_dir_info */
#define OVLFS_DIRENT_NORMAL	0	/* No special info */
#define OVLFS_DIRENT_UNLINKED	1	/* Directory entry was removed */


/**
 ** DEBUGGING CONSTANTS
 **
 ** NOTES:
 **	- Debug messages are printed using printk() and can be viewed by
 **	  looking at a console text screen or by running dmesg.
 **
 **	- KDEBUG's meanings are as follows:
 **		- if not defined (#undef KDEBUG), no debug messages will be
 **		  generated by the ovlfs regardless of the severity.
 **
 **		- if defined to 0 or non-numeric, only serious erros will
 **		  generate debug messages.
 **
 **		- if set to 1, warnings will generate debug messages in
 **		  addition to the serious error messages.
 **
 **		- if set above 1, be prepared for inundation of debugging
 **		  information.  It is HIGHLY recommended that this NEVER be
 **		  used unless absolutely necessary.
 **/

#define OVLFS_DEBUG_FILE	"/tmp/ovlfs.debug"
#define KDEBUG_CALL_PREFIX	"OVLFS: CALL: "
#define KDEBUG_RET_PREFIX	"OVLFS: RETURN: "
#ifndef KDEBUG_CALLS
# if KDEBUG > 1
#  define KDEBUG_CALLS 1
# endif
#endif

	/* lock/unlock debugging */
#define ovlfs_lock_inode0(i)	\
	printk("OVLFS: lock inode 0x%x @ %s: %d\n", (int) (i), __FILE__, \
	       __LINE__), \
	ovlfs_lock_inode(i)

#define ovlfs_unlock_inode0(i)	\
	printk("OVLFS: unlock inode 0x%x @ %s: %d\n", (int) (i), __FILE__, \
	       __LINE__),\
	ovlfs_unlock_inode(i)



/******************************************************************************
 ******************************************************************************
 **
 ** TYPE DEFINITIONS:
 **/

/* syscall_t is a function pointer used by interfacing to system calls */
typedef int		(*syscall_t)();

/* Shortcuts: trying to make these distinct from any in use */

typedef unsigned int	_uint;
typedef unsigned long	_ulong;
typedef unsigned char	_uchar;

/**
 ** STRUCTURE: ovlfs_super_info_struct
 **
 ** DESCRIPTION: Maintain the information and options for an overlayed
 **              filesystem.
 **
 ** MEMBERS:
 **	base_root	- root directory of the filesystem to overlay (i.e. the
 **			  original/real filesystem).
 **	storage_root	- root directory of the area to store changes to the
 **			  overlayed fs.
 **	root_inode	- the inode of the base filesystems root directory.
 **	storage_inode	- the inode of the storage fs's root directory.
 **	storage_opts	- options for storing changes to the overlayed fs.
 **	magic_opts	- options for turning on or off the use of the magic
 **			  files.
 **	stg_file	- path of the file to store the overlay filesystem
 **			  inode information; this is the persistent storage.
 **	inodes		- list of inodes for the overlay filesystem.
 **	base_to_ovl	- list of mappings from the base filesystem inodes
 **			  to the overlay filesystem's inodes.
 **	stg_to_ovl	- list of mappings from the storage filesystem inodes
 **			  to the overlay filesystem's inodes.
 **/

struct ovlfs_super_info_struct {
	char		*base_root;
	char		*storage_root;
	struct inode	*root_inode;
	struct inode	*storage_inode;
	char		storage_opts;
#if USE_MAGIC
	char		magic_opts;
	char		*Smagic_name;	/* Name of magic dir for the STG FS */
	char		*Bmagic_name;	/* Name of magic dir for the Base FS */
	int		Smagic_len;
	int		Bmagic_len;
#endif
	char		*stg_file;
	struct inode	*stg_file_inode;
	list_t		inodes;
	list_t		base_to_ovl;
	list_t		stg_to_ovl;
} ;

typedef struct ovlfs_super_info_struct	ovlfs_super_info_t;



/**
 ** STRUCTURE: ovlfs_inode_struct
 **
 ** DESCRIPTION: Maintain the information for the overlay filesystem inode.
 **/

struct ovlfs_inode_struct {
	struct super_block	*sb;

	umode_t		mode;
	off_t		size;
	uid_t		uid;
	gid_t		gid;
	time_t		atime;
	time_t		mtime;
	time_t		ctime;
	nlink_t		nlink;
	unsigned long	blksize;
	unsigned long	blocks;
	unsigned long	version;
	unsigned long	nrpages;

	union {
		kdev_t	rdev;	/* for device special files */
		char	*link;	/* for symbolic links */
		void	*generic;
		int	int_value;
	} u;

	_ulong	ino;
	_ulong	parent_ino;	/* At least, one of the parents ;) */
	kdev_t	base_dev;
	_ulong	base_ino;
	kdev_t	stg_dev;
	_ulong	stg_ino;
	list_t	dir_entries;
	int	flags;

	char	*name;		/* At least, one of the names ;) */
	int	len;
} ;

typedef struct ovlfs_inode_struct   ovlfs_inode_t;



/**
 ** STRUCTURE: ovlfs_inode_info
 **
 ** PURPOSE: This structure holds information for a single inode in the overlay
 **          filesystem.
 **/

struct ovlfs_inode_info {
#if STORE_REF_INODES
	struct inode	*ovlfs_dir;	/* for files, directory they are in */
	struct inode	*base_inode;
	struct inode	*overlay_inode;
#else
	char		*base_name;
	char		*overlay_name;
#endif

	loff_t		base_pos;
	loff_t		overlay_pos;
	char		*filename;
	int		name_len;
} ;

typedef struct ovlfs_inode_info	ovlfs_inode_info_t;



/**
 ** STRUCTURE: ovlfs_file_info_struct
 **
 **   PURPOSE: Maintain information on a file that references an ovlfs inode.
 **/

struct ovlfs_file_info_struct {
	struct file	file;
	char		is_base;
} ;

typedef struct ovlfs_file_info_struct	ovlfs_file_info_t;



/**
 ** STRUCTURE: ovlfs_externs_struct
 **
 **   PURPOSE: Maintain the references to all functions and data that are made
 **            externally available for the overlay filesystem.  The concept
 **            is to maintain as few actual external symbols as possible.
 **
 ** NOTES:
 **	- I am debating whether to do this or not since I have multiple source
 **	  files, and to export functions and variables that are static to
 **	  each one, I would need to setup the structure at load time by
 **	  calling one function for each source file (yet more externals!).
 **/

struct ovlfs_externs_struct {
} ;

typedef struct ovlfs_externs_struct	ovlfs_externs_t;



/**
 ** STRUCTURE: ovlfs_ino_map_struct
 **
 ** DESCRIPTION: Maintain mapping from a device number and inode number to
 **              the inode number of the overlay fs inode that it belongs to.
 **/

struct ovlfs_ino_map_struct {
	kdev_t	dev;
	_uint	ino;
	_uint	ovl_ino;
} ;

typedef struct ovlfs_ino_map_struct	ovlfs_ino_map_t;



/**
 ** STRUCTURE: ovlfs_dir_info_struct
 **/

struct ovlfs_dir_info_struct {
	char	*name;
	int	len;
	_ulong	ino;
	int	flags;
} ;

typedef struct ovlfs_dir_info_struct	ovlfs_dir_info_t;


/******************************************************************************
 ******************************************************************************
 **
 ** PROTOTYPES:
 **/

#ifdef __cplusplus
extern "C" {
#endif

#define dprintf	ovlfs_dprintf

	extern void	ovlfs_end_debug();
	extern void	ovlfs_dprintf(int, char *, ...);
	extern int	ovlfs_resolve_inode(struct inode *, struct inode **,
			                    char);
	extern void	ovlfs_debug_inodes(struct inode *);
	extern int	ovlfs_copy_file(struct inode *, const char *, int,
			                struct inode *, struct inode **);
	extern int	ovlfs_inode_copy_file(struct inode *, struct inode *,
			                      struct inode **);
	extern void	ovlfs_prepare_inode(struct inode *, int);

	extern int	ovlfs_make_hierarchy(struct inode *, struct inode **,
			                     int);

	extern struct inode	*ovlfs_get_parent_dir(struct inode *);

/* Directory operations */
	extern int		ovlfs_read_directory(struct inode *);

/* Storage operations */
	extern ovlfs_inode_t	*ovlfs_add_inode(struct super_block *,
				                 const char *, int);
	extern int		ovlfs_inode_set_base(struct super_block *,
				                     ovlfs_inode_t *, dev_t,
				                     _uint);
	extern int		ovlfs_inode_set_stg(struct super_block *,
				                    ovlfs_inode_t *, dev_t,
				                    _uint);
	extern ovlfs_inode_t	*ovlfs_setup_new_inode(struct super_block *,
				                       const char *, int,
				                       dev_t, _ulong, dev_t,
				                       _ulong);
	extern ovlfs_inode_t	*ovlfs_get_ino(struct super_block *, _ulong);
	extern _ulong		ovlfs_lookup_ino(struct super_block *, kdev_t,
				                 _ulong, char);
	extern int		ovlfs_ino_add_dirent(ovlfs_inode_t *,
				                     const char *, int, _ulong,
				                     int);
	extern ovlfs_dir_info_t	*ovlfs_ino_find_dirent(ovlfs_inode_t *,
				                       const char *, int);
	extern ovlfs_inode_t	*ovlfs_stg_read_ino(struct inode *);
	extern int		ovlfs_stg_read_super(struct super_block *);
	extern void		ovlfs_stg_put_super(struct super_block *);
	extern int		ovlfs_stg_write_super(struct super_block *);
	extern int		ovlfs_remove_dirent(ovlfs_inode_t *,
				                    const char *, int);
	extern int		ovlfs_unlink_dirent(ovlfs_inode_t *,
				                    const char *, int);

	extern int		ovlfs_set_attrib(struct inode *,
				                 ovlfs_inode_t *);


/**
 ** Kernel "replacements":
 **/

# if OVLFS_NEED_GET_SUPER
	extern struct super_block	*ovlfs_get_super(kdev_t);
# endif
# if OVLFS_NEED_OPEN_INODE
	int				ovlfs_open_inode(struct inode *, int,
					                 struct file *);
	void				ovlfs_close_file(struct file *);
# endif

		/* Inode locking operations */
# if OVLFS_NEED_WAIT_ON_INODE
	extern int		ovlfs_wait_on_inode(struct inode *);
# endif
# if OVLFS_NEED_LOCK_INODE
	extern int		ovlfs_lock_inode(struct inode *);
# endif
# if OVLFS_NEED_UNLOCK_INODE
	extern void		ovlfs_unlock_inode(struct inode *);
# endif

		/* Permission checking */

# if OVLFS_NEED_DEF_PERM
	extern int		ovlfs_ck_def_perm(struct inode *, int);
# endif

		/* System call interface setup */
	void	ovlfs_setup_syscalls();

#if VERIFY_MALLOC
	void	ovlfs_start_alloc(const char *);
	void	ovlfs_stop_alloc(const char *);
	void	*ovlfs_alloc(unsigned int);
	void	ovlfs_free(void *);
	void	ovlfs_free2(void *, unsigned int);
#endif

#ifdef __cplusplus
}
#endif



/******************************************************************************
 ******************************************************************************
 **
 ** EXTERNAL DECLARATIONS:
 **
 **	note that these are not all placed into the kernel symbol table.
 **/

extern struct file_operations	ovlfs_dir_fops;
extern struct inode_operations	ovlfs_ino_ops;
extern struct inode_operations	ovlfs_file_ino_ops;
extern struct file_operations	ovlfs_file_ops;

extern syscall_t	open;
extern syscall_t	close;
extern syscall_t	read;
extern syscall_t	lseek;
extern syscall_t	write;




/******************************************************************************
 ******************************************************************************
 **
 ** MACROS:
 **/

/* Macro to convert a #defined value to a string (i.e. put quotes around it) */
#define do_mk_str(x)	#x
#define macro2str(x)	do_mk_str(x)

/* The third parameter to open_namei() is the mode of the file; since we */
/*  are not creating files (yet?) this is set to 0.                      */

#define get_inode(f,i)		open_namei(f, 0, 0, i, (struct inode *) NULL)
#define get_inode_ro(f,i)	open_namei(f, 1, 0, i, (struct inode *) NULL)
#define get_inode_wo(f,i)	open_namei(f, 2, 0, i, (struct inode *) NULL)
#define get_inode_rw(f,i)	open_namei(f, 3, 0, i, (struct inode *) NULL)

#define opt_use_storage_p(o)	((o).storage_opts & OVLFS_USE_STORAGE)

#define ovlfs_resolve_base_inode(i,r)	ovlfs_resolve_inode((i),(r),'b')
#define ovlfs_resolve_ovl_inode(i,r)	ovlfs_resolve_inode((i),(r),'o')
#define ovlfs_resolve_stg_inode(i,r)	ovlfs_resolve_ovl_inode((i),(r))

#define ovlfs_get_super_opts(sb)	\
	((sb) == NULL? NULL :  (sb)->u.generic_sbp )
#define ovlfs_sb_opts(sb)		\
	((ovlfs_super_info_t *)((sb) == NULL? NULL : (sb)->u.generic_sbp))
#define ovlfs_magic_opt_p(sb,m)					\
	((sb) == NULL? 0 : ( (sb)->u.generic_sbp == NULL ?	\
	                     0 : ovlfs_sb_opts(sb)->magic_opts & (m)))
#define ovlfs_sb_opt(sb,opt)			\
	( (sb) == NULL?				\
	  0 : ( (sb)->u.generic_sbp == NULL ?	\
	        0 : ((ovlfs_super_info_t *) (sb)->u.generic_sbp)->opt ) )
#define ovlfs_sb_xmnt(sb)		\
	( ( ovlfs_sb_opt((sb), storage_opts) & OVLFS_CROSS_MNT_PTS ) ? 1 : 0 )

#define is_ovlfs_inode(i)	(((i)->i_sb != (struct super_block *) NULL) &&\
				 ((i)->i_sb->s_magic == OVLFS_SUPER_MAGIC))

/**
 ** MACRO: copy_inode_info
 **
 ** PURPOSE: Copy the inode information that is "copy-able" from the given
 **          source inode to the given target inode.
 **
 ** NOTES:
 **	- Do NOT copy the inode number.  Once an inode is obtained with iget,
 **	  its inode number must not change until it is iput for the last time.
 **/

#define copy_inode_info(t,f)			\
	(t)->i_mode    = (f)->i_mode,		\
	(t)->i_nlink   = (f)->i_nlink,		\
	(t)->i_uid     = (f)->i_uid,		\
	(t)->i_gid     = (f)->i_gid,		\
	(t)->i_size    = (f)->i_size,		\
	(t)->i_atime   = (f)->i_atime,		\
	(t)->i_ctime   = (f)->i_ctime,		\
	(t)->i_mtime   = (f)->i_mtime,		\
	(t)->i_blksize = (f)->i_blksize,	\
	(t)->i_blocks  = (f)->i_blocks,		\
	(t)->i_version = (f)->i_version,	\
	(t)->i_nrpages = (f)->i_nrpages


/* Inode storage macros */

#define create_ino_map()    (ovlfs_ino_map_t *) MALLOC(sizeof(ovlfs_ino_map_t))

#define ovlfs_lookup_base_ino(s,d,i)	ovlfs_lookup_ino((s), (d), (i), 'b')
#define ovlfs_lookup_stg_ino(s,d,i)		ovlfs_lookup_ino((s), (d), (i), 's')


/* file info stuff */
#define file_info_filp(i)	((i) == NULL ? (struct file *) NULL : \
				               &((i)->file))
#define file_info_is_base(i)	((i) == NULL ? 0 : (i)->is_base)


/* DEBUGGING */
#if KDEBUG > 1  ||  KDEBUG_IREF
# define IPUT(i)	dprintf(9, "OVLFS: file %s, line %d: putting inode " \
			        "%ld at 0x%x count before = %d\n", \
			        __FILE__, __LINE__, \
			        (i)->i_ino, (i), (i)->i_count), iput(i);
#else
# define IPUT(i)	iput(i)
#endif

#if KDEBUG > 1  ||  KDEBUG_IREF
# define IMARK(i)	dprintf(9, "OVLFS: file %s, line %d: marking inode " \
			        "%ld at 0x%x; count before = %d\n", \
			        __FILE__, __LINE__, \
			        (i)->i_ino, (i), (i)->i_count), (i)->i_count++
#else
# define IMARK(i)	(i)->i_count++
#endif


/** DEBUG_SRC# - include source if debug level is >= # **/

#ifndef DEBUG_SRC1
# if KDEBUG
#  define DEBUG_SRC1(s)	s
# else
#  define DEBUG_SRC1(s)	
# endif
#endif

#ifndef DEBUG_SRC2
# if KDEBUG > 1
#  define DEBUG_SRC2(s)	s
# else
#  define DEBUG_SRC2(s)
# endif
#endif

#ifndef DEBUG_PRINT1
# if KDEBUG
#  define DEBUG_PRINT1(s)	printk##s
# else
#  define DEBUG_PRINT1(s)
# endif
#endif

#ifndef DEBUG_PRINT2
# if KDEBUG > 1
#  define DEBUG_PRINT2(s)	printk##s
# else
#  define DEBUG_PRINT2(s)
# endif
#endif



/** Memory allocation **/

#undef MALLOC
#undef FREE
#if VERIFY_MALLOC
# define START_ALLOC(m)	ovlfs_start_alloc(m)
# define STOP_ALLOC(m)	ovlfs_stop_alloc(m)
# define MALLOC(s)	ovlfs_alloc(s)
# if VERIFY_MALLOC > 1
#  define FREE(p)	ovlfs_free2((p), sizeof((p)[0]))
# else
#  define FREE(p)	ovlfs_free(p)
# endif
# define Free		ovlfs_free
#else
# define START_ALLOC(m)
# define STOP_ALLOC(m)
# define MALLOC(s)	kmalloc((s), GFP_KERNEL)
# define FREE(p)	kfree(p)
# define Free		kfree
#endif



/******************************************************************************
 ******************************************************************************
 **
 ** INLINE FUNCTIONS
 **
 ** NOTE:
 **	I don't really like header file functions, but this is the easiest way
 **	to inline this code and make it available across source files.
 **/

/**
 ** FUNCTION: ovlfs_chown
 **
 **  PURPOSE: Change the ownership of the given inode to the specified user
 **           id and group id.  If the user id or the group id is -1, then that
 **           attribute is unchanged.
 **
 ** NOTES:
 **	- Any filesystem whose super block does not implement write_inode is
 **	  likely to fail this update.
 **
 **	- There is no permission checking here; the main intent of this
 **	  function is to allow the ovlfs to set the ownership of files and
 **	  directories it creates to the ownership of the matching file or
 **	  directory in the base filesystem.
 **
 ** RETURNS:
 **	0	- If both id arguments are -1.
 **	1	- If successful.
 **/

static inline int	ovlfs_chown(struct inode *inode, uid_t uid, gid_t gid)
{
	int	ret;

	if ( uid != (uid_t) -1 )
		inode->i_uid = uid;
	else
		if ( gid == (gid_t) -1 )
			return	0;

	ret = 1;

	if ( gid != (gid_t) -1 )
		inode->i_gid = gid;

	if ( ( inode->i_sb != NULL ) &&
	     ( inode->i_sb->s_op != NULL ) &&
	     ( inode->i_sb->s_op->write_inode != NULL ) )
	{
		inode->i_dirt = 1;
		ret = ovlfs_lock_inode(inode);

		if ( ret >= 0 )
		{
			inode->i_sb->s_op->write_inode(inode);
			ovlfs_unlock_inode(inode);
		}
	}

	return	ret;
}


#endif
