/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_main.c
 **
 ** DESCRIPTION: This file conatins the source code for the super block
 **              operations for the overlay filesystem.  In addition, if the
 **              filesystem is compiled as a module, this file contains the
 **              module handling code.
 **
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 09/27/1997	art            	Added to source code control.
 ** 02/21/1998	ARTHUR N.	Fixed ovl_fs_read_super() when the root inode
 **				 fails to be retrieved.
 ** 02/24/1998	ARTHUR N.	Added the ovlfs_write_super() method in order
 **				 to support sync'ing of the filesystem's
 **				 information.  Also, added support for hiding
 **				 or showing of MAGIC directories and added
 **				 several comments.
 ** 02/26/1998	ARTHUR N.	Added four more inode fields to the list of
 **				 fields retained in our storage.
 ** 02/27/1998	ARTHUR N.	Added the ovlfs_notify_change() operation and
 **				 moved the superblock operations structure
 **				 to the top of the file.
 ** 02/28/1998	ARTHUR N.	Make the statfs() operation somewhat
 **				 meaningful; approximate used memory.
 ** 02/28/1998	ARTHUR N.	Set superblock to be dirty on the write_inode()
 **				 operation.
 ** 02/28/1998	ARTHUR N.	After writing the superblock, mark it as clean.
 ** 03/01/1998	ARTHUR N.	Added the maxmem option and the magic directory
 **				 name options.
 ** 03/01/1998	ARTHUR N.	Allocate GFP_KERNEL priority memory only.
 ** 03/02/1998	ARTHUR N.	Keep the inode of the storage file.
 ** 03/02/1998	ARTHUR N.	Fixed bug in ovlfs_read_super() that prevented
 **				 a failed ovlfs_stg_read_super() call from
 **				 cleaning up properly.
 ** 03/03/1998	ARTHUR N.	Move common source for cleaning up the super
 **				 block's generic information to a single func.
 **				 Also, complete the fix for the
 **				 ovlfs_read_super() problem mentioned above.
 ** 03/03/1998	ARTHUR N.	Added support for the noxmnt option.
 ** 03/03/1998	ARTHUR N.	Use default file operations for non-directory,
 **				 non-regular file inodes.
 ** 03/05/1998	ARTHUR N.	Added support for the mapping storage options,
 **				 and fixed problem with root inode not being
 **				 updated properly in the storage file.
 ** 03/05/1998	ARTHUR N.	More error messages for error conditions.
 ** 03/05/1998	ARTHUR N.	Malloc-checking support added.
 ** 03/06/1998	ARTHUR N.	Free the memory used by the name of the
 **				  storage file.  Also, added code to check the
 **				  use of memory.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 ** 03/10/1998	ARTHUR N.	Improved comments and changed printk() calls
 **				  to use the DEBUG_PRINT macros.
 ** 03/10/1998	ARTHUR N.	Set the attributes of the root inode on mount.
 ** 03/10/1998	ARTHUR N.	Mark inode as clean after writing it.
 ** 03/10/1998	ARTHUR N.	Don't set the superblock as dirty if told so
 **				 by mount options.
 ** 03/10/1998	ARTHUR N.	Added shortcuts for mount options.
 ** 03/11/1998	ARTHUR N.	Added warning for invalid mount options.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_main.c 1.25"

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#ifdef MODULE
# if (! defined(__GENKSYMS__)) || (! defined(MODVERSIONS))
#  include <linux/module.h>
# endif
#endif

#include <linux/stddef.h>

#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/locks.h>
#include <linux/malloc.h>
#include <linux/vfs.h>
#include <asm/statfs.h>
#include <asm/segment.h>

#include "ovl_fs.h"


/**
 ** SUPER BLOCK OPERATIONS:
 **/

static void	ovlfs_read_inode(struct inode *);
static int	ovlfs_notify_change(struct inode *, struct iattr *);
static void	ovlfs_write_inode(struct inode *);
static void	ovlfs_put_inode(struct inode *);
static void	ovlfs_put_super(struct super_block *);
static void	ovlfs_write_super(struct super_block *);
static void	ovlfs_statfs(struct super_block *, struct statfs *, int);

static struct super_operations	ovl_super_ops = {
	ovlfs_read_inode,
	ovlfs_notify_change,
	ovlfs_write_inode,
	ovlfs_put_inode,
	ovlfs_put_super,
	ovlfs_write_super,
	ovlfs_statfs,
	(int (*)(struct super_block *, int *, char *)) NULL  /* Don't need it */
};



			/*******************/
			/** OVLFS OPTIONS **/
			/*******************/



/**
 ** FUNCTION: ovlfs_parse_opts
 **
 **  PURPOSE: Parse the options given to the mount command.
 **/

static void	ovlfs_parse_opts (char *opt_str, ovlfs_super_info_t *opts)
{
	char	*ptr;
	char	*tmp;

	if ( opt_str == (char *) NULL )
		return;

		/* Loop through the entire string of options */

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_parse_opts(%s)\n", opt_str);
#endif

	while ( ( opt_str != (char *) NULL ) && ( opt_str[0] != '\0' ) )
	{
			/* Look for the end of the current option */

		ptr = strchr(opt_str, ',');

		if ( ptr[0] == ',' )
		{
			ptr[0] = '\0';
			ptr++;
		}

		DEBUG_PRINT2(("OVLFS:    one_option '%s'\n", opt_str));

			/* Translate the option: see if this is an */
			/*  assignment.                            */

		if ( ( tmp = strchr(opt_str, '=') ) == (char *) NULL )
		{
			if ( ( strcmp(opt_str, "nostorage") == 0 ) ||
			     ( strcmp(opt_str, "nost") == 0 ) )
				opts->storage_opts &= ~OVLFS_USE_STORAGE;
			else if ( ( strcmp(opt_str, "maxmem") == 0 ) ||
			          ( strcmp(opt_str, "mm") == 0 ) )
				opts->storage_opts |= OVLFS_STG_CK_MEM;
			else if ( ( strcmp(opt_str, "xmnt") == 0 ) ||
			          ( strcmp(opt_str, "xm") == 0 ) )
				opts->storage_opts |= OVLFS_CROSS_MNT_PTS;
			else if ( ( strcmp(opt_str, "noxmnt") == 0 ) ||
			          ( strcmp(opt_str, "noxm") == 0 ) )
				opts->storage_opts &= ~OVLFS_CROSS_MNT_PTS;
			else if ( ( strcmp(opt_str, "updmntonly") == 0 ) ||
			          ( strcmp(opt_str, "um") == 0 ) )
				opts->storage_opts |= OVLFS_UPD_UMOUNT_ONLY;
			else if ( ( strcmp(opt_str, "noupdmntonly") == 0 ) ||
			          ( strcmp(opt_str, "noum") == 0 ) )
				opts->storage_opts &= ~OVLFS_UPD_UMOUNT_ONLY;
#if STORE_MAPPINGS
			else if ( ( strcmp(opt_str, "storemaps") == 0 ) ||
			          ( strcmp(opt_str, "ma") == 0 ) )
				opts->storage_opts |= OVLFS_STORE_ALL_MAPS;
			else if ( ( strcmp(opt_str, "nostoremaps") == 0 ) ||
			          ( strcmp(opt_str, "noma") == 0 ) )
				opts->storage_opts &= ~OVLFS_STORE_ALL_MAPS;
			else if ( ( strcmp(opt_str, "basemap") == 0 ) ||
			          ( strcmp(opt_str, "bm") == 0 ) )
				opts->storage_opts |= OVLFS_STORE_BASE_MAP;
			else if ( ( strcmp(opt_str, "stgmap") == 0 ) ||
			          ( strcmp(opt_str, "sm") == 0 ) )
				opts->storage_opts |= OVLFS_STORE_STG_MAP;
			else if ( ( strcmp(opt_str, "nobasemap") == 0 ) ||
			          ( strcmp(opt_str, "nobm") == 0 ) )
				opts->storage_opts &= ~OVLFS_STORE_BASE_MAP;
			else if ( ( strcmp(opt_str, "nostgmap") == 0 ) ||
			          ( strcmp(opt_str, "nosm") == 0 ) )
				opts->storage_opts &= ~OVLFS_STORE_STG_MAP;
#endif
#if USE_MAGIC
			else if ( ( strcmp(opt_str, "hidemagic") == 0 ) ||
			          ( strcmp(opt_str, "hm") == 0 ) )
				opts->magic_opts |= OVLFS_HIDE_MAGIC;
			else if ( ( strcmp(opt_str, "magic") == 0 ) ||
			          ( strcmp(opt_str, "mg") == 0 ) )
				opts->magic_opts = OVLFS_ALL_MAGIC;
			else if ( ( strcmp(opt_str, "basemagic") == 0 ) ||
			          ( strcmp(opt_str, "mb") == 0 ) )
				opts->magic_opts |= OVLFS_BASE_MAGIC;
			else if ( ( strcmp(opt_str, "ovlmagic") == 0 ) ||
			          ( strcmp(opt_str, "mo") == 0 ) )
				opts->magic_opts |= OVLFS_OVL_MAGIC;
			else if ( ( strcmp(opt_str, "nomagic") == 0 ) ||
			          ( strcmp(opt_str, "nomg") == 0 ) )
				opts->magic_opts = OVLFS_NO_MAGIC;
			else if ( ( strcmp(opt_str, "nobasemagic") == 0 ) ||
			          ( strcmp(opt_str, "nomb") == 0 ) )
				opts->magic_opts &= ~OVLFS_BASE_MAGIC;
			else if ( ( strcmp(opt_str, "noovlmagic") == 0 ) ||
			          ( strcmp(opt_str, "nomo") == 0 ) )
				opts->magic_opts &= ~OVLFS_OVL_MAGIC;
			else if ( ( strcmp(opt_str, "showmagic") == 0 ) ||
			          ( strcmp(opt_str, "ms") == 0 ) )
				opts->magic_opts &= ~OVLFS_HIDE_MAGIC;
#endif
			else
				DEBUG_PRINT1(("OVLFS: warning: unrecognized "
				              "mount option, %s\n", opt_str));
		}
		else
		{
			tmp[0] = '\0';
			tmp++;

				/* tmp points to the value, opt_str points */
				/*  to the name.                           */

			if ( ( strcmp(opt_str, "root" ) == 0 ) ||
			     ( strcmp(opt_str, "base_root") == 0 ) ||
			     ( strcmp(opt_str, "br") == 0 ) )
				opts->base_root = tmp;
			else if ( ( strcmp(opt_str, "storage") == 0 ) ||
			          ( strcmp(opt_str, "storage_root") == 0 ) ||
			          ( strcmp(opt_str, "sr") == 0 ) )
			{
				opts->storage_opts |= OVLFS_USE_STORAGE;
				opts->storage_root = tmp;
			}
			else if ( ( strcmp(opt_str, "stg_file") == 0 ) ||
			          ( strcmp(opt_str, "sf") == 0 ) )
			{
				opts->stg_file = tmp;
			}
#if USE_MAGIC
			else if ( ( strcmp(opt_str, "magic") == 0 ) ||
			          ( strcmp(opt_str, "smagic") == 0 ) ||
			          ( strcmp(opt_str, "sn") == 0 ) )
			{
					/* remember the name; the length */
					/*  will be determined later.    */

				opts->Smagic_name = tmp;
			}
			else if ( ( strcmp(opt_str, "magicr") == 0 ) ||
			          ( strcmp(opt_str, "bmagic") == 0 ) ||
			          ( strcmp(opt_str, "bn") == 0 ) )
			{
					/* remember the name; the length */
					/*  will be determined later.    */

				opts->Bmagic_name = tmp;
			}
#endif
			else
				DEBUG_PRINT1(("OVLFS: warning: unrecognized "
				              "mount option, %s=%s\n", opt_str,
				              tmp));
		}

			/* Move to the start of the next option, if there is */
			/*  one.                                             */

		opt_str = ptr;
	}
}



			/*****************************/
			/** OVLFS SUPPORT FUNCTIONS **/
			/*****************************/



/**
 ** FUNCTION: dup_string
 **/

static inline char	*dup_string (char *str)
{
	char	*result;

	if ( str != (char *) NULL )
	{
		result = (char *) MALLOC(strlen(str) + 1);

		if ( result != (char *) NULL )
			strcpy(result, str);
	}
	else
		result = NULL;

	return	result;
}



/**
 ** FUNCTION: get_stg_file_inode
 **
 **  PURPOSE: Obtain the inode for the storage file for the filesystem.  If
 **           the file does not exist, it is created.
 **/

static struct inode	*get_stg_file_inode (const char *stg_name)
{
	struct inode	*result = INODE_NULL;
	int		fd;
	unsigned long	orig_fs;

	if ( get_inode(stg_name, &result) < 0 )
	{
			/* Try to create the file; don't modify it now */

		orig_fs = get_fs();
		set_fs(KERNEL_DS);

    		fd = open(stg_name, O_RDONLY | O_CREAT, 0644);

		if ( fd < 0 )
		{
			DEBUG_PRINT1(("OVLFS: unable to open or create storage"
			              " file, %s: error %d\n", stg_name, fd));
		}
		else
		{
			close(fd);

			if ( get_inode(stg_name, &result) < 0 )
			{
				DEBUG_PRINT1(("OVLFS: unable to obtain inode "
				              "for storage file, %s: error "
				              "%d\n", stg_name, fd));
			}
		}

		set_fs(orig_fs);
	}

	return	result;
}



/**
 ** FUNCTION: fill_in_opt_info
 **
 **  PURPOSE: Fill in any information in the options structure that has
 **           not been defined and can be determined from the other
 **           information in the structure.
 **
 ** NOTES:
 **	- Currently retrieves the root and/or storage inodes.
 **/

static void	fill_in_opt_info (struct super_block *my_sb,
		                  ovlfs_super_info_t *opts)
{
#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "fill_in_opt_info\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
	if ( opts == (ovlfs_super_info_t *) NULL )
		return;
#endif


		/* Make sure the base filesystem root is defined and obtain */
		/*  the inode, if it is not already set.                    */

	if ( opts->root_inode == INODE_NULL )
		if ( opts->base_root != (char *) NULL )
		{
			if ( get_inode(opts->base_root,
			               &opts->root_inode) < 0 )
			{
				DEBUG_PRINT1(("OVLFS: unable to obtain the "
				              "inode for the base root, %s\n",
				              opts->base_root));
			}
			else if ( opts->root_inode->i_sb == my_sb )
			{
					/* This should never happen since my */
					/*  filesystem is not yet mounted!   */

				DEBUG_PRINT1(("OVLFS: base root inode is in my"
				              " own filesystem?!?\n"));

				IPUT(opts->root_inode);
				opts->root_inode = INODE_NULL;
			}
		}
		else
			DEBUG_PRINT1(("OVLFS: overlay root not defined!\n"));


		/* Make sure the storage root is defined and obtain the */
		/*  inode, if it is not already set.                    */

	if ( opts->storage_inode == INODE_NULL )
		if ( opts->storage_root != (char *) NULL )
		{
			if ( get_inode(opts->storage_root,
			               &opts->storage_inode) < 0 )
			{
				DEBUG_PRINT1(("OVLFS: unable to obtain the "
				              "inode for the storage root, "
				              "%s\n", opts->storage_root));
			}
			else if ( opts->storage_inode->i_sb == my_sb )
			{
					/* This should never happen since my */
					/*  filesystem is not yet mounted!   */

				DEBUG_PRINT1(("OVLFS: storage root inode is in"
				              " my own filesystem?!?\n"));

				IPUT(opts->storage_inode);
				opts->storage_inode = INODE_NULL;
			}
		}
		else
		{
#if KDEBUG
# if KDEBUG < 2
				/* If the debug level is not too high, only */
				/*  warn the user about the missing storage */
				/*  root if the storage option is set.      */

			if ( opt_use_storage_p(opts[0]) )
# endif
				printk("OVLFS: storage root not defined!\n");
#endif
		}


		/* If the storage file was specified, get its inode. */

	if ( opts->stg_file_inode == INODE_NULL )
		if ( opts->stg_file != (char *) NULL )
		{
			opts->stg_file_inode =
				get_stg_file_inode(opts->stg_file);
		}
		else
		{
			DEBUG_PRINT1(("OVLFS: no storage file specified\n"));
		}


#if USE_MAGIC
		/* Make sure the magic directory names are defined */

	if ( ( opts->Smagic_name == (char *) NULL ) ||
	     ( opts->Smagic_name[0] == '\0' ) )
		opts->Smagic_name = OVLFS_MAGIC_NAME;

	if ( ( opts->Bmagic_name == (char *) NULL ) ||
	     ( opts->Bmagic_name[0] == '\0' ) )
		opts->Bmagic_name = OVLFS_MAGICR_NAME;


		/* Determine the length of the names */

	opts->Smagic_len = strlen(opts->Smagic_name);
	opts->Bmagic_len = strlen(opts->Bmagic_name);
#endif
}



/**
 ** FUNCTION: dup_ovlfs_opts
 **
 **  PURPOSE: Allocate memory for a copy of an ovlfs_opt structure and its
 **           members, and then copy the information from the given structure
 **           to this newly allocated space.
 **
 ** NOTES:
 **	- Any inodes referenced by the given structure do NOT have their
 **	  reference counts updated; if the caller does not "replace" the
 **	  original with the duplicate, the counts must be updated so that both
 **	  copies of the structure are indicated as referring to the inodes.
 **/

static ovlfs_super_info_t	*dup_ovlfs_opts (struct super_block *my_sb,
				                 ovlfs_super_info_t *opts)
{
	ovlfs_super_info_t	*result;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "dup_ovlfs_opts\n");
#endif

	result = (ovlfs_super_info_t *) MALLOC(sizeof(ovlfs_super_info_t));

	if ( result == (ovlfs_super_info_t *) NULL )
	{
		DEBUG_PRINT1(("OVLFS: unable to allocate %d bytes\n",
		              sizeof(ovlfs_super_info_t)));

		return	(ovlfs_super_info_t *) NULL;
	}

		/* Initialize all pointers to NULL and integers to 0 */

	memset(result, '\0', sizeof(ovlfs_super_info_t));

	if ( opts != (ovlfs_super_info_t *) NULL )
	{
			/* Duplicate the names of the two root directories */

		if ( opts->base_root != (char *) NULL )
		{
			result->base_root = dup_string(opts->base_root);

			if ( result->base_root == (char *) NULL )
			{
				DEBUG_PRINT1(("OVLFS: dup_ovlfs_opts: unable "
				              "to dup string %s\n",
				              opts->base_root));

				FREE(result);
				return	(ovlfs_super_info_t *) NULL;
			}
		}
		else
			result->base_root = (char *) NULL;

		if ( opts->storage_root != (char *) NULL )
		{
			result->storage_root = dup_string(opts->storage_root);

			if ( result->storage_root == (char *) NULL )
			{
				DEBUG_PRINT1(("OVLFS: dup_ovlfs_opts: unable "
				              "to dup string %s\n",
				              result->storage_root));

				if ( result->base_root != (char *) NULL )
					FREE(result->base_root);

				FREE(result);
				return	(ovlfs_super_info_t *) NULL;
			}
		}
		else
			result->storage_root = (char *) NULL;


		if ( opts->stg_file != (char *) NULL )
		{
			result->stg_file = dup_string(opts->stg_file);

			if ( result->stg_file == (char *) NULL )
			{
				DEBUG_PRINT1(("OVLFS: dup_ovlfs_opts: unable "
				              "to dup string %s\n",
				              result->stg_file));

				if ( result->base_root != (char *) NULL )
					FREE(result->base_root);

				if ( result->storage_root != (char *) NULL )
					FREE(result->storage_root);

				FREE(result);
				return	(ovlfs_super_info_t *) NULL;
			}
		}
		else
			result->stg_file = (char *) NULL;


			/* Obtain the inodes of the two root directories */

		if ( opts->root_inode == INODE_NULL )
		{
			result->root_inode = INODE_NULL;

				/* Get the base filesystem root's inode; */
				/*  use the original string in case the  */
				/*  string copy failed.                  */

			if ( opts->base_root != (char *) NULL )
			{
				if ( get_inode(opts->base_root,
				               &result->root_inode) < 0 )
				{
					DEBUG_PRINT1(("OVLFS: unable to obtain"
					              " base root's inode\n"));
				}
				else if ( result->root_inode->i_sb == my_sb )
				{
						/* The result is not allowed */
						/*  to be in my own fs!      */ 

					DEBUG_PRINT1(("OVLFS: base root's (%s)"
					              " fs is my fs!?!\n",
					              opts->base_root));

					IPUT(result->root_inode);
					result->root_inode = INODE_NULL;
				}
			}
		}
		else
			result->root_inode = opts->root_inode;

		if ( opts->storage_inode == INODE_NULL )
		{
			result->storage_inode = INODE_NULL;

				/* Get the storage filesystem root's inode; */
				/*  use the original string in case the     */
				/*  string copy failed.                     */

			if ( result->storage_root != (char *) NULL )
			{
				if ( get_inode(opts->storage_root,
					       &result->storage_inode) < 0 )
				{
					DEBUG_PRINT1(("OVLFS: unable to obtain"
					              " inode for storage root"
					              " %s\n",
					              opts->storage_root));
				}
				else if ( result->storage_inode->i_sb ==
				          my_sb )
				{
						/* The result is not allowed */
						/*  to be in my own fs!      */

					DEBUG_PRINT1(("OVLFS: storage root's "
					              "(%s) fs is my fs!?!\n",
					              opts->storage_root));

					IPUT(result->root_inode);
					result->root_inode = INODE_NULL;
				}
			}
		}
		else
			result->storage_inode = opts->storage_inode;

			/* Copy the storage file inode */

		result->stg_file_inode = opts->stg_file_inode;

			/* Copy the option flags */

		result->storage_opts = opts->storage_opts;
#if USE_MAGIC
		result->magic_opts   = opts->magic_opts;

			/* It is not necessary to error these dup_string()'s */
			/*  if they fail since these names are checked later */

		if ( opts->Smagic_name != NULL )
		{
			result->Smagic_name = dup_string(opts->Smagic_name);
			result->Smagic_len = opts->Smagic_len;
		}

		if ( opts->Bmagic_name != NULL )
		{
			result->Bmagic_name = dup_string(opts->Bmagic_name);
			result->Bmagic_len = opts->Bmagic_len;
		}
#endif

		DEBUG_PRINT2(("OVLFS: root inode at 0x%x\n",
		              result->root_inode));
		DEBUG_PRINT2(("OVLFS: storage inode at 0x%x\n",
		              result->storage_inode));
	}
	else
	{
#if USE_MAGIC
		result->magic_opts = OVLFS_ALL_MAGIC;
#endif
	}

	return	result;
}



/**
 ** FUNCTION: free_super_info
 **
 **  PURPOSE: Free the memory allocated to the given option structure and its
 **           members.
 **
 ** NOTES:
 **	- The memory used by the inodes, base_to_ovl, and stg_to_ovl members
 **	  of the super block information structure must be freed before
 **	  calling this function.  This is done by the function
 **	  ovlfs_stg_put_super().
 **/

static void	free_super_info (ovlfs_super_info_t *sup_info)
{
#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "free_super_info\n");
#endif

	if ( sup_info != (ovlfs_super_info_t *) NULL )
	{
		if ( sup_info->base_root != (char *) NULL )
			FREE(sup_info->base_root);

		if ( sup_info->storage_root != (char *) NULL )
			FREE(sup_info->storage_root);

		if ( sup_info->stg_file != (char *) NULL )
			FREE(sup_info->stg_file);

#if USE_MAGIC
		if ( sup_info->Smagic_name != (char *) NULL )
			FREE(sup_info->Smagic_name);

		if ( sup_info->Bmagic_name != (char *) NULL )
			FREE(sup_info->Bmagic_name);
#endif

#if ! FASTER_AND_MORE_DANGEROUS
		memset(sup_info, '\0', sizeof(ovlfs_super_info_t));
#endif

		FREE(sup_info);
	}
}




/**
 ** FUNCTION: dup_inode_info
 **
 **  PURPOSE: Duplicate the inode information structure given.  The result
 **           must be freed with free_inode_info().
 **/

static ovlfs_inode_info_t	*dup_inode_info (ovlfs_inode_info_t *i_info)
{
	ovlfs_inode_info_t	*result;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "dup_inode_info\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS

	if ( i_info == (ovlfs_inode_info_t *) NULL )
		return	(ovlfs_inode_info_t *) NULL;

#endif

	result = (ovlfs_inode_info_t *) MALLOC(sizeof(ovlfs_inode_info_t));

	if ( result != (ovlfs_inode_info_t *) NULL )
	{
		if ( ( i_info->filename != NULL ) &&
		     ( i_info->name_len > 0 ) )
		{
			result->filename = MALLOC(i_info->name_len);

			if ( result->filename == (char *) NULL )
			{
				FREE(result);
				return	(ovlfs_inode_info_t *) NULL;
			}

			result->name_len = i_info->name_len;
		}
		else
		{
			result->filename = (char *) NULL;
			result->name_len = 0;
		}

#if STORE_REF_INODES
		result->ovlfs_dir     = i_info->ovlfs_dir;
		result->base_inode    = i_info->base_inode;
		result->overlay_inode = i_info->overlay_inode;
#else
		if ( i_info->base_name != (char *) NULL )
		{
			result->base_name = dup_string(i_info->base_name);

			if ( result->base_name == (char *) NULL )
			{
				if ( result->filename != (char *) NULL )
					FREE(result->filename);

				FREE(result);
				return	(ovlfs_inode_info_t *) NULL;
			}
		}
		else
			result->base_name = (char *) NULL;

		if ( i_info->overlay_name != (char *) NULL )
		{
			result->overlay_name = dup_string(i_info->overlay_name);
			if ( result->overlay_name == (char *) NULL )
			{
				if ( result->filename != (char *) NULL )
					FREE(result->filename);

				if ( result->base_name != (char *) NULL )
					FREE(result->base_name);

				FREE(result);
				return	(ovlfs_inode_info_t *) NULL;
			}
		}
		else
			result->overlay_name = (char *) NULL;
#endif
	}

	return	result;
}



/**
 ** FUNCTION: free_inode_info
 **
 **  PURPOSE: Free the kernel memory allocated to the given inode information
 **           structure.
 **/

static void	free_inode_info (ovlfs_inode_info_t *i_info)
{
#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "free_inode_info\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS

	if ( i_info == (ovlfs_inode_info_t *) NULL )
	{
		DEBUG_PRINT1(("OVLFS: free_inode_info called with NULL "
		              "pointer\n"));
		return;
	}

#endif

	if ( i_info->filename != (char *) NULL )
		FREE(i_info->filename);

#if ! STORE_REF_INODES
	if ( i_info->base_name != (char *) NULL )
		FREE(i_info->base_name);

	if ( i_info->overlay_name != (char *) NULL )
		FREE(result->overlay_name);
#endif

	FREE(i_info);
}



		/**********************************/
		/** OVLFS SUPER BLOCK OPERATIONS **/
		/**********************************/


/**
 ** FUNCTION: ovlfs_prepare_inode
 **
 **  PURPOSE: Prepare the inode for use in the overlay filesystem.
 **/

void	ovlfs_prepare_inode (struct inode *rd_ino, int type)
{
	ovlfs_inode_info_t	inode_info;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_prepare_inode(%ld, %d)\n",
	       rd_ino->i_ino, type);
#endif

		/* Set the entire structure to 0's (pointers to NULL) */

	memset(&inode_info, '\0', sizeof(ovlfs_inode_info_t));

	switch ( type )
	{
	    case OVLFS_ROOT_INO:
		if ( ( rd_ino->i_sb != (struct super_block *) NULL ) &&
		     ( rd_ino->i_sb->u.generic_sbp != NULL ) )
		{
			ovlfs_super_info_t	*opts;

			opts = (ovlfs_super_info_t *)
			       rd_ino->i_sb->u.generic_sbp;

#if STORE_REF_INODES
				/* Get the inodes from the other fs's */

			inode_info.base_inode     = opts->root_inode;
			inode_info.overlay_inode  = opts->storage_inode;

				/* Mark the inodes as used so that we don't */
				/*  lose them until this inode is put.      */

			if ( inode_info.base_inode != INODE_NULL )
				IMARK(inode_info.base_inode);

			if ( inode_info.overlay_inode != INODE_NULL )
				IMARK(inode_info.overlay_inode);

# if KDEBUG > 2
			if ( inode_info.base_inode != INODE_NULL )
				printk("OVLFS: ovlfs_prepare_inode: ROOT "
				       "INODE base is inode %d at 0x%x\n",
				       inode_info.base_inode->i_ino,
				       inode_info.base_inode);
			else
				printk("OVLFS: ovlfs_prepare_inode: ROOT "
				       "INODE base is null\n");
# endif
#else
			inode_info.base_name = opts->base_root;
			inode_info.overlay_name = opts->storage_root;
#endif
		}
		else
		{
			DEBUG_PRINT1(("OVLFS: strange, super block info "
			              "missing\n"));
		}

		rd_ino->i_mode = S_IFDIR | S_IRUGO | S_IXUGO;
		rd_ino->i_nlink = 2;
		break;

	    case OVLFS_DIR_INO:
/*
		rd_ino->i_op = &ovlfs_file_ino_ops;
*/

		if ( rd_ino->i_mode == 0 )
			rd_ino->i_mode = S_IFDIR | S_IRUGO | S_IXUGO;
		rd_ino->i_nlink = 2;
		break;

	    case OVLFS_FILE_INO:
		rd_ino->i_op = &ovlfs_file_ino_ops;

		if ( rd_ino->i_mode == 0 )
			rd_ino->i_mode = S_IFREG | S_IRUGO | S_IXUGO;
		break;

	    case OVLFS_GENERIC_INO:
		rd_ino->i_op = &ovlfs_file_ino_ops;
		break;

	    default:
		DEBUG_PRINT1(("OVLFS: unrecognized inode type %d\n", type));

			/* Don't set the generic_ip if type is not recognized */
		return;
	}

		/* If the inode already has inode info, just copy it in */

	if ( rd_ino->u.generic_ip == NULL )
		rd_ino->u.generic_ip = (void *) dup_inode_info(&inode_info);
	else
		memcpy(rd_ino->u.generic_ip, &inode_info, sizeof(inode_info));
}



/**
 ** FUNCTION: set_root_inode_attr
 **
 **  PURPOSE: Given the root inode of the overlay filesystem for the first
 **           time, copy the attributes of the storage or base filesystem's
 **           root inode into the pseudo filesystem's root inode.
 **/

static void	set_root_inode_attr (struct inode *inode,
		                     ovlfs_inode_t *ino)
{
	struct inode	*o_inode;
	nlink_t		cnt;
	int		ret;

		/* See if the storage information has already been used, */
		/*  and use it if not.                                   */

	if ( ino->stg_dev == NODEV )
	{
		ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

		if ( ret >= 0 )
		{
			ino->stg_dev = o_inode->i_dev;
			ino->stg_ino = o_inode->i_ino;

			cnt = inode->i_nlink;	/* maintain the link count */
			copy_inode_info(inode, o_inode);
			inode->i_nlink = cnt;

			IPUT(o_inode);
		}
		else if ( ino->base_dev == NODEV )
		{
			ret = ovlfs_resolve_base_inode(inode, &o_inode);

			if ( ret >= 0 )
			{
				ino->stg_dev = o_inode->i_dev;
				ino->stg_ino = o_inode->i_ino;

					/* maintain the link count */
				cnt = inode->i_nlink;
				copy_inode_info(inode, o_inode);
				inode->i_nlink = cnt;

				IPUT(o_inode);
			}
			else
				DEBUG_PRINT1(("OVLFS: set_root_inode_attr: "
				              "unable to obtain the storage "
					      "or base inode! (%d)\n", ret));
		}
	}
}



/**
 ** FUNCTION: ovlfs_read_inode
 **
 **  PURPOSE: Implement the read_inode super-block operation.
 **/

static void	ovlfs_read_inode (struct inode *rd_ino)
{
	ovlfs_inode_t	*ino;
	int		ret;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_read_inode(%ld at 0x%x)\n",
	       rd_ino->i_ino, (int) rd_ino);
#endif

	rd_ino->i_op = &ovlfs_ino_ops;
	rd_ino->i_mode = 0;
	rd_ino->i_uid = 0;
	rd_ino->i_gid = 0;
	rd_ino->i_nlink = 1;
	rd_ino->i_size = 0;
	rd_ino->i_mtime = CURRENT_TIME;
	rd_ino->i_atime = CURRENT_TIME;
	rd_ino->i_ctime = CURRENT_TIME;
	rd_ino->i_blocks = 0;
	rd_ino->i_blksize = 1024;
	rd_ino->u.generic_ip = NULL;

	if ( rd_ino->i_ino == OVLFS_ROOT_INO )
	{
			/* Fill in the inodes information structure */

		ovlfs_prepare_inode(rd_ino, OVLFS_ROOT_INO);



			/* Read the directory information; the number of   */
			/*  entries in the directory is returned if        */
			/*  successful; set nlink to 2 + ret, unless magic */
			/*  is compiled in, in which case, use 4 + ret.    */

#if USE_MAGIC
		rd_ino->i_nlink = 4;
#else
		rd_ino->i_nlink = 2;
#endif

		ret = ovlfs_read_directory(rd_ino);

		if ( ret < 0 )
		{
			DEBUG_PRINT1(("OVLFS: ovlfs_read_inode: read root "
			              "directory returned error %d\n", ret));
		}
		else
		{
				/* Find this inode's info from storage */

			ino = ovlfs_get_ino(rd_ino->i_sb, rd_ino->i_ino);

			if ( ino == (ovlfs_inode_t *) NULL )
			{
				ret = -ENOENT;

				DEBUG_PRINT1(("OVLFS: ovlfs_read_inode: unable"
				              "to obtain the root inode's info"
				              " from storage\n"));
			}
			else
			{
				rd_ino->i_nlink += ret;

					/* Set the attributes for the root */
				set_root_inode_attr(rd_ino, ino);

					/* Update the stg info for this inode */

				ino->mode    = rd_ino->i_mode;
				ino->size    = rd_ino->i_size;
				ino->uid     = rd_ino->i_uid;
				ino->gid     = rd_ino->i_gid;
				ino->atime   = rd_ino->i_atime;
				ino->mtime   = rd_ino->i_mtime;
				ino->ctime   = rd_ino->i_ctime;
				ino->nlink   = rd_ino->i_nlink;
				ino->blksize = rd_ino->i_blksize;
				ino->blocks  = rd_ino->i_blocks;
				ino->version = rd_ino->i_version;
				ino->nrpages = rd_ino->i_nrpages;
			}
		}
	}
	else
	{
			/* Find this inode's information from storage */
		ino = ovlfs_get_ino(rd_ino->i_sb, rd_ino->i_ino);

		if ( ino != (ovlfs_inode_t *) NULL )
		{
			rd_ino->i_mode    = ino->mode;
			rd_ino->i_size    = ino->size;
			rd_ino->i_uid     = ino->uid;
			rd_ino->i_gid     = ino->gid;
			rd_ino->i_atime   = ino->atime;
			rd_ino->i_mtime   = ino->mtime;
			rd_ino->i_ctime   = ino->ctime;
			rd_ino->i_nlink   = ino->nlink;
			rd_ino->i_blksize = ino->blksize;
			rd_ino->i_blocks  = ino->blocks;
			rd_ino->i_version = ino->version;
			rd_ino->i_nrpages = ino->nrpages;

			if ( S_ISBLK(ino->mode) || S_ISCHR(ino->mode) )
				rd_ino->i_rdev = ino->u.rdev;
			else
				rd_ino->i_rdev = 0;
		}

			/* If the inode is a directory, prepare it and fill */
			/*  in its contents.                                */

		if ( S_ISDIR(rd_ino->i_mode) )
		{
			ovlfs_prepare_inode(rd_ino, OVLFS_DIR_INO);

			ret = ovlfs_read_directory(rd_ino);

			if ( ret < 0 )
			{
				DEBUG_PRINT1(("OVLFS: ovlfs_read_inode: read"
				              " ovlfs directory, %ld, returned"
				              " error %d\n", rd_ino->i_ino,
				              ret));
			}
			else
				rd_ino->i_nlink += ret;
		}
		else if ( S_ISREG(rd_ino->i_mode) )
			ovlfs_prepare_inode(rd_ino, OVLFS_FILE_INO);
		else
			ovlfs_prepare_inode(rd_ino, OVLFS_GENERIC_INO);
	}
}



/**
 ** FUNCTION: ovlfs_notify_change
 **
 **  PURPOSE: This if the overlay filesystem's notify_change() superblock
 **           operation.
 **/

static int	ovlfs_notify_change (struct inode *inode, struct iattr *attr)
{
	struct inode	*o_inode;
	int		ret;

	if ( ! is_ovlfs_inode(inode) )
		return	-ENOENT;

		/* If an storage inode exists, update it; otherwise, just */
		/*  update our own inode's information.                   */

	ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

	if ( ret >= 0 )
	{
		if ( ( o_inode->i_sb != NULL ) &&
		     ( o_inode->i_sb->s_op != NULL ) &&
		     ( o_inode->i_sb->s_op->notify_change != NULL ) )
		{
			down(&(o_inode->i_sem));
			ret = o_inode->i_sb->s_op->notify_change(o_inode, attr);
			up(&(o_inode->i_sem));
		}
		else
		{
			ret = inode_change_ok(o_inode, attr);

			if ( ret == 0 )
				inode_setattr(o_inode, attr);
		}

		IPUT(o_inode);
	}
	else
		ret = 0;	/* It is OK to not find the OVL inode */


		/* Now update the attributes of the psuedo-fs' inode */

	if ( ret >= 0 )
	{
		ret = inode_change_ok(inode, attr);

		if ( ret == 0 )
			inode_setattr(inode, attr);
	}

	return	ret;
}



/**
 ** FUNCTION: ovlfs_write_inode
 **
 **  PURPOSE: Implement the write_inode super-block operation.
 **/

static void	ovlfs_write_inode (struct inode *rd_ino)
{
	ovlfs_inode_t	*ino;

#if ! FASTER_AND_MORE_DANGEROUS
	if ( ( rd_ino == INODE_NULL ) || ( rd_ino->i_sb == NULL ) ||
	     ( ! is_ovlfs_inode(rd_ino) ) )
	{
		DEBUG_PRINT1(("OVLFS: ovlfs_read_inode: invalid inode given at"
		              " 0x%x\n", (int) rd_ino));

		return;
	}
#endif

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_read_inode(%ld at 0x%x)\n",
	       rd_ino->i_ino, (int) rd_ino);
#endif

		/* Obtain the storage for the given inode */

	ino = ovlfs_get_ino(rd_ino->i_sb, rd_ino->i_ino);

	if ( ino != (ovlfs_inode_t *) NULL )
	{
			/* Update the attributes of the inode */

		ino->mode    = rd_ino->i_mode;
		ino->size    = rd_ino->i_size;
		ino->uid     = rd_ino->i_uid;
		ino->gid     = rd_ino->i_gid;
		ino->atime   = rd_ino->i_atime;
		ino->mtime   = rd_ino->i_mtime;
		ino->ctime   = rd_ino->i_ctime;
		ino->nlink   = rd_ino->i_nlink;
		ino->blksize = rd_ino->i_blksize;
		ino->blocks  = rd_ino->i_blocks;
		ino->version = rd_ino->i_version;
		ino->nrpages = rd_ino->i_nrpages;

		if ( S_ISBLK(ino->mode) || S_ISCHR(ino->mode) )
			ino->u.rdev = rd_ino->i_rdev;
		else
			ino->u.generic = NULL;


			/* Force the superblock to get updated before long */
			/*  unless told not to.                            */

		if ( ! ( ovlfs_sb_opt(rd_ino->i_sb, storage_opts) &
		         OVLFS_UPD_UMOUNT_ONLY ) )
		{
			rd_ino->i_sb->s_dirt = 1;
		}
	}
	else
		DEBUG_PRINT1(("OVLFS: write_inode: inode %d, device 0x%x: "
		              "unable to find storage info\n",
		              (int) rd_ino->i_ino, (int) rd_ino->i_dev));

	rd_ino->i_dirt = 0;
}



/**
 ** FUNCTION: ovlfs_put_inode
 **
 **  PURPOSE: Perform the fs put_inode operation for an ovlfs super block.
 **           The main purpose of the put_inode operation is to remove an
 **           inode/file from the filesystem when it is no longer used
 **           (i.e. when i_nlink = 0).
 **/

static void	ovlfs_put_inode (struct inode *an_inode)
{
	ovlfs_inode_info_t	*infop;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_put_inode(%ld)\n", an_inode->i_ino);
#endif

	if ( an_inode->u.generic_ip != NULL )
	{
		infop = (ovlfs_inode_info_t *) an_inode->u.generic_ip;

			/* Reset the inode's pointer before releasing its */
			/*  memory in order to prevent problems caused by */
			/*  obtaining the same inode from the inode hash. */

		an_inode->u.generic_ip = NULL;

#if STORE_REF_INODES
		if ( infop->ovlfs_dir != INODE_NULL )
			IPUT(infop->ovlfs_dir);

		if ( infop->base_inode != INODE_NULL )
			IPUT(infop->base_inode);

		if ( infop->overlay_inode != INODE_NULL )
			IPUT(infop->overlay_inode);
#endif

			/* Free the memory allocated to the inode's */
			/*  information structure.                  */

		free_inode_info(infop);
	}

	if ( an_inode->i_nlink == 0 )
		an_inode->i_size = 0;

		/* Prevent the inode from being re-used out of the pool  */
		/*  without doing a read_inode().  Before that, however, */
		/*  check if the inode is dirty and call write_inode()   */
		/*  if it is so that changes are not lost.               */

	if ( an_inode->i_dirt )
		ovlfs_write_inode(an_inode);

	clear_inode(an_inode);
}



/**
 ** FUNCTION: cleanup_super
 **
 **  PURPOSE: Cleanup the given superblock when an error occurs while
 **           reading it.
 **/

static void	cleanup_super (struct super_block *sb)
{
	ovlfs_super_info_t *s_info;

	if ( ( sb == NULL ) || ( sb->u.generic_sbp == NULL ) )
		return;

	s_info = (ovlfs_super_info_t *) sb->u.generic_sbp;

	if ( s_info->root_inode != INODE_NULL )
		IPUT(s_info->root_inode);

	if ( s_info->storage_inode != INODE_NULL )
		IPUT(s_info->storage_inode);

	if ( s_info->stg_file_inode != INODE_NULL )
		IPUT(s_info->stg_file_inode);


		/* Free the memory allocated for the generic_sbp structure */

	free_super_info(s_info);
	sb->u.generic_sbp = NULL;
}



/**
 ** FUNCTION: ovlfs_put_super
 **
 **  PURPOSE: Perform the fs put_super operation for an ovlfs super block.
 **/

static void	ovlfs_put_super (struct super_block *sup_blk)
{
#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_put_super\n");
#endif

	lock_super(sup_blk);
	sup_blk->s_dev = 0;

		/* Release the overlay and storage inodes */

	if ( sup_blk->u.generic_sbp != NULL )
	{
			/* Put the storage for the super block */

		ovlfs_stg_put_super(sup_blk);

		cleanup_super(sup_blk);
	}

	unlock_super(sup_blk);

	STOP_ALLOC("ovlfs_put_super");

#if MODULE
	MOD_DEC_USE_COUNT;
#endif

#if KDEBUG_CALLS
	printk("OVLFS: put super returning\n");
#endif
}



/**
 ** FUNCTION: ovlfs_write_super
 **
 **  PURPOSE: Perform the fs write_super operation for an ovlfs superblock.
 **/

static void	ovlfs_write_super (struct super_block *sup_blk)
{
	int	ret;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_write_super 0x%x\n", (int) sup_blk);
#endif

	lock_super(sup_blk);

	if ( ( ret = ovlfs_stg_write_super(sup_blk) ) >= 0 )
		sup_blk->s_dirt = 0;
	else
		DEBUG_PRINT1(("OVLFS: error %d returned by "
		              "ovlfs_stg_write_super\n", ret));

	unlock_super(sup_blk);

#if KDEBUG_CALLS
	printk("OVLFS: write super (0x%x) returning\n", (int) sup_blk);
#endif
}



/**
 ** FUNCTION: ovlfs_statfs
 **
 **  PURPOSE: Perform the fs statfs operation for an ovlfs super block.
 **
 ** NOTES:
 **	- The buffer is used to allows the values to be assigned in kernel
 **	  space before using memcpy_tofs() to copy them back to user space.
 **	  The altenative is to find the address of each member of the
 **	  structure in user space and use the put_user() macro.
 **
 ** TODO:
 **	- Make this stat the storage area to determine the space
 **	  available/used.
 **/

static void	ovlfs_statfs (struct super_block *sup_blk,
		              struct statfs *fs_info, int size)
{
	struct statfs		buf;
	ovlfs_super_info_t	*s_info;
	struct sysinfo		mem_info;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovlfs_statfs\n");
#endif

	if ( size < sizeof(struct statfs) )
	{
		DEBUG_PRINT1(("OVLFS: statfs structure size mismatch; called"
		              " size is %d, expected %d\n", size,
		              sizeof(struct statfs)));
		return;
	}

		/* How do we measure available and used blocks without */
		/*  some storage space?  For now, use the available    */
		/*  free memory as available space and calculate an    */
		/*  approximate memory use based on the number of      */
		/*  inodes "owned" by the psuedo filesystem.           */

	si_meminfo(&mem_info);

	buf.f_type = OVLFS_SUPER_MAGIC;
	buf.f_bsize = 1024;
	buf.f_bfree = 0;
	buf.f_files = 0;
	buf.f_ffree = 0xFF;
	buf.f_namelen = NAME_MAX;

		/* Just set available space to the same as free space; for */
		/*  memory, it is all the same...                          */

	buf.f_bfree = mem_info.freeram / BLOCK_SIZE;
	buf.f_bavail = buf.f_bfree;

	if ( ( sup_blk != (struct super_block *) NULL ) &&
	     ( sup_blk->u.generic_sbp != NULL ) )
	{
		s_info = (ovlfs_super_info_t *) sup_blk->u.generic_sbp;

		if ( s_info->inodes == NULL )
			buf.f_blocks = buf.f_bfree;
		else
		{
			int	result;
			int	n_ino;

			n_ino = klist_func(list_length)(s_info->inodes);
			result = (n_ino * sizeof(ovlfs_inode_t)) / BLOCK_SIZE;

				/* Set the total number of blocks to the */
				/*  free space plus the used space.      */

			buf.f_blocks = buf.f_bfree + result;
		}
	}

	memcpy_tofs(fs_info, &buf, sizeof(struct statfs));

#if KDEBUG > 1
	printk("OVLFS: ovlfs_statfs returning\n");
#endif
}



/**
 ** FUNCTION: ovl_fs_read_super
 **
 **  PURPOSE: Fill in the superblock information for the given superblock.
 **           This function is called to perform a mount().
 **/

static struct super_block	*ovl_fs_read_super (struct super_block *sup_blk,
				                    void *options, int silent)
{
	ovlfs_super_info_t	sup_info;
	struct inode		*mnt_pt;

#if KDEBUG_CALLS
	printk(KDEBUG_CALL_PREFIX "ovl_fs_read_super\n");
#endif

		/* Increment the use count before doing any real work   */
		/*  so that the module can not be unloaded while in use */

#if MODULE
	MOD_INC_USE_COUNT;
#endif

	START_ALLOC("ovlfs_read_super");

	lock_super(sup_blk);
	sup_blk->s_op = &ovl_super_ops;

	sup_blk->s_magic = OVLFS_SUPER_MAGIC;

	DEBUG_PRINT2(("OVLFS: device for super block = %d\n", sup_blk->s_dev));


		/* Initialize all pointers to NULL and integers to 0 */

	memset(&sup_info, '\0', sizeof(sup_info));

	sup_info.storage_opts  = OVLFS_STG_DEF_OPTS;
	sup_info.magic_opts    = OVLFS_ALL_MAGIC;

	ovlfs_parse_opts(options, &sup_info);
	fill_in_opt_info(sup_blk, &sup_info);
	sup_blk->u.generic_sbp = (void *) dup_ovlfs_opts(sup_blk, &sup_info);

	if ( sup_blk->u.generic_sbp == NULL )
	{
#if MODULE
		MOD_DEC_USE_COUNT;
#endif

		DEBUG_PRINT1(("OVLFS: ovlfs_read_super: unable to allocate "
		              "memory needed for the super block\n"));

		cleanup_super(sup_blk);
		unlock_super(sup_blk);
		STOP_ALLOC("ovlfs_read_super");
		return	(struct super_block *) NULL;
	}

		/* Read and setup any storage information associated */
		/*  with the super block.                            */

	if ( ovlfs_stg_read_super(sup_blk) < 0 )
	{
#if MODULE
		MOD_DEC_USE_COUNT;
#endif

		DEBUG_PRINT1(("OVLFS: ovlfs_read_super: error reading storage "
		              "information for the super block\n"));

			/* Need to clean up resources "allocated" above, */
			/*  unless the super block will be "put"...      */

		cleanup_super(sup_blk);
		unlock_super(sup_blk);
		STOP_ALLOC("ovlfs_read_super");
		return	(struct super_block *) NULL;
	}

		/* Get the root inode */

	mnt_pt = iget(sup_blk, OVLFS_ROOT_INO);

	if ( mnt_pt != INODE_NULL )
	{
		sup_blk->s_mounted = mnt_pt;

		DEBUG_PRINT2(("OVLFS: mount point: dev %d, inode %d\n",
		              mnt_pt->i_dev, mnt_pt->i_ino));
	}
	else
	{
		DEBUG_PRINT2(("OVLFS: unable to obtain inode for the mount "
		              "point.\n"));

#if MODULE
		MOD_DEC_USE_COUNT;
#endif

			/* Need to clean up resources "allocated" above, */
			/*  unless the super block will be "put"...      */

		cleanup_super(sup_blk);
		unlock_super(sup_blk);

		STOP_ALLOC("ovlfs_read_super");
		return	(struct super_block *) NULL;
	}

	unlock_super(sup_blk);

#if KDEBUG_CALLS
	DEBUG_PRINT2(("OVLFS: ovl_fs_read_super returning\n"));
#endif

	return	sup_blk;
}



/**
 ** FUNCTION: init_ovl_fs
 **
 **  PURPOSE: Register the overlay filesystem with the kernel.
 **/

static struct file_system_type	ovl_fs = { ovl_fs_read_super, "ovl", 0, NULL };

int	init_ovl_fs ()
{
	int	err_code;

	err_code = register_filesystem(&ovl_fs);

	return	err_code;
}



			/**** MODULE INTERFACE ****/

#ifdef MODULE

#if 1

/**
 ** Symbol Table: define symbols (versioned symbols) that are exported from or
 **               needed by this module.
 **/

static struct symbol_table	ovlfs_symtab = {
#include <linux/symtab_begin.h>
	X(init_ovl_fs),
	X(ovlfs_end_debug),
	X(ovlfs_dprintf),
	X(ovlfs_resolve_inode),
	X(ovlfs_debug_inodes),
	X(ovlfs_copy_file),
	X(ovlfs_prepare_inode),
	X(ovlfs_ino_find_dirent),
#include <linux/symtab_end.h>
} ;
#endif

int	init_module ()
{
	int	err_code;

#if KDEBUG > 1 || OVLFS_BANNER
	printk("starting ovl fs module (compiled %s)\n", macro2str(DATE_STAMP));
#endif

	START_ALLOC("init_module");

	err_code = init_ovl_fs();

	if ( err_code >= 0 )
	{
		register_symtab(&ovlfs_symtab);

		ovlfs_setup_syscalls();
	}

	return	err_code;
}

void	cleanup_module()
{
	STOP_ALLOC("cleanup_module");

#if KDEBUG > 1 || OVLFS_BANNER
	printk("removing ovl fs module (compiled %s)\n", macro2str(DATE_STAMP));
#endif

	unregister_filesystem(&ovl_fs);
}

#endif
