/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_kern.c
 **
 ** DESCRIPTION:
 **	This file contains functions and data that are needed to implement
 **	functions that otherwise exist in the kernel but are not available
 **	for use (since they are declared static).
 **
 ** NOTES:
 **	- Why copy?  These functions are defined here because their
 **	  equivalents in the kernel (at least the version I am working with)
 **	  are defined static.
 **
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 02/26/1998	ARTHUR N.	Added to source code control.
 ** 02/27/1998	ARTHUR N.	Added the ovlfs_open_inode() and
 **				 ovlfs_close_inode() functions here.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_kern.c 1.3"

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#include <linux/stddef.h>

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/malloc.h>
#include <asm/statfs.h>

#include "ovl_fs.h"



#if OVLFS_NEED_OPEN_INODE

/**
 ** FUNCTION: ovlfs_open_inode
 **
 ** NOTES:
 **     - Copied and modified from linux/fs/open.c do_open(). (in linux source
 **	  tree for kernel source version 2.0.30).
 **
 **     - The inode must already exist.
 **/

int ovlfs_open_inode(struct inode *inode, int flags, struct file *file)
{
    int     error;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_open_inode(%ld, %d)\n", inode->i_ino,
           flags);
#endif

    if (! file)
        return -ENFILE;

    file->f_count = 1;
    file->f_flags = flags;
    file->f_mode = (flags + 1) & O_ACCMODE;

    if (file->f_mode & FMODE_WRITE)
    {
        error = get_write_access(inode);

        if (error)
        {
            file->f_count--;
            return error;
        }
    }

    file->f_inode = inode;
    file->f_pos = 0;
    file->f_reada = 0;
    file->f_op = NULL;

    if (inode->i_op)
        file->f_op = inode->i_op->default_file_ops;

    if (file->f_op && file->f_op->open)
    {
        error = file->f_op->open(inode, file);

        if (error < 0)
        {
            if (file->f_mode & FMODE_WRITE)
                put_write_access(inode);

            file->f_count--;
            return error;
        }

        IMARK(inode);
    }
    else
        IMARK(inode);

    file->f_flags &= ~(O_CREAT | O_EXCL | O_NOCTTY | O_TRUNC);

    return 0;
}



/**
 ** FUNCTION: ovlfs_close_file
 **
 **  PURPOSE: Close a file opened with the ovlfs_open_inode() function.
 **/

void    ovlfs_close_file (struct file *file)
{
    if ( file == (struct file *) NULL )
        return;

    if ( ( file->f_op != NULL ) && ( file->f_op->release != NULL ) )
        file->f_op->release(file->f_inode, file);

    if ( file->f_mode & FMODE_WRITE )
        put_write_access(file->f_inode);

    IPUT(file->f_inode);
}

#endif



#if OVLFS_NEED_GET_SUPER

/**
 ** FUNCTION: ovlfs_get_super
 **
 **  PURPOSE: Obtain the super block for the specified device.
 **
 ** NOTES:
 **     - Copied from the kernel source file, fs/super.c. (in linux source
 **	  tree for kernel source version 2.0.30).
 **/

extern struct super_block super_blocks[NR_SUPER];

struct super_block  *ovlfs_get_super (kdev_t dev)
{
    struct super_block * s;

    if (!dev)
        return NULL;
    s = 0+super_blocks;
    while (s < NR_SUPER+super_blocks)
        if (s->s_dev == dev) {
            wait_on_super(s);
            if (s->s_dev == dev)
                return s;
            s = 0+super_blocks;  /* something changed while waiting, restart */
        } else
            s++;
    return NULL;
}

#endif



#if OVLFS_NEED_WAIT_ON_INODE

/**
 ** FUNCTION: ovlfs_wait_on_inode
 **
 **  PURPOSE: Wait for the inode to become free.
 **
 ** NOTES:
 **	- Copied from fs/inode.c (in linux source tree for kernel source
 **	  version 2.0.30).
 **/

int	ovlfs_wait_on_inode (struct inode *inode)
{
	struct wait_queue	wait = { current, NULL };
	int			ret;

	ret = 0;

	if ( inode->i_lock )
	{
		add_wait_queue(&inode->i_wait, &wait);

			/* Don't call schedule() if the inode is not locked */
			/*  since there may be noone to call wake_up()...   */

		while ( inode->i_lock )
		{
			current->state = TASK_INTERRUPTIBLE;
			schedule();

			if ( current->signal & (~current->blocked) )
			{
				ret = -EINTR;
				break;
			}
		}

		remove_wait_queue(&inode->i_wait, &wait);
		current->state = TASK_RUNNING;
	}

	return	ret;
}

#endif



#if OVLFS_NEED_LOCK_INODE

/**
 ** FUNCTION: ovlfs_lock_inode
 **
 **  PURPOSE: Lock the given inode.
 **
 ** NOTES:
 **	- This is a replacement for the kernel function lock_inode() within
 **	  fs/inode.c
 **	- After calling this function, the program MUST call the
 **	  ovlfs_unlock_inode function, or do its work (set i_lock = 0 and call
 **	  wake_up()).
 **/

int	ovlfs_lock_inode (struct inode *inode)
{
	int	ret;

		/* Only wait if the inode is already locked */

	if ( inode->i_lock )
	{
		ret = ovlfs_wait_on_inode(inode);

		if ( ret == 0 )
			inode->i_lock = 1;
	}
	else
	{
		inode->i_lock = 1;
		ret = 0;
	}

	return	ret;
}

#endif



#if OVLFS_NEED_UNLOCK_INODE

/**
 ** FUNCTION: ovlfs_unlock_inode
 **
 **  PURPOSE: Unlock the given inode, which was locked by the current process
 **           by the ovlfs_lock_inode function (or equivalent).
 **
 ** NOTES:
 **	- This is a replacement for the kernel function unlock_inode() within
 **	  fs/inode.c
 **/

void	ovlfs_unlock_inode (struct inode *inode)
{
	inode->i_lock = 0;
	wake_up(&inode->i_wait);
}

#endif


#if OVLFS_NEED_DEF_PERM

/**
 ** FUNCTION: ovlfs_ck_def_perm
 **
 **  PURPOSE: Perform the default permission checking for the specified inode.
 **
 ** NOTES:
 **	- Copied from fs/namei.c (in linux source tree for kernel source
 **	  version 2.0.30).
 **/

int	ovlfs_ck_def_perm (struct inode *inode, int mask)
{
	int	mode;

	mode = inode->i_mode;

	if ((mask & S_IWOTH) && IS_RDONLY(inode) &&
	    (S_ISREG(mode) || S_ISDIR(mode) || S_ISLNK(mode)))
		return -EROFS; /* Nobody gets write access to a read-only fs */
	else if ((mask & S_IWOTH) && IS_IMMUTABLE(inode))
		return -EACCES; /* Nobody gets write access to an immutable file */
	else if (current->fsuid == inode->i_uid)
		mode >>= 6;
	else if (in_group_p(inode->i_gid))
		mode >>= 3;
	if (((mode & mask & 0007) == mask) || fsuser())
		return 0;
	return -EACCES;
}

#endif
