/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_file.c
 **
 ** DESCRIPTION: This file contains the source code of the file operations
 **              for the overlay filesystem.
 **
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 09/27/1997	art            	Added to source code control.
 ** 02/24/1998	ARTHUR N.	Update the size of the pseudo-fs' inode when
 **				 write() extends the file's size.  Also, lock
 **				 inode's in the overlay fs before updating
 **				 their contents.
 ** 02/27/1998	ARTHUR N.	Added the select(), ioctl(), fasync(), and
 **				 mmap() file operations and moved the
 **				 ovlfs_open_inode() and ovlfs_close_inode()
 **				 functions into ovl_kern.c.
 ** 02/27/1998	ARTHUR N.	Corrected the #if ... compiler directive logic
 **				 used to remove support for file pointers that
 **				 are missing their private data.
 ** 02/28/1998	ARTHUR N.	Removed all support for file pointers that
 **				 are missing their private data.  Only files
 **				 opened via ovlfs_open() may be used here.
 ** 03/01/1998	ARTHUR N.	Allocate GFP_KERNEL priority memory only.
 ** 03/03/1998	ARTHUR N.	Added support for the noxmnt option.
 ** 03/03/1998	ARTHUR N.	Use the file operation's write instead of the
 **				 inode's write.
 ** 03/05/1998	ARTHUR N.	Added malloc-checking support.
 ** 03/06/1998	ARTHUR N.	Added the PROTECT_MMAP_WRITE compile flag.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 ** 03/09/1998	ARTHUR N.	Update the ctime, mtime, and atime attributes
 **				 of the inodes when read and written.
 ** 03/10/1998	ARTHUR N.	Improved comments.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_file.c 1.14"

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#include <linux/stddef.h>

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <asm/statfs.h>
#include <asm/segment.h>

#include "ovl_fs.h"



/**
 ** FUNCTION: ovlfs_create_stg_inode
 **
 **  PURPOSE: Given an ovlfs directory inode, the name of a new file, and
 **           an inode to use as a reference, create the file in the ovlfs
 **           directory
 **/

static int  ovlfs_create_stg_inode (struct inode *inode, const char *name,
                                    int len, struct inode *ref,
                                    struct inode **result)
{
    ovlfs_inode_t   *dir_ino;
    ovlfs_inode_t   *file_ino;
    int             ret;

    if ( ! is_ovlfs_inode(inode) )
        return  -ENOTDIR;


        /* Obtain the directory's storage information */

    dir_ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

    if ( dir_ino == (ovlfs_inode_t *) NULL )
        ret = -ENOTDIR;
    else
    {
        ovlfs_dir_info_t    *dent;

            /* See if the inode entry exists for the file in the given dir. */

        dent = ovlfs_ino_find_dirent(dir_ino, name, len);

        if ( dent == (ovlfs_dir_info_t *) NULL )
        {
                /* The entry does not exist; create it */

            if ( ( inode->i_op != NULL ) && ( inode->i_op->create != NULL ) )
            {
                IMARK(inode);

                down(&inode->i_sem);
                ret = inode->i_op->create(inode, name, len, ref->i_mode,
                                          result);
                up(&inode->i_sem);
            }
            else
                ret = -ENOTDIR;
        }
        else
        {
                /* The entry exists in the directory, see if the storage */
                /*  inode's information is set and get the inode if so.  */

            file_ino = ovlfs_get_ino(inode->i_sb, dent->ino);

            if ( file_ino == (ovlfs_inode_t *) NULL )
            {
#if KDEBUG
                printk("OVLFS: ovlfs_create_stg_inode: unable to obtain "
                       "inode %ld's storage info\n", dent->ino);
#endif

                ret = -ENOENT;
            }
            else
            {
                if ( file_ino->stg_dev != NODEV )
                {
                    struct super_block  *sb;

                    sb = ovlfs_get_super(file_ino->stg_dev);

                    if ( sb != (struct super_block *) NULL )
                    {
                        result[0] = __iget(sb, file_ino->stg_ino,
                                           ovlfs_sb_xmnt(inode->i_sb));

                        if ( result[0] != INODE_NULL )
                            ret = 0;
                        else
                        {
#if KDEBUG
                            printk("OVLFS: ovlfs_create_stg_inode: unable to "
                                   "obtain inode %ld in stg fs, dev %d\n",
                                   file_ino->stg_ino, file_ino->stg_dev);
#endif
                            ret = -ENOENT;
                        }
                    }
                    else
                    {
#if KDEBUG
                        printk("OVLFS: ovlfs_create_stg_inode: unable to "
                               "obtain super block for device %d\n",
                               file_ino->stg_dev);
#endif

                        ret = -ENOENT;
                    }
                }
                else
                {
                    struct inode    *o_dir;

                        /* Entry exists in directory, but storage inode */
                        /*  does not exist; create it.                  */

                    ret = ovlfs_resolve_stg_inode(inode, &o_dir);

                    if ( ret < 0 )
                        ret = ovlfs_make_hierarchy(inode, &o_dir, 0);

                    if ( ret >= 0 )
                    {
                        if ( ( o_dir->i_op != NULL ) &&
                             ( o_dir->i_op->create != NULL ) )
                        {
                            IMARK(o_dir);
                            down(&o_dir->i_sem);
                            ret = o_dir->i_op->create(o_dir, name, len,
                                                      ref->i_mode, result);
                            up(&o_dir->i_sem);

                            if ( ret >= 0 )
                            {
                                    /* Update the storage information for */
                                    /*  the file to refer to the new stg  */
                                    /*  inode.                            */

                                ret = ovlfs_inode_set_stg(inode->i_sb,
                                                          file_ino,
                                                          result[0]->i_dev,
                                                          result[0]->i_ino);
                            }
                        }
                        else
                            ret = -ENOTDIR;

                        IPUT(o_dir);
                    }
                }
            }
        }
    }

    return  ret;
}



/**
 ** FUNCTION: do_read
 **
 **  PURPOSE: Read up to the specified number of bytes from the given file
 **           into the given buffer.
 **/

static inline int   do_read (struct inode *inode, struct file *file,
                             char *buf, int count, int to_kmem)
{
    int             ret;
    unsigned long   orig_fs;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "do_read(%ld)\n", inode->i_ino);
#endif

    if ( ( file->f_op != NULL ) && ( file->f_op->read != NULL ) )
    {
        if ( to_kmem )
        {
            orig_fs = get_fs();
            set_fs(KERNEL_DS);

                /* Lock the inode before reading from it; this is not done */
                /*  in the VFS even though writes are locked, but I feel   */
                /*  that this is more robust.                              */

            down(&inode->i_sem);
            ret = file->f_op->read(inode, file, buf, count);
            up(&inode->i_sem);
            set_fs(orig_fs);
        }
        else
        {
            down(&inode->i_sem);
            ret = file->f_op->read(inode, file, buf, count);
            up(&inode->i_sem);
        }
    }
    else
        ret = -EINVAL;

    return  ret;
}



/**
 ** FUNCTION: do_write
 **
 **  PURPOSE: Write the specified number of bytes from the given buffer to
 **           the given file.
 **
 ** NOTES:
 **     - The entire buffer must be written; therefore, to be safe, this
 **       function loops until the total byte count written = the specified
 **       count, or an error occurs.
 **/

static inline int   do_write (struct inode *inode, struct file *file,
                              char *buf, int count, int from_kmem)
{
    unsigned int    orig_fs;
    int             total = count;
    int             ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "do_write(%ld)\n", inode->i_ino);
#endif

    if ( ( file->f_op != NULL ) && ( file->f_op->write != NULL ) )
    {
        ret = 0;

        while ( ( count > 0 ) && ( ret >= 0 ) )
        {
            if ( from_kmem )
            {
                orig_fs = get_fs();
                set_fs(KERNEL_DS);

                down(&inode->i_sem);
                ret = file->f_op->write(inode, file, buf, count);
                up(&inode->i_sem);

                set_fs(orig_fs);
            }
            else
            {
                down(&inode->i_sem);
                ret = file->f_op->write(inode, file, buf, count);
                up(&inode->i_sem);
            }

            if ( ret > 0 )
            {
                count -= ret;
                buf += ret;
            }
            else if ( ret == 0 )
            {
#if KDEBUG
                printk("OVLFS: do_write: file operation write returned 0 on "
                       "write %d bytes\n", count);
#endif

                count = 0;  /* We can't keep trying if nothing is writing */
            }
        }
    }
    else
        ret = -EINVAL;

    if ( ret >= 0 )
        ret = total;

    return  ret;
}



/**
 ** FUNCTION: ovlfs_copy_file
 **
 **  PURPOSE: Copy the base inode file given into the storage filesystem in
 **           the directory given with the specified name.
 **
 ** NOTES:
 **     - I wish I could simply use sys_open(), sys_read(), and sys_write(),
 **       but I do not know the path from the current working directory to
 **       the file to copy.
 **/

int ovlfs_copy_file (struct inode *dir, const char *fname, int len,
                     struct inode *base_inode, struct inode **result)
{
    struct inode        *new_inode;
    struct file         out_file;
    struct file         in_file;
    char                buffer[1024];   /* 1K buffer is a good size */
    int                 count;
    int                 ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_copy_file(%ld, %ld)\n",
           ovlfs_inode->i_ino, base_inode->i_ino);
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( dir == INODE_NULL ) || ( base_inode == INODE_NULL ) ||
         ( dir->u.generic_ip == NULL ) || ( ! is_ovlfs_inode(dir) ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_copy_file called with invalid arguments"
               "(0x%x, 0x%x)\n", (int) dir, (int) base_inode);
# endif

        return  -EINVAL;
    }
    else
        if ( ! S_ISREG(base_inode->i_mode) )
        {
# if KDEBUG
            printk("OVLFS: ovlfs_copy_file called with non-regular file\n");
# endif
            return  -EINVAL;
        }
#endif


        /* First, create the file in the storage filesystem */

    if ( ( dir->i_op != NULL ) && ( dir->i_op->create != NULL ) )
    {
#if KDEBUG > 1
        printk("OVLFS: ovlfs_copy_file: creating inode named %.*s length %d\n",
               len, fname, len);
#endif

        ret = ovlfs_create_stg_inode(dir, fname, len, base_inode, &new_inode);
    }
    else
    {
#if KDEBUG
        printk("OVLFS: ovlfs_copy_file: ovlfs directory lacking create()\n");
#endif

        ret = -EINVAL;
    }

    if ( ret >= 0 )
    {
            /* Set the ownership of the new inode to the ownership of the */
            /*  inode in the base filesystem.                             */

        ovlfs_chown(new_inode, base_inode->i_uid, base_inode->i_gid);


            /* Now read the original and write the new inode */

        ret = ovlfs_open_inode(new_inode, O_WRONLY, &out_file);

        if ( ret < 0 )
        {
#if KDEBUG
            printk("OVLFS: ovlfs_copy_file: error %d opening new inode for"
                   " writing\n", ret);
#endif
        }
        else
        {
                /* Now open the original for reading */

            ret = ovlfs_open_inode(base_inode, O_RDONLY, &in_file);

            if ( ret < 0 )
            {
#if KDEBUG
                printk("OVLFS: ovlfs_copy_file: error %d opening base inode"
                       " for reading\n", ret);
#endif
            }
            else
            {
                do
                {
                    count = do_read(base_inode, &in_file, buffer,
                                    sizeof(buffer), 1);

                    if ( count > 0 )
                    {
#if KDEBUG > 1
                        printk("OVLFS: ovlfs_copy_file: do_read returned %d",
                               count);
#endif
                        ret = do_write(new_inode, &out_file, buffer, count, 1);

#if KDEBUG > 1
                        printk("OVLFS: ovlfs_copy_file: do_write returned %d",
                               ret);
#endif
                    }
                    else
                    {
#if KDEBUG
# if KDEBUG < 2
                        if ( count < 0 )
# endif
                                printk("OVLFS: ovlfs_copy_file: do_read "
                                       "returned %d", count);
#endif

                        ret = count;
                    }
                } while ( ret > 0 );

                ovlfs_close_file(&in_file);
            }

            ovlfs_close_file(&out_file);
        }

                /* If the copy failed for any reason, remove the created */
                /*  file.                                                */

        if ( ret < 0 )
        {
            IPUT(new_inode);

            IMARK(dir);
            down(&dir->i_sem);
            dir->i_op->unlink(dir, fname, len);
            up(&dir->i_sem);
        }
        else
            result[0] = new_inode;
    }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_inode_copy_file
 **
 **  PURPOSE: Copy the base inode specified into the storage filesystem in
 **           the directory that the ovlfs_inode resides in.
 **
 ** NOTES:
 **     - I wish I could simply use sys_open(), sys_read(), and sys_write(),
 **       but I do not know the path from the current working directory to
 **       the file to copy.
 **/

int ovlfs_inode_copy_file (struct inode *ovlfs_inode, struct inode *base_inode,
                           struct inode **result)
{
    ovlfs_inode_t       *ino;
    ovlfs_inode_info_t  *i_info;
    struct inode        *ovlfs_dir;
    char                *fname;
    int                 len;
    int                 ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_inode_copy_file(%ld, %ld)\n",
           ovlfs_inode->i_ino, base_inode->i_ino);
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( ovlfs_inode == INODE_NULL ) || ( base_inode == INODE_NULL ) ||
         ( ovlfs_inode->u.generic_ip == NULL ) ||
         ( ! is_ovlfs_inode(ovlfs_inode) ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_inode_copy_file called with invalid arguments"
               "(0x%x, 0x%x)\n", (int) ovlfs_inode, (int) base_inode);
# endif

        return  -EINVAL;
    }
    else
        if ( ! S_ISREG(base_inode->i_mode) )
        {
# if KDEBUG
            printk("OVLFS: ovlfs_inode_copy_file called with non-regular "
                   "file\n");
# endif
            return  -EINVAL;
        }
#endif


    i_info = (ovlfs_inode_info_t *) ovlfs_inode->u.generic_ip;

    ino = ovlfs_get_ino(ovlfs_inode->i_sb, ovlfs_inode->i_ino);

    if ( ino == (ovlfs_inode_t *) NULL )
    {
#ifdef KDEBUG
        printk("OVLFS: ovlfs_inode_copy_file: unable to obtain ovlfs storage "
               " inode for sb dev %d, inode %ld\n", ovlfs_inode->i_sb->s_dev,
               ovlfs_inode->i_ino);
#endif

        return  -EINVAL;
    }
    else
    {
        fname = ino->name;
        len = ino->len;
    }

    if ( ( fname == (char *) NULL ) || ( fname[0] == '\0' ) || ( len <= 0 ) )
    {
#ifdef KDEBUG
        printk("OVLFS: ovlfs_inode_copy_file: ovlfs inode's file name "
               "missing\n");
#endif

        return  -EINVAL;
    }


        /* Obtain the parent directory of the ovlfs inode */

    ovlfs_dir = ovlfs_get_parent_dir(ovlfs_inode);

    if ( ovlfs_dir == INODE_NULL )
    {
#ifdef KDEBUG
        printk("OVLFS: ovlfs_inode_copy_file: ovlfs inode lacking directory"
               " reference\n");
#endif

        return  -EINVAL;
    }

    ret = ovlfs_copy_file(ovlfs_dir, fname, len, base_inode, result);

    if ( ret >= 0 )
    {
            /* Now set the storage inode for the given inode to */
            /*  the new inode.                                  */

        IMARK(result[0]);
        i_info->overlay_inode = result[0];

        ovlfs_inode_set_stg(ovlfs_inode->i_sb, ino, result[0]->i_dev,
                            result[0]->i_ino);
    }

    IPUT(ovlfs_dir);

    return  ret;
}



/**
 ** FUNCTION: default_lseek
 **
 **  PURPOSE: Perform the default lseek behavior when a file/inode does not
 **           implement the lseek entry point.
 **/

static inline int   default_lseek (struct file *file, off_t offset,
                                   int from_where)
{
    off_t   tmp = 0;
    int     ret = 0;

    switch (from_where) {
        case 0:
            tmp = offset;
            break;
        case 1:
            tmp = file->f_pos + offset;
            break;
        case 2:
            tmp = file->f_inode->i_size + offset;
            break;
        default:
            ret = -EINVAL;
            break;
    }

    if ( (tmp < 0) && ( ret >= 0 ) )
        ret = -EINVAL;
    else if (tmp != file->f_pos) {
        file->f_pos = tmp;
        file->f_reada = 0;
        file->f_version = ++event;
    }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_lseek
 **/

static int  ovlfs_lseek (struct inode *inode, struct file *file, off_t offset,
                         int from_where)
{
    int ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_lseek(%ld, 0x%x, %d, %d)\n",
           inode->i_ino, (int) file, (int) offset, from_where);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this is also */
        /*  useful for preventing problems due to bad programming ;).   */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_lseek called with non-ovlfs inode!\n");
# endif
        return  -EINVAL;
    }

#endif

    if ( file->private_data == NULL )
        return  -EINVAL;
    else
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->lseek != NULL ) )
        {
            ret = filp->f_op->lseek(filp->f_inode, filp, offset, from_where);
        }
        else
            ret = default_lseek(filp, offset, from_where);

        file->f_pos = filp->f_pos;
    }

    if ( ret >= 0 )
        ret = file->f_pos;

    return  ret;
}



/**
 ** FUNCTION: ovlfs_read
 **
 **  PURPOSE: This is the overlay filesystem's read() file operation.
 **/

int ovlfs_read (struct inode *inode, struct file *file, char *buf, int count)
{
    ovlfs_inode_t   *ino;
    _uint           max_size;
    int             ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_read(%ld, 0x%x, 0x%x, %d)\n",
           inode->i_ino, (int) file, (int) buf, count);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this */
        /*  is also useful for preventing problems due to bad   */
        /*  programming ;).                                     */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_read called with non-ovlfs inode!\n");
# endif
        return  -EINVAL;
    }

#endif

    ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

        /* Don't "believe" the size of a pipe */

    if ( ( ino != (ovlfs_inode_t *) NULL ) && ( ! S_ISFIFO(ino->mode) ) )
        max_size = ino->size;
    else
        max_size = (_uint) -1;


        /* Obtain the file information added by the open() call */

    if ( file->private_data == NULL )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_read(): file given has null private data\n");
#endif

        ret = -EINVAL;
    }
    else
    {
        struct file     *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

            /* Make sure the file information is valid */

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->read != NULL ) )
        {
                /* Call the read function on the real inode */

            down(&filp->f_inode->i_sem);
            ret = filp->f_op->read(filp->f_inode, filp, buf, count);
            up(&filp->f_inode->i_sem);

            if ( ( ret >= 0 ) && ( ! IS_RDONLY(inode) ) )
            {
                inode->i_ctime = inode->i_atime = CURRENT_TIME;
                inode->i_dirt = 1;
            }

            file->f_pos = filp->f_pos;
        }
        else
            ret = -EINVAL;
    }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_write
 **
 ** NOTES:
 **     - Only regular files are copied into the storage filesystem; other
 **       types of files that reside in the base filesystem may be written
 **       normally.
 **/

static int  ovlfs_write_op (struct inode *, struct inode *, ovlfs_inode_t *,
                            struct file *, const char *, int);

int ovlfs_write (struct inode *inode, struct file *file, const char *buf,
                 int count)
{
    struct inode        *o_inode;
    ovlfs_inode_t       *ino;
    int                 ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_write(%ld, 0x%x, 0x%x, %d)\n",
           inode->i_ino, (int) file, (int) buf, count);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this is also */
        /*  useful for preventing problems due to bad programming ;).   */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_write called with non-ovlfs inode!\n");
# endif
        return  -EINVAL;
    }

#endif

    ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);


    if ( file->private_data == NULL )
        ret = -EBADF;
    else
    {
        ovlfs_file_info_t   *f_info;
        struct file         *filp;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

            /* If the file was opened in the base filesystem, and it is a */
            /*  regular file, then it must be closed and copied, and the  */
            /*  result of the copy must be opened.                        */

        if ( ( f_info->is_base ) && ( S_ISREG(filp->f_inode->i_mode) ) )
        {
            ret = ovlfs_inode_copy_file(inode, f_info->file.f_inode, &o_inode);

                    /* If the file copied alright, try the update on the */
                    /*  new file.                                        */

            if ( ret >= 0 )
            {
                    /* Release the old inode, if it has a release op */

                if ( ( filp->f_op != NULL ) && ( filp->f_op->release != NULL ) )
                    filp->f_op->release(filp->f_inode, filp);

                IPUT(filp->f_inode);

                filp->f_inode = o_inode;
                f_info->is_base = 0;
                filp->f_op = o_inode->i_op->default_file_ops;

                if ( ( filp->f_op != NULL ) && ( filp->f_op->open != NULL ) )
                    ret = filp->f_op->open(o_inode, filp);
            }
        }
        else
            ret = 0;

        if ( ret >= 0 )
        {
            ret = ovlfs_write_op(inode, filp->f_inode, ino, filp, buf, count);

            if ( ret >= 0 )
            {
                inode->i_ctime = inode->i_mtime = CURRENT_TIME;
                inode->i_dirt = 1;
            }
        }


        file->f_pos = filp->f_pos;
    }

    return  ret;
}


    /* only write this code once: */

static int  ovlfs_write_op (struct inode *inode, struct inode *o_inode,
                            ovlfs_inode_t *ino, struct file *file,
                            const char *buf, int count)
{
    int ret;

    if ( ( o_inode != INODE_NULL ) && ( file->f_op != NULL ) &&
         ( file->f_op->write != NULL ) )
    {
        down(&o_inode->i_sem);
        ret = file->f_op->write(o_inode, file, buf, count);
        up(&o_inode->i_sem);

            /* Update the size of the file in the existing psuedo fs inode */
            /*  and the storage information for the inode if the file has  */
            /*  grown.                                                     */

        if ( file->f_pos > inode->i_size )
        {
            inode->i_size = file->f_pos;
            inode->i_dirt = 1;

            if ( ino != (ovlfs_inode_t *) NULL )
                ino->size = file->f_pos;
        }
    }
    else
    {
# if KDEBUG > 1
        printk("OVLFS: ovlfs_do_write: inode has no write() function; ino = "
               "%ld\n", o_inode->i_ino);
# endif

        ret = -EINVAL;
    }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_select
 **
 **  PURPOSE: This is the overlay filesystem's select() file operation.
 **/

static int  ovlfs_select (struct inode *inode, struct file *file, int flags,
                          select_table *tbl)
{
    int ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_select(%ld, 0x%x, %d, 0x%x)\n",
           inode->i_ino, (int) file, (int) flags, (int) tbl);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this is also */
        /*  useful for preventing problems due to bad programming ;).   */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_select called with non-ovlfs inode!\n");
# endif
        return  -EINVAL;
    }

#endif

    if ( file->private_data == NULL )
        ret = 0;
    else
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->select != NULL ) )
        {
            ret = filp->f_op->select(filp->f_inode, filp, flags, tbl);
        }
        else
            ret = 0;
    }


        /* On error, just return 0 becuase the return value is taken as */
        /*  a boolean.                                                  */

    if ( ret < 0 )
        ret = 0;

    return  ret;
}



/**
 ** FUNCTION: ovlfs_ioctl
 **
 **  PURPOSE: This is the overlay filesystem's ioctl() file operation.
 **/

static int  ovlfs_ioctl (struct inode *inode, struct file *file,
                         unsigned int cmd, unsigned long arg)
{
    int ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_ioctl(%ld, 0x%x, %ud, %ld)\n",
           inode->i_ino, (int) file, cmd, arg);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this is also */
        /*  useful for preventing problems due to bad programming ;).   */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_ioctl called with non-ovlfs inode!\n");
# endif
        return  -EINVAL;
    }

#endif

    if ( file->private_data == NULL )
        ret = -ENOTTY;
    else
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->ioctl != NULL ) )
        {
            ret = filp->f_op->ioctl(filp->f_inode, filp, cmd, arg);
        }
        else
            ret = -ENOTTY;
    }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_fsync
 **/

int ovlfs_fsync (struct inode *inode, struct file *file)
{
    int ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_fsync(%ld, 0x%x)\n", inode->i_ino,
           (int) file);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this */
        /*  is also useful for preventing problems due to bad   */
        /*  programming ;).                                     */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_fsync called with non-ovlfs inode!\n");
# endif
        return  -EINVAL;
    }
#endif

    if ( file->private_data == NULL )
        ret = -EINVAL;
    else
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->fsync != NULL ) )
            ret = filp->f_op->fsync(filp->f_inode, filp);
        else
            ret = -EINVAL;
    }

    return  ret;
}



/**
 ** FUNCTION: dup_file_info
 **
 **  PURPOSE: Allocate memory to store a temporary file structure and copy
 **           the given file structure into the result.
 **/

static ovlfs_file_info_t    *dup_file_info (ovlfs_file_info_t *f_info)
{
    ovlfs_file_info_t   *result;

    if ( f_info == (ovlfs_file_info_t *) NULL )
        return  (ovlfs_file_info_t *) NULL;


    result = (ovlfs_file_info_t *) MALLOC(sizeof(ovlfs_file_info_t));

    if ( result != (ovlfs_file_info_t *) NULL )
    {
        result[0] = f_info[0];
    }

    return  result;
}

static inline void  free_file_info (ovlfs_file_info_t *f_info)
{
    FREE(f_info);
}



/**
 ** FUNCTION: ovlfs_release
 **/

void    ovlfs_release (struct inode *inode, struct file *file)
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_release(%ld, 0x%x)\n", inode->i_ino,
           (int) file);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_release called with non-ovlfs inode!\n");
# endif

        return;
    }
#endif

    if ( file->private_data == NULL )
        ;
    else
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->release != NULL ) )
            filp->f_op->release(filp->f_inode, filp);

        IPUT(filp->f_inode);
        free_file_info(f_info);
    }
}



/**
 ** FUNCTION: ovlfs_mmap
 **
 **  PURPOSE: This is the overlay filesystem's mmap() file operation.
 **/

static int  ovlfs_mmap (struct inode *inode, struct file *file,
                        struct vm_area_struct *vm_str)
{
    int ret;

    if ( ! is_ovlfs_inode(inode) )
        return  -EINVAL;

    if ( file->private_data != NULL )
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

            /* Make sure the referenced file's information is valid */

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->mmap != NULL ) )
        {
#if PROTECT_MMAP_WRITE
            if ( vm_str != NULL )
            {
                    /* Do not allow shared write access to a file in the */
                    /*  base filesystem.                                 */

                if ( ( f_info->is_base ) && ( vm_str->vm_flags & VM_WRITE ) &&
                     ( vm_str->vm_flags & VM_SHARED ) )
                    ret = -EACCES;
                else
                    ret = filp->f_op->mmap(filp->f_inode, filp, vm_str);
            }
            else
                ret = -EINVAL;
#else
            ret = filp->f_op->mmap(filp->f_inode, filp, vm_str);
#endif
        }
        else
            ret = -ENODEV;
    }
    else
        ret = -EINVAL;

    return ret;
}



/**
 ** FUNCTION: ovlfs_fasync
 **
 **  PURPOSE: This is the overlay filesystem's fasync() file operation.
 **/

static int  ovlfs_fasync (struct inode *inode, struct file *file, int flags)
{
    int ret;

    if ( ! is_ovlfs_inode(inode) )
        return  -EINVAL;

    if ( file->private_data != NULL )
    {
        struct file         *filp;
        ovlfs_file_info_t   *f_info;

        f_info = (ovlfs_file_info_t *) file->private_data;
        filp = file_info_filp(f_info);

        if ( ( filp->f_inode != INODE_NULL ) && ( filp->f_op != NULL ) &&
             ( filp->f_op->fasync != NULL ) )
            ret = filp->f_op->fasync(filp->f_inode, filp, flags);
        else
            ret = 0;
    }
    else
        ret = 0;

    return ret;
}



/**
 ** FUNCTION: ovlfs_open
 **/

static int  ovlfs_open (struct inode *inode, struct file *file)
{
    struct inode        *o_inode;
    ovlfs_file_info_t   tmp;
    int                 ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_open(%ld, 0x%x)\n", inode->i_ino,
           (int) file);
#endif

#if ! FASTER_AND_MORE_DANGEROUS

        /* make certain the given inode is an ovlfs inode; this */
        /*  is also useful for preventing problems due to bad   */
        /*  programming ;).                                     */

    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_open called with non-ovlfs inode!\n");
# endif

        return  -EINVAL;
    }

#endif

    file->private_data = NULL;

        /* Look for the inode to "open" in the storage fs first; if it */
        /*  does not exist there, look for it in the base filesystem.  */

    ret = ovlfs_resolve_stg_inode(inode, &o_inode);

    if ( ret < 0 )
    {
        tmp.is_base = 1;
        ret = ovlfs_resolve_base_inode(inode, &o_inode);
    }
    else
        tmp.is_base = 0;

    if ( ret >= 0 )
    {
        tmp.file         = file[0];
        tmp.file.f_next  = NULL;
        tmp.file.f_prev  = NULL;
        tmp.file.f_inode = o_inode;

        if ( o_inode->i_op != NULL )
        {
            tmp.file.f_op = o_inode->i_op->default_file_ops;

                /* Call the real inode's open if it has one */

            if ( ( tmp.file.f_op != NULL ) &&
                 ( tmp.file.f_op->open != NULL ) )
            {
                ret = tmp.file.f_op->open(o_inode, &(tmp.file));
            }
        }
        else
            tmp.file.f_op = NULL;

        if ( ret >= 0 )
        {
                /* Copy the temporary file's info and store a reference */
                /*  to it in the given file's private data.             */

            file->private_data = dup_file_info(&tmp);

            if ( file->private_data == NULL )
            {
                ret = -ENOMEM;

                if ( tmp.file.f_op->release != NULL )
                    tmp.file.f_op->release(o_inode, &(tmp.file));
            }
        }

            /* If an error was encountered, release the inode */

        if ( ret < 0 )
            IPUT(o_inode);
    }

    return  ret;
}



struct file_operations  ovlfs_file_ops = {
    ovlfs_lseek,    /* (int (*)(struct inode *, struct file *, off_t, int)) NULL, */
    ovlfs_read,     /* (int (*)(struct inode *, struct file *, char *, int)) NULL, */
    ovlfs_write,    /* (int (*)(struct inode *, struct file *, const char *, int)) NULL, */
    (int (*)(struct inode *, struct file *, void *, filldir_t)) NULL,
    ovlfs_select,   /* (int (*)(struct inode *, struct file *, int, select_table *)) NULL, */
    ovlfs_ioctl,    /* (int (*)(struct inode *, struct file *, unsigned int, unsigned long)) NULL, */
    ovlfs_mmap,     /* (int (*)(struct inode *, struct file *, struct vm_area_struct *)) NULL, */
    ovlfs_open,     /* (int (*)(struct inode *, struct file *)) NULL, */
    ovlfs_release,  /* (void (*)(struct inode *, struct file *)) NULL, */
    ovlfs_fsync,    /* (int (*)(struct inode *, struct file *)) NULL, */
    ovlfs_fasync,   /* (int (*)(struct inode *, struct file *, int)) NULL, */
    (int (*)(kdev_t)) NULL,   /* How could we make use of this? need a file */
    (int (*)(kdev_t)) NULL    /* How could we make use of this? need a file */
} ;
